// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef OBJECT_INC
#define OBJECT_INC

#include "general.h"
#include "random.h"

namespace Torch {

//-----

typedef struct Option_
{
    char *name;
    char *help;
    int size;
    void *ptr;
    bool is_allowed_after_init;
} Option;

//-----

/** Provides a useful interface for managing options.
    Class inherited by almost all classes in Torch.
    
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class Object
{
  public:

    int n_options;
    Option *options;
    bool is_already_initialized;

    ///
    Object();
    
    //-----

    /** Initialize the object.
        All objects *MUST* call the #init()#
        function of their parents.
    */
    virtual void init();

    /** Add the option #name#.
        This one has a pointer on #ptr# and has the size #size#.
        You can provide an help in #help#.
        (Note that if #help# is empty, its an empty string and not NULL).
        If the option can be changed at any time, you can set
        #is_allowed_after_init# to #true#.
    */
    void addOption(const char *name, int size, void *ptr, const char *help="", bool is_allowed_after_init=false);

    /** Several wrappers of #addOption()# for
        Int, Real and Bool options.
        Be carefull: there is no type checking.
    */
    void addIOption(const char *name, int *ptr,  int init_value,  const char *help="", bool is_allowed_after_init=false);

    ///
    void addROption(const char *name, real *ptr, real init_value, const char *help="", bool is_allowed_after_init=false);

    ///
    void addBOption(const char *name, bool *ptr, bool init_value, const char *help="", bool is_allowed_after_init=false);

    /// Set the option #name# to the value contained at #ptr#.
    void setOption(const char *name, void *ptr);

    /** Several wrappers of #setOption()# for
        Int, Real and Bool options.
        Be carefull: there is no type checking.
    */
    void setIOption(const char *name, int option);

    ///
    void setROption(const char *name, real option);

    ///
    void setBOption(const char *name, bool option);

    /// Load the object from a file pointer (\emph{not the options})
    virtual void loadFILE(FILE *file);

    /// Save the object to a file pointer (\emph{not the options})
    virtual void saveFILE(FILE *file);

    /// Load the machine from a file name (\emph{not the options})
    void load(const char *filename);

    /// Save the machine to a file name (\emph{not the options})
    void save(const char *filename);

    //-----

    virtual ~Object();
};


}

#endif
