// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef IO_TORCH_INC
#define IO_TORCH_INC

#include "general.h"
#include "Object.h"

namespace Torch {

//-----

typedef struct BlkMem_
{
    void *internal_alloc;
    void **data;
    void **targets;
    int n_examples;
    bool data_is_internal;
    bool targets_is_internal;
} BlkMem;

//-----

/** Load and save file in the Torch format.
    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
class IOTorch : public Object
{
  public:
    
    BlkMem *blocks;
    int n_blocks;

    ///
    IOTorch();

    int addBlkMem(void *internal_alloc, void **data, void **targets, int n_examples, bool data_is_internal, bool targets_is_internal);
    void destroyBlkMem(int number);

    //-----

    /** Load the file #file#.
        Each example must have #n_inputs# inputs and #n_targets# targets.
        #bin# is #true# if the file is in binary format.
        If #max_load# > 0, load only the first #max_load# examples.
        Returns: \begin{itemize}
          \item #data_# which contains all inputs (the memory allocation is done by this function)
          \item #y_# which contains all targets (the memory allocation is done by this function)
          \item #n_examples# which contains the real number of loaded examples.
        \end{itemize}
    */
    int loadData(const char  *file,  real ***data_, real ***y_, int n_inputs, int n_targets, int *n_examples, bool bin=false, int max_load=-1);

    /// Sparse version of the previous function
    int loadData(const char  *file, sreal ***data_, real ***y_, int n_inputs, int n_targets, int *n_examples, bool bin=false, int max_load=-1);

    /// Similar function, but to save data
    void saveData(const char *file,  real **data, real **y, int l, int n_inputs, int n_targets, bool bin=false, int max_save=-1);

    /// Similar function, but to save sparse data
    void saveData(const char *file, sreal **data, real **y, int l, int n_inputs, int n_targets, bool bin=false, int max_save=-1);

    //-----

    ~IOTorch();
};


}

#endif
