// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef CONNECTED_MACHINE_INC
#define CONNECTED_MACHINE_INC

#include "GradientMachine.h"

namespace Torch {

/** Easy connections between several #GradientMachine#.
    GradientMachine has "layers" on which you can
    add some #GradientMachine#.

    The inputs of the machine on the first layer will
    be the inputs of the #ConnectedMachine#.

    The outputs of the #ConnectedMachine# are the union
    (in the order of adding) of all machines on the
    last layer.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class ConnectedMachine : public GradientMachine
{
  public:
    real *alpha_buff;
    int alpha_buff_size;
    int current_alpha_offset;

    List ***links;
    List ***alpha_links;

    int n_layers;
    int *n_machines_on_layer;
    GradientMachine ***machines;

    int current_layer;
    int current_machine;

    //-----

    ///
    ConnectedMachine();

    /** add a Full Connected Layer. The #machine# is fully connected
        to the previous layer. If necessary, a layer is added before
        adding the machine.
    */
    void addFCL(GradientMachine *machine);
    
    /// add a #machine# on the current layer
    void addMachine(GradientMachine *machine);

    /** connect the last added machine on #machine#.
        Note that #machine# \emph{must} be in a previous layer.
    */
    void connectOn(GradientMachine *machine);

    /// add a layer (you don't have to call that for the first layer)
    void addLayer();

    //-----

    virtual void reset();
    virtual void allocateMemory();
    virtual void freeMemory();
    virtual void iterInitialize();
    void checkInternalLinks();

    virtual int numberOfParams();
    virtual void forward(List *inputs);
    virtual void backward(List *inputs, real *alpha);

    virtual void loadFILE(FILE *file);
    virtual void saveFILE(FILE *file);

    virtual ~ConnectedMachine();
};


}

#endif

