// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "ClassMeasurer.h"

namespace Torch {

ClassMeasurer::ClassMeasurer(List *inputs_, DataSet *data_, ClassFormat *class_format_, FILE *file_) : Measurer(data_, file_)
{
  inputs = inputs_;
  class_format = class_format_;
  addBOption("confusion matrix", &calc_confusion, false, "compute the confusion matrix");
}

ClassMeasurer::ClassMeasurer(List *inputs_, DataSet *data_, ClassFormat *class_format_, const char *filename) : Measurer(data_, filename)
{
  inputs = inputs_;
  class_format = class_format_;
  addBOption("confusion matrix", &calc_confusion, false, "compute the confusion matrix");
}

void ClassMeasurer::init()
{
  Measurer::init();

  n_classes = class_format->getNumberOfClasses();

  if(calc_confusion)
  {
    measure = xalloc(sizeof(real)*2+sizeof(int)*2*n_classes*n_classes);
    
    real *r_measure = (real *)measure;
    erreur = r_measure;
    erreur_c = r_measure+1;

    confusion   = (int **)xalloc(sizeof(int *)*n_classes);
    confusion_c = (int **)xalloc(sizeof(int *)*n_classes);

    r_measure += 2;
    int *i_measure = (int *)r_measure;
    for(int i = 0; i < n_classes; i++, i_measure += n_classes)
      confusion[i]   = i_measure;

    for(int i = 0; i < n_classes; i++, i_measure += n_classes)
      confusion_c[i] = i_measure;
  }
  else
  {
    measure = xalloc(sizeof(real)*2);
    
    real *r_measure = (real *)measure;
    erreur = r_measure;
    erreur_c = r_measure+1;

    confusion   = NULL;
    confusion_c = NULL;
  }

  reset();
}

void ClassMeasurer::measureEx()
{
  int c_obs = class_format->getOutputClass(inputs);
  int c_des = class_format->getTargetClass(data->targets);

  if(c_obs != c_des)
    *erreur_c += 1;

  if(calc_confusion)
    confusion_c[c_obs][c_des]++;
}

void ClassMeasurer::measureIter()
{
  *erreur = *erreur_c / ((real)data->n_examples);

  fprintf(file, "%g\n", *erreur);
  fflush(file);

  if(calc_confusion)
  {
    for(int i = 0; i < n_classes; i++)
    {
      for(int j = 0; j < n_classes; j++)
        confusion[i][j] = confusion_c[i][j];
    }
  }

  reset();
}

void ClassMeasurer::measureEnd()
{
  if(calc_confusion)
  {
    fprintf(file, "# Labels of classes:\n");
    for(int i = 0; i < n_classes; i++)
    {
      for(int j = 0; j < class_format->getOutputSize(); j++)
        fprintf(file, "%g ", class_format->class_labels[i][j]);
      fprintf(file, "\n");
    }

    fprintf(file, "# Confusion matrix [rows: observed, colums: desired]:\n");
    for(int i = 0; i < n_classes; i++)
    {
      for(int j = 0; j < n_classes; j++)
        fprintf(file, "%d ", confusion[i][j]);
      fprintf(file, "\n");
    }
    fflush(file);
  }  
}

void ClassMeasurer::reset()
{
  *erreur_c = 0;
  if(calc_confusion)
  {
    for(int i = 0; i < n_classes; i++)
    {
      for(int j = 0; j < n_classes; j++)
        confusion_c[i][j] = 0;
    }
  }
}

ClassMeasurer::~ClassMeasurer()
{
  free(measure);
  free(confusion);
  free(confusion_c);
}

}

