// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "Bagging.h"
#include "random.h"

namespace Torch {

Bagging::Bagging(WeightedSumMachine* w_machine_, DataSet* data_) : Trainer(w_machine_,data_)
{
  w_machine = w_machine_;

  int n_examples = data->n_examples;
  n_trainers = w_machine->n_trainers;
  is_selected_examples = (int *)xalloc(n_examples*sizeof(int));
  n_unselected_examples = (int *)xalloc(n_trainers*sizeof(int));
  unselected_examples = (int **)xalloc(n_trainers*sizeof(int*));
  selected_examples = (int **)xalloc(n_trainers*sizeof(int*));
  for (int i=0;i<n_trainers;i++) {
    unselected_examples[i] = (int *)xalloc(n_examples*sizeof(int));
    selected_examples[i] = (int *)xalloc(n_examples*sizeof(int));
  } 

  for(int i = 0; i < n_trainers; i++)
    w_machine->weights[i] = 1./((real)n_trainers);
}

void Bagging::bootstrapData(int* selected, int* is_selected)
{
  int n = data->n_examples;
  for (int j=0;j<n;j++) {
    selected[j] = (int)floor(bounded_uniform(0,n));
    is_selected[selected[j]] = 1;
  }
}

void Bagging::train(List* measurers)
{
  message("Bagging: training");
  w_machine->n_trainers_trained = 0;

  int n = data->n_examples;
  for (int i=0;i<n_trainers;i++) {
    // initialization
    for (int j=0;j<n;j++) {
      is_selected_examples[j]=0;
    }

    // select a bootstrap
    bootstrapData(selected_examples[i],is_selected_examples);
    w_machine->trainers[i]->data->pushSubset(selected_examples[i],n);

    // keep in mind examples not used by trainers[i]
    int k=0;
    for (int j=0;j<n;j++) {
      if (!is_selected_examples[j])
        unselected_examples[i][k++] = j;
    }
    n_unselected_examples[i] = k;
    

    // train the trainer
    w_machine->trainers[i]->machine->reset();
    w_machine->trainers[i]->train(w_machine->trainers_measurers ? w_machine->trainers_measurers[i] : NULL);

    // put back the selected_examples
    w_machine->trainers[i]->data->popSubset();

    w_machine->n_trainers_trained = i+1;

    // if measurers is given, call the test method by fooling it
    // with the number of trainers
    if (measurers)
      test(measurers);
  }
}

Bagging::~Bagging()
{
  free(is_selected_examples);
  free(n_unselected_examples);
  for (int i=0;i<n_trainers;i++) {
    free(unselected_examples[i]);
    free(selected_examples[i]);
  }
  free(selected_examples);
  free(unselected_examples);
}

}

