/* dvi2esppage -  DVI => EPSON ESC/Page printer
 * by Hirotsugu Kakugawa
 *
 *  26 Feb 1997  First implementation.
 *  28 Feb 1997  Added offset, signal trap features. 
 *   2 Mar 1997  Added bitmap compression feature.
 *   3 Mar 1997  Added dynamic selection of bitmap compression feature.
 *   5 Mar 1997  Upgraded for DVIlib 2.3
 *   3 Jun 1997  Added vflibcap parameterlization feature for vflibcap.
 */
/*
 * Copyright (C) 1997-1998  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <varargs.h>
#endif
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_SIGNAL_H
#  include <signal.h>
#endif
#include "libdvi29.h"
#include "defs.h"
#include "cf-dvi2escpage.h"
#include "compiled.h"
#include "../params.h"

#define PR1(s)     fprintf(stderr, s);
#define PR2(s1,s2) fprintf(stderr, s1,s2);

int           param_init_mode      = INIT_MODE;
int           param_dpi            = 0;
char          *param_kpathsea_mode = NULL;
char          *param_vflibcap      = DEFAULT_VFLIBCAP;
char          *param_device        = NULL;
char          *param_dvi_file      = NULL;
int           param_paper_id       = 0;  /* casste size */
int           param_debug          = 0;
int           param_debug_slow     = 2;
int     interrupted;
char    temp_file[256];
double  offset_x_inch;
double  offset_y_inch;
int     offset_x;
int     offset_y;
int           lbp_compress_mode;
unsigned char lbp_buffer[1024*16];
int           lbp_buffer_ptr = 0;

struct s_paper {
  char  *name;
  int   id;
};
struct s_paper  paper_list[] = {
  {"A3", 13},   {"A4", 14},   {"A5", 15},   {"A6", 16},   
  {"a3", 13},   {"a4", 14},   {"a5", 15},   {"a6", 16},   
  {"B4", 24},   {"B5", 25},   {"LETTER", 30},   {"LEGAL", 32},
  {"b4", 24},   {"b5", 25},   {"letter", 30},   {"legal", 32},
  {NULL, -1}
};
#define ESC  (char)0x1b
#define GS   (char)0x1d

#define COMPRESS_YES    0
#define COMPRESS_NO     1

DVI_DEVICE create_dev(void);
void  DEV_put_bitmap(DVI_DEVICE,DVI,DVI_BITMAP,int,long,long,long,long);
int   send_bitmap(DVI_DEVICE,DVI_BITMAP,int,int);
void  DEV_put_rectangle(DVI_DEVICE,DVI,long,long,long,long);
int   DEV_poll(DVI_DEVICE,DVI,int);
void  DEV_message(DVI_DEVICE,DVI,char*,...);

void  parse_args(int,char**);
void  usage(void);
void  print_defaults(void);
void  clean_up(int);
void  stream2file(FILE*);
void  lbp_set_param(DVI_DEVICE);
void  lbp_bitmap_compress_mode(DVI_DEVICE,int);
void  lbp_reset_param(DVI_DEVICE);
void  lbp_form_feed(DVI_DEVICE);
void  lbp_mode_escp2escpage(DVI_DEVICE);
void  lbp_mode_escpage2escp(DVI_DEVICE);
void  lbp_send(DVI_DEVICE,char*,char*);
void  lbp_send_nbytes(DVI_DEVICE,char*,int,char*);
void  lbp_flush(DVI_DEVICE);
#if HAVE_SIGNAL_H
RETSIGTYPE  sig_handler(int);
#endif



int
main(int argc, char **argv)
{
  DVI          dvi     = NULL;
  DVI_DEVICE   dev     = NULL;
  FILE         *fp_out = NULL;
  int      ch, fd;
  char     *dvi_file; 
  FILE     *out;
  int      page;
  char     params[3*256];

  interrupted = 0;
#if HAVE_SIGNAL_H
  if (signal(SIGINT, SIG_IGN) == SIG_DFL)
    signal(SIGINT, sig_handler);
#endif

  parse_args(argc, argv);
  out = NULL;
  if ((dvi_file = param_dvi_file) == NULL){
#if HAVE_MKSTEMP
    sprintf(temp_file, "%s/dvi2escpage-XXXXXX", TEMP_DIR);
    if ((fd = mkstemp(temp_file)) >= 0)
      out = fdopen(fd, "w");
#else
    sprintf(temp_file, "%s/dvi2escpage-XXXXXX", TEMP_DIR);
    mktemp(temp_file);
    out = fopen(temp_file, "w");
#endif
    if (out == NULL){
      fprintf(stderr, "%s: Can't create temporary file\n", PROGRAM);
      clean_up(1);
    }
    while ((ch = getc(stdin)) != EOF)
      putc(ch, out);
    fclose(out);
    dvi_file = temp_file;
  }
  if (access(dvi_file, R_OK) < 0){
    fprintf(stderr, "%s: Can't read DVI file: %s\n", PROGRAM, dvi_file);
    clean_up(1);
  }

  if (param_device == NULL){
    fp_out = stdout;
  } else {
    if ((fp_out = fopen(param_device, "w")) == NULL){
      fprintf(stderr, "%s: Can't open output device: %s\n", 
	      PROGRAM, param_device);
      clean_up(1);
    }
  }

  if ((param_vflibcap != NULL) 
      && (access(param_vflibcap, R_OK) < 0)){
    fprintf(stderr, "%s: Can't read vflibcap file: %s\n",
	    PROGRAM, param_vflibcap);
    clean_up(1);
  }

  sprintf(params, "%s=%d, %s=%s, %s=%s", 
	  PARAM_NAME_DPI,  param_dpi,
	  PARAM_NAME_MODE, param_kpathsea_mode,
	  PARAM_NAME_PROG, "dvi2escpage");

  if (DVI_INIT(param_vflibcap, params) < 0){
    fprintf(stderr, "%s: Can't initialize DVI interpreter\n", PROGRAM);
    clean_up(1);
  }

  if ((dev = create_dev()) == NULL){
    fprintf(stderr, "%s: Can't initialize device\n", PROGRAM);
    clean_up(1);
  }

  if (interrupted == 1)
    clean_up(1);

  if ((dvi = DVI_CREATE(dev, dvi_file, NULL)) == NULL){
    fprintf(stderr, "%s: Can't open a DVI file: %s\n", PROGRAM, dvi_file);
    clean_up(1);
  }

  if (DVI_OPEN_FONT(dvi, dev) < 0){
    fprintf(stderr, "%s: Cannot find all fonts\n", PROGRAM);
    clean_up(1);
  }

  offset_x = offset_x_inch * param_dpi;
  offset_y = offset_y_inch * param_dpi;
  dev->private2 = fp_out;

  if (interrupted == 1)
    clean_up(1);

  if (param_init_mode == INIT_MODE_ESCP)
    lbp_mode_escp2escpage(dev);
  lbp_reset_param(dev);
  lbp_set_param(dev);
  for (page = 1; page <= dvi->pages; page++){
    if (DVI_DRAW_PAGE(dvi, dev, page, 1.0) < 0)
      break;
    lbp_form_feed(dev);
  }
  lbp_flush(dev);
  lbp_reset_param(dev);
  if (param_init_mode == INIT_MODE_ESCP)
    lbp_mode_escpage2escp(dev);

  clean_up(0);

  return 0;
}

void 
clean_up(int exit_val)
{
  if (param_dvi_file == NULL)
    unlink(temp_file);
  exit(exit_val);
}


void
parse_args(int argc, char **argv)
{
  int  dpi = DEFAULT_DPI;
  int  ox, oy, i;

  ox = 0; 
  oy = 0;
  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
      exit(0);
    } else if (strcmp(*argv, "-defaults") == 0){
      print_defaults();
      exit(0);
    } else if ((strcmp(*argv, "-v") == 0) && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if (strcmp(*argv, "-600") == 0){
      dpi = 600;
    } else if (strcmp(*argv, "-300") == 0){
      dpi = 300;
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      if ((strcmp(*argv, "300") == 0) || (strcmp(*argv, "600") == 0)){
	dpi = atoi(*argv);
      } else {
	fprintf(stderr, "%s: DPI value must be 300 or 600\n", PROGRAM);
	exit(1);
      }
    } else if ((strcmp(*argv, "-d") == 0) && (argc > 1)){
      argc--;  argv++;
      param_device = *argv;
    } else if ((strcmp(*argv, "-ox") == 0) && (argc > 1)){
      argc--;  argv++;
      offset_x_inch = atof(*argv);
      ox = 1; 
    } else if ((strcmp(*argv, "-oy") == 0) && (argc > 1)){
      argc--;  argv++;
      offset_y_inch = atof(*argv);
      oy = 1;
    } else if (strcmp(*argv, "-esc/p") == 0){
      param_init_mode = INIT_MODE_ESCP;
    } else if (strcmp(*argv, "-debug") == 0){
      param_debug = 1;
    } else if (strcmp(*argv, "-DEBUG") == 0){
      param_debug = 2;
    } else if (strcmp(*argv, "-slow") == 0){
      argc--;  argv++;
      if ((param_debug_slow = atoi(*argv)) < 0)
	param_debug_slow = 0;	
    } else if ((strcmp(*argv, "-f") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dvi_file = *argv;
    } else if (**argv != '-'){
      param_dvi_file = *argv;
    } else {
      for (i = 0; paper_list[i].name != NULL; i++){
	if (strcmp(&argv[0][1], paper_list[i].name) == 0){
	  param_paper_id = paper_list[i].id;
	  break;
	}
      }
      if (paper_list[i].name == NULL){
	fprintf(stderr, "%s: Unknow option: %s\n", PROGRAM, *argv);
	exit(1);
      }
    }
  }

  if (dpi == 300){
    param_dpi = 300;
    if (param_kpathsea_mode == NULL)
      param_kpathsea_mode = DEFAULT_MODE_300;
    if (ox == 0)
      offset_x_inch  = DEFAULT_OFFSET_X_300;
    if (oy == 0)
      offset_y_inch  = DEFAULT_OFFSET_Y_300;
  } else {
    param_dpi = 600;
    if (param_kpathsea_mode == NULL)
      param_kpathsea_mode = DEFAULT_MODE_600;
    if (ox == 0)
      offset_x_inch  = DEFAULT_OFFSET_X_600;
    if (oy == 0)
      offset_y_inch  = DEFAULT_OFFSET_Y_600;
  }
}

DVI_DEVICE  
create_dev(void)
{
  DVI_DEVICE  dev;
  
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->put_rectangle        = DEV_put_rectangle;
  dev->put_bitmap           = DEV_put_bitmap;
  dev->device_polling       = DEV_poll;
  dev->message_warning      = DEV_message;
  dev->message_error        = DEV_message;
  dev->message_fatal        = DEV_message;
  return dev;
}

void  
usage(void)
{
  PR1("dvi2escpage --- A DVI driver for ESC/Page printers\n");
  fprintf(stderr, "version %s of %s on %s\n", 
	  VERSION, COMPILED_DATE, COMPILED_HOST);
  PR1("\n");
  PR1("Usage: dvi2escpage [Options] [DVI-FILE]\n");
  PR1("Options:\n");
  PR2("  -v VFLIBCAP  A vflibcap file (default value is %s)\n", 
      DEFAULT_VFLIBCAP);
  PR1("  -mode MODE   Device mode name for kpathsea\n");
  PR1("  -dpi DPI     Select device resolution (DPI must be 300 or 600)\n");
  PR1("  -300         300 DPI mode (this is the default mode)\n");
  PR1("  -600         600 DPI mode\n");
  PR1("  -ox OFFSET   Horizontal offset (in inch)\n");
  PR1("  -oy OFFSET   Vertical offset (in inch)\n");
  PR1("  -A3, -A3, -A5, -A6, -B4, -B5, -letter, -legal\n");
  PR1("               Paper size selection\n");
  PR1("  -esc/p       Assume initial printer mode is ESC/P.\n");
  PR1("  -d DEV       Printer device file\n");
  PR1("  -defaults    Print defaut values\n");
  PR1("  -h           Print this\n");
  exit(0);
}

void
print_defaults(void)
{
  fprintf(stderr, "%s version %s default values\n", PROGRAM, VERSION);
  fprintf(stderr, "Default device resolution: %d\n", DEFAULT_DPI);
  fprintf(stderr, "Default vflibcap: %s\n", DEFAULT_VFLIBCAP);
  fprintf(stderr, "Initial printer mode: %s\n", 
	  (param_init_mode == INIT_MODE_ESCPAGE) ? "ESC/Page" : "ESC/P");
  fprintf(stderr, "* Defaults for 300 dpi mode (`-300' option)\n");
  fprintf(stderr, "    kpathsea mode name: %s\n", DEFAULT_MODE_300);
  fprintf(stderr, "    offset: (%.2f,%.2f) inch\n", 
	  DEFAULT_OFFSET_X_300, DEFAULT_OFFSET_Y_300);
  fprintf(stderr, "* Defaults for 600 dpi mode (`-600' option)\n");
  fprintf(stderr, "    kpathsea mode name: %s\n", DEFAULT_MODE_600);
  fprintf(stderr, "    offset: (%.2f,%.2f) inch\n", 
	  DEFAULT_OFFSET_X_600, DEFAULT_OFFSET_Y_600);
}

void  
DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
	       int font_id, long key2, long code_point, long x, long y)
{
  char  escseq[256];
  int   bm_length_compress_yes, bm_length_compress_no;

  if (param_debug == 1)
    fprintf(stderr, "Put bitmap\n");

  sprintf(escseq, "%c%ldX", GS, x+offset_x);         /* Set CP x */
  lbp_send(dev, escseq, "CP x");
  sprintf(escseq, "%c%ldY", GS, y+offset_y);         /* Set CP y */
  lbp_send(dev, escseq, "CP y");

  bm_length_compress_yes = send_bitmap(NULL, bm, COMPRESS_YES, -1);
  bm_length_compress_no  = send_bitmap(NULL, bm, COMPRESS_NO,  -1);
  if (lbp_compress_mode == COMPRESS_YES){
    if ((bm_length_compress_no + 5) < bm_length_compress_yes)
      send_bitmap(dev, bm, COMPRESS_NO,  bm_length_compress_no);
    else 
      send_bitmap(dev, bm, COMPRESS_YES, bm_length_compress_yes);
  } else {
    if (bm_length_compress_no > (bm_length_compress_yes + 5))
      send_bitmap(dev, bm, COMPRESS_YES, bm_length_compress_yes);
    else 
      send_bitmap(dev, bm, COMPRESS_NO,  bm_length_compress_no);
  }
}

int
send_bitmap(DVI_DEVICE dev, DVI_BITMAP bm, int compress_mode, int length)
{
  char  escseq[256], msg[256];
  int   yy, wb, bm_len;
  char  cmp[3];
  int   xx, r, cnt, t;

  if ((length >= 0) && (compress_mode != lbp_compress_mode))
    lbp_bitmap_compress_mode(dev, compress_mode);

  if (compress_mode == COMPRESS_YES){
    wb = (bm->width+7)/8;
    sprintf(escseq, "%c%d;%ld;%ld;%dbi{I",           /* Bitmap */
	    GS, length, bm->width, bm->height, 0);
    sprintf(msg, "Bitmap %ldbytes", wb*bm->height);
    if (length >= 0)
      lbp_send(dev, escseq, msg);
    bm_len = 0;
    for (yy = 0, r = 0; yy < bm->height; yy++, r += bm->raster){
      sprintf(msg, "  bitmap body %d/%ld", yy, bm->height);
      xx = r;
      while (xx < (r+wb)){
	if (xx == (r+wb-1)){
	  cmp[0] = cmp[1] = bm->bitmap[xx];
	  cmp[2] = 1;
	  if (length >= 0)
	    lbp_send_nbytes(dev, cmp, 3, NULL);
	  bm_len += 3;
	  xx++;
	} else if (bm->bitmap[xx] != bm->bitmap[xx+1]){
	  if (length >= 0)
	    lbp_send_nbytes(dev, (char *)&bm->bitmap[xx], 1, NULL);
	  bm_len += 1;
	  xx++;
	} else { /* (xx != (r+wb-1)) && (bm->bitmap[xx]==bm->bitmap[xx+1]) */
	  cmp[0] = cmp[1] = bm->bitmap[xx];
	  for (cnt = 2; (xx+cnt) < (r+wb); cnt++){
	    if (bm->bitmap[xx] != bm->bitmap[xx+cnt])
	      break;
	  }
	  xx = xx + cnt;
	  while (cnt > 0){
	    if ((t = cnt) > 255)
	      t = 255;
	    cnt = cnt - t;
	    cmp[2] = t;
	    if (length >= 0)
	      lbp_send_nbytes(dev, cmp, 3, NULL);
	    bm_len += 3;
	  }
	}
      }
    }
  } else {   /* compress_mode == COMPRESS_NO */
    wb = (bm->width+7)/8;
    if (length >= 0){
      sprintf(escseq, "%c%ld;%ld;%ld;%dbi{I",       /* Bitmap */
	      GS, bm->height * wb, bm->width, bm->height, 0);
      sprintf(msg, "Bitmap %ldbytes", wb*bm->height);
      lbp_send(dev, escseq, msg);
      for (yy = 0, r = 0; yy < bm->height; yy++, r += bm->raster){
	sprintf(msg, "  bitmap body %d/%ld", yy, bm->height);
	lbp_send_nbytes(dev, (char *)&bm->bitmap[r], wb, msg);
      }
    }
    bm_len = wb * bm->height;
  }
  return bm_len;
}

void  
DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		  long x, long y, long w, long h)
{
  char  mode[32], rect[256];
  int   x0, y0;
 
  if (param_debug == 1)
    fprintf(stderr, "Put rectangle\n");
  x0 = x + offset_x;
  y0 = y + offset_y;
  sprintf(mode, "%c1dmG", GS);
  sprintf(rect, "%c%d;%d;%ld;%ld;%drG", GS, x0, y0, x0+w, y0+h, 0);
  lbp_send(dev, mode, "inside fill mode");           /* Inside fill mode */
  lbp_send(dev, rect, "Rectangle");                  /* Rectangle Fill */
  sprintf(mode, "%c0dmG", GS);
  lbp_send(dev, mode, "frame mode");                 /* Frame mode */
  lbp_send(dev, rect, "Rectangle");                  /* Rectangle Frame */
}

int   
DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  if (interrupted == 1)
    return -1;
  return 0;
}

void  
DEV_message(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;
  
  va_start(ap, fmt);
  fprintf(stderr, "%s: ", PROGRAM);
  vfprintf(stderr, fmt, ap);  
  fprintf(stderr, "\n");
  va_end(ap);
}


void  
lbp_form_feed(DVI_DEVICE dev)
{
  char  ff[2] = { 0x0c, '\0'};                       /* Flush page */

  if (param_debug == 1)
    fprintf(stderr, "Next page\n");
  lbp_send(dev, ff, "Form Feed");
  lbp_flush(dev);
}

void  
lbp_set_param(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "Set param\n");
  sprintf(escseq, "%c%dpsE", GS, param_paper_id);    /* Paper Size */
  lbp_send(dev, escseq, "Paper Size");
  sprintf(escseq, "%c0;%smuE", GS,   
	  (param_dpi==600)?"0.12":"0.24");           /* Dimen & Resolution */
  lbp_send(dev, escseq, "Dimension");
  sprintf(escseq, "%c0;%d;%ddrE", GS, param_dpi, param_dpi);  /* Resolution */
  lbp_send(dev, escseq, "Resolution");
  sprintf(escseq, "%c0;%d;%d;%dlwG", GS, 1, 0, 0);   /* Line style */
  lbp_send(dev, escseq, "Line Style");
  lbp_send(dev, escseq, "Bitmap Compression");
  lbp_bitmap_compress_mode(dev, COMPRESS_YES);
}

void 
lbp_bitmap_compress_mode(DVI_DEVICE dev, int compress_mode)
{
  char  escseq[256];

  if (compress_mode == COMPRESS_NO){
    sprintf(escseq, "%c%dbcI", GS, 0);
    lbp_compress_mode = COMPRESS_NO;
  } else {
    sprintf(escseq, "%c%dbcI", GS, 1);               /* Bitmap compression */
    lbp_compress_mode = COMPRESS_YES;
  }
  lbp_send(dev, escseq, "Compression Mode");
}


void 
lbp_reset_param(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "Reset param\n");
  sprintf(escseq, "%crpE", GS);                      /* Parameter Reset */
  lbp_send(dev, escseq, "Param Reset");
}

void  
lbp_mode_escp2escpage(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "ESC/P->ESC/Page");
  sprintf(escseq, "%c%c%c%c", ESC, 0x7a, 0, 0);      /* ESC/P -> ESC/Page */
  lbp_send_nbytes(dev, escseq, 4, "ESC/P->ESC/Page");
}

void  
lbp_mode_escpage2escp(DVI_DEVICE dev)
{
  char  escseq[256];

  if (param_debug == 1)
    fprintf(stderr, "ESC/Page->ESC/P");
  sprintf(escseq, "%c1pmE", GS);                     /* ESC/Page -> ESC/P */
  lbp_send(dev, escseq, "ESC/Page->ESC/P");
  lbp_flush(dev);
}

void
lbp_send(DVI_DEVICE dev, char *s, char *msg)
{
  lbp_send_nbytes(dev, s, strlen(s), msg);
}

void
lbp_send_nbytes(DVI_DEVICE dev, char *s, int len, char *msg)
{
  int   m, m0;

  while (len > 0){
    if ((len + lbp_buffer_ptr) < sizeof(lbp_buffer))
      m0 = m = len;
    else
      m0 = m = sizeof(lbp_buffer) - lbp_buffer_ptr;
    while (m > 0){
      lbp_buffer[lbp_buffer_ptr++] = *(s++);
      m--;
    }
    len -= m0;
    if (lbp_buffer_ptr == sizeof(lbp_buffer))
      lbp_flush(dev);
  }
  if ((param_debug == 2) && (msg != NULL)){
    lbp_flush(dev);    
    fprintf(stderr, "%s\n", msg);
    if ((msg[0] != ' ') && (param_debug_slow > 0))	
      sleep(param_debug_slow);
  }
}

void 
lbp_flush(DVI_DEVICE dev)
{ 
  if (lbp_buffer_ptr == 0)
    return;
  if (param_debug >= 1)
    fprintf(stderr, "Flush buffer\n");
  fwrite(lbp_buffer, sizeof(char), lbp_buffer_ptr, dev->private2);
  lbp_buffer_ptr = 0;
  fflush(dev->private2);
}

#if HAVE_SIGNAL_H
RETSIGTYPE
sig_handler(int sig)
{
  fprintf(stderr, "%s: Interrupted.\n", PROGRAM);
  interrupted = 1;
}
#endif

/*EOF*/
