/* rpdl.c
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "dvi-2_6.h"
#include "defs.h"

#define	 SUB   0x1a
#define  ESC   0x1b
#define	 DC2   0x12
#define	 EM    0x19

static char
nx100_init[] = {
  ESC,0x0D, 'P',0x0D,				/* software reset */
  ESC, DC2, '!', '@', 'R', '0', '0', ESC, ' ',	/* RPDL mode */
  ESC, DC2, 'Y', 'W', ',', '3', ' ',		/* graphics (600dpi) */
  ESC, DC2, 'Y', 'A', '0', '4', ',', '3', ' ',	/* engine (600dpi) */
  ESC, DC2, 'Y', 'B', ',', '2', ' ',		/* max printable area */
  ESC, DC2, '#', '4', ' ',			/* coordination unit(600dpi) */
  ESC, DC2, 'Q', '5', ' ',			/* spaceing unit(600dpi) */
  ESC, DC2, 'M', '1', ' ',			/* normal CR,LF,FF */
  ESC, '!',					/* auto CR-LF disable */
  ESC, EM, 'M',					/* select multi feeder tray */
  ESC, DC2, '3', '3', ' ',			/* force exhausting time */
};

struct s_rpdl_page_setup {
  char   *name;
  char   *dbgname;
  char	*cmdstr; 
};

static struct s_rpdl_page_setup
rpdl_page_setup[] = {
  {"A6",     "A6", "\033\022" "52@A6\033 "},
  {"B6",     "B6", "\033\022" "52@B6\033 "},
  {"A5",     "A5", "\033\022" "52@A5\033 "},
  {"B5",     "B5", "\033\022" "52@B5\033 "},
  {"A4",     "A4", "\033\022" "52@A4\033 "},
  {"B4",     "B4", "\033\022" "52@B4\033 "},
  {"A3",     "A3", "\033\022" "52@A3\033 "},
  {"LETTER", "LT", "\033\022" "52@LT\033 "},
  {"LEGAL",  "LG", "\033\022" "52@LG\033 "},
  {NULL,     "?",  "\033\022" "D1 "}
};

static unsigned char *compressed_image = NULL;
static int           pcol;

static int     rpdl_prologue(DVI_DEVICE,int,long,long);
static void    rpdl_epilogue(DVI_DEVICE);
static void    rpdl_begin_page(DVI_DEVICE,int);
static void    rpdl_end_page(DVI_DEVICE,int);
static int     rpdl_compression(unsigned char*,unsigned char*,int,int);
extern int     param_verbose;





void
dvi2rpdl(char *dvi_file, long w, long h) 
{
  int  page, ppp;  

  pcol = 0;

  if (rpdl_prologue(dev, dvi->pages, w, h) < 0)
    return; 

  page = (param_page_rev == 0) ? param_page_from : param_page_to;
  do {
    if (param_verbose > 0){
      if (pcol > 70){
	fprintf(stderr, "\n");
	pcol = 0;
      }
      fprintf(stderr, "[%d] ", page);
      fflush(stderr);
      pcol += 4;
      ppp = page;
      while (ppp >= 10){
	pcol++;
	ppp /= 10;
      }
    }
    rpdl_begin_page(dev, page);
    if (DVI_DRAW_PAGE(dvi, dev, page, 1.0) < 0)
      break;
    rpdl_end_page(dev, page);
    if (interrupted == 1)
      break;
    page = page + ((param_page_rev == 0) ? 1 : -1);
  } while ((param_page_from <= page) && (page <= param_page_to));

  rpdl_epilogue(dev);

  if (param_verbose > 0)
    fprintf(stderr, "\n");
}


static int
rpdl_prologue(DVI_DEVICE dev, int pages, long w, long h)
{
  struct s_rpdl_page_setup  *setup_cmd = rpdl_page_setup;

  if ((compressed_image = (unsigned char*)malloc(framebuff->w_bytes)) == NULL){
    fprintf(stderr, "No memory\n");
    return -1;
  }  

  fwrite(nx100_init, (sizeof nx100_init), 1, output_fp);
  if (param_debug != 0)
    fprintf(stderr, "Initialize, w=%ld, h=%ld\n", w, h);

  while ((setup_cmd->name != NULL)
	 && (strcmp(setup_cmd->name, param_paper_name) != 0)){
    setup_cmd++;
  }
  fputs(setup_cmd->cmdstr, output_fp);
  if (param_landscape == 0)
    fputs( "\033\022D1 ", output_fp);
  else
    fputs( "\033\022D2 ", output_fp);
  if (param_debug != 0)
    fprintf(stderr, "Paper: %s\n", setup_cmd->dbgname);

  fprintf(output_fp, "%c%cN%d ", ESC, DC2, param_num_copies);
  if (param_debug != 0)
    fprintf(stderr, "Copies: %d\n", param_num_copies);

  return 0;
}

static void  
rpdl_epilogue(DVI_DEVICE dev)
{
  fflush(output_fp);
  free(compressed_image);
}

static void  
rpdl_begin_page(DVI_DEVICE dev, int seq_page)
{
  DVI_fb_clear(framebuff);
}

static void  
rpdl_end_page(DVI_DEVICE dev, int seq_page)
{
  int      y, x1, x2, n;
  unsigned char  *p;

  if (param_debug != 0)
    fprintf(stderr, "Page: %d\n", seq_page);

  p = framebuff->data;
  for (y = 0; y < framebuff->height; y++, p = p + framebuff->raster){
    if (interrupted == 1)
      break;
    for (x1 = 0; x1 < framebuff->w_bytes; x1++){
      if (p[x1] != 0)
	break;
    }
    if (x1 >= framebuff->w_bytes)
      continue;
    for (x2 = framebuff->w_bytes - 1; x2 >= 0; x2--){
      if (p[x2] != 0)
	break;
    }
    n = rpdl_compression(compressed_image, &p[x1], x2-x1+1, y);
    fprintf(output_fp, "\033\022G3,%d,1,1,4,%d,%d,%d@",
	    (x2 - x1 + 1) * 8, x1 * 8, y, n);
    fwrite(compressed_image, n, 1, output_fp);
    if (param_debug != 0)
      fprintf(stderr, "Image: (%d,%d)-(%d,%d), compressed %d bytes\n", 
	      (x2 - x1 + 1)*8, y, x1*8, y, n);
  }
  fputc('\f', output_fp);
  fflush(output_fp);
}

static int
rpdl_compression(unsigned char *compress_buff,
		 unsigned char *image, int image_size, int y)
{
  unsigned char  *cb, *ip;
  int             in, last_char, repeat_count;

  cb = compress_buff;
  repeat_count = 1;
  last_char = -1;

  for (in = 0, ip = image; in < image_size; in++, ip++){ 
    if ((*ip == last_char) && (repeat_count < 256)){
      if (repeat_count == 1)
	*cb++ = *ip;
      repeat_count++;
    } else {
      if (repeat_count != 1)
	*cb++ = repeat_count - 2;
      last_char = *cb++ = *ip;
      repeat_count = 1;
    }
  }
  if (repeat_count != 1)
    *cb++ = repeat_count - 2;

  return (cb - compress_buff);
}


/*EOF*/
