/************************************************************************/
/*									*/
/*  Read an RTF text file into a BufferDocument				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Consume a field.							*/
/*									*/
/************************************************************************/

static int docRtfReadFldrslt(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfFldinst(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			size	)
    {
    BufferDocument *		bd= rrc->rrcBd;
    DocumentField *		df;

    df= bd->bdFieldList.dflFields+ rrc->rrcFieldNumber;

    if  ( df->dfKind != DOCfkUNKNOWN )
	{ SDEB(docFieldKindStr(df->dfKind));	}

    if  ( docAddToFieldInst( df, text, size ) )
	{ LDEB(size); return -1;	}

    return 0;
    }

static int docRtfReadFldinst(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfFldinst );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfFieldGroups[]=
    {
	{ "fldrslt",	RTFidFLDRSLT,	DOClevPARA, docRtfReadFldrslt, },
	{ "fldinst",	RTFidFLDINST,	DOClevPARA, docRtfReadFldinst, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfFieldWords[]=
    {
	{ "flddirty",	RTFidFLDDIRTY,	DOClevPARA, docRtfIgnoreWord, },
	{ "fldedit",	RTFidFLDEDIT,	DOClevPARA, docRtfIgnoreWord, },
	{ "fldlock",	RTFidFLDLOCK,	DOClevPARA, docRtfIgnoreWord, },
	{ "fldpriv",	RTFidFLDPRIV,	DOClevPARA, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

static int docRtfReadStartField(	int *			pPart,
					int *			pFieldNumber,
					RtfReadingContext *	rrc	)
    {
    BufferDocument *	bd= rrc->rrcBd;
    BufferItem *	bi= rrc->rrcBi;

    DocumentField *	df;
    TextParticule *	tp;

    int			part= bi->biParaParticuleCount;

    if  ( docRtfFlushBookmarks( -1, bi, rrc ) )
	{ LDEB(1); return -1;	}

    df= docClaimField( &(rrc->rrcFieldNumber), &(bd->bdFieldList) );
    if  ( ! df )
	{ XDEB(df); return -1;	}
    df->dfKind= DOCfkUNKNOWN;

    tp= docInsertTextParticule( bi, -1, bi->biParaStrlen, 0,
			DOCkindFIELDSTART, rrc->rrcCurrentTextAttribute );
    if  ( ! tp )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    tp->tpObjectNumber= rrc->rrcFieldNumber;

    *pFieldNumber= rrc->rrcFieldNumber;
    *pPart= part;

    return 0;
    }

static int docRtfTerminateField(	BufferItem *		bi,
					int			fieldNumber,
					RtfReadingContext *	rrc )
    {
    TextParticule *	tp;

    if  ( rrc->rrcBi != bi )
	{ XXDEB(rrc->rrcBi,bi); }

    tp= docInsertTextParticule( bi, -1, bi->biParaStrlen, 0,
			DOCkindFIELDEND, rrc->rrcCurrentTextAttribute );
    if  ( ! tp )
	{ LDEB(bi->biParaParticuleCount); return -1;	}
    tp->tpObjectNumber= fieldNumber;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish a 'Real' field. I.E. A field that is a field in RTF terms.	*/
/*									*/
/************************************************************************/

static int docRtfFinishField(	BufferItem *		bi,
				int			fieldNumber,
				RtfReadingContext *	rrc )
    {
    DocumentField *			df;

    unsigned char *			s;
    int					n;
    unsigned char *			p;
    int					i;
    const FieldKindInformation *	fki;

    if  ( docRtfTerminateField( bi, fieldNumber, rrc ) )
	{ LDEB(fieldNumber); return -1;	}

    df= rrc->rrcBd->bdFieldList.dflFields+ fieldNumber;
    s= df->dfInstructions.mbBytes;
    n= df->dfInstructions.mbSize;

    i= 0;
    while( i < n && isspace( *s ) )
	{ i++; s++;	}
    p= s;
    while( i < n && isalpha( *p ) )
	{
	if  ( islower( *p ) )
	    { *p= toupper( *p );	}
	i++; p++;
	}

    if  ( DOC_FieldKindCount != DOCfk_COUNT )
	{ LLDEB(DOC_FieldKindCount,DOCfk_COUNT); return -1;	}

    /* LSDEB(p-s,(char *)s); */

    fki= DOC_FieldKinds;
    for ( i= 0; i < DOC_FieldKindCount; fki++, i++ )
	{
	if  ( ! fki->fkiIsFieldInRtf )
	    { continue;	}

	if  ( ! strncmp( (const char *)s, fki->fkiLabel, p- s )	&&
	      ! fki->fkiLabel[p- s]			)
	    { break;	}
	}

    if  ( i < DOC_FieldKindCount )
	{ df->dfKind= i;	}

    return 0;
    }

int docRtfReadField(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc	)
    {
    int			res;
    BufferItem *	bi= rrc->rrcBi;

    int			startParticule;
    int			fieldNumber= -1;

    if  ( docRtfReadStartField( &startParticule, &fieldNumber, rrc ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfFieldWords, docRtfFieldGroups,
				docRtfIgnoreText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    if  ( docRtfFlushBookmarks( -1, bi, rrc ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfFinishField( bi, fieldNumber, rrc ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    return res;
    }

static RtfControlWord	docRtfXE_Words[]=
    {
	{ "xef",	RTFidXEF,	DOClevPARA, docRtfIgnoreWord, },
	{ "bxe",	RTFidBXE,	DOClevPARA, docRtfIgnoreWord, },
	{ "ixe",	RTFidIXE,	DOClevPARA, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfTC_Words[]=
    {
	{ "tcf",	RTFidTCF,	DOClevPARA, docRtfIgnoreWord, },
	{ "tcl",	RTFidTCL,	DOClevPARA, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

int docRtfReadLookupEntry(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;
    BufferItem *	bi= rrc->rrcBi;

    int			particuleKind;
    int			fieldKind;

    int			startParticule;
    int			fieldNumber;

    RtfControlWord *	firstApply= (RtfControlWord *)0;
    RtfControlWord *	contolWords= (RtfControlWord *)0;
    RtfControlWord *	groupWords= (RtfControlWord *)0;

    switch( rcw->rcwId )
	{
	case RTFidXE:
	    particuleKind= DOCkindXE;
	    fieldKind= DOCfkXE;
	    contolWords= docRtfXE_Words;
	    break;

	case RTFidTC:
	    particuleKind= DOCkindTC;
	    fieldKind= DOCfkTC;
	    contolWords= docRtfTC_Words;
	    break;

	case RTFidTCN:
	    particuleKind= DOCkindTC;
	    fieldKind= DOCfkTCN;
	    contolWords= docRtfTC_Words;
	    break;

	default:
	    SDEB(rcw->rcwWord); return -1;
	}

    if  ( docRtfReadStartField( &startParticule, &fieldNumber, rrc ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    res= docRtfReadGroup( sis, DOClevPARA,
				firstApply, 0, 0, rrc,
				contolWords, groupWords,
				docRtfFldinst );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Process bookmarks.							*/
/*									*/
/************************************************************************/

static int docRtfBookmarkName(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len	)
    {
    if  ( rrc->rrcBookmarkSize+ len > DOCmaxBOOKMARK )
	{
	LLDEB(rrc->rrcBookmarkSize+ len,DOCmaxBOOKMARK);
	len= DOCmaxBOOKMARK- rrc->rrcBookmarkSize;
	if  ( len < 0 )
	    { len= 0;	}
	}

    memcpy( rrc->rrcBookmarkName+ rrc->rrcBookmarkSize, name, len );
    rrc->rrcBookmarkSize += len;
    rrc->rrcBookmarkName[rrc->rrcBookmarkSize]= '\0';

    return 0;
    }

static RtfControlWord	docRtfBkmkStartWords[]=
    {
	{ "bkmkcolf",	RTFidBKMKCOLF,	DOClevANY, docRtfIgnoreWord, },
	{ "bkmkcoll",	RTFidBKMKCOLL,	DOClevANY, docRtfIgnoreWord, },
	
	{ 0, 0, 0 }
    };

int docRtfBkmkStart(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc )
    {
    BufferDocument *	bd= rrc->rrcBd;
    RtfBookmarkLevel *	rbl;

    int			res;

    int			startParticule;
    int			fieldNumber;
    DocumentField *	df;

    rrc->rrcBookmarkName[0]= '\0';
    rrc->rrcBookmarkSize= 0;

    if  ( docRtfReadStartField( &startParticule, &fieldNumber, rrc ) )
	{ LDEB(1); return -1;	}

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfBkmkStartWords, docRtfEmptyTable,
				docRtfBookmarkName );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res); return res;	}

    df= bd->bdFieldList.dflFields+ fieldNumber;
    if  ( docFieldSetBookmark( df,
			    rrc->rrcBookmarkName, rrc->rrcBookmarkSize ) )
	{ LDEB(1); return -1;	}
    df->dfKind= DOCfkBOOKMARK;

    rbl= (RtfBookmarkLevel *)malloc( sizeof(RtfBookmarkLevel) );
    if  ( ! rbl )
	{ XDEB(rbl); return -1;	}
    rbl->rblFieldNumber= fieldNumber;
    rbl->rblPrevious= rrc->rrcTopBookmark;
    rrc->rrcTopBookmark= rbl;

    return res;
    }

int docRtfFlushBookmarks(	int			fieldNumber,
				BufferItem *		bi,
				RtfReadingContext *	rrc )
    {
    RtfBookmarkLevel *	rbl;

    if  ( fieldNumber >= 0 )
	{
	rbl= rrc->rrcTopBookmark;
	while( rbl )
	    {
	    if  ( rbl->rblFieldNumber == fieldNumber )
		{ break;	}

	    rbl= rbl->rblPrevious;
	    }

	if  ( ! rbl )
	    { return 0;	}
	}

    rbl= rrc->rrcTopBookmark;
    while( rbl )
	{
	if  ( rbl->rblFieldNumber == fieldNumber )
	    { break;	}

	if  ( docRtfFinishField( bi, rbl->rblFieldNumber, rrc ) )
	    { LDEB(1); return -1;	}

	rrc->rrcTopBookmark= rbl->rblPrevious;
	free( rbl );
	rbl= rrc->rrcTopBookmark;
	}

    return 0;
    }

int docRtfBkmkEnd(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc )
    {
    BufferDocument *	bd= rrc->rrcBd;

    int			res;
    int			fieldNumber;

    rrc->rrcBookmarkName[0]= '\0';
    rrc->rrcBookmarkSize= 0;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfEmptyTable,
				docRtfBookmarkName );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res); return res;	}

    for ( fieldNumber= bd->bdFieldList.dflFieldCount- 1;
					    fieldNumber >= 0; fieldNumber-- )
	{
	DocumentField *		df;
	const char *		markName;
	int			markSize;

	df= bd->bdFieldList.dflFields+ fieldNumber;

	if  ( df->dfKind != DOCfkBOOKMARK )
	    { continue;	}

	if  ( docFieldGetBookmark( df, &markName, &markSize ) )
	    { LDEB(1); continue;	}

	if  ( markSize != rrc->rrcBookmarkSize )
	    { continue;	}

	if  ( memcmp( markName, rrc->rrcBookmarkName, rrc->rrcBookmarkSize ) )
	    { continue;	}

	break;
	}

    if  ( fieldNumber < 0 )
	{ /*SDEB((char *)rrc->rrcBookmarkName);*/ return 0;	}

    if  ( docRtfFlushBookmarks( fieldNumber, rrc->rrcBi, rrc ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfFinishField( rrc->rrcBi, fieldNumber, rrc ) )
	{ LDEB(1); return -1;	}

    if  ( rrc->rrcTopBookmark )
	{
	RtfBookmarkLevel *	rbl= rrc->rrcTopBookmark;

	if  ( rbl->rblFieldNumber == fieldNumber )
	    {
	    rrc->rrcTopBookmark= rbl->rblPrevious;
	    free( rbl );
	    }
	}

    return res;
    }

int docRtfSpecialToField(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    BufferItem *	bi= rrc->rrcBi;
    BufferDocument *	bd= rrc->rrcBd;
    DocumentField *	df;

    int			startParticule;
    int			fieldNumber;

    switch( rcw->rcwId )
	{
	case RTFidCHPGN:
	    break;
	default:
	    SDEB(rcw->rcwWord); return -1;
	}

    if  ( docRtfReadStartField( &startParticule, &fieldNumber, rrc ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    df= bd->bdFieldList.dflFields+ fieldNumber;

    if  ( docRtfTextParticule( rrc, (unsigned char *)"?", 1 ) )
	{ LDEB(2); return -1;	}

    if  ( docSetFieldInst( df, (unsigned char *)" PAGE ", 6 ) )
	{ LDEB(6); return -1;	}

    if  ( docRtfFlushBookmarks( -1, bi, rrc ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfFinishField( bi, fieldNumber, rrc ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume Headers and Footers.					*/
/*									*/
/************************************************************************/

static int docRtfReadExternalItem(	BufferItem **		pBi,
					int *			pExtItKind,
					SimpleInputStream *	sis,
					RtfReadingContext *	rrc,
					const SelectionScope *	ss )
    {
    BufferItem *	bi;

    BufferItem *	savedBi;
    int			savedLevel;
    int			savedInField;
    int			savedExternalItemKind;

    int			res;

    savedBi= rrc->rrcBi;
    savedLevel= rrc->rrcLevel;
    savedInField= rrc->rrcParagraphProperties.ppInTable;
    savedExternalItemKind= rrc->rrcExternalItemKind;
    if  ( ! savedBi )
	{ XDEB(savedBi); return -1;	}

    bi= *pBi;
    if  ( ! bi )
	{
	bi= docMakeExternalItem( rrc->rrcBd,
				    ss, &(rrc->rrcSectionProperties) );
	if  ( ! bi )
	    { XDEB(bi); return -1;	}

	*pBi= bi;
	}

    rrc->rrcBi= bi;
    rrc->rrcLevel= DOClevSECT;
    rrc->rrcExternalItemKind= ss->ssInExternalItem;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ LDEB(res);	}

    if  ( docRtfFlushBookmarks( -1, bi, rrc ) )
	{ LDEB(1); return -1;	}

    *pExtItKind= rrc->rrcExternalItemKind;

    rrc->rrcExternalItemKind= savedExternalItemKind;
    rrc->rrcParagraphProperties.ppInTable= savedInField;
    rrc->rrcLevel= savedLevel;
    rrc->rrcBi= savedBi;

    return res;
    }

int docRtfReadExtIt(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    BufferItem *	sectBi= rrc->rrcBi;
    BufferDocument *	bd= rrc->rrcBd;
    ExternalItem *	ei;

    SelectionScope	ss;
    int			extItKind;

    docInitSelectionScope( &ss );

    switch( rcw->rcwId )
	{
	case DOCinSECT_HEADER:
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:
	case DOCinSECT_FOOTER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    if  ( ! sectBi || sectBi->biLevel != DOClevSECT )
		{ XDEB(sectBi); return -1;	}

	    ei= docSectionHeaderFooter( sectBi, rcw->rcwId );
	    if  ( ! ei )
		{ LXDEB(rcw->rcwId,ei); return -1;	}

	    ss.ssSectNrExternalTo= sectBi->biNumberInParent;

	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    if  ( ! bd )
		{ XDEB(bd); return -1;	}

	    ei= docDocumentNoteSeparator( bd, rcw->rcwId );
	    if  ( ! ei )
		{ LXDEB(rcw->rcwId,ei); return -1;	}

	    break;

	default:
	    SDEB(rcw->rcwWord); return -1;
	}

    if  ( ei->eiItem )
	{ SXDEB(rcw->rcwWord,ei->eiItem);	}

    ss.ssInExternalItem= rcw->rcwId;
    ss.ssNoteArrayIndex= -1;

    if  ( docRtfReadExternalItem( &(ei->eiItem), &extItKind, sis, rrc, &ss ) )
	{ SDEB(rcw->rcwWord); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume a footnote.							*/
/*									*/
/************************************************************************/

int docRtfReadFootnote(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    BufferDocument *	bd= rrc->rrcBd;
    BufferItem *	bi= rrc->rrcBi;

    TextParticule *	tp;

    DocumentNote *	dn;
    int			noteIndex;

    BufferItem *	sectBi= bi;
    SelectionScope	ss;

    int			externalItemKind;

    docInitSelectionScope( &ss );

    while( sectBi && sectBi->biLevel != DOClevSECT )
	{ sectBi= sectBi->biParent;	}

    if  ( ! sectBi )
	{ XDEB(sectBi); return -1;	}

    noteIndex= docInsertNote( &dn, bd, bi, bi->biParaStrlen );
    if  ( noteIndex < 0 )
	{ LDEB(noteIndex); return -1;	}

    tp= docInsertTextParticule( bi, -1, bi->biParaStrlen, 0,
			    DOCkindNOTE, rrc->rrcCurrentTextAttribute );
    if  ( ! tp )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    ss.ssInExternalItem= DOCinFOOTNOTE;
    ss.ssSectNrExternalTo= sectBi->biNumberInParent;
    ss.ssNoteArrayIndex= noteIndex;

    if  ( docRtfReadExternalItem( &(dn->dnExternalItem.eiItem),
					&externalItemKind, sis, rrc, &ss ) )
	{ SDEB(rcw->rcwWord); return -1;	}

    dn->dnExternalItemKind= externalItemKind;
    docSetExternalItemKind( dn->dnExternalItem.eiItem,
						externalItemKind );

    return 0;
    }

int docRtfChftn(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    BufferDocument *	bd= rrc->rrcBd;

    int			startParticule;
    int			fieldNumber= -1;
    DocumentField *	df;

    TextAttribute	ta= rrc->rrcCurrentTextAttribute;

    if  ( docRtfReadStartField( &startParticule, &fieldNumber, rrc ) )
	{ LDEB(1); return -1;	}

    df= bd->bdFieldList.dflFields+ fieldNumber;
    df->dfKind= DOCfkCHFTN;

    /*  No: just fake \\up in the formatter temporarily?
    ta.taSuperSub= DOCfontSUPERSCRIPT;
    */

    if  ( docSaveParticules( rrc->rrcBi, ta, (const unsigned char *)"1", 1 ) )
	{ LDEB(1);	}

    if  ( docRtfTerminateField( rrc->rrcBi, fieldNumber, rrc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }
