/***************************************************************************
 *   Copyright (C) 2004 by Ivan Forcada Atienza                            *
 *   ivan@forcada.info                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "shapehandler.h"


ShapeHandler::~ShapeHandler()
{
	qDebug("Deleting shapehandler...");
	mut.lock();
		if (shapeName != NULL) delete [] shapeName;
	mut.unlock();
}

void ShapeHandler::setShapeName(char * name)
{
	if (name == NULL)
	{
		shapeName=NULL;
		return;
	}
	delete shapeName;
	shapeName = new char[strlen(name) + 1];
	if (shapeName == NULL) ErrorMem();
	strcpy(shapeName, name);	
}
	
	
void * ShapeHandler::openShapeFile(int type, char * name)
{ // if type=0 --> .shp // type=1 --> .dbf

	SHPHandle	hSHP;
	DBFHandle	hDBF;
	char cName[strlen(name)+10];
	//char * completeName = new char(sizeof(name)+5);
	
	strcpy(cName,name);	// Backup of the name
	cout << "ShapeHandler: opening " << cName << "." << ((type==0)?"[shp/shx]":"dbf") << endl;
	if (type==0)	// .shp
	{	
		hSHP = SHPOpen( strcat(cName,".shp"), "r+b" );
		if( hSHP == NULL )
		{
			printf( "SHPOpen(%s,\"r\") failed.\n", name );
			//delete completeName;
			return( NULL );
		}
		//delete completeName;
		return (void *)hSHP;
	}
	else	// .dbf
	{
		hDBF = DBFOpen( strcat(cName,".dbf"), "r+b" );
		if( hDBF == NULL )
		{
			printf( "DBFOpen(%s,\"r\") failed.\n", name );
			//delete completeName;
			return( NULL );
		}
		//delete completeName;
		return (void *)hDBF;
	}	
}


int ShapeHandler::createShapefile(char * name)
{
	mut.lock();
	
	cout << "Shapehandler: Creating shapefiles in " << name << endl;
	// Create the shx and shp files
	shpH = SHPCreate( name, SHPT_POINT );	// returns the shapehandle private member
	if( shpH == NULL )
	{
		printf( "Unable to create:%s\n", name );
		return (-1);
	}
	
	setShapeName(name);	
	
	// Create the dbf attribute file:
	dbfH = DBFCreate( name );
	if( dbfH == NULL )
	{
		printf( "DBFCreate(%s) failed.\n", name );
		return (-2);
	}
	// Now I add the desired fields (id, mac, essid, signal+, wep)
	if( DBFAddField( dbfH, "id", FTInteger, 4, 0 ) == -1)
	{
		printf( "DBFAddField(id, FTInteger, 4, 0) failed.\n");
		return (-3);
	}	
	if( DBFAddField( dbfH, "mac", FTString, 18, 0 ) == -1)
	{
		printf( "DBFAddField(mac, FTString, 18, 0) failed.\n");
		return (-3);
	}
	if( DBFAddField( dbfH, "essid", FTString, 32, 0 ) == -1)
	{
		printf( "DBFAddField(essid, FTString, 32, 0) failed.\n");
		return (-3);
	}
	if( DBFAddField( dbfH, "chan", FTInteger, 4, 0 ) == -1)
	{
		printf( "DBFAddField(chan, FTInteger, 4, 0) failed.\n");
		return (-3);
	}	
	if( DBFAddField( dbfH, "signal+", FTInteger, 4, 0 ) == -1)
	{
		printf( "DBFAddField(signal+, FTInteger, 4, 0) failed.\n");
		return (-3);
	}	
	if( DBFAddField( dbfH, "wep", FTInteger, 1, 0 ) == -1)
	{
		printf( "DBFAddField(wep, FTInteger, 1, 0) failed.\n");
		return (-3);
	}	
	
	// Close both newly opened files
	SHPClose( shpH );
	DBFClose( dbfH );
	mut.unlock();
	return 0;	
}

int ShapeHandler::newData(APEntry * ap)
{
	int		id = -1;
	char		mac[18];
	
	mut.lock();
	
	cout << "ShapeHandler: new data arrived." << endl;
	// Must open and close the files with each change in order to keep them in sinchrony
	// Open both shapefile and dbf files:
	if((shpH = (SHPHandle)openShapeFile(SHPFILE, shapeName)) == NULL) 
		{ mut.unlock();return -1; }
	if((dbfH = (DBFHandle)openShapeFile(DBFFILE, shapeName)) == NULL)
		{ mut.unlock();return -1; };
	// I first search if it's a new ap or not
	if ( (id=isInShapes(ap->getMAC(mac))) >= 0 )	// The mac already exists
		updateRecord(ap, id);
	else
		newRecord(ap);	
	
	// Close all
	SHPClose( shpH );
	DBFClose( dbfH );
	
	mut.unlock();
	return 0;
}

int ShapeHandler::isInShapes(char * mac)
{
	int iField=-1;
	int numFields=0;
	
	if((numFields=DBFGetFieldCount(dbfH)) == 0)
	{
		printf( "There are no fields in this table!\n" );
		return( -1 );
	}
	// I have to open the dbf file and find if mac matches with any record 'mac' field
	if ((iField = DBFGetFieldIndex(dbfH, "mac")) == -1) return -2;
	for( int i = 0; i < DBFGetRecordCount(dbfH); i++ )
		if (strcmp(mac,DBFReadStringAttribute( dbfH, i, iField ))==0) return i;
	
	// If the mac isn't there:
	return -1;	
}

void ShapeHandler::newRecord(APEntry * ap)
{
	int iField;
	char temp[20];
	int lRecord = DBFGetRecordCount( dbfH );
	
	cout << "ShapeHandler: adding new point (" << nextID << ")to shapefile." << endl;
	// Modify the shapefile:
	writePoint(nextID, ap->getLongitude(), ap->getLatitude());
	
	// Modify the dbf attributes file:
	if ((iField = DBFGetFieldIndex(dbfH, "id")) >= 0)
		DBFWriteIntegerAttribute(dbfH, lRecord, iField, nextID );
	if ((iField = DBFGetFieldIndex(dbfH, "mac")) >= 0)
		DBFWriteStringAttribute(dbfH, lRecord, iField, ap->getMAC(temp) );
	if ((iField = DBFGetFieldIndex(dbfH, "essid")) >= 0)
		DBFWriteStringAttribute(dbfH, lRecord, iField, ap->getEssid(temp) );
	if ((iField = DBFGetFieldIndex(dbfH, "chan")) >= 0)
		DBFWriteIntegerAttribute(dbfH, lRecord, iField, ap->getChannel() );
	if ((iField = DBFGetFieldIndex(dbfH, "signal+")) >= 0)
		DBFWriteIntegerAttribute(dbfH, lRecord, iField, ap->getSignal() );
	if ((iField = DBFGetFieldIndex(dbfH, "wep")) >= 0)
		DBFWriteIntegerAttribute(dbfH, lRecord, iField, ap->getWEP() ? 1 : 0 );
	
	nextID++;
}


void ShapeHandler::updateRecord(APEntry * ap, int id)
{
	int iField;
//	char temp[20];
	
	cout << "ShapeHandler: updating point " << id << " to shapefile." << endl;
	// See if the new signal is bigger than old one: in this case don't do anything
	if ((iField = DBFGetFieldIndex(dbfH, "signal+")) >= 0)
		if ( DBFReadIntegerAttribute( dbfH, id, iField ) >= ap->getSignal() ) return;
			
	// Modify the shapefile:	
	updatePoint(id, ap->getLongitude(), ap->getLatitude());
	
	// Modify the dbf attributes file:
//	if ((iField = DBFGetFieldIndex(dbfH, "id")) >= 0)
//		DBFWriteIntegerAttribute(dbfH, id, iField, nextID );
//	if ((iField = DBFGetFieldIndex(dbfH, "mac")) >= 0)
//		DBFWriteStringAttribute(dbfH, id, iField, ap->getMAC(temp) );
//	if ((iField = DBFGetFieldIndex(dbfH, "essid")) >= 0)
//		DBFWriteStringAttribute(dbfH, id, iField, ap->getEssid(temp) );
	if ((iField = DBFGetFieldIndex(dbfH, "signal+")) >= 0)
		DBFWriteIntegerAttribute(dbfH, id, iField, ap->getSignal() );
//	if ((iField = DBFGetFieldIndex(dbfH, "wep")) >= 0)
//		DBFWriteIntegerAttribute(dbfH, id, iField, ap->getWEP() ? 1 : 0 );
}

void ShapeHandler::writePoint(int rec, double x, double y ) 
{
	SHPObject	*psShape;

	psShape = SHPCreateObject( SHPT_POINT, rec, 0, NULL, NULL, 1, &x, &y, NULL, NULL );
	SHPWriteObject( shpH, -1, psShape );	// -1 = new record
	SHPDestroyObject( psShape );
}

void ShapeHandler::updatePoint(int rec, double x, double y ) 
{
	SHPObject	*psShape;

	psShape = SHPCreateObject( SHPT_POINT, rec, 0, NULL, NULL, 1, &x, &y, NULL, NULL );
	SHPWriteObject( shpH, rec, psShape );	// Write it in an existing id
	SHPDestroyObject( psShape );
}

void ShapeHandler::ErrorMem()
{
	cerr << "ShapeHandler: Not enough memory\n";
	exit(-1);
}
