/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "RepositoryDialog.h"
#include "RepositoryModel.h"
#include "PostCmdResult.h"
#include "CursorSupport.h"
#include "events/ScParamEvent.h"
#include "commands/ListCmd.h"
#include "commands/ListParam.h"
#include "commands/DetailsParam.h"
#include "sublib/Gui.h"
#include "sublib/Utility.h"
#include "svn/DirEntry.h"

// qt
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlistview.h>
#include <qheader.h>
#include <qregexp.h>



class RDParamVisitor :
  public ParamVisitor<ListParam>,
  public ParamVisitor<DetailsParam>
{
public:
  RDParamVisitor( RepositoryDialog* dlg ) : _dlg(dlg)
  {
  }

  void visit( ListParam* p )
  {
    QListView*       list    = _dlg->getListView();
    svn::DirEntries& entries = p->getEntries();

    for( svn::DirEntries::iterator it = entries.begin(); it != entries.end(); it++ )
    {
      svn::DirEntryPtr e = *it;

      if( e->isDir() || ! _dlg->showDirsOnly() )
      {
        QString stripped = QString::fromUtf8(e->getName());
        stripped.remove( 0, (uint)p->getPathOrUrl().getCharCnt()+1 );

        QListViewItem* lvi = new QListViewItem( list->selectedItem(),
          stripped, QString::fromUtf8(e->getName()) );

        if( e->isDir() )
        {
          lvi->setExpandable(true);
          lvi->setPixmap(0,_dlg->getFolderPixmap());
        }
      }
    }
    list->setOpen( list->selectedItem(), true );
  }

  void visit( DetailsParam* p )
  {
    _dlg->setRootRep( QString::fromUtf8(p->getReposRoot()) );
  }

private:
  RepositoryDialog* _dlg;
};


RepositoryDialog::RepositoryDialog( RepositoryModel* model, QWidget *parent )
  : super( parent, 0, true ), TargetId(this), _model(model), _running(false)
{
  setCaption( _q("subcommander:select repository url") );
  _dirsOnly = false;

  _folder = new QPixmap( getIconDir() + "FolderSvn.png" );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    QGridLayout* gl = new QGridLayout(vbl,1,1);
    {
      _list = new QListView(this);
      //_list->header()->hide();
      _list->setRootIsDecorated(true);
      //_list->setAllColumnsShowFocus(true);
      _list->addColumn( _q("url") );
      //_list->addColumn( "url" );
      //_list->setColumnWidthMode(0,QListView::Maximum);
      //_list->setColumnWidthMode(1,QListView::Maximum);
      _list->setResizeMode( QListView::LastColumn );
      //_list->hideColumn(0);
      //_list->hideColumn(1);

      connect( _list, SIGNAL(selectionChanged(QListViewItem*)),
        this, SLOT(selectionChanged(QListViewItem*)) );

      connect( _list, SIGNAL(expanded(QListViewItem*)),
        this, SLOT(expanded(QListViewItem*)) );

      gl->addWidget(_list,0,0);
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      _parent = new QPushButton( _q("up"), this);
      _parent->setEnabled(true);
      hu->addWidget(_parent);

      connect( _parent, SIGNAL(clicked()), this, SLOT(parent()) );

      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      QPushButton* ok = new QPushButton(this);
      ok->setText( _q("&Ok") );
      ok->setDefault(true);
      hu->addWidget(ok);
      QPushButton* ca = new QPushButton(this);
      ca->setText( _q("&Cancel") );
      hu->addWidget(ca);
      
      hu->addSpacing(getSizeGripSpacing());

      connect( ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( ca, SIGNAL(clicked()), SLOT(reject()) );
    }
  }

  resize( QSize( 300, 400 ) );
}

RepositoryDialog::~RepositoryDialog()
{
  cleanupCursor();
  delete _folder;
}

void RepositoryDialog::setSource( const QString& root )
{
  _root = root;

  QListViewItem* lvi = new QListViewItem( _list, _root, _root );
  lvi->setExpandable(true);

  // fires selectionChanged to load first level
  _list->setSelected(lvi,true);

  // get details only at first run..
  if( _rootRep.isEmpty() )
  {
    getRootDetails();
  }
}

void RepositoryDialog::setRootRep( const QString& rootRep )
{
  _rootRep = rootRep;
}

QString RepositoryDialog::getSelection()
{
  // return full url from hidden column
  return _list->selectedItem()->text(1);
}

QListView* RepositoryDialog::getListView()
{
  return _list;
}

const QPixmap& RepositoryDialog::getFolderPixmap() const
{
  return *_folder;
}

void RepositoryDialog::setDirsOnly( bool b )
{
  _dirsOnly = b;
}

bool RepositoryDialog::showDirsOnly()
{
  return _dirsOnly;
}

void RepositoryDialog::customEvent( QCustomEvent* ce )
{
  switch( ce->type() )
  {
  case ScParameterEvent:
    {
      ScParamEvent* pe = dynamic_cast<ScParamEvent*>(ce);

      RDParamVisitor v(this);
      pe->getParam()->accept(&v);

      _running = false;
      restoreCursor();
      break;
    }
  default:
    {
      printf( "RepositoryDialog: unknown custom event type %d!\n", ce->type() );
    }
  }
}

void RepositoryDialog::selectionChanged( QListViewItem* lvi )
{
  if( _running )
  {
    return;
  }

  if( lvi->text(1) == _root && _root != _rootRep )
  {
    _parent->setEnabled(true);
  }
  else
  {
    _parent->setEnabled(false);
  }

  if( lvi->childCount() != 0 || !lvi->isExpandable() )
  {
    return;
  }

  ListParam*     param = new ListParam( sc::String(lvi->text(1).utf8()),
    new svn::Revision(svn::Revision_Unspecified), false, false );
  PostCmdResult* pcres = new PostCmdResult( this );

  setWaitCursor();
  _running = true;

  _model->list( param, pcres );
}

void RepositoryDialog::getRootDetails()
{
  DetailsParam*  param = new DetailsParam( sc::String(_root.utf8()) );
  PostCmdResult* pcres = new PostCmdResult( this );

  _model->details( param, pcres );
}

void RepositoryDialog::expanded( QListViewItem* lvi )
{
  _list->setSelected(lvi,true);
}

void RepositoryDialog::parent()
{
  if( _running )
  {
    return;
  }

  QRegExp reg("(.*)/.+$");
  reg.exactMatch( _root );

  if( reg.cap(1) != _root )
  {
    _list->clear();
    setSource( reg.cap(1) );
  }
}

