// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/*
  names.cc
  Contains the functions for the starconvert utility which allow it to
  extract star names and designations from a text record.
*/

#include "convert.h"
#include "../classes/constellations.h"
#include "../classes/greek.h"

bool bayer_comm(StringList *, name, StringList *);
bool flam_comm (StringList *, name, StringList *);
bool cspec_comm(StringList *, name, StringList *);
bool dm_comm   (StringList *, name, StringList *) { return false; } // no-op
bool other_comm(StringList *, name, StringList *);

bool bayer(const char *, name, StringList *);
bool flam (const char *, name, StringList *);
bool cspec(const char *, name, StringList *);
bool dm   (const char *, name, StringList *);
bool other(const char *, name, StringList *);

// arrays of function pointers to the above for ease of calling in get_names()

bool (*commentfns[NUM_NAME_TYPES])(StringList *, name, StringList *) =
{ bayer_comm, flam_comm, cspec_comm, dm_comm, other_comm };

bool (*recordfns[NUM_NAME_TYPES])(const char *, name, StringList *) =
{ bayer, flam, cspec, dm, other };


// get_names(): This is the top-level function which loops through the
//  name specifications and calls the appropriate function for each spec.
//  It then takes care of all the substitutions, if any.

void get_names(const char *record, namedata nmd, 
	       StringList *sn, StringList *sc)
               // sn: starnames; sc: starcomments
{
  *sn = StringList();

  // first find all the names
  for (unsigned int i = 0; i < nmd.names.size(); i++) {
    if (nmd.names[i].isNameCommented)
      commentfns[nmd.names[i].type](sc, nmd.names[i], sn);
    else
      recordfns[nmd.names[i].type](record, nmd.names[i], sn);
  }

  // then perform any substitutions
  bool found;
  do {
    unsigned int i = 0;
    found = false;

    if (nmd.isSubstCaseSensitive)
      while (i < nmd.subst1.size() && strcmp(nmd.subst1[i], (*sn)[0]) != 0)
	i++;
    else
      while (i < nmd.subst1.size() && strcasecmp(nmd.subst1[i], (*sn)[0]) != 0)
	i++;
    
    if (i < nmd.subst1.size()) {
      sn->insert(nmd.subst2[i], 0);
      found = true;
    }
  } while (found) ;

  return;
}



// The following functions do the actual work of extracting names from records
//  and comments.  They return true if successful at extracting a name, false
//  otherwise.

bool bayer(const char *record, name namespec, StringList *starnames)
{
  if (namespec.name_len < 6) // 3 chars for Greek letter, 3 for constellation
    return false;

  char *bname = new char[namespec.name_len + 1];
  char fullname[15];
  unsigned int constnum = 0, greeknum = 0;
  unsigned int ptr = 0, supscript;

  strncpy(bname, record + namespec.name_start, namespec.name_len);
  bname[namespec.name_len] = 0;
  stripspace(bname);
  if (!bname[0] || strlen(bname) < 5) { delete [] bname; return false; }

  // assume that Greek letters are abbreviated with at least 3 ASCII chars
  //  (except for those whose English name has only two)
  while (greeknum < NUM_GREEK_LETTERS
	 && (strncasecmp(bname, Greek[greeknum].abbrev, 
			 Greek[greeknum].abbrlen) != 0))
    greeknum++;
  if (greeknum == NUM_GREEK_LETTERS) { delete [] bname; return false; }
  
  // assume that the standard 3-letter constellation abbreviations are used
  while (constnum < NUM_CONSTELLATIONS 
	 && (strcasecmp(bname + strlen(bname) - 3, constellations[constnum])
	     != 0))
    constnum++;
  if (constnum == NUM_CONSTELLATIONS) { delete [] bname; return false; }

  while (ptr < strlen(bname) && !isdigit(bname[ptr]))
    ptr++;
  supscript = myatoi(bname + ptr);
  
  if (supscript) 
    snprintf(fullname, 15, "%s(%d) %s", Greek[greeknum].name, supscript,
	     constellations[constnum]);
  else
    snprintf(fullname, 15, "%s %s", Greek[greeknum].name,
	     constellations[constnum]);
  fullname[14] = 0;
  
  starnames->append(fullname);
  delete [] bname;
  return true;
}


bool flam(const char *record, name namespec, StringList *starnames)
{
  if (namespec.name_len < 6) // 3 chars for number, 3 for constellation
    return false;

  char *fname = new char[namespec.name_len + 1];
  char fullname[15];
  char *endptr;
  unsigned int constnum = 0;
  int flamnum = 0;

  strncpy(fname, record + namespec.name_start, namespec.name_len);
  fname[namespec.name_len] = 0;
  stripspace(fname);
  if (!fname[0] || strlen(fname) < 4) goto end;

  flamnum = strtol(fname, &endptr, 10);
  if (flamnum <= 0 || flamnum > 140 /* what's the largest Flamsteed number? */)
    { flamnum = 0; goto end; }
  if (*endptr == 0 || fname + strlen(fname) - endptr < 3)
    { flamnum = 0; goto end; }
  if (isalpha(fname[strlen(fname) - 4]))
    // then the purported constellation name is actually part of a longer word
    { flamnum = 0; goto end; }

  // assume that the standard 3-letter constellation abbreviations are used
  while (constnum < NUM_CONSTELLATIONS 
	 && (strcasecmp(fname + strlen(fname) - 3, constellations[constnum])
	     != 0))
    constnum++;
  if (constnum == NUM_CONSTELLATIONS)
    { flamnum = 0; goto end; }

  snprintf(fullname, 15, "%d %s", flamnum, constellations[constnum]);
  starnames->append(fullname);

 end:
  delete [] fname;
  return (flamnum > 0);
}


bool cspec(const char *record, name namespec, StringList *starnames)
{
  if (namespec.name_len < 4) // 3 chars for constellation + at least 1 more
    return false;

  char *cname = new char[namespec.name_len + 1];
  char *temp = new char[namespec.name_len + 1];
  unsigned int constnum = 0;

  strncpy(cname, record + namespec.name_start, namespec.name_len);
  cname[namespec.name_len] = 0;
  stripspace(cname);
  if (!cname[0] || strlen(cname) < 4) 
    { delete [] cname; delete [] temp; return false; }

  // assume that the standard 3-letter constellation abbreviations are used
  while (constnum < NUM_CONSTELLATIONS 
	 && (strcasecmp(cname + strlen(cname) - 3, constellations[constnum])
	     != 0))
    constnum++;
  if (constnum == NUM_CONSTELLATIONS) 
    { delete [] cname; delete [] temp; return false; }

  strncpy(temp, cname, strlen(cname) - 3);
  temp[strlen(cname) - 3] = 0;
  stripspace(temp);

  // ensure that this isn't a Bayer / Flamsteed designation
  unsigned int greeknum = 0;
  if (myatoi(temp) > 0) { delete [] cname; delete [] temp; return false; }

  while (greeknum < NUM_GREEK_LETTERS
	 && (strncasecmp(temp, Greek[greeknum].abbrev, 
			 Greek[greeknum].abbrlen) != 0))
    greeknum++;
  if (greeknum < NUM_GREEK_LETTERS) 
    { delete [] cname; delete [] temp; return false; }

  strcat(temp, " ");
  strcat(temp, constellations[constnum]);
  starnames->append(temp);

  delete [] cname;
  delete [] temp;
  return true;
}


bool dm(const char *record, name namespec, StringList *starnames)
{
  if (namespec.name_len < 10) return false;

  char *dname = new char[namespec.name_len + 1];
  strncpy(dname, record + namespec.name_start, namespec.name_len);
  dname[namespec.name_len] = 0;
  stripspace(dname);

  char catalog[3], degrees[10], number[10], sign;
  unsigned int ptr = 0, temp;

  // which catalog: CP, SD, BD or CP
  while (dname[ptr] && !isdigit(dname[ptr]) && ptr < 2)
    catalog[ptr] = dname[ptr++];
  catalog[2] = 0;
  if (strcmp(catalog, "CP") && strcmp(catalog, "BD") && strcmp(catalog, "CD")
      && strcmp(catalog, "SD"))
    { delete [] dname; return false; }

  // value of degree sign (+ or -)
  while (dname[ptr] && isspace(dname[ptr])) ptr++;
  if (isdigit(dname[ptr])) sign = '+';
  else if (dname[ptr] == '+' || dname[ptr] == '-')
    sign = dname[ptr++];
  else { delete [] dname; return false; }
    
  // degree part of identifier
  while (dname[ptr] && !isdigit(dname[ptr])) ptr++;
  temp = ptr;
  while (dname[ptr] && isdigit(dname[ptr]) && ptr - temp < 9)
    degrees[ptr - temp] = dname[ptr++];
  degrees[ptr - temp] = 0;
  if (strlen(degrees) == 0) { delete [] dname; return false; }

  // numerical part of identifier
  while (dname[ptr] && !isdigit(dname[ptr])) ptr++;
  temp = ptr;
  while (dname[ptr] && !isspace(dname[ptr]) && ptr - temp < 9)
    number[ptr - temp] = dname[ptr++];
  number[ptr - temp] = 0;

  if (strlen(degrees) == 1) { // put a leading zero for consistency
    degrees[2] = 0; degrees[1] = degrees[0]; degrees[0] = '0';
  }

  if (strlen(number) == 0) { // then the two items were contiguous
    for (unsigned int i = 2; i <= strlen(degrees); i++)
      number[i - 2] = degrees[i];
    degrees[2] = 0;
  }

  if (strlen(number) && strlen(degrees)) {
    char *fullname = new char[20];
#ifdef EXTENDED_ASCII
    snprintf(fullname, 20, "%s %c%s" DEGREE_SYMBOL "%s",
	     catalog, sign, degrees, number);
#else
    snprintf(fullname, 20, "%s %c%s %s", catalog, sign, degrees, number);
#endif
    starnames->append(fullname);
    delete [] fullname;
    delete [] dname;
    return true;
  }
  else {
    delete [] dname;
    return false;
  }
}


bool other(const char *record, name namespec, StringList *starnames)
{
  if (namespec.name_len <= 0) return false;

  char *sname = new char[namespec.name_len + 1];
  strncpy(sname, record + namespec.name_start, namespec.name_len);
  sname[namespec.name_len] = 0;

  stripspace(sname);
  if (!sname[0]) { delete [] sname; return false; }

  if (!isempty(namespec.name_prefixes[0])) {
    unsigned int len = strlen(sname) + strlen(namespec.name_prefixes[0]) + 2;
    char *tmp = new char[len], *tmp2 = sname;
    snprintf(tmp, len, "%s %s", namespec.name_prefixes[0], sname);
    sname = tmp;
    delete [] tmp2;
  }
  else if (strlen(sname) < 3) { delete [] sname; return false; }

  // remove gratuitous internal spaces, e.g. "Gl   3" -> "Gl 3"
  StringList s = StringList(sname, ' ');
  for (int i = s.size() - 1; i >= 0; i--)
    if (isempty(s[i])) s.remove(i);
  
  unsigned int i;
  sname[0] = 0;
  for (i = 0; i + 1 < s.size(); i++) {
    strcat(sname, s[i]);
    strcat(sname, " ");
  }
  strcat(sname, s[i]);
  
  starnames->append(sname);
  delete [] sname;

  return true;
}


// We'll use the existing Bayer / Flamsteed functions in the comment versions
//  via the miracle of function pointers.

bool var_comm(StringList *starcomments, name namespec, StringList *starnames,
	      bool (*star_fn)(const char *, name, StringList *))
{
  if (starcomments->size() < 2) return false;
  bool test;
  unsigned int oldlen = starnames->size();
  name testspec;
  testspec.name_start = 0;

  for (int j = 0; j + 1 < (int)starcomments->size(); j++) {
    int testlen = strlen((*starcomments)[j]) 
      + strlen((*starcomments)[j + 1]) + 2;
    char * teststring = new char[testlen];
    testspec.name_len = testlen;
    snprintf(teststring, testlen, "%s %s", (*starcomments)[j],
	     (*starcomments)[j + 1]);

    // this does most of the work:
    test = (*star_fn)(teststring, testspec, starnames);

    if (test) {
      starcomments->remove(j + 1);
      starcomments->remove(j);
      j--;
    }
    delete [] teststring;
  }
  return (oldlen - starnames->size() > 0);
}


bool bayer_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, bayer); }

bool flam_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, flam); }

bool cspec_comm(StringList *starcomments, name namespec, StringList *starnames)
{ return var_comm(starcomments, namespec, starnames, cspec); }


bool other_comm(StringList *starcomments, name namespec, StringList *starnames)
{
  if (starcomments->size() < 2) return false;

  unsigned int oldlen = starnames->size();

  for (unsigned int i = 0; i < namespec.name_prefixes.size(); i++) {
    StringList prefix = StringList(namespec.name_prefixes[i], '=');
    prefix.stripspace();

    if (strcmp(prefix[0], "") != 0) {
      if (prefix.size() < 2) prefix.append(prefix[0]);

      for (int j = 0; j + 1 < (int)starcomments->size(); j++) {
	if (strcmp((*starcomments)[j], prefix[0]) == 0) {
	  unsigned int len = strlen(prefix[1]) 
	    + strlen((*starcomments)[j + 1]) + 2;
	  char *sn = new char[len];
	  snprintf(sn, len, "%s %s", prefix[1], (*starcomments)[j + 1]);
	  starnames->append(sn);
	  
	  delete [] sn;
	  starcomments->remove(j + 1);
	  starcomments->remove(j);
	  j--;
	}
      }
    }
  }
  return (oldlen - starnames->size() > 0);
}
