<?php
/**
 * functions/decode/gb2312.php
 *
 * This file contains gb2312-euk decoding function that is needed to read
 * gb2313 encoded mails in non-gb2312 locale.
 * 
 * Original data taken from:
 *  ftp://ftp.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/GB/GB2312.TXT
 *
 *   Name:             GB2312-80 to Unicode table (complete, hex format)
 *   Unicode version:  3.0
 *   Table version:    1.0
 *   Table format:     Format A
 *   Date:             1999 October 8
 *
 * Copyright (c) 1991-1999 Unicode, Inc.  All Rights reserved.
 *
 * This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
 * No claims are made as to fitness for any particular purpose.  No
 * warranties of any kind are expressed or implied.  The recipient
 * agrees to determine applicability of information provided.  If this
 * file has been provided on optical media by Unicode, Inc., the sole
 * remedy for any claim will be exchange of defective media within 90
 * days of receipt.
 * 
 * Unicode, Inc. hereby grants the right to freely use the information
 * supplied in this file in the creation of products supporting the
 * Unicode Standard, and to make copies of this file in any form for
 * internal or external distribution as long as this notice remains
 * attached.
 *
 * @copyright (c) 2003-2005 The SquirrelMail Project Team
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version $Id: gb2312.php,v 1.10 2005/11/17 21:10:27 tokul Exp $
 * @package decode
 * @subpackage eastasia
 */

/**
 * Decode gb2312-euk encoded string
 * @param string $string Encoded string
 * @param boolean $save_html don't html encode special characters if true
 * @return string $string Decoded string
 */
function charset_decode_gb2312 ($string,$save_html=false) {
    global $aggressive_decoding;

    // this is CPU intensive task. Use recode functions if they are available. 
    if (function_exists('recode_string')) {
        // if string is already sanitized, undo htmlspecial chars
        if (! $save_html)
            $string=str_replace(array('&amp;','&quot;','&lt;','&gt;'),array('&','"','<','>'),$string);

        $string = recode_string("gb2312..html",$string);

        // if string sanitizing is not needed, undo htmlspecialchars applied by recode.
        if ($save_html)
            $string=str_replace(array('&amp;','&quot;','&lt;','&gt;'),array('&','"','<','>'),$string);

        return $string;
    }

    /*
     * iconv does not support html target, but internal utf-8 decoding is faster 
     * than pure php implementation. 
     */
    if (function_exists('iconv') && file_exists(SM_PATH . 'functions/decode/utf_8.php') ) {
        include_once(SM_PATH . 'functions/decode/utf_8.php');
        $string = iconv('gb2312','utf-8',$string);
        return charset_decode_utf_8($string);
    }

    // try mbstring
    // TODO: check sanitizing of html special chars.
    if (function_exists('mb_convert_encoding') && 
        function_exists('sq_mb_list_encodings') &&
        check_php_version(4,3,0) &&
        in_array('gb2312',sq_mb_list_encodings())) {
        return mb_convert_encoding($string,'HTML-ENTITIES','GB2312');
    }

    if (!$aggressive_decoding) return $string;

    // don't do decoding when there are no 8bit symbols
    if (! sq_is8bit($string,'gb2312'))
        return $string;

    $gb2312 = array(
	"\xA1\xA1" => '&#12288;',
	"\xA1\xA2" => '&#12289;',
	"\xA1\xA3" => '&#12290;',
	"\xA1\xA4" => '&#12539;',
	"\xA1\xA5" => '&#713;',
	"\xA1\xA6" => '&#711;',
	"\xA1\xA7" => '&#168;',
	"\xA1\xA8" => '&#12291;',
	"\xA1\xA9" => '&#12293;',
	"\xA1\xAA" => '&#8213;',
	"\xA1\xAB" => '&#65374;',
	"\xA1\xAC" => '&#8214;',
	"\xA1\xAD" => '&#8230;',
	"\xA1\xAE" => '&#8216;',
	"\xA1\xAF" => '&#8217;',
	"\xA1\xB0" => '&#8220;',
	"\xA1\xB1" => '&#8221;',
	"\xA1\xB2" => '&#12308;',
	"\xA1\xB3" => '&#12309;',
	"\xA1\xB4" => '&#12296;',
	"\xA1\xB5" => '&#12297;',
	"\xA1\xB6" => '&#12298;',
	"\xA1\xB7" => '&#12299;',
	"\xA1\xB8" => '&#12300;',
	"\xA1\xB9" => '&#12301;',
	"\xA1\xBA" => '&#12302;',
	"\xA1\xBB" => '&#12303;',
	"\xA1\xBC" => '&#12310;',
	"\xA1\xBD" => '&#12311;',
	"\xA1\xBE" => '&#12304;',
	"\xA1\xBF" => '&#12305;',
	"\xA1\xC0" => '&#177;',
	"\xA1\xC1" => '&#215;',
	"\xA1\xC2" => '&#247;',
	"\xA1\xC3" => '&#8758;',
	"\xA1\xC4" => '&#8743;',
	"\xA1\xC5" => '&#8744;',
	"\xA1\xC6" => '&#8721;',
	"\xA1\xC7" => '&#8719;',
	"\xA1\xC8" => '&#8746;',
	"\xA1\xC9" => '&#8745;',
	"\xA1\xCA" => '&#8712;',
	"\xA1\xCB" => '&#8759;',
	"\xA1\xCC" => '&#8730;',
	"\xA1\xCD" => '&#8869;',
	"\xA1\xCE" => '&#8741;',
	"\xA1\xCF" => '&#8736;',
	"\xA1\xD0" => '&#8978;',
	"\xA1\xD1" => '&#8857;',
	"\xA1\xD2" => '&#8747;',
	"\xA1\xD3" => '&#8750;',
	"\xA1\xD4" => '&#8801;',
	"\xA1\xD5" => '&#8780;',
	"\xA1\xD6" => '&#8776;',
	"\xA1\xD7" => '&#8765;',
	"\xA1\xD8" => '&#8733;',
	"\xA1\xD9" => '&#8800;',
	"\xA1\xDA" => '&#8814;',
	"\xA1\xDB" => '&#8815;',
	"\xA1\xDC" => '&#8804;',
	"\xA1\xDD" => '&#8805;',
	"\xA1\xDE" => '&#8734;',
	"\xA1\xDF" => '&#8757;',
	"\xA1\xE0" => '&#8756;',
	"\xA1\xE1" => '&#9794;',
	"\xA1\xE2" => '&#9792;',
	"\xA1\xE3" => '&#176;',
	"\xA1\xE4" => '&#8242;',
	"\xA1\xE5" => '&#8243;',
	"\xA1\xE6" => '&#8451;',
	"\xA1\xE7" => '&#65284;',
	"\xA1\xE8" => '&#164;',
	"\xA1\xE9" => '&#65504;',
	"\xA1\xEA" => '&#65505;',
	"\xA1\xEB" => '&#8240;',
	"\xA1\xEC" => '&#167;',
	"\xA1\xED" => '&#8470;',
	"\xA1\xEE" => '&#9734;',
	"\xA1\xEF" => '&#9733;',
	"\xA1\xF0" => '&#9675;',
	"\xA1\xF1" => '&#9679;',
	"\xA1\xF2" => '&#9678;',
	"\xA1\xF3" => '&#9671;',
	"\xA1\xF4" => '&#9670;',
	"\xA1\xF5" => '&#9633;',
	"\xA1\xF6" => '&#9632;',
	"\xA1\xF7" => '&#9651;',
	"\xA1\xF8" => '&#9650;',
	"\xA1\xF9" => '&#8251;',
	"\xA1\xFA" => '&#8594;',
	"\xA1\xFB" => '&#8592;',
	"\xA1\xFC" => '&#8593;',
	"\xA1\xFD" => '&#8595;',
	"\xA1\xFE" => '&#12307;',
	"\xA2\xB1" => '&#9352;',
	"\xA2\xB2" => '&#9353;',
	"\xA2\xB3" => '&#9354;',
	"\xA2\xB4" => '&#9355;',
	"\xA2\xB5" => '&#9356;',
	"\xA2\xB6" => '&#9357;',
	"\xA2\xB7" => '&#9358;',
	"\xA2\xB8" => '&#9359;',
	"\xA2\xB9" => '&#9360;',
	"\xA2\xBA" => '&#9361;',
	"\xA2\xBB" => '&#9362;',
	"\xA2\xBC" => '&#9363;',
	"\xA2\xBD" => '&#9364;',
	"\xA2\xBE" => '&#9365;',
	"\xA2\xBF" => '&#9366;',
	"\xA2\xC0" => '&#9367;',
	"\xA2\xC1" => '&#9368;',
	"\xA2\xC2" => '&#9369;',
	"\xA2\xC3" => '&#9370;',
	"\xA2\xC4" => '&#9371;',
	"\xA2\xC5" => '&#9332;',
	"\xA2\xC6" => '&#9333;',
	"\xA2\xC7" => '&#9334;',
	"\xA2\xC8" => '&#9335;',
	"\xA2\xC9" => '&#9336;',
	"\xA2\xCA" => '&#9337;',
	"\xA2\xCB" => '&#9338;',
	"\xA2\xCC" => '&#9339;',
	"\xA2\xCD" => '&#9340;',
	"\xA2\xCE" => '&#9341;',
	"\xA2\xCF" => '&#9342;',
	"\xA2\xD0" => '&#9343;',
	"\xA2\xD1" => '&#9344;',
	"\xA2\xD2" => '&#9345;',
	"\xA2\xD3" => '&#9346;',
	"\xA2\xD4" => '&#9347;',
	"\xA2\xD5" => '&#9348;',
	"\xA2\xD6" => '&#9349;',
	"\xA2\xD7" => '&#9350;',
	"\xA2\xD8" => '&#9351;',
	"\xA2\xD9" => '&#9312;',
	"\xA2\xDA" => '&#9313;',
	"\xA2\xDB" => '&#9314;',
	"\xA2\xDC" => '&#9315;',
	"\xA2\xDD" => '&#9316;',
	"\xA2\xDE" => '&#9317;',
	"\xA2\xDF" => '&#9318;',
	"\xA2\xE0" => '&#9319;',
	"\xA2\xE1" => '&#9320;',
	"\xA2\xE2" => '&#9321;',
	"\xA2\xE5" => '&#12832;',
	"\xA2\xE6" => '&#12833;',
	"\xA2\xE7" => '&#12834;',
	"\xA2\xE8" => '&#12835;',
	"\xA2\xE9" => '&#12836;',
	"\xA2\xEA" => '&#12837;',
	"\xA2\xEB" => '&#12838;',
	"\xA2\xEC" => '&#12839;',
	"\xA2\xED" => '&#12840;',
	"\xA2\xEE" => '&#12841;',
	"\xA2\xF1" => '&#8544;',
	"\xA2\xF2" => '&#8545;',
	"\xA2\xF3" => '&#8546;',
	"\xA2\xF4" => '&#8547;',
	"\xA2\xF5" => '&#8548;',
	"\xA2\xF6" => '&#8549;',
	"\xA2\xF7" => '&#8550;',
	"\xA2\xF8" => '&#8551;',
	"\xA2\xF9" => '&#8552;',
	"\xA2\xFA" => '&#8553;',
	"\xA2\xFB" => '&#8554;',
	"\xA2\xFC" => '&#8555;',
	"\xA3\xA1" => '&#65281;',
	"\xA3\xA2" => '&#65282;',
	"\xA3\xA3" => '&#65283;',
	"\xA3\xA4" => '&#65509;',
	"\xA3\xA5" => '&#65285;',
	"\xA3\xA6" => '&#65286;',
	"\xA3\xA7" => '&#65287;',
	"\xA3\xA8" => '&#65288;',
	"\xA3\xA9" => '&#65289;',
	"\xA3\xAA" => '&#65290;',
	"\xA3\xAB" => '&#65291;',
	"\xA3\xAC" => '&#65292;',
	"\xA3\xAD" => '&#65293;',
	"\xA3\xAE" => '&#65294;',
	"\xA3\xAF" => '&#65295;',
	"\xA3\xB0" => '&#65296;',
	"\xA3\xB1" => '&#65297;',
	"\xA3\xB2" => '&#65298;',
	"\xA3\xB3" => '&#65299;',
	"\xA3\xB4" => '&#65300;',
	"\xA3\xB5" => '&#65301;',
	"\xA3\xB6" => '&#65302;',
	"\xA3\xB7" => '&#65303;',
	"\xA3\xB8" => '&#65304;',
	"\xA3\xB9" => '&#65305;',
	"\xA3\xBA" => '&#65306;',
	"\xA3\xBB" => '&#65307;',
	"\xA3\xBC" => '&#65308;',
	"\xA3\xBD" => '&#65309;',
	"\xA3\xBE" => '&#65310;',
	"\xA3\xBF" => '&#65311;',
	"\xA3\xC0" => '&#65312;',
	"\xA3\xC1" => '&#65313;',
	"\xA3\xC2" => '&#65314;',
	"\xA3\xC3" => '&#65315;',
	"\xA3\xC4" => '&#65316;',
	"\xA3\xC5" => '&#65317;',
	"\xA3\xC6" => '&#65318;',
	"\xA3\xC7" => '&#65319;',
	"\xA3\xC8" => '&#65320;',
	"\xA3\xC9" => '&#65321;',
	"\xA3\xCA" => '&#65322;',
	"\xA3\xCB" => '&#65323;',
	"\xA3\xCC" => '&#65324;',
	"\xA3\xCD" => '&#65325;',
	"\xA3\xCE" => '&#65326;',
	"\xA3\xCF" => '&#65327;',
	"\xA3\xD0" => '&#65328;',
	"\xA3\xD1" => '&#65329;',
	"\xA3\xD2" => '&#65330;',
	"\xA3\xD3" => '&#65331;',
	"\xA3\xD4" => '&#65332;',
	"\xA3\xD5" => '&#65333;',
	"\xA3\xD6" => '&#65334;',
	"\xA3\xD7" => '&#65335;',
	"\xA3\xD8" => '&#65336;',
	"\xA3\xD9" => '&#65337;',
	"\xA3\xDA" => '&#65338;',
	"\xA3\xDB" => '&#65339;',
	"\xA3\xDC" => '&#65340;',
	"\xA3\xDD" => '&#65341;',
	"\xA3\xDE" => '&#65342;',
	"\xA3\xDF" => '&#65343;',
	"\xA3\xE0" => '&#65344;',
	"\xA3\xE1" => '&#65345;',
	"\xA3\xE2" => '&#65346;',
	"\xA3\xE3" => '&#65347;',
	"\xA3\xE4" => '&#65348;',
	"\xA3\xE5" => '&#65349;',
	"\xA3\xE6" => '&#65350;',
	"\xA3\xE7" => '&#65351;',
	"\xA3\xE8" => '&#65352;',
	"\xA3\xE9" => '&#65353;',
	"\xA3\xEA" => '&#65354;',
	"\xA3\xEB" => '&#65355;',
	"\xA3\xEC" => '&#65356;',
	"\xA3\xED" => '&#65357;',
	"\xA3\xEE" => '&#65358;',
	"\xA3\xEF" => '&#65359;',
	"\xA3\xF0" => '&#65360;',
	"\xA3\xF1" => '&#65361;',
	"\xA3\xF2" => '&#65362;',
	"\xA3\xF3" => '&#65363;',
	"\xA3\xF4" => '&#65364;',
	"\xA3\xF5" => '&#65365;',
	"\xA3\xF6" => '&#65366;',
	"\xA3\xF7" => '&#65367;',
	"\xA3\xF8" => '&#65368;',
	"\xA3\xF9" => '&#65369;',
	"\xA3\xFA" => '&#65370;',
	"\xA3\xFB" => '&#65371;',
	"\xA3\xFC" => '&#65372;',
	"\xA3\xFD" => '&#65373;',
	"\xA3\xFE" => '&#65507;',
	"\xA4\xA1" => '&#12353;',
	"\xA4\xA2" => '&#12354;',
	"\xA4\xA3" => '&#12355;',
	"\xA4\xA4" => '&#12356;',
	"\xA4\xA5" => '&#12357;',
	"\xA4\xA6" => '&#12358;',
	"\xA4\xA7" => '&#12359;',
	"\xA4\xA8" => '&#12360;',
	"\xA4\xA9" => '&#12361;',
	"\xA4\xAA" => '&#12362;',
	"\xA4\xAB" => '&#12363;',
	"\xA4\xAC" => '&#12364;',
	"\xA4\xAD" => '&#12365;',
	"\xA4\xAE" => '&#12366;',
	"\xA4\xAF" => '&#12367;',
	"\xA4\xB0" => '&#12368;',
	"\xA4\xB1" => '&#12369;',
	"\xA4\xB2" => '&#12370;',
	"\xA4\xB3" => '&#12371;',
	"\xA4\xB4" => '&#12372;',
	"\xA4\xB5" => '&#12373;',
	"\xA4\xB6" => '&#12374;',
	"\xA4\xB7" => '&#12375;',
	"\xA4\xB8" => '&#12376;',
	"\xA4\xB9" => '&#12377;',
	"\xA4\xBA" => '&#12378;',
	"\xA4\xBB" => '&#12379;',
	"\xA4\xBC" => '&#12380;',
	"\xA4\xBD" => '&#12381;',
	"\xA4\xBE" => '&#12382;',
	"\xA4\xBF" => '&#12383;',
	"\xA4\xC0" => '&#12384;',
	"\xA4\xC1" => '&#12385;',
	"\xA4\xC2" => '&#12386;',
	"\xA4\xC3" => '&#12387;',
	"\xA4\xC4" => '&#12388;',
	"\xA4\xC5" => '&#12389;',
	"\xA4\xC6" => '&#12390;',
	"\xA4\xC7" => '&#12391;',
	"\xA4\xC8" => '&#12392;',
	"\xA4\xC9" => '&#12393;',
	"\xA4\xCA" => '&#12394;',
	"\xA4\xCB" => '&#12395;',
	"\xA4\xCC" => '&#12396;',
	"\xA4\xCD" => '&#12397;',
	"\xA4\xCE" => '&#12398;',
	"\xA4\xCF" => '&#12399;',
	"\xA4\xD0" => '&#12400;',
	"\xA4\xD1" => '&#12401;',
	"\xA4\xD2" => '&#12402;',
	"\xA4\xD3" => '&#12403;',
	"\xA4\xD4" => '&#12404;',
	"\xA4\xD5" => '&#12405;',
	"\xA4\xD6" => '&#12406;',
	"\xA4\xD7" => '&#12407;',
	"\xA4\xD8" => '&#12408;',
	"\xA4\xD9" => '&#12409;',
	"\xA4\xDA" => '&#12410;',
	"\xA4\xDB" => '&#12411;',
	"\xA4\xDC" => '&#12412;',
	"\xA4\xDD" => '&#12413;',
	"\xA4\xDE" => '&#12414;',
	"\xA4\xDF" => '&#12415;',
	"\xA4\xE0" => '&#12416;',
	"\xA4\xE1" => '&#12417;',
	"\xA4\xE2" => '&#12418;',
	"\xA4\xE3" => '&#12419;',
	"\xA4\xE4" => '&#12420;',
	"\xA4\xE5" => '&#12421;',
	"\xA4\xE6" => '&#12422;',
	"\xA4\xE7" => '&#12423;',
	"\xA4\xE8" => '&#12424;',
	"\xA4\xE9" => '&#12425;',
	"\xA4\xEA" => '&#12426;',
	"\xA4\xEB" => '&#12427;',
	"\xA4\xEC" => '&#12428;',
	"\xA4\xED" => '&#12429;',
	"\xA4\xEE" => '&#12430;',
	"\xA4\xEF" => '&#12431;',
	"\xA4\xF0" => '&#12432;',
	"\xA4\xF1" => '&#12433;',
	"\xA4\xF2" => '&#12434;',
	"\xA4\xF3" => '&#12435;',
	"\xA5\xA1" => '&#12449;',
	"\xA5\xA2" => '&#12450;',
	"\xA5\xA3" => '&#12451;',
	"\xA5\xA4" => '&#12452;',
	"\xA5\xA5" => '&#12453;',
	"\xA5\xA6" => '&#12454;',
	"\xA5\xA7" => '&#12455;',
	"\xA5\xA8" => '&#12456;',
	"\xA5\xA9" => '&#12457;',
	"\xA5\xAA" => '&#12458;',
	"\xA5\xAB" => '&#12459;',
	"\xA5\xAC" => '&#12460;',
	"\xA5\xAD" => '&#12461;',
	"\xA5\xAE" => '&#12462;',
	"\xA5\xAF" => '&#12463;',
	"\xA5\xB0" => '&#12464;',
	"\xA5\xB1" => '&#12465;',
	"\xA5\xB2" => '&#12466;',
	"\xA5\xB3" => '&#12467;',
	"\xA5\xB4" => '&#12468;',
	"\xA5\xB5" => '&#12469;',
	"\xA5\xB6" => '&#12470;',
	"\xA5\xB7" => '&#12471;',
	"\xA5\xB8" => '&#12472;',
	"\xA5\xB9" => '&#12473;',
	"\xA5\xBA" => '&#12474;',
	"\xA5\xBB" => '&#12475;',
	"\xA5\xBC" => '&#12476;',
	"\xA5\xBD" => '&#12477;',
	"\xA5\xBE" => '&#12478;',
	"\xA5\xBF" => '&#12479;',
	"\xA5\xC0" => '&#12480;',
	"\xA5\xC1" => '&#12481;',
	"\xA5\xC2" => '&#12482;',
	"\xA5\xC3" => '&#12483;',
	"\xA5\xC4" => '&#12484;',
	"\xA5\xC5" => '&#12485;',
	"\xA5\xC6" => '&#12486;',
	"\xA5\xC7" => '&#12487;',
	"\xA5\xC8" => '&#12488;',
	"\xA5\xC9" => '&#12489;',
	"\xA5\xCA" => '&#12490;',
	"\xA5\xCB" => '&#12491;',
	"\xA5\xCC" => '&#12492;',
	"\xA5\xCD" => '&#12493;',
	"\xA5\xCE" => '&#12494;',
	"\xA5\xCF" => '&#12495;',
	"\xA5\xD0" => '&#12496;',
	"\xA5\xD1" => '&#12497;',
	"\xA5\xD2" => '&#12498;',
	"\xA5\xD3" => '&#12499;',
	"\xA5\xD4" => '&#12500;',
	"\xA5\xD5" => '&#12501;',
	"\xA5\xD6" => '&#12502;',
	"\xA5\xD7" => '&#12503;',
	"\xA5\xD8" => '&#12504;',
	"\xA5\xD9" => '&#12505;',
	"\xA5\xDA" => '&#12506;',
	"\xA5\xDB" => '&#12507;',
	"\xA5\xDC" => '&#12508;',
	"\xA5\xDD" => '&#12509;',
	"\xA5\xDE" => '&#12510;',
	"\xA5\xDF" => '&#12511;',
	"\xA5\xE0" => '&#12512;',
	"\xA5\xE1" => '&#12513;',
	"\xA5\xE2" => '&#12514;',
	"\xA5\xE3" => '&#12515;',
	"\xA5\xE4" => '&#12516;',
	"\xA5\xE5" => '&#12517;',
	"\xA5\xE6" => '&#12518;',
	"\xA5\xE7" => '&#12519;',
	"\xA5\xE8" => '&#12520;',
	"\xA5\xE9" => '&#12521;',
	"\xA5\xEA" => '&#12522;',
	"\xA5\xEB" => '&#12523;',
	"\xA5\xEC" => '&#12524;',
	"\xA5\xED" => '&#12525;',
	"\xA5\xEE" => '&#12526;',
	"\xA5\xEF" => '&#12527;',
	"\xA5\xF0" => '&#12528;',
	"\xA5\xF1" => '&#12529;',
	"\xA5\xF2" => '&#12530;',
	"\xA5\xF3" => '&#12531;',
	"\xA5\xF4" => '&#12532;',
	"\xA5\xF5" => '&#12533;',
	"\xA5\xF6" => '&#12534;',
	"\xA6\xA1" => '&#913;',
	"\xA6\xA2" => '&#914;',
	"\xA6\xA3" => '&#915;',
	"\xA6\xA4" => '&#916;',
	"\xA6\xA5" => '&#917;',
	"\xA6\xA6" => '&#918;',
	"\xA6\xA7" => '&#919;',
	"\xA6\xA8" => '&#920;',
	"\xA6\xA9" => '&#921;',
	"\xA6\xAA" => '&#922;',
	"\xA6\xAB" => '&#923;',
	"\xA6\xAC" => '&#924;',
	"\xA6\xAD" => '&#925;',
	"\xA6\xAE" => '&#926;',
	"\xA6\xAF" => '&#927;',
	"\xA6\xB0" => '&#928;',
	"\xA6\xB1" => '&#929;',
	"\xA6\xB2" => '&#931;',
	"\xA6\xB3" => '&#932;',
	"\xA6\xB4" => '&#933;',
	"\xA6\xB5" => '&#934;',
	"\xA6\xB6" => '&#935;',
	"\xA6\xB7" => '&#936;',
	"\xA6\xB8" => '&#937;',
	"\xA6\xC1" => '&#945;',
	"\xA6\xC2" => '&#946;',
	"\xA6\xC3" => '&#947;',
	"\xA6\xC4" => '&#948;',
	"\xA6\xC5" => '&#949;',
	"\xA6\xC6" => '&#950;',
	"\xA6\xC7" => '&#951;',
	"\xA6\xC8" => '&#952;',
	"\xA6\xC9" => '&#953;',
	"\xA6\xCA" => '&#954;',
	"\xA6\xCB" => '&#955;',
	"\xA6\xCC" => '&#956;',
	"\xA6\xCD" => '&#957;',
	"\xA6\xCE" => '&#958;',
	"\xA6\xCF" => '&#959;',
	"\xA6\xD0" => '&#960;',
	"\xA6\xD1" => '&#961;',
	"\xA6\xD2" => '&#963;',
	"\xA6\xD3" => '&#964;',
	"\xA6\xD4" => '&#965;',
	"\xA6\xD5" => '&#966;',
	"\xA6\xD6" => '&#967;',
	"\xA6\xD7" => '&#968;',
	"\xA6\xD8" => '&#969;',
	"\xA7\xA1" => '&#1040;',
	"\xA7\xA2" => '&#1041;',
	"\xA7\xA3" => '&#1042;',
	"\xA7\xA4" => '&#1043;',
	"\xA7\xA5" => '&#1044;',
	"\xA7\xA6" => '&#1045;',
	"\xA7\xA7" => '&#1025;',
	"\xA7\xA8" => '&#1046;',
	"\xA7\xA9" => '&#1047;',
	"\xA7\xAA" => '&#1048;',
	"\xA7\xAB" => '&#1049;',
	"\xA7\xAC" => '&#1050;',
	"\xA7\xAD" => '&#1051;',
	"\xA7\xAE" => '&#1052;',
	"\xA7\xAF" => '&#1053;',
	"\xA7\xB0" => '&#1054;',
	"\xA7\xB1" => '&#1055;',
	"\xA7\xB2" => '&#1056;',
	"\xA7\xB3" => '&#1057;',
	"\xA7\xB4" => '&#1058;',
	"\xA7\xB5" => '&#1059;',
	"\xA7\xB6" => '&#1060;',
	"\xA7\xB7" => '&#1061;',
	"\xA7\xB8" => '&#1062;',
	"\xA7\xB9" => '&#1063;',
	"\xA7\xBA" => '&#1064;',
	"\xA7\xBB" => '&#1065;',
	"\xA7\xBC" => '&#1066;',
	"\xA7\xBD" => '&#1067;',
	"\xA7\xBE" => '&#1068;',
	"\xA7\xBF" => '&#1069;',
	"\xA7\xC0" => '&#1070;',
	"\xA7\xC1" => '&#1071;',
	"\xA7\xD1" => '&#1072;',
	"\xA7\xD2" => '&#1073;',
	"\xA7\xD3" => '&#1074;',
	"\xA7\xD4" => '&#1075;',
	"\xA7\xD5" => '&#1076;',
	"\xA7\xD6" => '&#1077;',
	"\xA7\xD7" => '&#1105;',
	"\xA7\xD8" => '&#1078;',
	"\xA7\xD9" => '&#1079;',
	"\xA7\xDA" => '&#1080;',
	"\xA7\xDB" => '&#1081;',
	"\xA7\xDC" => '&#1082;',
	"\xA7\xDD" => '&#1083;',
	"\xA7\xDE" => '&#1084;',
	"\xA7\xDF" => '&#1085;',
	"\xA7\xE0" => '&#1086;',
	"\xA7\xE1" => '&#1087;',
	"\xA7\xE2" => '&#1088;',
	"\xA7\xE3" => '&#1089;',
	"\xA7\xE4" => '&#1090;',
	"\xA7\xE5" => '&#1091;',
	"\xA7\xE6" => '&#1092;',
	"\xA7\xE7" => '&#1093;',
	"\xA7\xE8" => '&#1094;',
	"\xA7\xE9" => '&#1095;',
	"\xA7\xEA" => '&#1096;',
	"\xA7\xEB" => '&#1097;',
	"\xA7\xEC" => '&#1098;',
	"\xA7\xED" => '&#1099;',
	"\xA7\xEE" => '&#1100;',
	"\xA7\xEF" => '&#1101;',
	"\xA7\xF0" => '&#1102;',
	"\xA7\xF1" => '&#1103;',
	"\xA8\xA1" => '&#257;',
	"\xA8\xA2" => '&#225;',
	"\xA8\xA3" => '&#462;',
	"\xA8\xA4" => '&#224;',
	"\xA8\xA5" => '&#275;',
	"\xA8\xA6" => '&#233;',
	"\xA8\xA7" => '&#283;',
	"\xA8\xA8" => '&#232;',
	"\xA8\xA9" => '&#299;',
	"\xA8\xAA" => '&#237;',
	"\xA8\xAB" => '&#464;',
	"\xA8\xAC" => '&#236;',
	"\xA8\xAD" => '&#333;',
	"\xA8\xAE" => '&#243;',
	"\xA8\xAF" => '&#466;',
	"\xA8\xB0" => '&#242;',
	"\xA8\xB1" => '&#363;',
	"\xA8\xB2" => '&#250;',
	"\xA8\xB3" => '&#468;',
	"\xA8\xB4" => '&#249;',
	"\xA8\xB5" => '&#470;',
	"\xA8\xB6" => '&#472;',
	"\xA8\xB7" => '&#474;',
	"\xA8\xB8" => '&#476;',
	"\xA8\xB9" => '&#252;',
	"\xA8\xBA" => '&#234;',
	"\xA8\xC5" => '&#12549;',
	"\xA8\xC6" => '&#12550;',
	"\xA8\xC7" => '&#12551;',
	"\xA8\xC8" => '&#12552;',
	"\xA8\xC9" => '&#12553;',
	"\xA8\xCA" => '&#12554;',
	"\xA8\xCB" => '&#12555;',
	"\xA8\xCC" => '&#12556;',
	"\xA8\xCD" => '&#12557;',
	"\xA8\xCE" => '&#12558;',
	"\xA8\xCF" => '&#12559;',
	"\xA8\xD0" => '&#12560;',
	"\xA8\xD1" => '&#12561;',
	"\xA8\xD2" => '&#12562;',
	"\xA8\xD3" => '&#12563;',
	"\xA8\xD4" => '&#12564;',
	"\xA8\xD5" => '&#12565;',
	"\xA8\xD6" => '&#12566;',
	"\xA8\xD7" => '&#12567;',
	"\xA8\xD8" => '&#12568;',
	"\xA8\xD9" => '&#12569;',
	"\xA8\xDA" => '&#12570;',
	"\xA8\xDB" => '&#12571;',
	"\xA8\xDC" => '&#12572;',
	"\xA8\xDD" => '&#12573;',
	"\xA8\xDE" => '&#12574;',
	"\xA8\xDF" => '&#12575;',
	"\xA8\xE0" => '&#12576;',
	"\xA8\xE1" => '&#12577;',
	"\xA8\xE2" => '&#12578;',
	"\xA8\xE3" => '&#12579;',
	"\xA8\xE4" => '&#12580;',
	"\xA8\xE5" => '&#12581;',
	"\xA8\xE6" => '&#12582;',
	"\xA8\xE7" => '&#12583;',
	"\xA8\xE8" => '&#12584;',
	"\xA8\xE9" => '&#12585;',
	"\xA9\xA4" => '&#9472;',
	"\xA9\xA5" => '&#9473;',
	"\xA9\xA6" => '&#9474;',
	"\xA9\xA7" => '&#9475;',
	"\xA9\xA8" => '&#9476;',
	"\xA9\xA9" => '&#9477;',
	"\xA9\xAA" => '&#9478;',
	"\xA9\xAB" => '&#9479;',
	"\xA9\xAC" => '&#9480;',
	"\xA9\xAD" => '&#9481;',
	"\xA9\xAE" => '&#9482;',
	"\xA9\xAF" => '&#9483;',
	"\xA9\xB0" => '&#9484;',
	"\xA9\xB1" => '&#9485;',
	"\xA9\xB2" => '&#9486;',
	"\xA9\xB3" => '&#9487;',
	"\xA9\xB4" => '&#9488;',
	"\xA9\xB5" => '&#9489;',
	"\xA9\xB6" => '&#9490;',
	"\xA9\xB7" => '&#9491;',
	"\xA9\xB8" => '&#9492;',
	"\xA9\xB9" => '&#9493;',
	"\xA9\xBA" => '&#9494;',
	"\xA9\xBB" => '&#9495;',
	"\xA9\xBC" => '&#9496;',
	"\xA9\xBD" => '&#9497;',
	"\xA9\xBE" => '&#9498;',
	"\xA9\xBF" => '&#9499;',
	"\xA9\xC0" => '&#9500;',
	"\xA9\xC1" => '&#9501;',
	"\xA9\xC2" => '&#9502;',
	"\xA9\xC3" => '&#9503;',
	"\xA9\xC4" => '&#9504;',
	"\xA9\xC5" => '&#9505;',
	"\xA9\xC6" => '&#9506;',
	"\xA9\xC7" => '&#9507;',
	"\xA9\xC8" => '&#9508;',
	"\xA9\xC9" => '&#9509;',
	"\xA9\xCA" => '&#9510;',
	"\xA9\xCB" => '&#9511;',
	"\xA9\xCC" => '&#9512;',
	"\xA9\xCD" => '&#9513;',
	"\xA9\xCE" => '&#9514;',
	"\xA9\xCF" => '&#9515;',
	"\xA9\xD0" => '&#9516;',
	"\xA9\xD1" => '&#9517;',
	"\xA9\xD2" => '&#9518;',
	"\xA9\xD3" => '&#9519;',
	"\xA9\xD4" => '&#9520;',
	"\xA9\xD5" => '&#9521;',
	"\xA9\xD6" => '&#9522;',
	"\xA9\xD7" => '&#9523;',
	"\xA9\xD8" => '&#9524;',
	"\xA9\xD9" => '&#9525;',
	"\xA9\xDA" => '&#9526;',
	"\xA9\xDB" => '&#9527;',
	"\xA9\xDC" => '&#9528;',
	"\xA9\xDD" => '&#9529;',
	"\xA9\xDE" => '&#9530;',
	"\xA9\xDF" => '&#9531;',
	"\xA9\xE0" => '&#9532;',
	"\xA9\xE1" => '&#9533;',
	"\xA9\xE2" => '&#9534;',
	"\xA9\xE3" => '&#9535;',
	"\xA9\xE4" => '&#9536;',
	"\xA9\xE5" => '&#9537;',
	"\xA9\xE6" => '&#9538;',
	"\xA9\xE7" => '&#9539;',
	"\xA9\xE8" => '&#9540;',
	"\xA9\xE9" => '&#9541;',
	"\xA9\xEA" => '&#9542;',
	"\xA9\xEB" => '&#9543;',
	"\xA9\xEC" => '&#9544;',
	"\xA9\xED" => '&#9545;',
	"\xA9\xEE" => '&#9546;',
	"\xA9\xEF" => '&#9547;',
	"\xB0\xA1" => '&#21834;',
	"\xB0\xA2" => '&#38463;',
	"\xB0\xA3" => '&#22467;',
	"\xB0\xA4" => '&#25384;',
	"\xB0\xA5" => '&#21710;',
	"\xB0\xA6" => '&#21769;',
	"\xB0\xA7" => '&#21696;',
	"\xB0\xA8" => '&#30353;',
	"\xB0\xA9" => '&#30284;',
	"\xB0\xAA" => '&#34108;',
	"\xB0\xAB" => '&#30702;',
	"\xB0\xAC" => '&#33406;',
	"\xB0\xAD" => '&#30861;',
	"\xB0\xAE" => '&#29233;',
	"\xB0\xAF" => '&#38552;',
	"\xB0\xB0" => '&#38797;',
	"\xB0\xB1" => '&#27688;',
	"\xB0\xB2" => '&#23433;',
	"\xB0\xB3" => '&#20474;',
	"\xB0\xB4" => '&#25353;',
	"\xB0\xB5" => '&#26263;',
	"\xB0\xB6" => '&#23736;',
	"\xB0\xB7" => '&#33018;',
	"\xB0\xB8" => '&#26696;',
	"\xB0\xB9" => '&#32942;',
	"\xB0\xBA" => '&#26114;',
	"\xB0\xBB" => '&#30414;',
	"\xB0\xBC" => '&#20985;',
	"\xB0\xBD" => '&#25942;',
	"\xB0\xBE" => '&#29100;',
	"\xB0\xBF" => '&#32753;',
	"\xB0\xC0" => '&#34948;',
	"\xB0\xC1" => '&#20658;',
	"\xB0\xC2" => '&#22885;',
	"\xB0\xC3" => '&#25034;',
	"\xB0\xC4" => '&#28595;',
	"\xB0\xC5" => '&#33453;',
	"\xB0\xC6" => '&#25420;',
	"\xB0\xC7" => '&#25170;',
	"\xB0\xC8" => '&#21485;',
	"\xB0\xC9" => '&#21543;',
	"\xB0\xCA" => '&#31494;',
	"\xB0\xCB" => '&#20843;',
	"\xB0\xCC" => '&#30116;',
	"\xB0\xCD" => '&#24052;',
	"\xB0\xCE" => '&#25300;',
	"\xB0\xCF" => '&#36299;',
	"\xB0\xD0" => '&#38774;',
	"\xB0\xD1" => '&#25226;',
	"\xB0\xD2" => '&#32793;',
	"\xB0\xD3" => '&#22365;',
	"\xB0\xD4" => '&#38712;',
	"\xB0\xD5" => '&#32610;',
	"\xB0\xD6" => '&#29240;',
	"\xB0\xD7" => '&#30333;',
	"\xB0\xD8" => '&#26575;',
	"\xB0\xD9" => '&#30334;',
	"\xB0\xDA" => '&#25670;',
	"\xB0\xDB" => '&#20336;',
	"\xB0\xDC" => '&#36133;',
	"\xB0\xDD" => '&#25308;',
	"\xB0\xDE" => '&#31255;',
	"\xB0\xDF" => '&#26001;',
	"\xB0\xE0" => '&#29677;',
	"\xB0\xE1" => '&#25644;',
	"\xB0\xE2" => '&#25203;',
	"\xB0\xE3" => '&#33324;',
	"\xB0\xE4" => '&#39041;',
	"\xB0\xE5" => '&#26495;',
	"\xB0\xE6" => '&#29256;',
	"\xB0\xE7" => '&#25198;',
	"\xB0\xE8" => '&#25292;',
	"\xB0\xE9" => '&#20276;',
	"\xB0\xEA" => '&#29923;',
	"\xB0\xEB" => '&#21322;',
	"\xB0\xEC" => '&#21150;',
	"\xB0\xED" => '&#32458;',
	"\xB0\xEE" => '&#37030;',
	"\xB0\xEF" => '&#24110;',
	"\xB0\xF0" => '&#26758;',
	"\xB0\xF1" => '&#27036;',
	"\xB0\xF2" => '&#33152;',
	"\xB0\xF3" => '&#32465;',
	"\xB0\xF4" => '&#26834;',
	"\xB0\xF5" => '&#30917;',
	"\xB0\xF6" => '&#34444;',
	"\xB0\xF7" => '&#38225;',
	"\xB0\xF8" => '&#20621;',
	"\xB0\xF9" => '&#35876;',
	"\xB0\xFA" => '&#33502;',
	"\xB0\xFB" => '&#32990;',
	"\xB0\xFC" => '&#21253;',
	"\xB0\xFD" => '&#35090;',
	"\xB0\xFE" => '&#21093;',
	"\xB1\xA1" => '&#34180;',
	"\xB1\xA2" => '&#38649;',
	"\xB1\xA3" => '&#20445;',
	"\xB1\xA4" => '&#22561;',
	"\xB1\xA5" => '&#39281;',
	"\xB1\xA6" => '&#23453;',
	"\xB1\xA7" => '&#25265;',
	"\xB1\xA8" => '&#25253;',
	"\xB1\xA9" => '&#26292;',
	"\xB1\xAA" => '&#35961;',
	"\xB1\xAB" => '&#40077;',
	"\xB1\xAC" => '&#29190;',
	"\xB1\xAD" => '&#26479;',
	"\xB1\xAE" => '&#30865;',
	"\xB1\xAF" => '&#24754;',
	"\xB1\xB0" => '&#21329;',
	"\xB1\xB1" => '&#21271;',
	"\xB1\xB2" => '&#36744;',
	"\xB1\xB3" => '&#32972;',
	"\xB1\xB4" => '&#36125;',
	"\xB1\xB5" => '&#38049;',
	"\xB1\xB6" => '&#20493;',
	"\xB1\xB7" => '&#29384;',
	"\xB1\xB8" => '&#22791;',
	"\xB1\xB9" => '&#24811;',
	"\xB1\xBA" => '&#28953;',
	"\xB1\xBB" => '&#34987;',
	"\xB1\xBC" => '&#22868;',
	"\xB1\xBD" => '&#33519;',
	"\xB1\xBE" => '&#26412;',
	"\xB1\xBF" => '&#31528;',
	"\xB1\xC0" => '&#23849;',
	"\xB1\xC1" => '&#32503;',
	"\xB1\xC2" => '&#29997;',
	"\xB1\xC3" => '&#27893;',
	"\xB1\xC4" => '&#36454;',
	"\xB1\xC5" => '&#36856;',
	"\xB1\xC6" => '&#36924;',
	"\xB1\xC7" => '&#40763;',
	"\xB1\xC8" => '&#27604;',
	"\xB1\xC9" => '&#37145;',
	"\xB1\xCA" => '&#31508;',
	"\xB1\xCB" => '&#24444;',
	"\xB1\xCC" => '&#30887;',
	"\xB1\xCD" => '&#34006;',
	"\xB1\xCE" => '&#34109;',
	"\xB1\xCF" => '&#27605;',
	"\xB1\xD0" => '&#27609;',
	"\xB1\xD1" => '&#27606;',
	"\xB1\xD2" => '&#24065;',
	"\xB1\xD3" => '&#24199;',
	"\xB1\xD4" => '&#30201;',
	"\xB1\xD5" => '&#38381;',
	"\xB1\xD6" => '&#25949;',
	"\xB1\xD7" => '&#24330;',
	"\xB1\xD8" => '&#24517;',
	"\xB1\xD9" => '&#36767;',
	"\xB1\xDA" => '&#22721;',
	"\xB1\xDB" => '&#33218;',
	"\xB1\xDC" => '&#36991;',
	"\xB1\xDD" => '&#38491;',
	"\xB1\xDE" => '&#38829;',
	"\xB1\xDF" => '&#36793;',
	"\xB1\xE0" => '&#32534;',
	"\xB1\xE1" => '&#36140;',
	"\xB1\xE2" => '&#25153;',
	"\xB1\xE3" => '&#20415;',
	"\xB1\xE4" => '&#21464;',
	"\xB1\xE5" => '&#21342;',
	"\xB1\xE6" => '&#36776;',
	"\xB1\xE7" => '&#36777;',
	"\xB1\xE8" => '&#36779;',
	"\xB1\xE9" => '&#36941;',
	"\xB1\xEA" => '&#26631;',
	"\xB1\xEB" => '&#24426;',
	"\xB1\xEC" => '&#33176;',
	"\xB1\xED" => '&#34920;',
	"\xB1\xEE" => '&#40150;',
	"\xB1\xEF" => '&#24971;',
	"\xB1\xF0" => '&#21035;',
	"\xB1\xF1" => '&#30250;',
	"\xB1\xF2" => '&#24428;',
	"\xB1\xF3" => '&#25996;',
	"\xB1\xF4" => '&#28626;',
	"\xB1\xF5" => '&#28392;',
	"\xB1\xF6" => '&#23486;',
	"\xB1\xF7" => '&#25672;',
	"\xB1\xF8" => '&#20853;',
	"\xB1\xF9" => '&#20912;',
	"\xB1\xFA" => '&#26564;',
	"\xB1\xFB" => '&#19993;',
	"\xB1\xFC" => '&#31177;',
	"\xB1\xFD" => '&#39292;',
	"\xB1\xFE" => '&#28851;',
	"\xB2\xA1" => '&#30149;',
	"\xB2\xA2" => '&#24182;',
	"\xB2\xA3" => '&#29627;',
	"\xB2\xA4" => '&#33760;',
	"\xB2\xA5" => '&#25773;',
	"\xB2\xA6" => '&#25320;',
	"\xB2\xA7" => '&#38069;',
	"\xB2\xA8" => '&#27874;',
	"\xB2\xA9" => '&#21338;',
	"\xB2\xAA" => '&#21187;',
	"\xB2\xAB" => '&#25615;',
	"\xB2\xAC" => '&#38082;',
	"\xB2\xAD" => '&#31636;',
	"\xB2\xAE" => '&#20271;',
	"\xB2\xAF" => '&#24091;',
	"\xB2\xB0" => '&#33334;',
	"\xB2\xB1" => '&#33046;',
	"\xB2\xB2" => '&#33162;',
	"\xB2\xB3" => '&#28196;',
	"\xB2\xB4" => '&#27850;',
	"\xB2\xB5" => '&#39539;',
	"\xB2\xB6" => '&#25429;',
	"\xB2\xB7" => '&#21340;',
	"\xB2\xB8" => '&#21754;',
	"\xB2\xB9" => '&#34917;',
	"\xB2\xBA" => '&#22496;',
	"\xB2\xBB" => '&#19981;',
	"\xB2\xBC" => '&#24067;',
	"\xB2\xBD" => '&#27493;',
	"\xB2\xBE" => '&#31807;',
	"\xB2\xBF" => '&#37096;',
	"\xB2\xC0" => '&#24598;',
	"\xB2\xC1" => '&#25830;',
	"\xB2\xC2" => '&#29468;',
	"\xB2\xC3" => '&#35009;',
	"\xB2\xC4" => '&#26448;',
	"\xB2\xC5" => '&#25165;',
	"\xB2\xC6" => '&#36130;',
	"\xB2\xC7" => '&#30572;',
	"\xB2\xC8" => '&#36393;',
	"\xB2\xC9" => '&#37319;',
	"\xB2\xCA" => '&#24425;',
	"\xB2\xCB" => '&#33756;',
	"\xB2\xCC" => '&#34081;',
	"\xB2\xCD" => '&#39184;',
	"\xB2\xCE" => '&#21442;',
	"\xB2\xCF" => '&#34453;',
	"\xB2\xD0" => '&#27531;',
	"\xB2\xD1" => '&#24813;',
	"\xB2\xD2" => '&#24808;',
	"\xB2\xD3" => '&#28799;',
	"\xB2\xD4" => '&#33485;',
	"\xB2\xD5" => '&#33329;',
	"\xB2\xD6" => '&#20179;',
	"\xB2\xD7" => '&#27815;',
	"\xB2\xD8" => '&#34255;',
	"\xB2\xD9" => '&#25805;',
	"\xB2\xDA" => '&#31961;',
	"\xB2\xDB" => '&#27133;',
	"\xB2\xDC" => '&#26361;',
	"\xB2\xDD" => '&#33609;',
	"\xB2\xDE" => '&#21397;',
	"\xB2\xDF" => '&#31574;',
	"\xB2\xE0" => '&#20391;',
	"\xB2\xE1" => '&#20876;',
	"\xB2\xE2" => '&#27979;',
	"\xB2\xE3" => '&#23618;',
	"\xB2\xE4" => '&#36461;',
	"\xB2\xE5" => '&#25554;',
	"\xB2\xE6" => '&#21449;',
	"\xB2\xE7" => '&#33580;',
	"\xB2\xE8" => '&#33590;',
	"\xB2\xE9" => '&#26597;',
	"\xB2\xEA" => '&#30900;',
	"\xB2\xEB" => '&#25661;',
	"\xB2\xEC" => '&#23519;',
	"\xB2\xED" => '&#23700;',
	"\xB2\xEE" => '&#24046;',
	"\xB2\xEF" => '&#35815;',
	"\xB2\xF0" => '&#25286;',
	"\xB2\xF1" => '&#26612;',
	"\xB2\xF2" => '&#35962;',
	"\xB2\xF3" => '&#25600;',
	"\xB2\xF4" => '&#25530;',
	"\xB2\xF5" => '&#34633;',
	"\xB2\xF6" => '&#39307;',
	"\xB2\xF7" => '&#35863;',
	"\xB2\xF8" => '&#32544;',
	"\xB2\xF9" => '&#38130;',
	"\xB2\xFA" => '&#20135;',
	"\xB2\xFB" => '&#38416;',
	"\xB2\xFC" => '&#39076;',
	"\xB2\xFD" => '&#26124;',
	"\xB2\xFE" => '&#29462;',
	"\xB3\xA1" => '&#22330;',
	"\xB3\xA2" => '&#23581;',
	"\xB3\xA3" => '&#24120;',
	"\xB3\xA4" => '&#38271;',
	"\xB3\xA5" => '&#20607;',
	"\xB3\xA6" => '&#32928;',
	"\xB3\xA7" => '&#21378;',
	"\xB3\xA8" => '&#25950;',
	"\xB3\xA9" => '&#30021;',
	"\xB3\xAA" => '&#21809;',
	"\xB3\xAB" => '&#20513;',
	"\xB3\xAC" => '&#36229;',
	"\xB3\xAD" => '&#25220;',
	"\xB3\xAE" => '&#38046;',
	"\xB3\xAF" => '&#26397;',
	"\xB3\xB0" => '&#22066;',
	"\xB3\xB1" => '&#28526;',
	"\xB3\xB2" => '&#24034;',
	"\xB3\xB3" => '&#21557;',
	"\xB3\xB4" => '&#28818;',
	"\xB3\xB5" => '&#36710;',
	"\xB3\xB6" => '&#25199;',
	"\xB3\xB7" => '&#25764;',
	"\xB3\xB8" => '&#25507;',
	"\xB3\xB9" => '&#24443;',
	"\xB3\xBA" => '&#28552;',
	"\xB3\xBB" => '&#37108;',
	"\xB3\xBC" => '&#33251;',
	"\xB3\xBD" => '&#36784;',
	"\xB3\xBE" => '&#23576;',
	"\xB3\xBF" => '&#26216;',
	"\xB3\xC0" => '&#24561;',
	"\xB3\xC1" => '&#27785;',
	"\xB3\xC2" => '&#38472;',
	"\xB3\xC3" => '&#36225;',
	"\xB3\xC4" => '&#34924;',
	"\xB3\xC5" => '&#25745;',
	"\xB3\xC6" => '&#31216;',
	"\xB3\xC7" => '&#22478;',
	"\xB3\xC8" => '&#27225;',
	"\xB3\xC9" => '&#25104;',
	"\xB3\xCA" => '&#21576;',
	"\xB3\xCB" => '&#20056;',
	"\xB3\xCC" => '&#31243;',
	"\xB3\xCD" => '&#24809;',
	"\xB3\xCE" => '&#28548;',
	"\xB3\xCF" => '&#35802;',
	"\xB3\xD0" => '&#25215;',
	"\xB3\xD1" => '&#36894;',
	"\xB3\xD2" => '&#39563;',
	"\xB3\xD3" => '&#31204;',
	"\xB3\xD4" => '&#21507;',
	"\xB3\xD5" => '&#30196;',
	"\xB3\xD6" => '&#25345;',
	"\xB3\xD7" => '&#21273;',
	"\xB3\xD8" => '&#27744;',
	"\xB3\xD9" => '&#36831;',
	"\xB3\xDA" => '&#24347;',
	"\xB3\xDB" => '&#39536;',
	"\xB3\xDC" => '&#32827;',
	"\xB3\xDD" => '&#40831;',
	"\xB3\xDE" => '&#20360;',
	"\xB3\xDF" => '&#23610;',
	"\xB3\xE0" => '&#36196;',
	"\xB3\xE1" => '&#32709;',
	"\xB3\xE2" => '&#26021;',
	"\xB3\xE3" => '&#28861;',
	"\xB3\xE4" => '&#20805;',
	"\xB3\xE5" => '&#20914;',
	"\xB3\xE6" => '&#34411;',
	"\xB3\xE7" => '&#23815;',
	"\xB3\xE8" => '&#23456;',
	"\xB3\xE9" => '&#25277;',
	"\xB3\xEA" => '&#37228;',
	"\xB3\xEB" => '&#30068;',
	"\xB3\xEC" => '&#36364;',
	"\xB3\xED" => '&#31264;',
	"\xB3\xEE" => '&#24833;',
	"\xB3\xEF" => '&#31609;',
	"\xB3\xF0" => '&#20167;',
	"\xB3\xF1" => '&#32504;',
	"\xB3\xF2" => '&#30597;',
	"\xB3\xF3" => '&#19985;',
	"\xB3\xF4" => '&#33261;',
	"\xB3\xF5" => '&#21021;',
	"\xB3\xF6" => '&#20986;',
	"\xB3\xF7" => '&#27249;',
	"\xB3\xF8" => '&#21416;',
	"\xB3\xF9" => '&#36487;',
	"\xB3\xFA" => '&#38148;',
	"\xB3\xFB" => '&#38607;',
	"\xB3\xFC" => '&#28353;',
	"\xB3\xFD" => '&#38500;',
	"\xB3\xFE" => '&#26970;',
	"\xB4\xA1" => '&#30784;',
	"\xB4\xA2" => '&#20648;',
	"\xB4\xA3" => '&#30679;',
	"\xB4\xA4" => '&#25616;',
	"\xB4\xA5" => '&#35302;',
	"\xB4\xA6" => '&#22788;',
	"\xB4\xA7" => '&#25571;',
	"\xB4\xA8" => '&#24029;',
	"\xB4\xA9" => '&#31359;',
	"\xB4\xAA" => '&#26941;',
	"\xB4\xAB" => '&#20256;',
	"\xB4\xAC" => '&#33337;',
	"\xB4\xAD" => '&#21912;',
	"\xB4\xAE" => '&#20018;',
	"\xB4\xAF" => '&#30126;',
	"\xB4\xB0" => '&#31383;',
	"\xB4\xB1" => '&#24162;',
	"\xB4\xB2" => '&#24202;',
	"\xB4\xB3" => '&#38383;',
	"\xB4\xB4" => '&#21019;',
	"\xB4\xB5" => '&#21561;',
	"\xB4\xB6" => '&#28810;',
	"\xB4\xB7" => '&#25462;',
	"\xB4\xB8" => '&#38180;',
	"\xB4\xB9" => '&#22402;',
	"\xB4\xBA" => '&#26149;',
	"\xB4\xBB" => '&#26943;',
	"\xB4\xBC" => '&#37255;',
	"\xB4\xBD" => '&#21767;',
	"\xB4\xBE" => '&#28147;',
	"\xB4\xBF" => '&#32431;',
	"\xB4\xC0" => '&#34850;',
	"\xB4\xC1" => '&#25139;',
	"\xB4\xC2" => '&#32496;',
	"\xB4\xC3" => '&#30133;',
	"\xB4\xC4" => '&#33576;',
	"\xB4\xC5" => '&#30913;',
	"\xB4\xC6" => '&#38604;',
	"\xB4\xC7" => '&#36766;',
	"\xB4\xC8" => '&#24904;',
	"\xB4\xC9" => '&#29943;',
	"\xB4\xCA" => '&#35789;',
	"\xB4\xCB" => '&#27492;',
	"\xB4\xCC" => '&#21050;',
	"\xB4\xCD" => '&#36176;',
	"\xB4\xCE" => '&#27425;',
	"\xB4\xCF" => '&#32874;',
	"\xB4\xD0" => '&#33905;',
	"\xB4\xD1" => '&#22257;',
	"\xB4\xD2" => '&#21254;',
	"\xB4\xD3" => '&#20174;',
	"\xB4\xD4" => '&#19995;',
	"\xB4\xD5" => '&#20945;',
	"\xB4\xD6" => '&#31895;',
	"\xB4\xD7" => '&#37259;',
	"\xB4\xD8" => '&#31751;',
	"\xB4\xD9" => '&#20419;',
	"\xB4\xDA" => '&#36479;',
	"\xB4\xDB" => '&#31713;',
	"\xB4\xDC" => '&#31388;',
	"\xB4\xDD" => '&#25703;',
	"\xB4\xDE" => '&#23828;',
	"\xB4\xDF" => '&#20652;',
	"\xB4\xE0" => '&#33030;',
	"\xB4\xE1" => '&#30209;',
	"\xB4\xE2" => '&#31929;',
	"\xB4\xE3" => '&#28140;',
	"\xB4\xE4" => '&#32736;',
	"\xB4\xE5" => '&#26449;',
	"\xB4\xE6" => '&#23384;',
	"\xB4\xE7" => '&#23544;',
	"\xB4\xE8" => '&#30923;',
	"\xB4\xE9" => '&#25774;',
	"\xB4\xEA" => '&#25619;',
	"\xB4\xEB" => '&#25514;',
	"\xB4\xEC" => '&#25387;',
	"\xB4\xED" => '&#38169;',
	"\xB4\xEE" => '&#25645;',
	"\xB4\xEF" => '&#36798;',
	"\xB4\xF0" => '&#31572;',
	"\xB4\xF1" => '&#30249;',
	"\xB4\xF2" => '&#25171;',
	"\xB4\xF3" => '&#22823;',
	"\xB4\xF4" => '&#21574;',
	"\xB4\xF5" => '&#27513;',
	"\xB4\xF6" => '&#20643;',
	"\xB4\xF7" => '&#25140;',
	"\xB4\xF8" => '&#24102;',
	"\xB4\xF9" => '&#27526;',
	"\xB4\xFA" => '&#20195;',
	"\xB4\xFB" => '&#36151;',
	"\xB4\xFC" => '&#34955;',
	"\xB4\xFD" => '&#24453;',
	"\xB4\xFE" => '&#36910;',
	"\xB5\xA1" => '&#24608;',
	"\xB5\xA2" => '&#32829;',
	"\xB5\xA3" => '&#25285;',
	"\xB5\xA4" => '&#20025;',
	"\xB5\xA5" => '&#21333;',
	"\xB5\xA6" => '&#37112;',
	"\xB5\xA7" => '&#25528;',
	"\xB5\xA8" => '&#32966;',
	"\xB5\xA9" => '&#26086;',
	"\xB5\xAA" => '&#27694;',
	"\xB5\xAB" => '&#20294;',
	"\xB5\xAC" => '&#24814;',
	"\xB5\xAD" => '&#28129;',
	"\xB5\xAE" => '&#35806;',
	"\xB5\xAF" => '&#24377;',
	"\xB5\xB0" => '&#34507;',
	"\xB5\xB1" => '&#24403;',
	"\xB5\xB2" => '&#25377;',
	"\xB5\xB3" => '&#20826;',
	"\xB5\xB4" => '&#33633;',
	"\xB5\xB5" => '&#26723;',
	"\xB5\xB6" => '&#20992;',
	"\xB5\xB7" => '&#25443;',
	"\xB5\xB8" => '&#36424;',
	"\xB5\xB9" => '&#20498;',
	"\xB5\xBA" => '&#23707;',
	"\xB5\xBB" => '&#31095;',
	"\xB5\xBC" => '&#23548;',
	"\xB5\xBD" => '&#21040;',
	"\xB5\xBE" => '&#31291;',
	"\xB5\xBF" => '&#24764;',
	"\xB5\xC0" => '&#36947;',
	"\xB5\xC1" => '&#30423;',
	"\xB5\xC2" => '&#24503;',
	"\xB5\xC3" => '&#24471;',
	"\xB5\xC4" => '&#30340;',
	"\xB5\xC5" => '&#36460;',
	"\xB5\xC6" => '&#28783;',
	"\xB5\xC7" => '&#30331;',
	"\xB5\xC8" => '&#31561;',
	"\xB5\xC9" => '&#30634;',
	"\xB5\xCA" => '&#20979;',
	"\xB5\xCB" => '&#37011;',
	"\xB5\xCC" => '&#22564;',
	"\xB5\xCD" => '&#20302;',
	"\xB5\xCE" => '&#28404;',
	"\xB5\xCF" => '&#36842;',
	"\xB5\xD0" => '&#25932;',
	"\xB5\xD1" => '&#31515;',
	"\xB5\xD2" => '&#29380;',
	"\xB5\xD3" => '&#28068;',
	"\xB5\xD4" => '&#32735;',
	"\xB5\xD5" => '&#23265;',
	"\xB5\xD6" => '&#25269;',
	"\xB5\xD7" => '&#24213;',
	"\xB5\xD8" => '&#22320;',
	"\xB5\xD9" => '&#33922;',
	"\xB5\xDA" => '&#31532;',
	"\xB5\xDB" => '&#24093;',
	"\xB5\xDC" => '&#24351;',
	"\xB5\xDD" => '&#36882;',
	"\xB5\xDE" => '&#32532;',
	"\xB5\xDF" => '&#39072;',
	"\xB5\xE0" => '&#25474;',
	"\xB5\xE1" => '&#28359;',
	"\xB5\xE2" => '&#30872;',
	"\xB5\xE3" => '&#28857;',
	"\xB5\xE4" => '&#20856;',
	"\xB5\xE5" => '&#38747;',
	"\xB5\xE6" => '&#22443;',
	"\xB5\xE7" => '&#30005;',
	"\xB5\xE8" => '&#20291;',
	"\xB5\xE9" => '&#30008;',
	"\xB5\xEA" => '&#24215;',
	"\xB5\xEB" => '&#24806;',
	"\xB5\xEC" => '&#22880;',
	"\xB5\xED" => '&#28096;',
	"\xB5\xEE" => '&#27583;',
	"\xB5\xEF" => '&#30857;',
	"\xB5\xF0" => '&#21500;',
	"\xB5\xF1" => '&#38613;',
	"\xB5\xF2" => '&#20939;',
	"\xB5\xF3" => '&#20993;',
	"\xB5\xF4" => '&#25481;',
	"\xB5\xF5" => '&#21514;',
	"\xB5\xF6" => '&#38035;',
	"\xB5\xF7" => '&#35843;',
	"\xB5\xF8" => '&#36300;',
	"\xB5\xF9" => '&#29241;',
	"\xB5\xFA" => '&#30879;',
	"\xB5\xFB" => '&#34678;',
	"\xB5\xFC" => '&#36845;',
	"\xB5\xFD" => '&#35853;',
	"\xB5\xFE" => '&#21472;',
	"\xB6\xA1" => '&#19969;',
	"\xB6\xA2" => '&#30447;',
	"\xB6\xA3" => '&#21486;',
	"\xB6\xA4" => '&#38025;',
	"\xB6\xA5" => '&#39030;',
	"\xB6\xA6" => '&#40718;',
	"\xB6\xA7" => '&#38189;',
	"\xB6\xA8" => '&#23450;',
	"\xB6\xA9" => '&#35746;',
	"\xB6\xAA" => '&#20002;',
	"\xB6\xAB" => '&#19996;',
	"\xB6\xAC" => '&#20908;',
	"\xB6\xAD" => '&#33891;',
	"\xB6\xAE" => '&#25026;',
	"\xB6\xAF" => '&#21160;',
	"\xB6\xB0" => '&#26635;',
	"\xB6\xB1" => '&#20375;',
	"\xB6\xB2" => '&#24683;',
	"\xB6\xB3" => '&#20923;',
	"\xB6\xB4" => '&#27934;',
	"\xB6\xB5" => '&#20828;',
	"\xB6\xB6" => '&#25238;',
	"\xB6\xB7" => '&#26007;',
	"\xB6\xB8" => '&#38497;',
	"\xB6\xB9" => '&#35910;',
	"\xB6\xBA" => '&#36887;',
	"\xB6\xBB" => '&#30168;',
	"\xB6\xBC" => '&#37117;',
	"\xB6\xBD" => '&#30563;',
	"\xB6\xBE" => '&#27602;',
	"\xB6\xBF" => '&#29322;',
	"\xB6\xC0" => '&#29420;',
	"\xB6\xC1" => '&#35835;',
	"\xB6\xC2" => '&#22581;',
	"\xB6\xC3" => '&#30585;',
	"\xB6\xC4" => '&#36172;',
	"\xB6\xC5" => '&#26460;',
	"\xB6\xC6" => '&#38208;',
	"\xB6\xC7" => '&#32922;',
	"\xB6\xC8" => '&#24230;',
	"\xB6\xC9" => '&#28193;',
	"\xB6\xCA" => '&#22930;',
	"\xB6\xCB" => '&#31471;',
	"\xB6\xCC" => '&#30701;',
	"\xB6\xCD" => '&#38203;',
	"\xB6\xCE" => '&#27573;',
	"\xB6\xCF" => '&#26029;',
	"\xB6\xD0" => '&#32526;',
	"\xB6\xD1" => '&#22534;',
	"\xB6\xD2" => '&#20817;',
	"\xB6\xD3" => '&#38431;',
	"\xB6\xD4" => '&#23545;',
	"\xB6\xD5" => '&#22697;',
	"\xB6\xD6" => '&#21544;',
	"\xB6\xD7" => '&#36466;',
	"\xB6\xD8" => '&#25958;',
	"\xB6\xD9" => '&#39039;',
	"\xB6\xDA" => '&#22244;',
	"\xB6\xDB" => '&#38045;',
	"\xB6\xDC" => '&#30462;',
	"\xB6\xDD" => '&#36929;',
	"\xB6\xDE" => '&#25479;',
	"\xB6\xDF" => '&#21702;',
	"\xB6\xE0" => '&#22810;',
	"\xB6\xE1" => '&#22842;',
	"\xB6\xE2" => '&#22427;',
	"\xB6\xE3" => '&#36530;',
	"\xB6\xE4" => '&#26421;',
	"\xB6\xE5" => '&#36346;',
	"\xB6\xE6" => '&#33333;',
	"\xB6\xE7" => '&#21057;',
	"\xB6\xE8" => '&#24816;',
	"\xB6\xE9" => '&#22549;',
	"\xB6\xEA" => '&#34558;',
	"\xB6\xEB" => '&#23784;',
	"\xB6\xEC" => '&#40517;',
	"\xB6\xED" => '&#20420;',
	"\xB6\xEE" => '&#39069;',
	"\xB6\xEF" => '&#35769;',
	"\xB6\xF0" => '&#23077;',
	"\xB6\xF1" => '&#24694;',
	"\xB6\xF2" => '&#21380;',
	"\xB6\xF3" => '&#25212;',
	"\xB6\xF4" => '&#36943;',
	"\xB6\xF5" => '&#37122;',
	"\xB6\xF6" => '&#39295;',
	"\xB6\xF7" => '&#24681;',
	"\xB6\xF8" => '&#32780;',
	"\xB6\xF9" => '&#20799;',
	"\xB6\xFA" => '&#32819;',
	"\xB6\xFB" => '&#23572;',
	"\xB6\xFC" => '&#39285;',
	"\xB6\xFD" => '&#27953;',
	"\xB6\xFE" => '&#20108;',
	"\xB7\xA1" => '&#36144;',
	"\xB7\xA2" => '&#21457;',
	"\xB7\xA3" => '&#32602;',
	"\xB7\xA4" => '&#31567;',
	"\xB7\xA5" => '&#20240;',
	"\xB7\xA6" => '&#20047;',
	"\xB7\xA7" => '&#38400;',
	"\xB7\xA8" => '&#27861;',
	"\xB7\xA9" => '&#29648;',
	"\xB7\xAA" => '&#34281;',
	"\xB7\xAB" => '&#24070;',
	"\xB7\xAC" => '&#30058;',
	"\xB7\xAD" => '&#32763;',
	"\xB7\xAE" => '&#27146;',
	"\xB7\xAF" => '&#30718;',
	"\xB7\xB0" => '&#38034;',
	"\xB7\xB1" => '&#32321;',
	"\xB7\xB2" => '&#20961;',
	"\xB7\xB3" => '&#28902;',
	"\xB7\xB4" => '&#21453;',
	"\xB7\xB5" => '&#36820;',
	"\xB7\xB6" => '&#33539;',
	"\xB7\xB7" => '&#36137;',
	"\xB7\xB8" => '&#29359;',
	"\xB7\xB9" => '&#39277;',
	"\xB7\xBA" => '&#27867;',
	"\xB7\xBB" => '&#22346;',
	"\xB7\xBC" => '&#33459;',
	"\xB7\xBD" => '&#26041;',
	"\xB7\xBE" => '&#32938;',
	"\xB7\xBF" => '&#25151;',
	"\xB7\xC0" => '&#38450;',
	"\xB7\xC1" => '&#22952;',
	"\xB7\xC2" => '&#20223;',
	"\xB7\xC3" => '&#35775;',
	"\xB7\xC4" => '&#32442;',
	"\xB7\xC5" => '&#25918;',
	"\xB7\xC6" => '&#33778;',
	"\xB7\xC7" => '&#38750;',
	"\xB7\xC8" => '&#21857;',
	"\xB7\xC9" => '&#39134;',
	"\xB7\xCA" => '&#32933;',
	"\xB7\xCB" => '&#21290;',
	"\xB7\xCC" => '&#35837;',
	"\xB7\xCD" => '&#21536;',
	"\xB7\xCE" => '&#32954;',
	"\xB7\xCF" => '&#24223;',
	"\xB7\xD0" => '&#27832;',
	"\xB7\xD1" => '&#36153;',
	"\xB7\xD2" => '&#33452;',
	"\xB7\xD3" => '&#37210;',
	"\xB7\xD4" => '&#21545;',
	"\xB7\xD5" => '&#27675;',
	"\xB7\xD6" => '&#20998;',
	"\xB7\xD7" => '&#32439;',
	"\xB7\xD8" => '&#22367;',
	"\xB7\xD9" => '&#28954;',
	"\xB7\xDA" => '&#27774;',
	"\xB7\xDB" => '&#31881;',
	"\xB7\xDC" => '&#22859;',
	"\xB7\xDD" => '&#20221;',
	"\xB7\xDE" => '&#24575;',
	"\xB7\xDF" => '&#24868;',
	"\xB7\xE0" => '&#31914;',
	"\xB7\xE1" => '&#20016;',
	"\xB7\xE2" => '&#23553;',
	"\xB7\xE3" => '&#26539;',
	"\xB7\xE4" => '&#34562;',
	"\xB7\xE5" => '&#23792;',
	"\xB7\xE6" => '&#38155;',
	"\xB7\xE7" => '&#39118;',
	"\xB7\xE8" => '&#30127;',
	"\xB7\xE9" => '&#28925;',
	"\xB7\xEA" => '&#36898;',
	"\xB7\xEB" => '&#20911;',
	"\xB7\xEC" => '&#32541;',
	"\xB7\xED" => '&#35773;',
	"\xB7\xEE" => '&#22857;',
	"\xB7\xEF" => '&#20964;',
	"\xB7\xF0" => '&#20315;',
	"\xB7\xF1" => '&#21542;',
	"\xB7\xF2" => '&#22827;',
	"\xB7\xF3" => '&#25975;',
	"\xB7\xF4" => '&#32932;',
	"\xB7\xF5" => '&#23413;',
	"\xB7\xF6" => '&#25206;',
	"\xB7\xF7" => '&#25282;',
	"\xB7\xF8" => '&#36752;',
	"\xB7\xF9" => '&#24133;',
	"\xB7\xFA" => '&#27679;',
	"\xB7\xFB" => '&#31526;',
	"\xB7\xFC" => '&#20239;',
	"\xB7\xFD" => '&#20440;',
	"\xB7\xFE" => '&#26381;',
	"\xB8\xA1" => '&#28014;',
	"\xB8\xA2" => '&#28074;',
	"\xB8\xA3" => '&#31119;',
	"\xB8\xA4" => '&#34993;',
	"\xB8\xA5" => '&#24343;',
	"\xB8\xA6" => '&#29995;',
	"\xB8\xA7" => '&#25242;',
	"\xB8\xA8" => '&#36741;',
	"\xB8\xA9" => '&#20463;',
	"\xB8\xAA" => '&#37340;',
	"\xB8\xAB" => '&#26023;',
	"\xB8\xAC" => '&#33071;',
	"\xB8\xAD" => '&#33105;',
	"\xB8\xAE" => '&#24220;',
	"\xB8\xAF" => '&#33104;',
	"\xB8\xB0" => '&#36212;',
	"\xB8\xB1" => '&#21103;',
	"\xB8\xB2" => '&#35206;',
	"\xB8\xB3" => '&#36171;',
	"\xB8\xB4" => '&#22797;',
	"\xB8\xB5" => '&#20613;',
	"\xB8\xB6" => '&#20184;',
	"\xB8\xB7" => '&#38428;',
	"\xB8\xB8" => '&#29238;',
	"\xB8\xB9" => '&#33145;',
	"\xB8\xBA" => '&#36127;',
	"\xB8\xBB" => '&#23500;',
	"\xB8\xBC" => '&#35747;',
	"\xB8\xBD" => '&#38468;',
	"\xB8\xBE" => '&#22919;',
	"\xB8\xBF" => '&#32538;',
	"\xB8\xC0" => '&#21648;',
	"\xB8\xC1" => '&#22134;',
	"\xB8\xC2" => '&#22030;',
	"\xB8\xC3" => '&#35813;',
	"\xB8\xC4" => '&#25913;',
	"\xB8\xC5" => '&#27010;',
	"\xB8\xC6" => '&#38041;',
	"\xB8\xC7" => '&#30422;',
	"\xB8\xC8" => '&#28297;',
	"\xB8\xC9" => '&#24178;',
	"\xB8\xCA" => '&#29976;',
	"\xB8\xCB" => '&#26438;',
	"\xB8\xCC" => '&#26577;',
	"\xB8\xCD" => '&#31487;',
	"\xB8\xCE" => '&#32925;',
	"\xB8\xCF" => '&#36214;',
	"\xB8\xD0" => '&#24863;',
	"\xB8\xD1" => '&#31174;',
	"\xB8\xD2" => '&#25954;',
	"\xB8\xD3" => '&#36195;',
	"\xB8\xD4" => '&#20872;',
	"\xB8\xD5" => '&#21018;',
	"\xB8\xD6" => '&#38050;',
	"\xB8\xD7" => '&#32568;',
	"\xB8\xD8" => '&#32923;',
	"\xB8\xD9" => '&#32434;',
	"\xB8\xDA" => '&#23703;',
	"\xB8\xDB" => '&#28207;',
	"\xB8\xDC" => '&#26464;',
	"\xB8\xDD" => '&#31705;',
	"\xB8\xDE" => '&#30347;',
	"\xB8\xDF" => '&#39640;',
	"\xB8\xE0" => '&#33167;',
	"\xB8\xE1" => '&#32660;',
	"\xB8\xE2" => '&#31957;',
	"\xB8\xE3" => '&#25630;',
	"\xB8\xE4" => '&#38224;',
	"\xB8\xE5" => '&#31295;',
	"\xB8\xE6" => '&#21578;',
	"\xB8\xE7" => '&#21733;',
	"\xB8\xE8" => '&#27468;',
	"\xB8\xE9" => '&#25601;',
	"\xB8\xEA" => '&#25096;',
	"\xB8\xEB" => '&#40509;',
	"\xB8\xEC" => '&#33011;',
	"\xB8\xED" => '&#30105;',
	"\xB8\xEE" => '&#21106;',
	"\xB8\xEF" => '&#38761;',
	"\xB8\xF0" => '&#33883;',
	"\xB8\xF1" => '&#26684;',
	"\xB8\xF2" => '&#34532;',
	"\xB8\xF3" => '&#38401;',
	"\xB8\xF4" => '&#38548;',
	"\xB8\xF5" => '&#38124;',
	"\xB8\xF6" => '&#20010;',
	"\xB8\xF7" => '&#21508;',
	"\xB8\xF8" => '&#32473;',
	"\xB8\xF9" => '&#26681;',
	"\xB8\xFA" => '&#36319;',
	"\xB8\xFB" => '&#32789;',
	"\xB8\xFC" => '&#26356;',
	"\xB8\xFD" => '&#24218;',
	"\xB8\xFE" => '&#32697;',
	"\xB9\xA1" => '&#22466;',
	"\xB9\xA2" => '&#32831;',
	"\xB9\xA3" => '&#26775;',
	"\xB9\xA4" => '&#24037;',
	"\xB9\xA5" => '&#25915;',
	"\xB9\xA6" => '&#21151;',
	"\xB9\xA7" => '&#24685;',
	"\xB9\xA8" => '&#40858;',
	"\xB9\xA9" => '&#20379;',
	"\xB9\xAA" => '&#36524;',
	"\xB9\xAB" => '&#20844;',
	"\xB9\xAC" => '&#23467;',
	"\xB9\xAD" => '&#24339;',
	"\xB9\xAE" => '&#24041;',
	"\xB9\xAF" => '&#27742;',
	"\xB9\xB0" => '&#25329;',
	"\xB9\xB1" => '&#36129;',
	"\xB9\xB2" => '&#20849;',
	"\xB9\xB3" => '&#38057;',
	"\xB9\xB4" => '&#21246;',
	"\xB9\xB5" => '&#27807;',
	"\xB9\xB6" => '&#33503;',
	"\xB9\xB7" => '&#29399;',
	"\xB9\xB8" => '&#22434;',
	"\xB9\xB9" => '&#26500;',
	"\xB9\xBA" => '&#36141;',
	"\xB9\xBB" => '&#22815;',
	"\xB9\xBC" => '&#36764;',
	"\xB9\xBD" => '&#33735;',
	"\xB9\xBE" => '&#21653;',
	"\xB9\xBF" => '&#31629;',
	"\xB9\xC0" => '&#20272;',
	"\xB9\xC1" => '&#27837;',
	"\xB9\xC2" => '&#23396;',
	"\xB9\xC3" => '&#22993;',
	"\xB9\xC4" => '&#40723;',
	"\xB9\xC5" => '&#21476;',
	"\xB9\xC6" => '&#34506;',
	"\xB9\xC7" => '&#39592;',
	"\xB9\xC8" => '&#35895;',
	"\xB9\xC9" => '&#32929;',
	"\xB9\xCA" => '&#25925;',
	"\xB9\xCB" => '&#39038;',
	"\xB9\xCC" => '&#22266;',
	"\xB9\xCD" => '&#38599;',
	"\xB9\xCE" => '&#21038;',
	"\xB9\xCF" => '&#29916;',
	"\xB9\xD0" => '&#21072;',
	"\xB9\xD1" => '&#23521;',
	"\xB9\xD2" => '&#25346;',
	"\xB9\xD3" => '&#35074;',
	"\xB9\xD4" => '&#20054;',
	"\xB9\xD5" => '&#25296;',
	"\xB9\xD6" => '&#24618;',
	"\xB9\xD7" => '&#26874;',
	"\xB9\xD8" => '&#20851;',
	"\xB9\xD9" => '&#23448;',
	"\xB9\xDA" => '&#20896;',
	"\xB9\xDB" => '&#35266;',
	"\xB9\xDC" => '&#31649;',
	"\xB9\xDD" => '&#39302;',
	"\xB9\xDE" => '&#32592;',
	"\xB9\xDF" => '&#24815;',
	"\xB9\xE0" => '&#28748;',
	"\xB9\xE1" => '&#36143;',
	"\xB9\xE2" => '&#20809;',
	"\xB9\xE3" => '&#24191;',
	"\xB9\xE4" => '&#36891;',
	"\xB9\xE5" => '&#29808;',
	"\xB9\xE6" => '&#35268;',
	"\xB9\xE7" => '&#22317;',
	"\xB9\xE8" => '&#30789;',
	"\xB9\xE9" => '&#24402;',
	"\xB9\xEA" => '&#40863;',
	"\xB9\xEB" => '&#38394;',
	"\xB9\xEC" => '&#36712;',
	"\xB9\xED" => '&#39740;',
	"\xB9\xEE" => '&#35809;',
	"\xB9\xEF" => '&#30328;',
	"\xB9\xF0" => '&#26690;',
	"\xB9\xF1" => '&#26588;',
	"\xB9\xF2" => '&#36330;',
	"\xB9\xF3" => '&#36149;',
	"\xB9\xF4" => '&#21053;',
	"\xB9\xF5" => '&#36746;',
	"\xB9\xF6" => '&#28378;',
	"\xB9\xF7" => '&#26829;',
	"\xB9\xF8" => '&#38149;',
	"\xB9\xF9" => '&#37101;',
	"\xB9\xFA" => '&#22269;',
	"\xB9\xFB" => '&#26524;',
	"\xB9\xFC" => '&#35065;',
	"\xB9\xFD" => '&#36807;',
	"\xB9\xFE" => '&#21704;',
	"\xBA\xA1" => '&#39608;',
	"\xBA\xA2" => '&#23401;',
	"\xBA\xA3" => '&#28023;',
	"\xBA\xA4" => '&#27686;',
	"\xBA\xA5" => '&#20133;',
	"\xBA\xA6" => '&#23475;',
	"\xBA\xA7" => '&#39559;',
	"\xBA\xA8" => '&#37219;',
	"\xBA\xA9" => '&#25000;',
	"\xBA\xAA" => '&#37039;',
	"\xBA\xAB" => '&#38889;',
	"\xBA\xAC" => '&#21547;',
	"\xBA\xAD" => '&#28085;',
	"\xBA\xAE" => '&#23506;',
	"\xBA\xAF" => '&#20989;',
	"\xBA\xB0" => '&#21898;',
	"\xBA\xB1" => '&#32597;',
	"\xBA\xB2" => '&#32752;',
	"\xBA\xB3" => '&#25788;',
	"\xBA\xB4" => '&#25421;',
	"\xBA\xB5" => '&#26097;',
	"\xBA\xB6" => '&#25022;',
	"\xBA\xB7" => '&#24717;',
	"\xBA\xB8" => '&#28938;',
	"\xBA\xB9" => '&#27735;',
	"\xBA\xBA" => '&#27721;',
	"\xBA\xBB" => '&#22831;',
	"\xBA\xBC" => '&#26477;',
	"\xBA\xBD" => '&#33322;',
	"\xBA\xBE" => '&#22741;',
	"\xBA\xBF" => '&#22158;',
	"\xBA\xC0" => '&#35946;',
	"\xBA\xC1" => '&#27627;',
	"\xBA\xC2" => '&#37085;',
	"\xBA\xC3" => '&#22909;',
	"\xBA\xC4" => '&#32791;',
	"\xBA\xC5" => '&#21495;',
	"\xBA\xC6" => '&#28009;',
	"\xBA\xC7" => '&#21621;',
	"\xBA\xC8" => '&#21917;',
	"\xBA\xC9" => '&#33655;',
	"\xBA\xCA" => '&#33743;',
	"\xBA\xCB" => '&#26680;',
	"\xBA\xCC" => '&#31166;',
	"\xBA\xCD" => '&#21644;',
	"\xBA\xCE" => '&#20309;',
	"\xBA\xCF" => '&#21512;',
	"\xBA\xD0" => '&#30418;',
	"\xBA\xD1" => '&#35977;',
	"\xBA\xD2" => '&#38402;',
	"\xBA\xD3" => '&#27827;',
	"\xBA\xD4" => '&#28088;',
	"\xBA\xD5" => '&#36203;',
	"\xBA\xD6" => '&#35088;',
	"\xBA\xD7" => '&#40548;',
	"\xBA\xD8" => '&#36154;',
	"\xBA\xD9" => '&#22079;',
	"\xBA\xDA" => '&#40657;',
	"\xBA\xDB" => '&#30165;',
	"\xBA\xDC" => '&#24456;',
	"\xBA\xDD" => '&#29408;',
	"\xBA\xDE" => '&#24680;',
	"\xBA\xDF" => '&#21756;',
	"\xBA\xE0" => '&#20136;',
	"\xBA\xE1" => '&#27178;',
	"\xBA\xE2" => '&#34913;',
	"\xBA\xE3" => '&#24658;',
	"\xBA\xE4" => '&#36720;',
	"\xBA\xE5" => '&#21700;',
	"\xBA\xE6" => '&#28888;',
	"\xBA\xE7" => '&#34425;',
	"\xBA\xE8" => '&#40511;',
	"\xBA\xE9" => '&#27946;',
	"\xBA\xEA" => '&#23439;',
	"\xBA\xEB" => '&#24344;',
	"\xBA\xEC" => '&#32418;',
	"\xBA\xED" => '&#21897;',
	"\xBA\xEE" => '&#20399;',
	"\xBA\xEF" => '&#29492;',
	"\xBA\xF0" => '&#21564;',
	"\xBA\xF1" => '&#21402;',
	"\xBA\xF2" => '&#20505;',
	"\xBA\xF3" => '&#21518;',
	"\xBA\xF4" => '&#21628;',
	"\xBA\xF5" => '&#20046;',
	"\xBA\xF6" => '&#24573;',
	"\xBA\xF7" => '&#29786;',
	"\xBA\xF8" => '&#22774;',
	"\xBA\xF9" => '&#33899;',
	"\xBA\xFA" => '&#32993;',
	"\xBA\xFB" => '&#34676;',
	"\xBA\xFC" => '&#29392;',
	"\xBA\xFD" => '&#31946;',
	"\xBA\xFE" => '&#28246;',
	"\xBB\xA1" => '&#24359;',
	"\xBB\xA2" => '&#34382;',
	"\xBB\xA3" => '&#21804;',
	"\xBB\xA4" => '&#25252;',
	"\xBB\xA5" => '&#20114;',
	"\xBB\xA6" => '&#27818;',
	"\xBB\xA7" => '&#25143;',
	"\xBB\xA8" => '&#33457;',
	"\xBB\xA9" => '&#21719;',
	"\xBB\xAA" => '&#21326;',
	"\xBB\xAB" => '&#29502;',
	"\xBB\xAC" => '&#28369;',
	"\xBB\xAD" => '&#30011;',
	"\xBB\xAE" => '&#21010;',
	"\xBB\xAF" => '&#21270;',
	"\xBB\xB0" => '&#35805;',
	"\xBB\xB1" => '&#27088;',
	"\xBB\xB2" => '&#24458;',
	"\xBB\xB3" => '&#24576;',
	"\xBB\xB4" => '&#28142;',
	"\xBB\xB5" => '&#22351;',
	"\xBB\xB6" => '&#27426;',
	"\xBB\xB7" => '&#29615;',
	"\xBB\xB8" => '&#26707;',
	"\xBB\xB9" => '&#36824;',
	"\xBB\xBA" => '&#32531;',
	"\xBB\xBB" => '&#25442;',
	"\xBB\xBC" => '&#24739;',
	"\xBB\xBD" => '&#21796;',
	"\xBB\xBE" => '&#30186;',
	"\xBB\xBF" => '&#35938;',
	"\xBB\xC0" => '&#28949;',
	"\xBB\xC1" => '&#28067;',
	"\xBB\xC2" => '&#23462;',
	"\xBB\xC3" => '&#24187;',
	"\xBB\xC4" => '&#33618;',
	"\xBB\xC5" => '&#24908;',
	"\xBB\xC6" => '&#40644;',
	"\xBB\xC7" => '&#30970;',
	"\xBB\xC8" => '&#34647;',
	"\xBB\xC9" => '&#31783;',
	"\xBB\xCA" => '&#30343;',
	"\xBB\xCB" => '&#20976;',
	"\xBB\xCC" => '&#24822;',
	"\xBB\xCD" => '&#29004;',
	"\xBB\xCE" => '&#26179;',
	"\xBB\xCF" => '&#24140;',
	"\xBB\xD0" => '&#24653;',
	"\xBB\xD1" => '&#35854;',
	"\xBB\xD2" => '&#28784;',
	"\xBB\xD3" => '&#25381;',
	"\xBB\xD4" => '&#36745;',
	"\xBB\xD5" => '&#24509;',
	"\xBB\xD6" => '&#24674;',
	"\xBB\xD7" => '&#34516;',
	"\xBB\xD8" => '&#22238;',
	"\xBB\xD9" => '&#27585;',
	"\xBB\xDA" => '&#24724;',
	"\xBB\xDB" => '&#24935;',
	"\xBB\xDC" => '&#21321;',
	"\xBB\xDD" => '&#24800;',
	"\xBB\xDE" => '&#26214;',
	"\xBB\xDF" => '&#36159;',
	"\xBB\xE0" => '&#31229;',
	"\xBB\xE1" => '&#20250;',
	"\xBB\xE2" => '&#28905;',
	"\xBB\xE3" => '&#27719;',
	"\xBB\xE4" => '&#35763;',
	"\xBB\xE5" => '&#35826;',
	"\xBB\xE6" => '&#32472;',
	"\xBB\xE7" => '&#33636;',
	"\xBB\xE8" => '&#26127;',
	"\xBB\xE9" => '&#23130;',
	"\xBB\xEA" => '&#39746;',
	"\xBB\xEB" => '&#27985;',
	"\xBB\xEC" => '&#28151;',
	"\xBB\xED" => '&#35905;',
	"\xBB\xEE" => '&#27963;',
	"\xBB\xEF" => '&#20249;',
	"\xBB\xF0" => '&#28779;',
	"\xBB\xF1" => '&#33719;',
	"\xBB\xF2" => '&#25110;',
	"\xBB\xF3" => '&#24785;',
	"\xBB\xF4" => '&#38669;',
	"\xBB\xF5" => '&#36135;',
	"\xBB\xF6" => '&#31096;',
	"\xBB\xF7" => '&#20987;',
	"\xBB\xF8" => '&#22334;',
	"\xBB\xF9" => '&#22522;',
	"\xBB\xFA" => '&#26426;',
	"\xBB\xFB" => '&#30072;',
	"\xBB\xFC" => '&#31293;',
	"\xBB\xFD" => '&#31215;',
	"\xBB\xFE" => '&#31637;',
	"\xBC\xA1" => '&#32908;',
	"\xBC\xA2" => '&#39269;',
	"\xBC\xA3" => '&#36857;',
	"\xBC\xA4" => '&#28608;',
	"\xBC\xA5" => '&#35749;',
	"\xBC\xA6" => '&#40481;',
	"\xBC\xA7" => '&#23020;',
	"\xBC\xA8" => '&#32489;',
	"\xBC\xA9" => '&#32521;',
	"\xBC\xAA" => '&#21513;',
	"\xBC\xAB" => '&#26497;',
	"\xBC\xAC" => '&#26840;',
	"\xBC\xAD" => '&#36753;',
	"\xBC\xAE" => '&#31821;',
	"\xBC\xAF" => '&#38598;',
	"\xBC\xB0" => '&#21450;',
	"\xBC\xB1" => '&#24613;',
	"\xBC\xB2" => '&#30142;',
	"\xBC\xB3" => '&#27762;',
	"\xBC\xB4" => '&#21363;',
	"\xBC\xB5" => '&#23241;',
	"\xBC\xB6" => '&#32423;',
	"\xBC\xB7" => '&#25380;',
	"\xBC\xB8" => '&#20960;',
	"\xBC\xB9" => '&#33034;',
	"\xBC\xBA" => '&#24049;',
	"\xBC\xBB" => '&#34015;',
	"\xBC\xBC" => '&#25216;',
	"\xBC\xBD" => '&#20864;',
	"\xBC\xBE" => '&#23395;',
	"\xBC\xBF" => '&#20238;',
	"\xBC\xC0" => '&#31085;',
	"\xBC\xC1" => '&#21058;',
	"\xBC\xC2" => '&#24760;',
	"\xBC\xC3" => '&#27982;',
	"\xBC\xC4" => '&#23492;',
	"\xBC\xC5" => '&#23490;',
	"\xBC\xC6" => '&#35745;',
	"\xBC\xC7" => '&#35760;',
	"\xBC\xC8" => '&#26082;',
	"\xBC\xC9" => '&#24524;',
	"\xBC\xCA" => '&#38469;',
	"\xBC\xCB" => '&#22931;',
	"\xBC\xCC" => '&#32487;',
	"\xBC\xCD" => '&#32426;',
	"\xBC\xCE" => '&#22025;',
	"\xBC\xCF" => '&#26551;',
	"\xBC\xD0" => '&#22841;',
	"\xBC\xD1" => '&#20339;',
	"\xBC\xD2" => '&#23478;',
	"\xBC\xD3" => '&#21152;',
	"\xBC\xD4" => '&#33626;',
	"\xBC\xD5" => '&#39050;',
	"\xBC\xD6" => '&#36158;',
	"\xBC\xD7" => '&#30002;',
	"\xBC\xD8" => '&#38078;',
	"\xBC\xD9" => '&#20551;',
	"\xBC\xDA" => '&#31292;',
	"\xBC\xDB" => '&#20215;',
	"\xBC\xDC" => '&#26550;',
	"\xBC\xDD" => '&#39550;',
	"\xBC\xDE" => '&#23233;',
	"\xBC\xDF" => '&#27516;',
	"\xBC\xE0" => '&#30417;',
	"\xBC\xE1" => '&#22362;',
	"\xBC\xE2" => '&#23574;',
	"\xBC\xE3" => '&#31546;',
	"\xBC\xE4" => '&#38388;',
	"\xBC\xE5" => '&#29006;',
	"\xBC\xE6" => '&#20860;',
	"\xBC\xE7" => '&#32937;',
	"\xBC\xE8" => '&#33392;',
	"\xBC\xE9" => '&#22904;',
	"\xBC\xEA" => '&#32516;',
	"\xBC\xEB" => '&#33575;',
	"\xBC\xEC" => '&#26816;',
	"\xBC\xED" => '&#26604;',
	"\xBC\xEE" => '&#30897;',
	"\xBC\xEF" => '&#30839;',
	"\xBC\xF0" => '&#25315;',
	"\xBC\xF1" => '&#25441;',
	"\xBC\xF2" => '&#31616;',
	"\xBC\xF3" => '&#20461;',
	"\xBC\xF4" => '&#21098;',
	"\xBC\xF5" => '&#20943;',
	"\xBC\xF6" => '&#33616;',
	"\xBC\xF7" => '&#27099;',
	"\xBC\xF8" => '&#37492;',
	"\xBC\xF9" => '&#36341;',
	"\xBC\xFA" => '&#36145;',
	"\xBC\xFB" => '&#35265;',
	"\xBC\xFC" => '&#38190;',
	"\xBC\xFD" => '&#31661;',
	"\xBC\xFE" => '&#20214;',
	"\xBD\xA1" => '&#20581;',
	"\xBD\xA2" => '&#33328;',
	"\xBD\xA3" => '&#21073;',
	"\xBD\xA4" => '&#39279;',
	"\xBD\xA5" => '&#28176;',
	"\xBD\xA6" => '&#28293;',
	"\xBD\xA7" => '&#28071;',
	"\xBD\xA8" => '&#24314;',
	"\xBD\xA9" => '&#20725;',
	"\xBD\xAA" => '&#23004;',
	"\xBD\xAB" => '&#23558;',
	"\xBD\xAC" => '&#27974;',
	"\xBD\xAD" => '&#27743;',
	"\xBD\xAE" => '&#30086;',
	"\xBD\xAF" => '&#33931;',
	"\xBD\xB0" => '&#26728;',
	"\xBD\xB1" => '&#22870;',
	"\xBD\xB2" => '&#35762;',
	"\xBD\xB3" => '&#21280;',
	"\xBD\xB4" => '&#37233;',
	"\xBD\xB5" => '&#38477;',
	"\xBD\xB6" => '&#34121;',
	"\xBD\xB7" => '&#26898;',
	"\xBD\xB8" => '&#30977;',
	"\xBD\xB9" => '&#28966;',
	"\xBD\xBA" => '&#33014;',
	"\xBD\xBB" => '&#20132;',
	"\xBD\xBC" => '&#37066;',
	"\xBD\xBD" => '&#27975;',
	"\xBD\xBE" => '&#39556;',
	"\xBD\xBF" => '&#23047;',
	"\xBD\xC0" => '&#22204;',
	"\xBD\xC1" => '&#25605;',
	"\xBD\xC2" => '&#38128;',
	"\xBD\xC3" => '&#30699;',
	"\xBD\xC4" => '&#20389;',
	"\xBD\xC5" => '&#33050;',
	"\xBD\xC6" => '&#29409;',
	"\xBD\xC7" => '&#35282;',
	"\xBD\xC8" => '&#39290;',
	"\xBD\xC9" => '&#32564;',
	"\xBD\xCA" => '&#32478;',
	"\xBD\xCB" => '&#21119;',
	"\xBD\xCC" => '&#25945;',
	"\xBD\xCD" => '&#37237;',
	"\xBD\xCE" => '&#36735;',
	"\xBD\xCF" => '&#36739;',
	"\xBD\xD0" => '&#21483;',
	"\xBD\xD1" => '&#31382;',
	"\xBD\xD2" => '&#25581;',
	"\xBD\xD3" => '&#25509;',
	"\xBD\xD4" => '&#30342;',
	"\xBD\xD5" => '&#31224;',
	"\xBD\xD6" => '&#34903;',
	"\xBD\xD7" => '&#38454;',
	"\xBD\xD8" => '&#25130;',
	"\xBD\xD9" => '&#21163;',
	"\xBD\xDA" => '&#33410;',
	"\xBD\xDB" => '&#26708;',
	"\xBD\xDC" => '&#26480;',
	"\xBD\xDD" => '&#25463;',
	"\xBD\xDE" => '&#30571;',
	"\xBD\xDF" => '&#31469;',
	"\xBD\xE0" => '&#27905;',
	"\xBD\xE1" => '&#32467;',
	"\xBD\xE2" => '&#35299;',
	"\xBD\xE3" => '&#22992;',
	"\xBD\xE4" => '&#25106;',
	"\xBD\xE5" => '&#34249;',
	"\xBD\xE6" => '&#33445;',
	"\xBD\xE7" => '&#30028;',
	"\xBD\xE8" => '&#20511;',
	"\xBD\xE9" => '&#20171;',
	"\xBD\xEA" => '&#30117;',
	"\xBD\xEB" => '&#35819;',
	"\xBD\xEC" => '&#23626;',
	"\xBD\xED" => '&#24062;',
	"\xBD\xEE" => '&#31563;',
	"\xBD\xEF" => '&#26020;',
	"\xBD\xF0" => '&#37329;',
	"\xBD\xF1" => '&#20170;',
	"\xBD\xF2" => '&#27941;',
	"\xBD\xF3" => '&#35167;',
	"\xBD\xF4" => '&#32039;',
	"\xBD\xF5" => '&#38182;',
	"\xBD\xF6" => '&#20165;',
	"\xBD\xF7" => '&#35880;',
	"\xBD\xF8" => '&#36827;',
	"\xBD\xF9" => '&#38771;',
	"\xBD\xFA" => '&#26187;',
	"\xBD\xFB" => '&#31105;',
	"\xBD\xFC" => '&#36817;',
	"\xBD\xFD" => '&#28908;',
	"\xBD\xFE" => '&#28024;',
	"\xBE\xA1" => '&#23613;',
	"\xBE\xA2" => '&#21170;',
	"\xBE\xA3" => '&#33606;',
	"\xBE\xA4" => '&#20834;',
	"\xBE\xA5" => '&#33550;',
	"\xBE\xA6" => '&#30555;',
	"\xBE\xA7" => '&#26230;',
	"\xBE\xA8" => '&#40120;',
	"\xBE\xA9" => '&#20140;',
	"\xBE\xAA" => '&#24778;',
	"\xBE\xAB" => '&#31934;',
	"\xBE\xAC" => '&#31923;',
	"\xBE\xAD" => '&#32463;',
	"\xBE\xAE" => '&#20117;',
	"\xBE\xAF" => '&#35686;',
	"\xBE\xB0" => '&#26223;',
	"\xBE\xB1" => '&#39048;',
	"\xBE\xB2" => '&#38745;',
	"\xBE\xB3" => '&#22659;',
	"\xBE\xB4" => '&#25964;',
	"\xBE\xB5" => '&#38236;',
	"\xBE\xB6" => '&#24452;',
	"\xBE\xB7" => '&#30153;',
	"\xBE\xB8" => '&#38742;',
	"\xBE\xB9" => '&#31455;',
	"\xBE\xBA" => '&#31454;',
	"\xBE\xBB" => '&#20928;',
	"\xBE\xBC" => '&#28847;',
	"\xBE\xBD" => '&#31384;',
	"\xBE\xBE" => '&#25578;',
	"\xBE\xBF" => '&#31350;',
	"\xBE\xC0" => '&#32416;',
	"\xBE\xC1" => '&#29590;',
	"\xBE\xC2" => '&#38893;',
	"\xBE\xC3" => '&#20037;',
	"\xBE\xC4" => '&#28792;',
	"\xBE\xC5" => '&#20061;',
	"\xBE\xC6" => '&#37202;',
	"\xBE\xC7" => '&#21417;',
	"\xBE\xC8" => '&#25937;',
	"\xBE\xC9" => '&#26087;',
	"\xBE\xCA" => '&#33276;',
	"\xBE\xCB" => '&#33285;',
	"\xBE\xCC" => '&#21646;',
	"\xBE\xCD" => '&#23601;',
	"\xBE\xCE" => '&#30106;',
	"\xBE\xCF" => '&#38816;',
	"\xBE\xD0" => '&#25304;',
	"\xBE\xD1" => '&#29401;',
	"\xBE\xD2" => '&#30141;',
	"\xBE\xD3" => '&#23621;',
	"\xBE\xD4" => '&#39545;',
	"\xBE\xD5" => '&#33738;',
	"\xBE\xD6" => '&#23616;',
	"\xBE\xD7" => '&#21632;',
	"\xBE\xD8" => '&#30697;',
	"\xBE\xD9" => '&#20030;',
	"\xBE\xDA" => '&#27822;',
	"\xBE\xDB" => '&#32858;',
	"\xBE\xDC" => '&#25298;',
	"\xBE\xDD" => '&#25454;',
	"\xBE\xDE" => '&#24040;',
	"\xBE\xDF" => '&#20855;',
	"\xBE\xE0" => '&#36317;',
	"\xBE\xE1" => '&#36382;',
	"\xBE\xE2" => '&#38191;',
	"\xBE\xE3" => '&#20465;',
	"\xBE\xE4" => '&#21477;',
	"\xBE\xE5" => '&#24807;',
	"\xBE\xE6" => '&#28844;',
	"\xBE\xE7" => '&#21095;',
	"\xBE\xE8" => '&#25424;',
	"\xBE\xE9" => '&#40515;',
	"\xBE\xEA" => '&#23071;',
	"\xBE\xEB" => '&#20518;',
	"\xBE\xEC" => '&#30519;',
	"\xBE\xED" => '&#21367;',
	"\xBE\xEE" => '&#32482;',
	"\xBE\xEF" => '&#25733;',
	"\xBE\xF0" => '&#25899;',
	"\xBE\xF1" => '&#25225;',
	"\xBE\xF2" => '&#25496;',
	"\xBE\xF3" => '&#20500;',
	"\xBE\xF4" => '&#29237;',
	"\xBE\xF5" => '&#35273;',
	"\xBE\xF6" => '&#20915;',
	"\xBE\xF7" => '&#35776;',
	"\xBE\xF8" => '&#32477;',
	"\xBE\xF9" => '&#22343;',
	"\xBE\xFA" => '&#33740;',
	"\xBE\xFB" => '&#38055;',
	"\xBE\xFC" => '&#20891;',
	"\xBE\xFD" => '&#21531;',
	"\xBE\xFE" => '&#23803;',
	"\xBF\xA1" => '&#20426;',
	"\xBF\xA2" => '&#31459;',
	"\xBF\xA3" => '&#27994;',
	"\xBF\xA4" => '&#37089;',
	"\xBF\xA5" => '&#39567;',
	"\xBF\xA6" => '&#21888;',
	"\xBF\xA7" => '&#21654;',
	"\xBF\xA8" => '&#21345;',
	"\xBF\xA9" => '&#21679;',
	"\xBF\xAA" => '&#24320;',
	"\xBF\xAB" => '&#25577;',
	"\xBF\xAC" => '&#26999;',
	"\xBF\xAD" => '&#20975;',
	"\xBF\xAE" => '&#24936;',
	"\xBF\xAF" => '&#21002;',
	"\xBF\xB0" => '&#22570;',
	"\xBF\xB1" => '&#21208;',
	"\xBF\xB2" => '&#22350;',
	"\xBF\xB3" => '&#30733;',
	"\xBF\xB4" => '&#30475;',
	"\xBF\xB5" => '&#24247;',
	"\xBF\xB6" => '&#24951;',
	"\xBF\xB7" => '&#31968;',
	"\xBF\xB8" => '&#25179;',
	"\xBF\xB9" => '&#25239;',
	"\xBF\xBA" => '&#20130;',
	"\xBF\xBB" => '&#28821;',
	"\xBF\xBC" => '&#32771;',
	"\xBF\xBD" => '&#25335;',
	"\xBF\xBE" => '&#28900;',
	"\xBF\xBF" => '&#38752;',
	"\xBF\xC0" => '&#22391;',
	"\xBF\xC1" => '&#33499;',
	"\xBF\xC2" => '&#26607;',
	"\xBF\xC3" => '&#26869;',
	"\xBF\xC4" => '&#30933;',
	"\xBF\xC5" => '&#39063;',
	"\xBF\xC6" => '&#31185;',
	"\xBF\xC7" => '&#22771;',
	"\xBF\xC8" => '&#21683;',
	"\xBF\xC9" => '&#21487;',
	"\xBF\xCA" => '&#28212;',
	"\xBF\xCB" => '&#20811;',
	"\xBF\xCC" => '&#21051;',
	"\xBF\xCD" => '&#23458;',
	"\xBF\xCE" => '&#35838;',
	"\xBF\xCF" => '&#32943;',
	"\xBF\xD0" => '&#21827;',
	"\xBF\xD1" => '&#22438;',
	"\xBF\xD2" => '&#24691;',
	"\xBF\xD3" => '&#22353;',
	"\xBF\xD4" => '&#21549;',
	"\xBF\xD5" => '&#31354;',
	"\xBF\xD6" => '&#24656;',
	"\xBF\xD7" => '&#23380;',
	"\xBF\xD8" => '&#25511;',
	"\xBF\xD9" => '&#25248;',
	"\xBF\xDA" => '&#21475;',
	"\xBF\xDB" => '&#25187;',
	"\xBF\xDC" => '&#23495;',
	"\xBF\xDD" => '&#26543;',
	"\xBF\xDE" => '&#21741;',
	"\xBF\xDF" => '&#31391;',
	"\xBF\xE0" => '&#33510;',
	"\xBF\xE1" => '&#37239;',
	"\xBF\xE2" => '&#24211;',
	"\xBF\xE3" => '&#35044;',
	"\xBF\xE4" => '&#22840;',
	"\xBF\xE5" => '&#22446;',
	"\xBF\xE6" => '&#25358;',
	"\xBF\xE7" => '&#36328;',
	"\xBF\xE8" => '&#33007;',
	"\xBF\xE9" => '&#22359;',
	"\xBF\xEA" => '&#31607;',
	"\xBF\xEB" => '&#20393;',
	"\xBF\xEC" => '&#24555;',
	"\xBF\xED" => '&#23485;',
	"\xBF\xEE" => '&#27454;',
	"\xBF\xEF" => '&#21281;',
	"\xBF\xF0" => '&#31568;',
	"\xBF\xF1" => '&#29378;',
	"\xBF\xF2" => '&#26694;',
	"\xBF\xF3" => '&#30719;',
	"\xBF\xF4" => '&#30518;',
	"\xBF\xF5" => '&#26103;',
	"\xBF\xF6" => '&#20917;',
	"\xBF\xF7" => '&#20111;',
	"\xBF\xF8" => '&#30420;',
	"\xBF\xF9" => '&#23743;',
	"\xBF\xFA" => '&#31397;',
	"\xBF\xFB" => '&#33909;',
	"\xBF\xFC" => '&#22862;',
	"\xBF\xFD" => '&#39745;',
	"\xBF\xFE" => '&#20608;',
	"\xC0\xA1" => '&#39304;',
	"\xC0\xA2" => '&#24871;',
	"\xC0\xA3" => '&#28291;',
	"\xC0\xA4" => '&#22372;',
	"\xC0\xA5" => '&#26118;',
	"\xC0\xA6" => '&#25414;',
	"\xC0\xA7" => '&#22256;',
	"\xC0\xA8" => '&#25324;',
	"\xC0\xA9" => '&#25193;',
	"\xC0\xAA" => '&#24275;',
	"\xC0\xAB" => '&#38420;',
	"\xC0\xAC" => '&#22403;',
	"\xC0\xAD" => '&#25289;',
	"\xC0\xAE" => '&#21895;',
	"\xC0\xAF" => '&#34593;',
	"\xC0\xB0" => '&#33098;',
	"\xC0\xB1" => '&#36771;',
	"\xC0\xB2" => '&#21862;',
	"\xC0\xB3" => '&#33713;',
	"\xC0\xB4" => '&#26469;',
	"\xC0\xB5" => '&#36182;',
	"\xC0\xB6" => '&#34013;',
	"\xC0\xB7" => '&#23146;',
	"\xC0\xB8" => '&#26639;',
	"\xC0\xB9" => '&#25318;',
	"\xC0\xBA" => '&#31726;',
	"\xC0\xBB" => '&#38417;',
	"\xC0\xBC" => '&#20848;',
	"\xC0\xBD" => '&#28572;',
	"\xC0\xBE" => '&#35888;',
	"\xC0\xBF" => '&#25597;',
	"\xC0\xC0" => '&#35272;',
	"\xC0\xC1" => '&#25042;',
	"\xC0\xC2" => '&#32518;',
	"\xC0\xC3" => '&#28866;',
	"\xC0\xC4" => '&#28389;',
	"\xC0\xC5" => '&#29701;',
	"\xC0\xC6" => '&#27028;',
	"\xC0\xC7" => '&#29436;',
	"\xC0\xC8" => '&#24266;',
	"\xC0\xC9" => '&#37070;',
	"\xC0\xCA" => '&#26391;',
	"\xC0\xCB" => '&#28010;',
	"\xC0\xCC" => '&#25438;',
	"\xC0\xCD" => '&#21171;',
	"\xC0\xCE" => '&#29282;',
	"\xC0\xCF" => '&#32769;',
	"\xC0\xD0" => '&#20332;',
	"\xC0\xD1" => '&#23013;',
	"\xC0\xD2" => '&#37226;',
	"\xC0\xD3" => '&#28889;',
	"\xC0\xD4" => '&#28061;',
	"\xC0\xD5" => '&#21202;',
	"\xC0\xD6" => '&#20048;',
	"\xC0\xD7" => '&#38647;',
	"\xC0\xD8" => '&#38253;',
	"\xC0\xD9" => '&#34174;',
	"\xC0\xDA" => '&#30922;',
	"\xC0\xDB" => '&#32047;',
	"\xC0\xDC" => '&#20769;',
	"\xC0\xDD" => '&#22418;',
	"\xC0\xDE" => '&#25794;',
	"\xC0\xDF" => '&#32907;',
	"\xC0\xE0" => '&#31867;',
	"\xC0\xE1" => '&#27882;',
	"\xC0\xE2" => '&#26865;',
	"\xC0\xE3" => '&#26974;',
	"\xC0\xE4" => '&#20919;',
	"\xC0\xE5" => '&#21400;',
	"\xC0\xE6" => '&#26792;',
	"\xC0\xE7" => '&#29313;',
	"\xC0\xE8" => '&#40654;',
	"\xC0\xE9" => '&#31729;',
	"\xC0\xEA" => '&#29432;',
	"\xC0\xEB" => '&#31163;',
	"\xC0\xEC" => '&#28435;',
	"\xC0\xED" => '&#29702;',
	"\xC0\xEE" => '&#26446;',
	"\xC0\xEF" => '&#37324;',
	"\xC0\xF0" => '&#40100;',
	"\xC0\xF1" => '&#31036;',
	"\xC0\xF2" => '&#33673;',
	"\xC0\xF3" => '&#33620;',
	"\xC0\xF4" => '&#21519;',
	"\xC0\xF5" => '&#26647;',
	"\xC0\xF6" => '&#20029;',
	"\xC0\xF7" => '&#21385;',
	"\xC0\xF8" => '&#21169;',
	"\xC0\xF9" => '&#30782;',
	"\xC0\xFA" => '&#21382;',
	"\xC0\xFB" => '&#21033;',
	"\xC0\xFC" => '&#20616;',
	"\xC0\xFD" => '&#20363;',
	"\xC0\xFE" => '&#20432;',
	"\xC1\xA1" => '&#30178;',
	"\xC1\xA2" => '&#31435;',
	"\xC1\xA3" => '&#31890;',
	"\xC1\xA4" => '&#27813;',
	"\xC1\xA5" => '&#38582;',
	"\xC1\xA6" => '&#21147;',
	"\xC1\xA7" => '&#29827;',
	"\xC1\xA8" => '&#21737;',
	"\xC1\xA9" => '&#20457;',
	"\xC1\xAA" => '&#32852;',
	"\xC1\xAB" => '&#33714;',
	"\xC1\xAC" => '&#36830;',
	"\xC1\xAD" => '&#38256;',
	"\xC1\xAE" => '&#24265;',
	"\xC1\xAF" => '&#24604;',
	"\xC1\xB0" => '&#28063;',
	"\xC1\xB1" => '&#24088;',
	"\xC1\xB2" => '&#25947;',
	"\xC1\xB3" => '&#33080;',
	"\xC1\xB4" => '&#38142;',
	"\xC1\xB5" => '&#24651;',
	"\xC1\xB6" => '&#28860;',
	"\xC1\xB7" => '&#32451;',
	"\xC1\xB8" => '&#31918;',
	"\xC1\xB9" => '&#20937;',
	"\xC1\xBA" => '&#26753;',
	"\xC1\xBB" => '&#31921;',
	"\xC1\xBC" => '&#33391;',
	"\xC1\xBD" => '&#20004;',
	"\xC1\xBE" => '&#36742;',
	"\xC1\xBF" => '&#37327;',
	"\xC1\xC0" => '&#26238;',
	"\xC1\xC1" => '&#20142;',
	"\xC1\xC2" => '&#35845;',
	"\xC1\xC3" => '&#25769;',
	"\xC1\xC4" => '&#32842;',
	"\xC1\xC5" => '&#20698;',
	"\xC1\xC6" => '&#30103;',
	"\xC1\xC7" => '&#29134;',
	"\xC1\xC8" => '&#23525;',
	"\xC1\xC9" => '&#36797;',
	"\xC1\xCA" => '&#28518;',
	"\xC1\xCB" => '&#20102;',
	"\xC1\xCC" => '&#25730;',
	"\xC1\xCD" => '&#38243;',
	"\xC1\xCE" => '&#24278;',
	"\xC1\xCF" => '&#26009;',
	"\xC1\xD0" => '&#21015;',
	"\xC1\xD1" => '&#35010;',
	"\xC1\xD2" => '&#28872;',
	"\xC1\xD3" => '&#21155;',
	"\xC1\xD4" => '&#29454;',
	"\xC1\xD5" => '&#29747;',
	"\xC1\xD6" => '&#26519;',
	"\xC1\xD7" => '&#30967;',
	"\xC1\xD8" => '&#38678;',
	"\xC1\xD9" => '&#20020;',
	"\xC1\xDA" => '&#37051;',
	"\xC1\xDB" => '&#40158;',
	"\xC1\xDC" => '&#28107;',
	"\xC1\xDD" => '&#20955;',
	"\xC1\xDE" => '&#36161;',
	"\xC1\xDF" => '&#21533;',
	"\xC1\xE0" => '&#25294;',
	"\xC1\xE1" => '&#29618;',
	"\xC1\xE2" => '&#33777;',
	"\xC1\xE3" => '&#38646;',
	"\xC1\xE4" => '&#40836;',
	"\xC1\xE5" => '&#38083;',
	"\xC1\xE6" => '&#20278;',
	"\xC1\xE7" => '&#32666;',
	"\xC1\xE8" => '&#20940;',
	"\xC1\xE9" => '&#28789;',
	"\xC1\xEA" => '&#38517;',
	"\xC1\xEB" => '&#23725;',
	"\xC1\xEC" => '&#39046;',
	"\xC1\xED" => '&#21478;',
	"\xC1\xEE" => '&#20196;',
	"\xC1\xEF" => '&#28316;',
	"\xC1\xF0" => '&#29705;',
	"\xC1\xF1" => '&#27060;',
	"\xC1\xF2" => '&#30827;',
	"\xC1\xF3" => '&#39311;',
	"\xC1\xF4" => '&#30041;',
	"\xC1\xF5" => '&#21016;',
	"\xC1\xF6" => '&#30244;',
	"\xC1\xF7" => '&#27969;',
	"\xC1\xF8" => '&#26611;',
	"\xC1\xF9" => '&#20845;',
	"\xC1\xFA" => '&#40857;',
	"\xC1\xFB" => '&#32843;',
	"\xC1\xFC" => '&#21657;',
	"\xC1\xFD" => '&#31548;',
	"\xC1\xFE" => '&#31423;',
	"\xC2\xA1" => '&#38534;',
	"\xC2\xA2" => '&#22404;',
	"\xC2\xA3" => '&#25314;',
	"\xC2\xA4" => '&#38471;',
	"\xC2\xA5" => '&#27004;',
	"\xC2\xA6" => '&#23044;',
	"\xC2\xA7" => '&#25602;',
	"\xC2\xA8" => '&#31699;',
	"\xC2\xA9" => '&#28431;',
	"\xC2\xAA" => '&#38475;',
	"\xC2\xAB" => '&#33446;',
	"\xC2\xAC" => '&#21346;',
	"\xC2\xAD" => '&#39045;',
	"\xC2\xAE" => '&#24208;',
	"\xC2\xAF" => '&#28809;',
	"\xC2\xB0" => '&#25523;',
	"\xC2\xB1" => '&#21348;',
	"\xC2\xB2" => '&#34383;',
	"\xC2\xB3" => '&#40065;',
	"\xC2\xB4" => '&#40595;',
	"\xC2\xB5" => '&#30860;',
	"\xC2\xB6" => '&#38706;',
	"\xC2\xB7" => '&#36335;',
	"\xC2\xB8" => '&#36162;',
	"\xC2\xB9" => '&#40575;',
	"\xC2\xBA" => '&#28510;',
	"\xC2\xBB" => '&#31108;',
	"\xC2\xBC" => '&#24405;',
	"\xC2\xBD" => '&#38470;',
	"\xC2\xBE" => '&#25134;',
	"\xC2\xBF" => '&#39540;',
	"\xC2\xC0" => '&#21525;',
	"\xC2\xC1" => '&#38109;',
	"\xC2\xC2" => '&#20387;',
	"\xC2\xC3" => '&#26053;',
	"\xC2\xC4" => '&#23653;',
	"\xC2\xC5" => '&#23649;',
	"\xC2\xC6" => '&#32533;',
	"\xC2\xC7" => '&#34385;',
	"\xC2\xC8" => '&#27695;',
	"\xC2\xC9" => '&#24459;',
	"\xC2\xCA" => '&#29575;',
	"\xC2\xCB" => '&#28388;',
	"\xC2\xCC" => '&#32511;',
	"\xC2\xCD" => '&#23782;',
	"\xC2\xCE" => '&#25371;',
	"\xC2\xCF" => '&#23402;',
	"\xC2\xD0" => '&#28390;',
	"\xC2\xD1" => '&#21365;',
	"\xC2\xD2" => '&#20081;',
	"\xC2\xD3" => '&#25504;',
	"\xC2\xD4" => '&#30053;',
	"\xC2\xD5" => '&#25249;',
	"\xC2\xD6" => '&#36718;',
	"\xC2\xD7" => '&#20262;',
	"\xC2\xD8" => '&#20177;',
	"\xC2\xD9" => '&#27814;',
	"\xC2\xDA" => '&#32438;',
	"\xC2\xDB" => '&#35770;',
	"\xC2\xDC" => '&#33821;',
	"\xC2\xDD" => '&#34746;',
	"\xC2\xDE" => '&#32599;',
	"\xC2\xDF" => '&#36923;',
	"\xC2\xE0" => '&#38179;',
	"\xC2\xE1" => '&#31657;',
	"\xC2\xE2" => '&#39585;',
	"\xC2\xE3" => '&#35064;',
	"\xC2\xE4" => '&#33853;',
	"\xC2\xE5" => '&#27931;',
	"\xC2\xE6" => '&#39558;',
	"\xC2\xE7" => '&#32476;',
	"\xC2\xE8" => '&#22920;',
	"\xC2\xE9" => '&#40635;',
	"\xC2\xEA" => '&#29595;',
	"\xC2\xEB" => '&#30721;',
	"\xC2\xEC" => '&#34434;',
	"\xC2\xED" => '&#39532;',
	"\xC2\xEE" => '&#39554;',
	"\xC2\xEF" => '&#22043;',
	"\xC2\xF0" => '&#21527;',
	"\xC2\xF1" => '&#22475;',
	"\xC2\xF2" => '&#20080;',
	"\xC2\xF3" => '&#40614;',
	"\xC2\xF4" => '&#21334;',
	"\xC2\xF5" => '&#36808;',
	"\xC2\xF6" => '&#33033;',
	"\xC2\xF7" => '&#30610;',
	"\xC2\xF8" => '&#39314;',
	"\xC2\xF9" => '&#34542;',
	"\xC2\xFA" => '&#28385;',
	"\xC2\xFB" => '&#34067;',
	"\xC2\xFC" => '&#26364;',
	"\xC2\xFD" => '&#24930;',
	"\xC2\xFE" => '&#28459;',
	"\xC3\xA1" => '&#35881;',
	"\xC3\xA2" => '&#33426;',
	"\xC3\xA3" => '&#33579;',
	"\xC3\xA4" => '&#30450;',
	"\xC3\xA5" => '&#27667;',
	"\xC3\xA6" => '&#24537;',
	"\xC3\xA7" => '&#33725;',
	"\xC3\xA8" => '&#29483;',
	"\xC3\xA9" => '&#33541;',
	"\xC3\xAA" => '&#38170;',
	"\xC3\xAB" => '&#27611;',
	"\xC3\xAC" => '&#30683;',
	"\xC3\xAD" => '&#38086;',
	"\xC3\xAE" => '&#21359;',
	"\xC3\xAF" => '&#33538;',
	"\xC3\xB0" => '&#20882;',
	"\xC3\xB1" => '&#24125;',
	"\xC3\xB2" => '&#35980;',
	"\xC3\xB3" => '&#36152;',
	"\xC3\xB4" => '&#20040;',
	"\xC3\xB5" => '&#29611;',
	"\xC3\xB6" => '&#26522;',
	"\xC3\xB7" => '&#26757;',
	"\xC3\xB8" => '&#37238;',
	"\xC3\xB9" => '&#38665;',
	"\xC3\xBA" => '&#29028;',
	"\xC3\xBB" => '&#27809;',
	"\xC3\xBC" => '&#30473;',
	"\xC3\xBD" => '&#23186;',
	"\xC3\xBE" => '&#38209;',
	"\xC3\xBF" => '&#27599;',
	"\xC3\xC0" => '&#32654;',
	"\xC3\xC1" => '&#26151;',
	"\xC3\xC2" => '&#23504;',
	"\xC3\xC3" => '&#22969;',
	"\xC3\xC4" => '&#23194;',
	"\xC3\xC5" => '&#38376;',
	"\xC3\xC6" => '&#38391;',
	"\xC3\xC7" => '&#20204;',
	"\xC3\xC8" => '&#33804;',
	"\xC3\xC9" => '&#33945;',
	"\xC3\xCA" => '&#27308;',
	"\xC3\xCB" => '&#30431;',
	"\xC3\xCC" => '&#38192;',
	"\xC3\xCD" => '&#29467;',
	"\xC3\xCE" => '&#26790;',
	"\xC3\xCF" => '&#23391;',
	"\xC3\xD0" => '&#30511;',
	"\xC3\xD1" => '&#37274;',
	"\xC3\xD2" => '&#38753;',
	"\xC3\xD3" => '&#31964;',
	"\xC3\xD4" => '&#36855;',
	"\xC3\xD5" => '&#35868;',
	"\xC3\xD6" => '&#24357;',
	"\xC3\xD7" => '&#31859;',
	"\xC3\xD8" => '&#31192;',
	"\xC3\xD9" => '&#35269;',
	"\xC3\xDA" => '&#27852;',
	"\xC3\xDB" => '&#34588;',
	"\xC3\xDC" => '&#23494;',
	"\xC3\xDD" => '&#24130;',
	"\xC3\xDE" => '&#26825;',
	"\xC3\xDF" => '&#30496;',
	"\xC3\xE0" => '&#32501;',
	"\xC3\xE1" => '&#20885;',
	"\xC3\xE2" => '&#20813;',
	"\xC3\xE3" => '&#21193;',
	"\xC3\xE4" => '&#23081;',
	"\xC3\xE5" => '&#32517;',
	"\xC3\xE6" => '&#38754;',
	"\xC3\xE7" => '&#33495;',
	"\xC3\xE8" => '&#25551;',
	"\xC3\xE9" => '&#30596;',
	"\xC3\xEA" => '&#34256;',
	"\xC3\xEB" => '&#31186;',
	"\xC3\xEC" => '&#28218;',
	"\xC3\xED" => '&#24217;',
	"\xC3\xEE" => '&#22937;',
	"\xC3\xEF" => '&#34065;',
	"\xC3\xF0" => '&#28781;',
	"\xC3\xF1" => '&#27665;',
	"\xC3\xF2" => '&#25279;',
	"\xC3\xF3" => '&#30399;',
	"\xC3\xF4" => '&#25935;',
	"\xC3\xF5" => '&#24751;',
	"\xC3\xF6" => '&#38397;',
	"\xC3\xF7" => '&#26126;',
	"\xC3\xF8" => '&#34719;',
	"\xC3\xF9" => '&#40483;',
	"\xC3\xFA" => '&#38125;',
	"\xC3\xFB" => '&#21517;',
	"\xC3\xFC" => '&#21629;',
	"\xC3\xFD" => '&#35884;',
	"\xC3\xFE" => '&#25720;',
	"\xC4\xA1" => '&#25721;',
	"\xC4\xA2" => '&#34321;',
	"\xC4\xA3" => '&#27169;',
	"\xC4\xA4" => '&#33180;',
	"\xC4\xA5" => '&#30952;',
	"\xC4\xA6" => '&#25705;',
	"\xC4\xA7" => '&#39764;',
	"\xC4\xA8" => '&#25273;',
	"\xC4\xA9" => '&#26411;',
	"\xC4\xAA" => '&#33707;',
	"\xC4\xAB" => '&#22696;',
	"\xC4\xAC" => '&#40664;',
	"\xC4\xAD" => '&#27819;',
	"\xC4\xAE" => '&#28448;',
	"\xC4\xAF" => '&#23518;',
	"\xC4\xB0" => '&#38476;',
	"\xC4\xB1" => '&#35851;',
	"\xC4\xB2" => '&#29279;',
	"\xC4\xB3" => '&#26576;',
	"\xC4\xB4" => '&#25287;',
	"\xC4\xB5" => '&#29281;',
	"\xC4\xB6" => '&#20137;',
	"\xC4\xB7" => '&#22982;',
	"\xC4\xB8" => '&#27597;',
	"\xC4\xB9" => '&#22675;',
	"\xC4\xBA" => '&#26286;',
	"\xC4\xBB" => '&#24149;',
	"\xC4\xBC" => '&#21215;',
	"\xC4\xBD" => '&#24917;',
	"\xC4\xBE" => '&#26408;',
	"\xC4\xBF" => '&#30446;',
	"\xC4\xC0" => '&#30566;',
	"\xC4\xC1" => '&#29287;',
	"\xC4\xC2" => '&#31302;',
	"\xC4\xC3" => '&#25343;',
	"\xC4\xC4" => '&#21738;',
	"\xC4\xC5" => '&#21584;',
	"\xC4\xC6" => '&#38048;',
	"\xC4\xC7" => '&#37027;',
	"\xC4\xC8" => '&#23068;',
	"\xC4\xC9" => '&#32435;',
	"\xC4\xCA" => '&#27670;',
	"\xC4\xCB" => '&#20035;',
	"\xC4\xCC" => '&#22902;',
	"\xC4\xCD" => '&#32784;',
	"\xC4\xCE" => '&#22856;',
	"\xC4\xCF" => '&#21335;',
	"\xC4\xD0" => '&#30007;',
	"\xC4\xD1" => '&#38590;',
	"\xC4\xD2" => '&#22218;',
	"\xC4\xD3" => '&#25376;',
	"\xC4\xD4" => '&#33041;',
	"\xC4\xD5" => '&#24700;',
	"\xC4\xD6" => '&#38393;',
	"\xC4\xD7" => '&#28118;',
	"\xC4\xD8" => '&#21602;',
	"\xC4\xD9" => '&#39297;',
	"\xC4\xDA" => '&#20869;',
	"\xC4\xDB" => '&#23273;',
	"\xC4\xDC" => '&#33021;',
	"\xC4\xDD" => '&#22958;',
	"\xC4\xDE" => '&#38675;',
	"\xC4\xDF" => '&#20522;',
	"\xC4\xE0" => '&#27877;',
	"\xC4\xE1" => '&#23612;',
	"\xC4\xE2" => '&#25311;',
	"\xC4\xE3" => '&#20320;',
	"\xC4\xE4" => '&#21311;',
	"\xC4\xE5" => '&#33147;',
	"\xC4\xE6" => '&#36870;',
	"\xC4\xE7" => '&#28346;',
	"\xC4\xE8" => '&#34091;',
	"\xC4\xE9" => '&#25288;',
	"\xC4\xEA" => '&#24180;',
	"\xC4\xEB" => '&#30910;',
	"\xC4\xEC" => '&#25781;',
	"\xC4\xED" => '&#25467;',
	"\xC4\xEE" => '&#24565;',
	"\xC4\xEF" => '&#23064;',
	"\xC4\xF0" => '&#37247;',
	"\xC4\xF1" => '&#40479;',
	"\xC4\xF2" => '&#23615;',
	"\xC4\xF3" => '&#25423;',
	"\xC4\xF4" => '&#32834;',
	"\xC4\xF5" => '&#23421;',
	"\xC4\xF6" => '&#21870;',
	"\xC4\xF7" => '&#38218;',
	"\xC4\xF8" => '&#38221;',
	"\xC4\xF9" => '&#28037;',
	"\xC4\xFA" => '&#24744;',
	"\xC4\xFB" => '&#26592;',
	"\xC4\xFC" => '&#29406;',
	"\xC4\xFD" => '&#20957;',
	"\xC4\xFE" => '&#23425;',
	"\xC5\xA1" => '&#25319;',
	"\xC5\xA2" => '&#27870;',
	"\xC5\xA3" => '&#29275;',
	"\xC5\xA4" => '&#25197;',
	"\xC5\xA5" => '&#38062;',
	"\xC5\xA6" => '&#32445;',
	"\xC5\xA7" => '&#33043;',
	"\xC5\xA8" => '&#27987;',
	"\xC5\xA9" => '&#20892;',
	"\xC5\xAA" => '&#24324;',
	"\xC5\xAB" => '&#22900;',
	"\xC5\xAC" => '&#21162;',
	"\xC5\xAD" => '&#24594;',
	"\xC5\xAE" => '&#22899;',
	"\xC5\xAF" => '&#26262;',
	"\xC5\xB0" => '&#34384;',
	"\xC5\xB1" => '&#30111;',
	"\xC5\xB2" => '&#25386;',
	"\xC5\xB3" => '&#25062;',
	"\xC5\xB4" => '&#31983;',
	"\xC5\xB5" => '&#35834;',
	"\xC5\xB6" => '&#21734;',
	"\xC5\xB7" => '&#27431;',
	"\xC5\xB8" => '&#40485;',
	"\xC5\xB9" => '&#27572;',
	"\xC5\xBA" => '&#34261;',
	"\xC5\xBB" => '&#21589;',
	"\xC5\xBC" => '&#20598;',
	"\xC5\xBD" => '&#27812;',
	"\xC5\xBE" => '&#21866;',
	"\xC5\xBF" => '&#36276;',
	"\xC5\xC0" => '&#29228;',
	"\xC5\xC1" => '&#24085;',
	"\xC5\xC2" => '&#24597;',
	"\xC5\xC3" => '&#29750;',
	"\xC5\xC4" => '&#25293;',
	"\xC5\xC5" => '&#25490;',
	"\xC5\xC6" => '&#29260;',
	"\xC5\xC7" => '&#24472;',
	"\xC5\xC8" => '&#28227;',
	"\xC5\xC9" => '&#27966;',
	"\xC5\xCA" => '&#25856;',
	"\xC5\xCB" => '&#28504;',
	"\xC5\xCC" => '&#30424;',
	"\xC5\xCD" => '&#30928;',
	"\xC5\xCE" => '&#30460;',
	"\xC5\xCF" => '&#30036;',
	"\xC5\xD0" => '&#21028;',
	"\xC5\xD1" => '&#21467;',
	"\xC5\xD2" => '&#20051;',
	"\xC5\xD3" => '&#24222;',
	"\xC5\xD4" => '&#26049;',
	"\xC5\xD5" => '&#32810;',
	"\xC5\xD6" => '&#32982;',
	"\xC5\xD7" => '&#25243;',
	"\xC5\xD8" => '&#21638;',
	"\xC5\xD9" => '&#21032;',
	"\xC5\xDA" => '&#28846;',
	"\xC5\xDB" => '&#34957;',
	"\xC5\xDC" => '&#36305;',
	"\xC5\xDD" => '&#27873;',
	"\xC5\xDE" => '&#21624;',
	"\xC5\xDF" => '&#32986;',
	"\xC5\xE0" => '&#22521;',
	"\xC5\xE1" => '&#35060;',
	"\xC5\xE2" => '&#36180;',
	"\xC5\xE3" => '&#38506;',
	"\xC5\xE4" => '&#37197;',
	"\xC5\xE5" => '&#20329;',
	"\xC5\xE6" => '&#27803;',
	"\xC5\xE7" => '&#21943;',
	"\xC5\xE8" => '&#30406;',
	"\xC5\xE9" => '&#30768;',
	"\xC5\xEA" => '&#25256;',
	"\xC5\xEB" => '&#28921;',
	"\xC5\xEC" => '&#28558;',
	"\xC5\xED" => '&#24429;',
	"\xC5\xEE" => '&#34028;',
	"\xC5\xEF" => '&#26842;',
	"\xC5\xF0" => '&#30844;',
	"\xC5\xF1" => '&#31735;',
	"\xC5\xF2" => '&#33192;',
	"\xC5\xF3" => '&#26379;',
	"\xC5\xF4" => '&#40527;',
	"\xC5\xF5" => '&#25447;',
	"\xC5\xF6" => '&#30896;',
	"\xC5\xF7" => '&#22383;',
	"\xC5\xF8" => '&#30738;',
	"\xC5\xF9" => '&#38713;',
	"\xC5\xFA" => '&#25209;',
	"\xC5\xFB" => '&#25259;',
	"\xC5\xFC" => '&#21128;',
	"\xC5\xFD" => '&#29749;',
	"\xC5\xFE" => '&#27607;',
	"\xC6\xA1" => '&#21860;',
	"\xC6\xA2" => '&#33086;',
	"\xC6\xA3" => '&#30130;',
	"\xC6\xA4" => '&#30382;',
	"\xC6\xA5" => '&#21305;',
	"\xC6\xA6" => '&#30174;',
	"\xC6\xA7" => '&#20731;',
	"\xC6\xA8" => '&#23617;',
	"\xC6\xA9" => '&#35692;',
	"\xC6\xAA" => '&#31687;',
	"\xC6\xAB" => '&#20559;',
	"\xC6\xAC" => '&#29255;',
	"\xC6\xAD" => '&#39575;',
	"\xC6\xAE" => '&#39128;',
	"\xC6\xAF" => '&#28418;',
	"\xC6\xB0" => '&#29922;',
	"\xC6\xB1" => '&#31080;',
	"\xC6\xB2" => '&#25735;',
	"\xC6\xB3" => '&#30629;',
	"\xC6\xB4" => '&#25340;',
	"\xC6\xB5" => '&#39057;',
	"\xC6\xB6" => '&#36139;',
	"\xC6\xB7" => '&#21697;',
	"\xC6\xB8" => '&#32856;',
	"\xC6\xB9" => '&#20050;',
	"\xC6\xBA" => '&#22378;',
	"\xC6\xBB" => '&#33529;',
	"\xC6\xBC" => '&#33805;',
	"\xC6\xBD" => '&#24179;',
	"\xC6\xBE" => '&#20973;',
	"\xC6\xBF" => '&#29942;',
	"\xC6\xC0" => '&#35780;',
	"\xC6\xC1" => '&#23631;',
	"\xC6\xC2" => '&#22369;',
	"\xC6\xC3" => '&#27900;',
	"\xC6\xC4" => '&#39047;',
	"\xC6\xC5" => '&#23110;',
	"\xC6\xC6" => '&#30772;',
	"\xC6\xC7" => '&#39748;',
	"\xC6\xC8" => '&#36843;',
	"\xC6\xC9" => '&#31893;',
	"\xC6\xCA" => '&#21078;',
	"\xC6\xCB" => '&#25169;',
	"\xC6\xCC" => '&#38138;',
	"\xC6\xCD" => '&#20166;',
	"\xC6\xCE" => '&#33670;',
	"\xC6\xCF" => '&#33889;',
	"\xC6\xD0" => '&#33769;',
	"\xC6\xD1" => '&#33970;',
	"\xC6\xD2" => '&#22484;',
	"\xC6\xD3" => '&#26420;',
	"\xC6\xD4" => '&#22275;',
	"\xC6\xD5" => '&#26222;',
	"\xC6\xD6" => '&#28006;',
	"\xC6\xD7" => '&#35889;',
	"\xC6\xD8" => '&#26333;',
	"\xC6\xD9" => '&#28689;',
	"\xC6\xDA" => '&#26399;',
	"\xC6\xDB" => '&#27450;',
	"\xC6\xDC" => '&#26646;',
	"\xC6\xDD" => '&#25114;',
	"\xC6\xDE" => '&#22971;',
	"\xC6\xDF" => '&#19971;',
	"\xC6\xE0" => '&#20932;',
	"\xC6\xE1" => '&#28422;',
	"\xC6\xE2" => '&#26578;',
	"\xC6\xE3" => '&#27791;',
	"\xC6\xE4" => '&#20854;',
	"\xC6\xE5" => '&#26827;',
	"\xC6\xE6" => '&#22855;',
	"\xC6\xE7" => '&#27495;',
	"\xC6\xE8" => '&#30054;',
	"\xC6\xE9" => '&#23822;',
	"\xC6\xEA" => '&#33040;',
	"\xC6\xEB" => '&#40784;',
	"\xC6\xEC" => '&#26071;',
	"\xC6\xED" => '&#31048;',
	"\xC6\xEE" => '&#31041;',
	"\xC6\xEF" => '&#39569;',
	"\xC6\xF0" => '&#36215;',
	"\xC6\xF1" => '&#23682;',
	"\xC6\xF2" => '&#20062;',
	"\xC6\xF3" => '&#20225;',
	"\xC6\xF4" => '&#21551;',
	"\xC6\xF5" => '&#22865;',
	"\xC6\xF6" => '&#30732;',
	"\xC6\xF7" => '&#22120;',
	"\xC6\xF8" => '&#27668;',
	"\xC6\xF9" => '&#36804;',
	"\xC6\xFA" => '&#24323;',
	"\xC6\xFB" => '&#27773;',
	"\xC6\xFC" => '&#27875;',
	"\xC6\xFD" => '&#35755;',
	"\xC6\xFE" => '&#25488;',
	"\xC7\xA1" => '&#24688;',
	"\xC7\xA2" => '&#27965;',
	"\xC7\xA3" => '&#29301;',
	"\xC7\xA4" => '&#25190;',
	"\xC7\xA5" => '&#38030;',
	"\xC7\xA6" => '&#38085;',
	"\xC7\xA7" => '&#21315;',
	"\xC7\xA8" => '&#36801;',
	"\xC7\xA9" => '&#31614;',
	"\xC7\xAA" => '&#20191;',
	"\xC7\xAB" => '&#35878;',
	"\xC7\xAC" => '&#20094;',
	"\xC7\xAD" => '&#40660;',
	"\xC7\xAE" => '&#38065;',
	"\xC7\xAF" => '&#38067;',
	"\xC7\xB0" => '&#21069;',
	"\xC7\xB1" => '&#28508;',
	"\xC7\xB2" => '&#36963;',
	"\xC7\xB3" => '&#27973;',
	"\xC7\xB4" => '&#35892;',
	"\xC7\xB5" => '&#22545;',
	"\xC7\xB6" => '&#23884;',
	"\xC7\xB7" => '&#27424;',
	"\xC7\xB8" => '&#27465;',
	"\xC7\xB9" => '&#26538;',
	"\xC7\xBA" => '&#21595;',
	"\xC7\xBB" => '&#33108;',
	"\xC7\xBC" => '&#32652;',
	"\xC7\xBD" => '&#22681;',
	"\xC7\xBE" => '&#34103;',
	"\xC7\xBF" => '&#24378;',
	"\xC7\xC0" => '&#25250;',
	"\xC7\xC1" => '&#27207;',
	"\xC7\xC2" => '&#38201;',
	"\xC7\xC3" => '&#25970;',
	"\xC7\xC4" => '&#24708;',
	"\xC7\xC5" => '&#26725;',
	"\xC7\xC6" => '&#30631;',
	"\xC7\xC7" => '&#20052;',
	"\xC7\xC8" => '&#20392;',
	"\xC7\xC9" => '&#24039;',
	"\xC7\xCA" => '&#38808;',
	"\xC7\xCB" => '&#25772;',
	"\xC7\xCC" => '&#32728;',
	"\xC7\xCD" => '&#23789;',
	"\xC7\xCE" => '&#20431;',
	"\xC7\xCF" => '&#31373;',
	"\xC7\xD0" => '&#20999;',
	"\xC7\xD1" => '&#33540;',
	"\xC7\xD2" => '&#19988;',
	"\xC7\xD3" => '&#24623;',
	"\xC7\xD4" => '&#31363;',
	"\xC7\xD5" => '&#38054;',
	"\xC7\xD6" => '&#20405;',
	"\xC7\xD7" => '&#20146;',
	"\xC7\xD8" => '&#31206;',
	"\xC7\xD9" => '&#29748;',
	"\xC7\xDA" => '&#21220;',
	"\xC7\xDB" => '&#33465;',
	"\xC7\xDC" => '&#25810;',
	"\xC7\xDD" => '&#31165;',
	"\xC7\xDE" => '&#23517;',
	"\xC7\xDF" => '&#27777;',
	"\xC7\xE0" => '&#38738;',
	"\xC7\xE1" => '&#36731;',
	"\xC7\xE2" => '&#27682;',
	"\xC7\xE3" => '&#20542;',
	"\xC7\xE4" => '&#21375;',
	"\xC7\xE5" => '&#28165;',
	"\xC7\xE6" => '&#25806;',
	"\xC7\xE7" => '&#26228;',
	"\xC7\xE8" => '&#27696;',
	"\xC7\xE9" => '&#24773;',
	"\xC7\xEA" => '&#39031;',
	"\xC7\xEB" => '&#35831;',
	"\xC7\xEC" => '&#24198;',
	"\xC7\xED" => '&#29756;',
	"\xC7\xEE" => '&#31351;',
	"\xC7\xEF" => '&#31179;',
	"\xC7\xF0" => '&#19992;',
	"\xC7\xF1" => '&#37041;',
	"\xC7\xF2" => '&#29699;',
	"\xC7\xF3" => '&#27714;',
	"\xC7\xF4" => '&#22234;',
	"\xC7\xF5" => '&#37195;',
	"\xC7\xF6" => '&#27845;',
	"\xC7\xF7" => '&#36235;',
	"\xC7\xF8" => '&#21306;',
	"\xC7\xF9" => '&#34502;',
	"\xC7\xFA" => '&#26354;',
	"\xC7\xFB" => '&#36527;',
	"\xC7\xFC" => '&#23624;',
	"\xC7\xFD" => '&#39537;',
	"\xC7\xFE" => '&#28192;',
	"\xC8\xA1" => '&#21462;',
	"\xC8\xA2" => '&#23094;',
	"\xC8\xA3" => '&#40843;',
	"\xC8\xA4" => '&#36259;',
	"\xC8\xA5" => '&#21435;',
	"\xC8\xA6" => '&#22280;',
	"\xC8\xA7" => '&#39079;',
	"\xC8\xA8" => '&#26435;',
	"\xC8\xA9" => '&#37275;',
	"\xC8\xAA" => '&#27849;',
	"\xC8\xAB" => '&#20840;',
	"\xC8\xAC" => '&#30154;',
	"\xC8\xAD" => '&#25331;',
	"\xC8\xAE" => '&#29356;',
	"\xC8\xAF" => '&#21048;',
	"\xC8\xB0" => '&#21149;',
	"\xC8\xB1" => '&#32570;',
	"\xC8\xB2" => '&#28820;',
	"\xC8\xB3" => '&#30264;',
	"\xC8\xB4" => '&#21364;',
	"\xC8\xB5" => '&#40522;',
	"\xC8\xB6" => '&#27063;',
	"\xC8\xB7" => '&#30830;',
	"\xC8\xB8" => '&#38592;',
	"\xC8\xB9" => '&#35033;',
	"\xC8\xBA" => '&#32676;',
	"\xC8\xBB" => '&#28982;',
	"\xC8\xBC" => '&#29123;',
	"\xC8\xBD" => '&#20873;',
	"\xC8\xBE" => '&#26579;',
	"\xC8\xBF" => '&#29924;',
	"\xC8\xC0" => '&#22756;',
	"\xC8\xC1" => '&#25880;',
	"\xC8\xC2" => '&#22199;',
	"\xC8\xC3" => '&#35753;',
	"\xC8\xC4" => '&#39286;',
	"\xC8\xC5" => '&#25200;',
	"\xC8\xC6" => '&#32469;',
	"\xC8\xC7" => '&#24825;',
	"\xC8\xC8" => '&#28909;',
	"\xC8\xC9" => '&#22764;',
	"\xC8\xCA" => '&#20161;',
	"\xC8\xCB" => '&#20154;',
	"\xC8\xCC" => '&#24525;',
	"\xC8\xCD" => '&#38887;',
	"\xC8\xCE" => '&#20219;',
	"\xC8\xCF" => '&#35748;',
	"\xC8\xD0" => '&#20995;',
	"\xC8\xD1" => '&#22922;',
	"\xC8\xD2" => '&#32427;',
	"\xC8\xD3" => '&#25172;',
	"\xC8\xD4" => '&#20173;',
	"\xC8\xD5" => '&#26085;',
	"\xC8\xD6" => '&#25102;',
	"\xC8\xD7" => '&#33592;',
	"\xC8\xD8" => '&#33993;',
	"\xC8\xD9" => '&#33635;',
	"\xC8\xDA" => '&#34701;',
	"\xC8\xDB" => '&#29076;',
	"\xC8\xDC" => '&#28342;',
	"\xC8\xDD" => '&#23481;',
	"\xC8\xDE" => '&#32466;',
	"\xC8\xDF" => '&#20887;',
	"\xC8\xE0" => '&#25545;',
	"\xC8\xE1" => '&#26580;',
	"\xC8\xE2" => '&#32905;',
	"\xC8\xE3" => '&#33593;',
	"\xC8\xE4" => '&#34837;',
	"\xC8\xE5" => '&#20754;',
	"\xC8\xE6" => '&#23418;',
	"\xC8\xE7" => '&#22914;',
	"\xC8\xE8" => '&#36785;',
	"\xC8\xE9" => '&#20083;',
	"\xC8\xEA" => '&#27741;',
	"\xC8\xEB" => '&#20837;',
	"\xC8\xEC" => '&#35109;',
	"\xC8\xED" => '&#36719;',
	"\xC8\xEE" => '&#38446;',
	"\xC8\xEF" => '&#34122;',
	"\xC8\xF0" => '&#29790;',
	"\xC8\xF1" => '&#38160;',
	"\xC8\xF2" => '&#38384;',
	"\xC8\xF3" => '&#28070;',
	"\xC8\xF4" => '&#33509;',
	"\xC8\xF5" => '&#24369;',
	"\xC8\xF6" => '&#25746;',
	"\xC8\xF7" => '&#27922;',
	"\xC8\xF8" => '&#33832;',
	"\xC8\xF9" => '&#33134;',
	"\xC8\xFA" => '&#40131;',
	"\xC8\xFB" => '&#22622;',
	"\xC8\xFC" => '&#36187;',
	"\xC8\xFD" => '&#19977;',
	"\xC8\xFE" => '&#21441;',
	"\xC9\xA1" => '&#20254;',
	"\xC9\xA2" => '&#25955;',
	"\xC9\xA3" => '&#26705;',
	"\xC9\xA4" => '&#21971;',
	"\xC9\xA5" => '&#20007;',
	"\xC9\xA6" => '&#25620;',
	"\xC9\xA7" => '&#39578;',
	"\xC9\xA8" => '&#25195;',
	"\xC9\xA9" => '&#23234;',
	"\xC9\xAA" => '&#29791;',
	"\xC9\xAB" => '&#33394;',
	"\xC9\xAC" => '&#28073;',
	"\xC9\xAD" => '&#26862;',
	"\xC9\xAE" => '&#20711;',
	"\xC9\xAF" => '&#33678;',
	"\xC9\xB0" => '&#30722;',
	"\xC9\xB1" => '&#26432;',
	"\xC9\xB2" => '&#21049;',
	"\xC9\xB3" => '&#27801;',
	"\xC9\xB4" => '&#32433;',
	"\xC9\xB5" => '&#20667;',
	"\xC9\xB6" => '&#21861;',
	"\xC9\xB7" => '&#29022;',
	"\xC9\xB8" => '&#31579;',
	"\xC9\xB9" => '&#26194;',
	"\xC9\xBA" => '&#29642;',
	"\xC9\xBB" => '&#33515;',
	"\xC9\xBC" => '&#26441;',
	"\xC9\xBD" => '&#23665;',
	"\xC9\xBE" => '&#21024;',
	"\xC9\xBF" => '&#29053;',
	"\xC9\xC0" => '&#34923;',
	"\xC9\xC1" => '&#38378;',
	"\xC9\xC2" => '&#38485;',
	"\xC9\xC3" => '&#25797;',
	"\xC9\xC4" => '&#36193;',
	"\xC9\xC5" => '&#33203;',
	"\xC9\xC6" => '&#21892;',
	"\xC9\xC7" => '&#27733;',
	"\xC9\xC8" => '&#25159;',
	"\xC9\xC9" => '&#32558;',
	"\xC9\xCA" => '&#22674;',
	"\xC9\xCB" => '&#20260;',
	"\xC9\xCC" => '&#21830;',
	"\xC9\xCD" => '&#36175;',
	"\xC9\xCE" => '&#26188;',
	"\xC9\xCF" => '&#19978;',
	"\xC9\xD0" => '&#23578;',
	"\xC9\xD1" => '&#35059;',
	"\xC9\xD2" => '&#26786;',
	"\xC9\xD3" => '&#25422;',
	"\xC9\xD4" => '&#31245;',
	"\xC9\xD5" => '&#28903;',
	"\xC9\xD6" => '&#33421;',
	"\xC9\xD7" => '&#21242;',
	"\xC9\xD8" => '&#38902;',
	"\xC9\xD9" => '&#23569;',
	"\xC9\xDA" => '&#21736;',
	"\xC9\xDB" => '&#37045;',
	"\xC9\xDC" => '&#32461;',
	"\xC9\xDD" => '&#22882;',
	"\xC9\xDE" => '&#36170;',
	"\xC9\xDF" => '&#34503;',
	"\xC9\xE0" => '&#33292;',
	"\xC9\xE1" => '&#33293;',
	"\xC9\xE2" => '&#36198;',
	"\xC9\xE3" => '&#25668;',
	"\xC9\xE4" => '&#23556;',
	"\xC9\xE5" => '&#24913;',
	"\xC9\xE6" => '&#28041;',
	"\xC9\xE7" => '&#31038;',
	"\xC9\xE8" => '&#35774;',
	"\xC9\xE9" => '&#30775;',
	"\xC9\xEA" => '&#30003;',
	"\xC9\xEB" => '&#21627;',
	"\xC9\xEC" => '&#20280;',
	"\xC9\xED" => '&#36523;',
	"\xC9\xEE" => '&#28145;',
	"\xC9\xEF" => '&#23072;',
	"\xC9\xF0" => '&#32453;',
	"\xC9\xF1" => '&#31070;',
	"\xC9\xF2" => '&#27784;',
	"\xC9\xF3" => '&#23457;',
	"\xC9\xF4" => '&#23158;',
	"\xC9\xF5" => '&#29978;',
	"\xC9\xF6" => '&#32958;',
	"\xC9\xF7" => '&#24910;',
	"\xC9\xF8" => '&#28183;',
	"\xC9\xF9" => '&#22768;',
	"\xC9\xFA" => '&#29983;',
	"\xC9\xFB" => '&#29989;',
	"\xC9\xFC" => '&#29298;',
	"\xC9\xFD" => '&#21319;',
	"\xC9\xFE" => '&#32499;',
	"\xCA\xA1" => '&#30465;',
	"\xCA\xA2" => '&#30427;',
	"\xCA\xA3" => '&#21097;',
	"\xCA\xA4" => '&#32988;',
	"\xCA\xA5" => '&#22307;',
	"\xCA\xA6" => '&#24072;',
	"\xCA\xA7" => '&#22833;',
	"\xCA\xA8" => '&#29422;',
	"\xCA\xA9" => '&#26045;',
	"\xCA\xAA" => '&#28287;',
	"\xCA\xAB" => '&#35799;',
	"\xCA\xAC" => '&#23608;',
	"\xCA\xAD" => '&#34417;',
	"\xCA\xAE" => '&#21313;',
	"\xCA\xAF" => '&#30707;',
	"\xCA\xB0" => '&#25342;',
	"\xCA\xB1" => '&#26102;',
	"\xCA\xB2" => '&#20160;',
	"\xCA\xB3" => '&#39135;',
	"\xCA\xB4" => '&#34432;',
	"\xCA\xB5" => '&#23454;',
	"\xCA\xB6" => '&#35782;',
	"\xCA\xB7" => '&#21490;',
	"\xCA\xB8" => '&#30690;',
	"\xCA\xB9" => '&#20351;',
	"\xCA\xBA" => '&#23630;',
	"\xCA\xBB" => '&#39542;',
	"\xCA\xBC" => '&#22987;',
	"\xCA\xBD" => '&#24335;',
	"\xCA\xBE" => '&#31034;',
	"\xCA\xBF" => '&#22763;',
	"\xCA\xC0" => '&#19990;',
	"\xCA\xC1" => '&#26623;',
	"\xCA\xC2" => '&#20107;',
	"\xCA\xC3" => '&#25325;',
	"\xCA\xC4" => '&#35475;',
	"\xCA\xC5" => '&#36893;',
	"\xCA\xC6" => '&#21183;',
	"\xCA\xC7" => '&#26159;',
	"\xCA\xC8" => '&#21980;',
	"\xCA\xC9" => '&#22124;',
	"\xCA\xCA" => '&#36866;',
	"\xCA\xCB" => '&#20181;',
	"\xCA\xCC" => '&#20365;',
	"\xCA\xCD" => '&#37322;',
	"\xCA\xCE" => '&#39280;',
	"\xCA\xCF" => '&#27663;',
	"\xCA\xD0" => '&#24066;',
	"\xCA\xD1" => '&#24643;',
	"\xCA\xD2" => '&#23460;',
	"\xCA\xD3" => '&#35270;',
	"\xCA\xD4" => '&#35797;',
	"\xCA\xD5" => '&#25910;',
	"\xCA\xD6" => '&#25163;',
	"\xCA\xD7" => '&#39318;',
	"\xCA\xD8" => '&#23432;',
	"\xCA\xD9" => '&#23551;',
	"\xCA\xDA" => '&#25480;',
	"\xCA\xDB" => '&#21806;',
	"\xCA\xDC" => '&#21463;',
	"\xCA\xDD" => '&#30246;',
	"\xCA\xDE" => '&#20861;',
	"\xCA\xDF" => '&#34092;',
	"\xCA\xE0" => '&#26530;',
	"\xCA\xE1" => '&#26803;',
	"\xCA\xE2" => '&#27530;',
	"\xCA\xE3" => '&#25234;',
	"\xCA\xE4" => '&#36755;',
	"\xCA\xE5" => '&#21460;',
	"\xCA\xE6" => '&#33298;',
	"\xCA\xE7" => '&#28113;',
	"\xCA\xE8" => '&#30095;',
	"\xCA\xE9" => '&#20070;',
	"\xCA\xEA" => '&#36174;',
	"\xCA\xEB" => '&#23408;',
	"\xCA\xEC" => '&#29087;',
	"\xCA\xED" => '&#34223;',
	"\xCA\xEE" => '&#26257;',
	"\xCA\xEF" => '&#26329;',
	"\xCA\xF0" => '&#32626;',
	"\xCA\xF1" => '&#34560;',
	"\xCA\xF2" => '&#40653;',
	"\xCA\xF3" => '&#40736;',
	"\xCA\xF4" => '&#23646;',
	"\xCA\xF5" => '&#26415;',
	"\xCA\xF6" => '&#36848;',
	"\xCA\xF7" => '&#26641;',
	"\xCA\xF8" => '&#26463;',
	"\xCA\xF9" => '&#25101;',
	"\xCA\xFA" => '&#31446;',
	"\xCA\xFB" => '&#22661;',
	"\xCA\xFC" => '&#24246;',
	"\xCA\xFD" => '&#25968;',
	"\xCA\xFE" => '&#28465;',
	"\xCB\xA1" => '&#24661;',
	"\xCB\xA2" => '&#21047;',
	"\xCB\xA3" => '&#32781;',
	"\xCB\xA4" => '&#25684;',
	"\xCB\xA5" => '&#34928;',
	"\xCB\xA6" => '&#29993;',
	"\xCB\xA7" => '&#24069;',
	"\xCB\xA8" => '&#26643;',
	"\xCB\xA9" => '&#25332;',
	"\xCB\xAA" => '&#38684;',
	"\xCB\xAB" => '&#21452;',
	"\xCB\xAC" => '&#29245;',
	"\xCB\xAD" => '&#35841;',
	"\xCB\xAE" => '&#27700;',
	"\xCB\xAF" => '&#30561;',
	"\xCB\xB0" => '&#31246;',
	"\xCB\xB1" => '&#21550;',
	"\xCB\xB2" => '&#30636;',
	"\xCB\xB3" => '&#39034;',
	"\xCB\xB4" => '&#33308;',
	"\xCB\xB5" => '&#35828;',
	"\xCB\xB6" => '&#30805;',
	"\xCB\xB7" => '&#26388;',
	"\xCB\xB8" => '&#28865;',
	"\xCB\xB9" => '&#26031;',
	"\xCB\xBA" => '&#25749;',
	"\xCB\xBB" => '&#22070;',
	"\xCB\xBC" => '&#24605;',
	"\xCB\xBD" => '&#31169;',
	"\xCB\xBE" => '&#21496;',
	"\xCB\xBF" => '&#19997;',
	"\xCB\xC0" => '&#27515;',
	"\xCB\xC1" => '&#32902;',
	"\xCB\xC2" => '&#23546;',
	"\xCB\xC3" => '&#21987;',
	"\xCB\xC4" => '&#22235;',
	"\xCB\xC5" => '&#20282;',
	"\xCB\xC6" => '&#20284;',
	"\xCB\xC7" => '&#39282;',
	"\xCB\xC8" => '&#24051;',
	"\xCB\xC9" => '&#26494;',
	"\xCB\xCA" => '&#32824;',
	"\xCB\xCB" => '&#24578;',
	"\xCB\xCC" => '&#39042;',
	"\xCB\xCD" => '&#36865;',
	"\xCB\xCE" => '&#23435;',
	"\xCB\xCF" => '&#35772;',
	"\xCB\xD0" => '&#35829;',
	"\xCB\xD1" => '&#25628;',
	"\xCB\xD2" => '&#33368;',
	"\xCB\xD3" => '&#25822;',
	"\xCB\xD4" => '&#22013;',
	"\xCB\xD5" => '&#33487;',
	"\xCB\xD6" => '&#37221;',
	"\xCB\xD7" => '&#20439;',
	"\xCB\xD8" => '&#32032;',
	"\xCB\xD9" => '&#36895;',
	"\xCB\xDA" => '&#31903;',
	"\xCB\xDB" => '&#20723;',
	"\xCB\xDC" => '&#22609;',
	"\xCB\xDD" => '&#28335;',
	"\xCB\xDE" => '&#23487;',
	"\xCB\xDF" => '&#35785;',
	"\xCB\xE0" => '&#32899;',
	"\xCB\xE1" => '&#37240;',
	"\xCB\xE2" => '&#33948;',
	"\xCB\xE3" => '&#31639;',
	"\xCB\xE4" => '&#34429;',
	"\xCB\xE5" => '&#38539;',
	"\xCB\xE6" => '&#38543;',
	"\xCB\xE7" => '&#32485;',
	"\xCB\xE8" => '&#39635;',
	"\xCB\xE9" => '&#30862;',
	"\xCB\xEA" => '&#23681;',
	"\xCB\xEB" => '&#31319;',
	"\xCB\xEC" => '&#36930;',
	"\xCB\xED" => '&#38567;',
	"\xCB\xEE" => '&#31071;',
	"\xCB\xEF" => '&#23385;',
	"\xCB\xF0" => '&#25439;',
	"\xCB\xF1" => '&#31499;',
	"\xCB\xF2" => '&#34001;',
	"\xCB\xF3" => '&#26797;',
	"\xCB\xF4" => '&#21766;',
	"\xCB\xF5" => '&#32553;',
	"\xCB\xF6" => '&#29712;',
	"\xCB\xF7" => '&#32034;',
	"\xCB\xF8" => '&#38145;',
	"\xCB\xF9" => '&#25152;',
	"\xCB\xFA" => '&#22604;',
	"\xCB\xFB" => '&#20182;',
	"\xCB\xFC" => '&#23427;',
	"\xCB\xFD" => '&#22905;',
	"\xCB\xFE" => '&#22612;',
	"\xCC\xA1" => '&#29549;',
	"\xCC\xA2" => '&#25374;',
	"\xCC\xA3" => '&#36427;',
	"\xCC\xA4" => '&#36367;',
	"\xCC\xA5" => '&#32974;',
	"\xCC\xA6" => '&#33492;',
	"\xCC\xA7" => '&#25260;',
	"\xCC\xA8" => '&#21488;',
	"\xCC\xA9" => '&#27888;',
	"\xCC\xAA" => '&#37214;',
	"\xCC\xAB" => '&#22826;',
	"\xCC\xAC" => '&#24577;',
	"\xCC\xAD" => '&#27760;',
	"\xCC\xAE" => '&#22349;',
	"\xCC\xAF" => '&#25674;',
	"\xCC\xB0" => '&#36138;',
	"\xCC\xB1" => '&#30251;',
	"\xCC\xB2" => '&#28393;',
	"\xCC\xB3" => '&#22363;',
	"\xCC\xB4" => '&#27264;',
	"\xCC\xB5" => '&#30192;',
	"\xCC\xB6" => '&#28525;',
	"\xCC\xB7" => '&#35885;',
	"\xCC\xB8" => '&#35848;',
	"\xCC\xB9" => '&#22374;',
	"\xCC\xBA" => '&#27631;',
	"\xCC\xBB" => '&#34962;',
	"\xCC\xBC" => '&#30899;',
	"\xCC\xBD" => '&#25506;',
	"\xCC\xBE" => '&#21497;',
	"\xCC\xBF" => '&#28845;',
	"\xCC\xC0" => '&#27748;',
	"\xCC\xC1" => '&#22616;',
	"\xCC\xC2" => '&#25642;',
	"\xCC\xC3" => '&#22530;',
	"\xCC\xC4" => '&#26848;',
	"\xCC\xC5" => '&#33179;',
	"\xCC\xC6" => '&#21776;',
	"\xCC\xC7" => '&#31958;',
	"\xCC\xC8" => '&#20504;',
	"\xCC\xC9" => '&#36538;',
	"\xCC\xCA" => '&#28108;',
	"\xCC\xCB" => '&#36255;',
	"\xCC\xCC" => '&#28907;',
	"\xCC\xCD" => '&#25487;',
	"\xCC\xCE" => '&#28059;',
	"\xCC\xCF" => '&#28372;',
	"\xCC\xD0" => '&#32486;',
	"\xCC\xD1" => '&#33796;',
	"\xCC\xD2" => '&#26691;',
	"\xCC\xD3" => '&#36867;',
	"\xCC\xD4" => '&#28120;',
	"\xCC\xD5" => '&#38518;',
	"\xCC\xD6" => '&#35752;',
	"\xCC\xD7" => '&#22871;',
	"\xCC\xD8" => '&#29305;',
	"\xCC\xD9" => '&#34276;',
	"\xCC\xDA" => '&#33150;',
	"\xCC\xDB" => '&#30140;',
	"\xCC\xDC" => '&#35466;',
	"\xCC\xDD" => '&#26799;',
	"\xCC\xDE" => '&#21076;',
	"\xCC\xDF" => '&#36386;',
	"\xCC\xE0" => '&#38161;',
	"\xCC\xE1" => '&#25552;',
	"\xCC\xE2" => '&#39064;',
	"\xCC\xE3" => '&#36420;',
	"\xCC\xE4" => '&#21884;',
	"\xCC\xE5" => '&#20307;',
	"\xCC\xE6" => '&#26367;',
	"\xCC\xE7" => '&#22159;',
	"\xCC\xE8" => '&#24789;',
	"\xCC\xE9" => '&#28053;',
	"\xCC\xEA" => '&#21059;',
	"\xCC\xEB" => '&#23625;',
	"\xCC\xEC" => '&#22825;',
	"\xCC\xED" => '&#28155;',
	"\xCC\xEE" => '&#22635;',
	"\xCC\xEF" => '&#30000;',
	"\xCC\xF0" => '&#29980;',
	"\xCC\xF1" => '&#24684;',
	"\xCC\xF2" => '&#33300;',
	"\xCC\xF3" => '&#33094;',
	"\xCC\xF4" => '&#25361;',
	"\xCC\xF5" => '&#26465;',
	"\xCC\xF6" => '&#36834;',
	"\xCC\xF7" => '&#30522;',
	"\xCC\xF8" => '&#36339;',
	"\xCC\xF9" => '&#36148;',
	"\xCC\xFA" => '&#38081;',
	"\xCC\xFB" => '&#24086;',
	"\xCC\xFC" => '&#21381;',
	"\xCC\xFD" => '&#21548;',
	"\xCC\xFE" => '&#28867;',
	"\xCD\xA1" => '&#27712;',
	"\xCD\xA2" => '&#24311;',
	"\xCD\xA3" => '&#20572;',
	"\xCD\xA4" => '&#20141;',
	"\xCD\xA5" => '&#24237;',
	"\xCD\xA6" => '&#25402;',
	"\xCD\xA7" => '&#33351;',
	"\xCD\xA8" => '&#36890;',
	"\xCD\xA9" => '&#26704;',
	"\xCD\xAA" => '&#37230;',
	"\xCD\xAB" => '&#30643;',
	"\xCD\xAC" => '&#21516;',
	"\xCD\xAD" => '&#38108;',
	"\xCD\xAE" => '&#24420;',
	"\xCD\xAF" => '&#31461;',
	"\xCD\xB0" => '&#26742;',
	"\xCD\xB1" => '&#25413;',
	"\xCD\xB2" => '&#31570;',
	"\xCD\xB3" => '&#32479;',
	"\xCD\xB4" => '&#30171;',
	"\xCD\xB5" => '&#20599;',
	"\xCD\xB6" => '&#25237;',
	"\xCD\xB7" => '&#22836;',
	"\xCD\xB8" => '&#36879;',
	"\xCD\xB9" => '&#20984;',
	"\xCD\xBA" => '&#31171;',
	"\xCD\xBB" => '&#31361;',
	"\xCD\xBC" => '&#22270;',
	"\xCD\xBD" => '&#24466;',
	"\xCD\xBE" => '&#36884;',
	"\xCD\xBF" => '&#28034;',
	"\xCD\xC0" => '&#23648;',
	"\xCD\xC1" => '&#22303;',
	"\xCD\xC2" => '&#21520;',
	"\xCD\xC3" => '&#20820;',
	"\xCD\xC4" => '&#28237;',
	"\xCD\xC5" => '&#22242;',
	"\xCD\xC6" => '&#25512;',
	"\xCD\xC7" => '&#39059;',
	"\xCD\xC8" => '&#33151;',
	"\xCD\xC9" => '&#34581;',
	"\xCD\xCA" => '&#35114;',
	"\xCD\xCB" => '&#36864;',
	"\xCD\xCC" => '&#21534;',
	"\xCD\xCD" => '&#23663;',
	"\xCD\xCE" => '&#33216;',
	"\xCD\xCF" => '&#25302;',
	"\xCD\xD0" => '&#25176;',
	"\xCD\xD1" => '&#33073;',
	"\xCD\xD2" => '&#40501;',
	"\xCD\xD3" => '&#38464;',
	"\xCD\xD4" => '&#39534;',
	"\xCD\xD5" => '&#39548;',
	"\xCD\xD6" => '&#26925;',
	"\xCD\xD7" => '&#22949;',
	"\xCD\xD8" => '&#25299;',
	"\xCD\xD9" => '&#21822;',
	"\xCD\xDA" => '&#25366;',
	"\xCD\xDB" => '&#21703;',
	"\xCD\xDC" => '&#34521;',
	"\xCD\xDD" => '&#27964;',
	"\xCD\xDE" => '&#23043;',
	"\xCD\xDF" => '&#29926;',
	"\xCD\xE0" => '&#34972;',
	"\xCD\xE1" => '&#27498;',
	"\xCD\xE2" => '&#22806;',
	"\xCD\xE3" => '&#35916;',
	"\xCD\xE4" => '&#24367;',
	"\xCD\xE5" => '&#28286;',
	"\xCD\xE6" => '&#29609;',
	"\xCD\xE7" => '&#39037;',
	"\xCD\xE8" => '&#20024;',
	"\xCD\xE9" => '&#28919;',
	"\xCD\xEA" => '&#23436;',
	"\xCD\xEB" => '&#30871;',
	"\xCD\xEC" => '&#25405;',
	"\xCD\xED" => '&#26202;',
	"\xCD\xEE" => '&#30358;',
	"\xCD\xEF" => '&#24779;',
	"\xCD\xF0" => '&#23451;',
	"\xCD\xF1" => '&#23113;',
	"\xCD\xF2" => '&#19975;',
	"\xCD\xF3" => '&#33109;',
	"\xCD\xF4" => '&#27754;',
	"\xCD\xF5" => '&#29579;',
	"\xCD\xF6" => '&#20129;',
	"\xCD\xF7" => '&#26505;',
	"\xCD\xF8" => '&#32593;',
	"\xCD\xF9" => '&#24448;',
	"\xCD\xFA" => '&#26106;',
	"\xCD\xFB" => '&#26395;',
	"\xCD\xFC" => '&#24536;',
	"\xCD\xFD" => '&#22916;',
	"\xCD\xFE" => '&#23041;',
	"\xCE\xA1" => '&#24013;',
	"\xCE\xA2" => '&#24494;',
	"\xCE\xA3" => '&#21361;',
	"\xCE\xA4" => '&#38886;',
	"\xCE\xA5" => '&#36829;',
	"\xCE\xA6" => '&#26693;',
	"\xCE\xA7" => '&#22260;',
	"\xCE\xA8" => '&#21807;',
	"\xCE\xA9" => '&#24799;',
	"\xCE\xAA" => '&#20026;',
	"\xCE\xAB" => '&#28493;',
	"\xCE\xAC" => '&#32500;',
	"\xCE\xAD" => '&#33479;',
	"\xCE\xAE" => '&#33806;',
	"\xCE\xAF" => '&#22996;',
	"\xCE\xB0" => '&#20255;',
	"\xCE\xB1" => '&#20266;',
	"\xCE\xB2" => '&#23614;',
	"\xCE\xB3" => '&#32428;',
	"\xCE\xB4" => '&#26410;',
	"\xCE\xB5" => '&#34074;',
	"\xCE\xB6" => '&#21619;',
	"\xCE\xB7" => '&#30031;',
	"\xCE\xB8" => '&#32963;',
	"\xCE\xB9" => '&#21890;',
	"\xCE\xBA" => '&#39759;',
	"\xCE\xBB" => '&#20301;',
	"\xCE\xBC" => '&#28205;',
	"\xCE\xBD" => '&#35859;',
	"\xCE\xBE" => '&#23561;',
	"\xCE\xBF" => '&#24944;',
	"\xCE\xC0" => '&#21355;',
	"\xCE\xC1" => '&#30239;',
	"\xCE\xC2" => '&#28201;',
	"\xCE\xC3" => '&#34442;',
	"\xCE\xC4" => '&#25991;',
	"\xCE\xC5" => '&#38395;',
	"\xCE\xC6" => '&#32441;',
	"\xCE\xC7" => '&#21563;',
	"\xCE\xC8" => '&#31283;',
	"\xCE\xC9" => '&#32010;',
	"\xCE\xCA" => '&#38382;',
	"\xCE\xCB" => '&#21985;',
	"\xCE\xCC" => '&#32705;',
	"\xCE\xCD" => '&#29934;',
	"\xCE\xCE" => '&#25373;',
	"\xCE\xCF" => '&#34583;',
	"\xCE\xD0" => '&#28065;',
	"\xCE\xD1" => '&#31389;',
	"\xCE\xD2" => '&#25105;',
	"\xCE\xD3" => '&#26017;',
	"\xCE\xD4" => '&#21351;',
	"\xCE\xD5" => '&#25569;',
	"\xCE\xD6" => '&#27779;',
	"\xCE\xD7" => '&#24043;',
	"\xCE\xD8" => '&#21596;',
	"\xCE\xD9" => '&#38056;',
	"\xCE\xDA" => '&#20044;',
	"\xCE\xDB" => '&#27745;',
	"\xCE\xDC" => '&#35820;',
	"\xCE\xDD" => '&#23627;',
	"\xCE\xDE" => '&#26080;',
	"\xCE\xDF" => '&#33436;',
	"\xCE\xE0" => '&#26791;',
	"\xCE\xE1" => '&#21566;',
	"\xCE\xE2" => '&#21556;',
	"\xCE\xE3" => '&#27595;',
	"\xCE\xE4" => '&#27494;',
	"\xCE\xE5" => '&#20116;',
	"\xCE\xE6" => '&#25410;',
	"\xCE\xE7" => '&#21320;',
	"\xCE\xE8" => '&#33310;',
	"\xCE\xE9" => '&#20237;',
	"\xCE\xEA" => '&#20398;',
	"\xCE\xEB" => '&#22366;',
	"\xCE\xEC" => '&#25098;',
	"\xCE\xED" => '&#38654;',
	"\xCE\xEE" => '&#26212;',
	"\xCE\xEF" => '&#29289;',
	"\xCE\xF0" => '&#21247;',
	"\xCE\xF1" => '&#21153;',
	"\xCE\xF2" => '&#24735;',
	"\xCE\xF3" => '&#35823;',
	"\xCE\xF4" => '&#26132;',
	"\xCE\xF5" => '&#29081;',
	"\xCE\xF6" => '&#26512;',
	"\xCE\xF7" => '&#35199;',
	"\xCE\xF8" => '&#30802;',
	"\xCE\xF9" => '&#30717;',
	"\xCE\xFA" => '&#26224;',
	"\xCE\xFB" => '&#22075;',
	"\xCE\xFC" => '&#21560;',
	"\xCE\xFD" => '&#38177;',
	"\xCE\xFE" => '&#29306;',
	"\xCF\xA1" => '&#31232;',
	"\xCF\xA2" => '&#24687;',
	"\xCF\xA3" => '&#24076;',
	"\xCF\xA4" => '&#24713;',
	"\xCF\xA5" => '&#33181;',
	"\xCF\xA6" => '&#22805;',
	"\xCF\xA7" => '&#24796;',
	"\xCF\xA8" => '&#29060;',
	"\xCF\xA9" => '&#28911;',
	"\xCF\xAA" => '&#28330;',
	"\xCF\xAB" => '&#27728;',
	"\xCF\xAC" => '&#29312;',
	"\xCF\xAD" => '&#27268;',
	"\xCF\xAE" => '&#34989;',
	"\xCF\xAF" => '&#24109;',
	"\xCF\xB0" => '&#20064;',
	"\xCF\xB1" => '&#23219;',
	"\xCF\xB2" => '&#21916;',
	"\xCF\xB3" => '&#38115;',
	"\xCF\xB4" => '&#27927;',
	"\xCF\xB5" => '&#31995;',
	"\xCF\xB6" => '&#38553;',
	"\xCF\xB7" => '&#25103;',
	"\xCF\xB8" => '&#32454;',
	"\xCF\xB9" => '&#30606;',
	"\xCF\xBA" => '&#34430;',
	"\xCF\xBB" => '&#21283;',
	"\xCF\xBC" => '&#38686;',
	"\xCF\xBD" => '&#36758;',
	"\xCF\xBE" => '&#26247;',
	"\xCF\xBF" => '&#23777;',
	"\xCF\xC0" => '&#20384;',
	"\xCF\xC1" => '&#29421;',
	"\xCF\xC2" => '&#19979;',
	"\xCF\xC3" => '&#21414;',
	"\xCF\xC4" => '&#22799;',
	"\xCF\xC5" => '&#21523;',
	"\xCF\xC6" => '&#25472;',
	"\xCF\xC7" => '&#38184;',
	"\xCF\xC8" => '&#20808;',
	"\xCF\xC9" => '&#20185;',
	"\xCF\xCA" => '&#40092;',
	"\xCF\xCB" => '&#32420;',
	"\xCF\xCC" => '&#21688;',
	"\xCF\xCD" => '&#36132;',
	"\xCF\xCE" => '&#34900;',
	"\xCF\xCF" => '&#33335;',
	"\xCF\xD0" => '&#38386;',
	"\xCF\xD1" => '&#28046;',
	"\xCF\xD2" => '&#24358;',
	"\xCF\xD3" => '&#23244;',
	"\xCF\xD4" => '&#26174;',
	"\xCF\xD5" => '&#38505;',
	"\xCF\xD6" => '&#29616;',
	"\xCF\xD7" => '&#29486;',
	"\xCF\xD8" => '&#21439;',
	"\xCF\xD9" => '&#33146;',
	"\xCF\xDA" => '&#39301;',
	"\xCF\xDB" => '&#32673;',
	"\xCF\xDC" => '&#23466;',
	"\xCF\xDD" => '&#38519;',
	"\xCF\xDE" => '&#38480;',
	"\xCF\xDF" => '&#32447;',
	"\xCF\xE0" => '&#30456;',
	"\xCF\xE1" => '&#21410;',
	"\xCF\xE2" => '&#38262;',
	"\xCF\xE3" => '&#39321;',
	"\xCF\xE4" => '&#31665;',
	"\xCF\xE5" => '&#35140;',
	"\xCF\xE6" => '&#28248;',
	"\xCF\xE7" => '&#20065;',
	"\xCF\xE8" => '&#32724;',
	"\xCF\xE9" => '&#31077;',
	"\xCF\xEA" => '&#35814;',
	"\xCF\xEB" => '&#24819;',
	"\xCF\xEC" => '&#21709;',
	"\xCF\xED" => '&#20139;',
	"\xCF\xEE" => '&#39033;',
	"\xCF\xEF" => '&#24055;',
	"\xCF\xF0" => '&#27233;',
	"\xCF\xF1" => '&#20687;',
	"\xCF\xF2" => '&#21521;',
	"\xCF\xF3" => '&#35937;',
	"\xCF\xF4" => '&#33831;',
	"\xCF\xF5" => '&#30813;',
	"\xCF\xF6" => '&#38660;',
	"\xCF\xF7" => '&#21066;',
	"\xCF\xF8" => '&#21742;',
	"\xCF\xF9" => '&#22179;',
	"\xCF\xFA" => '&#38144;',
	"\xCF\xFB" => '&#28040;',
	"\xCF\xFC" => '&#23477;',
	"\xCF\xFD" => '&#28102;',
	"\xCF\xFE" => '&#26195;',
	"\xD0\xA1" => '&#23567;',
	"\xD0\xA2" => '&#23389;',
	"\xD0\xA3" => '&#26657;',
	"\xD0\xA4" => '&#32918;',
	"\xD0\xA5" => '&#21880;',
	"\xD0\xA6" => '&#31505;',
	"\xD0\xA7" => '&#25928;',
	"\xD0\xA8" => '&#26964;',
	"\xD0\xA9" => '&#20123;',
	"\xD0\xAA" => '&#27463;',
	"\xD0\xAB" => '&#34638;',
	"\xD0\xAC" => '&#38795;',
	"\xD0\xAD" => '&#21327;',
	"\xD0\xAE" => '&#25375;',
	"\xD0\xAF" => '&#25658;',
	"\xD0\xB0" => '&#37034;',
	"\xD0\xB1" => '&#26012;',
	"\xD0\xB2" => '&#32961;',
	"\xD0\xB3" => '&#35856;',
	"\xD0\xB4" => '&#20889;',
	"\xD0\xB5" => '&#26800;',
	"\xD0\xB6" => '&#21368;',
	"\xD0\xB7" => '&#34809;',
	"\xD0\xB8" => '&#25032;',
	"\xD0\xB9" => '&#27844;',
	"\xD0\xBA" => '&#27899;',
	"\xD0\xBB" => '&#35874;',
	"\xD0\xBC" => '&#23633;',
	"\xD0\xBD" => '&#34218;',
	"\xD0\xBE" => '&#33455;',
	"\xD0\xBF" => '&#38156;',
	"\xD0\xC0" => '&#27427;',
	"\xD0\xC1" => '&#36763;',
	"\xD0\xC2" => '&#26032;',
	"\xD0\xC3" => '&#24571;',
	"\xD0\xC4" => '&#24515;',
	"\xD0\xC5" => '&#20449;',
	"\xD0\xC6" => '&#34885;',
	"\xD0\xC7" => '&#26143;',
	"\xD0\xC8" => '&#33125;',
	"\xD0\xC9" => '&#29481;',
	"\xD0\xCA" => '&#24826;',
	"\xD0\xCB" => '&#20852;',
	"\xD0\xCC" => '&#21009;',
	"\xD0\xCD" => '&#22411;',
	"\xD0\xCE" => '&#24418;',
	"\xD0\xCF" => '&#37026;',
	"\xD0\xD0" => '&#34892;',
	"\xD0\xD1" => '&#37266;',
	"\xD0\xD2" => '&#24184;',
	"\xD0\xD3" => '&#26447;',
	"\xD0\xD4" => '&#24615;',
	"\xD0\xD5" => '&#22995;',
	"\xD0\xD6" => '&#20804;',
	"\xD0\xD7" => '&#20982;',
	"\xD0\xD8" => '&#33016;',
	"\xD0\xD9" => '&#21256;',
	"\xD0\xDA" => '&#27769;',
	"\xD0\xDB" => '&#38596;',
	"\xD0\xDC" => '&#29066;',
	"\xD0\xDD" => '&#20241;',
	"\xD0\xDE" => '&#20462;',
	"\xD0\xDF" => '&#32670;',
	"\xD0\xE0" => '&#26429;',
	"\xD0\xE1" => '&#21957;',
	"\xD0\xE2" => '&#38152;',
	"\xD0\xE3" => '&#31168;',
	"\xD0\xE4" => '&#34966;',
	"\xD0\xE5" => '&#32483;',
	"\xD0\xE6" => '&#22687;',
	"\xD0\xE7" => '&#25100;',
	"\xD0\xE8" => '&#38656;',
	"\xD0\xE9" => '&#34394;',
	"\xD0\xEA" => '&#22040;',
	"\xD0\xEB" => '&#39035;',
	"\xD0\xEC" => '&#24464;',
	"\xD0\xED" => '&#35768;',
	"\xD0\xEE" => '&#33988;',
	"\xD0\xEF" => '&#37207;',
	"\xD0\xF0" => '&#21465;',
	"\xD0\xF1" => '&#26093;',
	"\xD0\xF2" => '&#24207;',
	"\xD0\xF3" => '&#30044;',
	"\xD0\xF4" => '&#24676;',
	"\xD0\xF5" => '&#32110;',
	"\xD0\xF6" => '&#23167;',
	"\xD0\xF7" => '&#32490;',
	"\xD0\xF8" => '&#32493;',
	"\xD0\xF9" => '&#36713;',
	"\xD0\xFA" => '&#21927;',
	"\xD0\xFB" => '&#23459;',
	"\xD0\xFC" => '&#24748;',
	"\xD0\xFD" => '&#26059;',
	"\xD0\xFE" => '&#29572;',
	"\xD1\xA1" => '&#36873;',
	"\xD1\xA2" => '&#30307;',
	"\xD1\xA3" => '&#30505;',
	"\xD1\xA4" => '&#32474;',
	"\xD1\xA5" => '&#38772;',
	"\xD1\xA6" => '&#34203;',
	"\xD1\xA7" => '&#23398;',
	"\xD1\xA8" => '&#31348;',
	"\xD1\xA9" => '&#38634;',
	"\xD1\xAA" => '&#34880;',
	"\xD1\xAB" => '&#21195;',
	"\xD1\xAC" => '&#29071;',
	"\xD1\xAD" => '&#24490;',
	"\xD1\xAE" => '&#26092;',
	"\xD1\xAF" => '&#35810;',
	"\xD1\xB0" => '&#23547;',
	"\xD1\xB1" => '&#39535;',
	"\xD1\xB2" => '&#24033;',
	"\xD1\xB3" => '&#27529;',
	"\xD1\xB4" => '&#27739;',
	"\xD1\xB5" => '&#35757;',
	"\xD1\xB6" => '&#35759;',
	"\xD1\xB7" => '&#36874;',
	"\xD1\xB8" => '&#36805;',
	"\xD1\xB9" => '&#21387;',
	"\xD1\xBA" => '&#25276;',
	"\xD1\xBB" => '&#40486;',
	"\xD1\xBC" => '&#40493;',
	"\xD1\xBD" => '&#21568;',
	"\xD1\xBE" => '&#20011;',
	"\xD1\xBF" => '&#33469;',
	"\xD1\xC0" => '&#29273;',
	"\xD1\xC1" => '&#34460;',
	"\xD1\xC2" => '&#23830;',
	"\xD1\xC3" => '&#34905;',
	"\xD1\xC4" => '&#28079;',
	"\xD1\xC5" => '&#38597;',
	"\xD1\xC6" => '&#21713;',
	"\xD1\xC7" => '&#20122;',
	"\xD1\xC8" => '&#35766;',
	"\xD1\xC9" => '&#28937;',
	"\xD1\xCA" => '&#21693;',
	"\xD1\xCB" => '&#38409;',
	"\xD1\xCC" => '&#28895;',
	"\xD1\xCD" => '&#28153;',
	"\xD1\xCE" => '&#30416;',
	"\xD1\xCF" => '&#20005;',
	"\xD1\xD0" => '&#30740;',
	"\xD1\xD1" => '&#34578;',
	"\xD1\xD2" => '&#23721;',
	"\xD1\xD3" => '&#24310;',
	"\xD1\xD4" => '&#35328;',
	"\xD1\xD5" => '&#39068;',
	"\xD1\xD6" => '&#38414;',
	"\xD1\xD7" => '&#28814;',
	"\xD1\xD8" => '&#27839;',
	"\xD1\xD9" => '&#22852;',
	"\xD1\xDA" => '&#25513;',
	"\xD1\xDB" => '&#30524;',
	"\xD1\xDC" => '&#34893;',
	"\xD1\xDD" => '&#28436;',
	"\xD1\xDE" => '&#33395;',
	"\xD1\xDF" => '&#22576;',
	"\xD1\xE0" => '&#29141;',
	"\xD1\xE1" => '&#21388;',
	"\xD1\xE2" => '&#30746;',
	"\xD1\xE3" => '&#38593;',
	"\xD1\xE4" => '&#21761;',
	"\xD1\xE5" => '&#24422;',
	"\xD1\xE6" => '&#28976;',
	"\xD1\xE7" => '&#23476;',
	"\xD1\xE8" => '&#35866;',
	"\xD1\xE9" => '&#39564;',
	"\xD1\xEA" => '&#27523;',
	"\xD1\xEB" => '&#22830;',
	"\xD1\xEC" => '&#40495;',
	"\xD1\xED" => '&#31207;',
	"\xD1\xEE" => '&#26472;',
	"\xD1\xEF" => '&#25196;',
	"\xD1\xF0" => '&#20335;',
	"\xD1\xF1" => '&#30113;',
	"\xD1\xF2" => '&#32650;',
	"\xD1\xF3" => '&#27915;',
	"\xD1\xF4" => '&#38451;',
	"\xD1\xF5" => '&#27687;',
	"\xD1\xF6" => '&#20208;',
	"\xD1\xF7" => '&#30162;',
	"\xD1\xF8" => '&#20859;',
	"\xD1\xF9" => '&#26679;',
	"\xD1\xFA" => '&#28478;',
	"\xD1\xFB" => '&#36992;',
	"\xD1\xFC" => '&#33136;',
	"\xD1\xFD" => '&#22934;',
	"\xD1\xFE" => '&#29814;',
	"\xD2\xA1" => '&#25671;',
	"\xD2\xA2" => '&#23591;',
	"\xD2\xA3" => '&#36965;',
	"\xD2\xA4" => '&#31377;',
	"\xD2\xA5" => '&#35875;',
	"\xD2\xA6" => '&#23002;',
	"\xD2\xA7" => '&#21676;',
	"\xD2\xA8" => '&#33280;',
	"\xD2\xA9" => '&#33647;',
	"\xD2\xAA" => '&#35201;',
	"\xD2\xAB" => '&#32768;',
	"\xD2\xAC" => '&#26928;',
	"\xD2\xAD" => '&#22094;',
	"\xD2\xAE" => '&#32822;',
	"\xD2\xAF" => '&#29239;',
	"\xD2\xB0" => '&#37326;',
	"\xD2\xB1" => '&#20918;',
	"\xD2\xB2" => '&#20063;',
	"\xD2\xB3" => '&#39029;',
	"\xD2\xB4" => '&#25494;',
	"\xD2\xB5" => '&#19994;',
	"\xD2\xB6" => '&#21494;',
	"\xD2\xB7" => '&#26355;',
	"\xD2\xB8" => '&#33099;',
	"\xD2\xB9" => '&#22812;',
	"\xD2\xBA" => '&#28082;',
	"\xD2\xBB" => '&#19968;',
	"\xD2\xBC" => '&#22777;',
	"\xD2\xBD" => '&#21307;',
	"\xD2\xBE" => '&#25558;',
	"\xD2\xBF" => '&#38129;',
	"\xD2\xC0" => '&#20381;',
	"\xD2\xC1" => '&#20234;',
	"\xD2\xC2" => '&#34915;',
	"\xD2\xC3" => '&#39056;',
	"\xD2\xC4" => '&#22839;',
	"\xD2\xC5" => '&#36951;',
	"\xD2\xC6" => '&#31227;',
	"\xD2\xC7" => '&#20202;',
	"\xD2\xC8" => '&#33008;',
	"\xD2\xC9" => '&#30097;',
	"\xD2\xCA" => '&#27778;',
	"\xD2\xCB" => '&#23452;',
	"\xD2\xCC" => '&#23016;',
	"\xD2\xCD" => '&#24413;',
	"\xD2\xCE" => '&#26885;',
	"\xD2\xCF" => '&#34433;',
	"\xD2\xD0" => '&#20506;',
	"\xD2\xD1" => '&#24050;',
	"\xD2\xD2" => '&#20057;',
	"\xD2\xD3" => '&#30691;',
	"\xD2\xD4" => '&#20197;',
	"\xD2\xD5" => '&#33402;',
	"\xD2\xD6" => '&#25233;',
	"\xD2\xD7" => '&#26131;',
	"\xD2\xD8" => '&#37009;',
	"\xD2\xD9" => '&#23673;',
	"\xD2\xDA" => '&#20159;',
	"\xD2\xDB" => '&#24441;',
	"\xD2\xDC" => '&#33222;',
	"\xD2\xDD" => '&#36920;',
	"\xD2\xDE" => '&#32900;',
	"\xD2\xDF" => '&#30123;',
	"\xD2\xE0" => '&#20134;',
	"\xD2\xE1" => '&#35028;',
	"\xD2\xE2" => '&#24847;',
	"\xD2\xE3" => '&#27589;',
	"\xD2\xE4" => '&#24518;',
	"\xD2\xE5" => '&#20041;',
	"\xD2\xE6" => '&#30410;',
	"\xD2\xE7" => '&#28322;',
	"\xD2\xE8" => '&#35811;',
	"\xD2\xE9" => '&#35758;',
	"\xD2\xEA" => '&#35850;',
	"\xD2\xEB" => '&#35793;',
	"\xD2\xEC" => '&#24322;',
	"\xD2\xED" => '&#32764;',
	"\xD2\xEE" => '&#32716;',
	"\xD2\xEF" => '&#32462;',
	"\xD2\xF0" => '&#33589;',
	"\xD2\xF1" => '&#33643;',
	"\xD2\xF2" => '&#22240;',
	"\xD2\xF3" => '&#27575;',
	"\xD2\xF4" => '&#38899;',
	"\xD2\xF5" => '&#38452;',
	"\xD2\xF6" => '&#23035;',
	"\xD2\xF7" => '&#21535;',
	"\xD2\xF8" => '&#38134;',
	"\xD2\xF9" => '&#28139;',
	"\xD2\xFA" => '&#23493;',
	"\xD2\xFB" => '&#39278;',
	"\xD2\xFC" => '&#23609;',
	"\xD2\xFD" => '&#24341;',
	"\xD2\xFE" => '&#38544;',
	"\xD3\xA1" => '&#21360;',
	"\xD3\xA2" => '&#33521;',
	"\xD3\xA3" => '&#27185;',
	"\xD3\xA4" => '&#23156;',
	"\xD3\xA5" => '&#40560;',
	"\xD3\xA6" => '&#24212;',
	"\xD3\xA7" => '&#32552;',
	"\xD3\xA8" => '&#33721;',
	"\xD3\xA9" => '&#33828;',
	"\xD3\xAA" => '&#33829;',
	"\xD3\xAB" => '&#33639;',
	"\xD3\xAC" => '&#34631;',
	"\xD3\xAD" => '&#36814;',
	"\xD3\xAE" => '&#36194;',
	"\xD3\xAF" => '&#30408;',
	"\xD3\xB0" => '&#24433;',
	"\xD3\xB1" => '&#39062;',
	"\xD3\xB2" => '&#30828;',
	"\xD3\xB3" => '&#26144;',
	"\xD3\xB4" => '&#21727;',
	"\xD3\xB5" => '&#25317;',
	"\xD3\xB6" => '&#20323;',
	"\xD3\xB7" => '&#33219;',
	"\xD3\xB8" => '&#30152;',
	"\xD3\xB9" => '&#24248;',
	"\xD3\xBA" => '&#38605;',
	"\xD3\xBB" => '&#36362;',
	"\xD3\xBC" => '&#34553;',
	"\xD3\xBD" => '&#21647;',
	"\xD3\xBE" => '&#27891;',
	"\xD3\xBF" => '&#28044;',
	"\xD3\xC0" => '&#27704;',
	"\xD3\xC1" => '&#24703;',
	"\xD3\xC2" => '&#21191;',
	"\xD3\xC3" => '&#29992;',
	"\xD3\xC4" => '&#24189;',
	"\xD3\xC5" => '&#20248;',
	"\xD3\xC6" => '&#24736;',
	"\xD3\xC7" => '&#24551;',
	"\xD3\xC8" => '&#23588;',
	"\xD3\xC9" => '&#30001;',
	"\xD3\xCA" => '&#37038;',
	"\xD3\xCB" => '&#38080;',
	"\xD3\xCC" => '&#29369;',
	"\xD3\xCD" => '&#27833;',
	"\xD3\xCE" => '&#28216;',
	"\xD3\xCF" => '&#37193;',
	"\xD3\xD0" => '&#26377;',
	"\xD3\xD1" => '&#21451;',
	"\xD3\xD2" => '&#21491;',
	"\xD3\xD3" => '&#20305;',
	"\xD3\xD4" => '&#37321;',
	"\xD3\xD5" => '&#35825;',
	"\xD3\xD6" => '&#21448;',
	"\xD3\xD7" => '&#24188;',
	"\xD3\xD8" => '&#36802;',
	"\xD3\xD9" => '&#28132;',
	"\xD3\xDA" => '&#20110;',
	"\xD3\xDB" => '&#30402;',
	"\xD3\xDC" => '&#27014;',
	"\xD3\xDD" => '&#34398;',
	"\xD3\xDE" => '&#24858;',
	"\xD3\xDF" => '&#33286;',
	"\xD3\xE0" => '&#20313;',
	"\xD3\xE1" => '&#20446;',
	"\xD3\xE2" => '&#36926;',
	"\xD3\xE3" => '&#40060;',
	"\xD3\xE4" => '&#24841;',
	"\xD3\xE5" => '&#28189;',
	"\xD3\xE6" => '&#28180;',
	"\xD3\xE7" => '&#38533;',
	"\xD3\xE8" => '&#20104;',
	"\xD3\xE9" => '&#23089;',
	"\xD3\xEA" => '&#38632;',
	"\xD3\xEB" => '&#19982;',
	"\xD3\xEC" => '&#23679;',
	"\xD3\xED" => '&#31161;',
	"\xD3\xEE" => '&#23431;',
	"\xD3\xEF" => '&#35821;',
	"\xD3\xF0" => '&#32701;',
	"\xD3\xF1" => '&#29577;',
	"\xD3\xF2" => '&#22495;',
	"\xD3\xF3" => '&#33419;',
	"\xD3\xF4" => '&#37057;',
	"\xD3\xF5" => '&#21505;',
	"\xD3\xF6" => '&#36935;',
	"\xD3\xF7" => '&#21947;',
	"\xD3\xF8" => '&#23786;',
	"\xD3\xF9" => '&#24481;',
	"\xD3\xFA" => '&#24840;',
	"\xD3\xFB" => '&#27442;',
	"\xD3\xFC" => '&#29425;',
	"\xD3\xFD" => '&#32946;',
	"\xD3\xFE" => '&#35465;',
	"\xD4\xA1" => '&#28020;',
	"\xD4\xA2" => '&#23507;',
	"\xD4\xA3" => '&#35029;',
	"\xD4\xA4" => '&#39044;',
	"\xD4\xA5" => '&#35947;',
	"\xD4\xA6" => '&#39533;',
	"\xD4\xA7" => '&#40499;',
	"\xD4\xA8" => '&#28170;',
	"\xD4\xA9" => '&#20900;',
	"\xD4\xAA" => '&#20803;',
	"\xD4\xAB" => '&#22435;',
	"\xD4\xAC" => '&#34945;',
	"\xD4\xAD" => '&#21407;',
	"\xD4\xAE" => '&#25588;',
	"\xD4\xAF" => '&#36757;',
	"\xD4\xB0" => '&#22253;',
	"\xD4\xB1" => '&#21592;',
	"\xD4\xB2" => '&#22278;',
	"\xD4\xB3" => '&#29503;',
	"\xD4\xB4" => '&#28304;',
	"\xD4\xB5" => '&#32536;',
	"\xD4\xB6" => '&#36828;',
	"\xD4\xB7" => '&#33489;',
	"\xD4\xB8" => '&#24895;',
	"\xD4\xB9" => '&#24616;',
	"\xD4\xBA" => '&#38498;',
	"\xD4\xBB" => '&#26352;',
	"\xD4\xBC" => '&#32422;',
	"\xD4\xBD" => '&#36234;',
	"\xD4\xBE" => '&#36291;',
	"\xD4\xBF" => '&#38053;',
	"\xD4\xC0" => '&#23731;',
	"\xD4\xC1" => '&#31908;',
	"\xD4\xC2" => '&#26376;',
	"\xD4\xC3" => '&#24742;',
	"\xD4\xC4" => '&#38405;',
	"\xD4\xC5" => '&#32792;',
	"\xD4\xC6" => '&#20113;',
	"\xD4\xC7" => '&#37095;',
	"\xD4\xC8" => '&#21248;',
	"\xD4\xC9" => '&#38504;',
	"\xD4\xCA" => '&#20801;',
	"\xD4\xCB" => '&#36816;',
	"\xD4\xCC" => '&#34164;',
	"\xD4\xCD" => '&#37213;',
	"\xD4\xCE" => '&#26197;',
	"\xD4\xCF" => '&#38901;',
	"\xD4\xD0" => '&#23381;',
	"\xD4\xD1" => '&#21277;',
	"\xD4\xD2" => '&#30776;',
	"\xD4\xD3" => '&#26434;',
	"\xD4\xD4" => '&#26685;',
	"\xD4\xD5" => '&#21705;',
	"\xD4\xD6" => '&#28798;',
	"\xD4\xD7" => '&#23472;',
	"\xD4\xD8" => '&#36733;',
	"\xD4\xD9" => '&#20877;',
	"\xD4\xDA" => '&#22312;',
	"\xD4\xDB" => '&#21681;',
	"\xD4\xDC" => '&#25874;',
	"\xD4\xDD" => '&#26242;',
	"\xD4\xDE" => '&#36190;',
	"\xD4\xDF" => '&#36163;',
	"\xD4\xE0" => '&#33039;',
	"\xD4\xE1" => '&#33900;',
	"\xD4\xE2" => '&#36973;',
	"\xD4\xE3" => '&#31967;',
	"\xD4\xE4" => '&#20991;',
	"\xD4\xE5" => '&#34299;',
	"\xD4\xE6" => '&#26531;',
	"\xD4\xE7" => '&#26089;',
	"\xD4\xE8" => '&#28577;',
	"\xD4\xE9" => '&#34468;',
	"\xD4\xEA" => '&#36481;',
	"\xD4\xEB" => '&#22122;',
	"\xD4\xEC" => '&#36896;',
	"\xD4\xED" => '&#30338;',
	"\xD4\xEE" => '&#28790;',
	"\xD4\xEF" => '&#29157;',
	"\xD4\xF0" => '&#36131;',
	"\xD4\xF1" => '&#25321;',
	"\xD4\xF2" => '&#21017;',
	"\xD4\xF3" => '&#27901;',
	"\xD4\xF4" => '&#36156;',
	"\xD4\xF5" => '&#24590;',
	"\xD4\xF6" => '&#22686;',
	"\xD4\xF7" => '&#24974;',
	"\xD4\xF8" => '&#26366;',
	"\xD4\xF9" => '&#36192;',
	"\xD4\xFA" => '&#25166;',
	"\xD4\xFB" => '&#21939;',
	"\xD4\xFC" => '&#28195;',
	"\xD4\xFD" => '&#26413;',
	"\xD4\xFE" => '&#36711;',
	"\xD5\xA1" => '&#38113;',
	"\xD5\xA2" => '&#38392;',
	"\xD5\xA3" => '&#30504;',
	"\xD5\xA4" => '&#26629;',
	"\xD5\xA5" => '&#27048;',
	"\xD5\xA6" => '&#21643;',
	"\xD5\xA7" => '&#20045;',
	"\xD5\xA8" => '&#28856;',
	"\xD5\xA9" => '&#35784;',
	"\xD5\xAA" => '&#25688;',
	"\xD5\xAB" => '&#25995;',
	"\xD5\xAC" => '&#23429;',
	"\xD5\xAD" => '&#31364;',
	"\xD5\xAE" => '&#20538;',
	"\xD5\xAF" => '&#23528;',
	"\xD5\xB0" => '&#30651;',
	"\xD5\xB1" => '&#27617;',
	"\xD5\xB2" => '&#35449;',
	"\xD5\xB3" => '&#31896;',
	"\xD5\xB4" => '&#27838;',
	"\xD5\xB5" => '&#30415;',
	"\xD5\xB6" => '&#26025;',
	"\xD5\xB7" => '&#36759;',
	"\xD5\xB8" => '&#23853;',
	"\xD5\xB9" => '&#23637;',
	"\xD5\xBA" => '&#34360;',
	"\xD5\xBB" => '&#26632;',
	"\xD5\xBC" => '&#21344;',
	"\xD5\xBD" => '&#25112;',
	"\xD5\xBE" => '&#31449;',
	"\xD5\xBF" => '&#28251;',
	"\xD5\xC0" => '&#32509;',
	"\xD5\xC1" => '&#27167;',
	"\xD5\xC2" => '&#31456;',
	"\xD5\xC3" => '&#24432;',
	"\xD5\xC4" => '&#28467;',
	"\xD5\xC5" => '&#24352;',
	"\xD5\xC6" => '&#25484;',
	"\xD5\xC7" => '&#28072;',
	"\xD5\xC8" => '&#26454;',
	"\xD5\xC9" => '&#19976;',
	"\xD5\xCA" => '&#24080;',
	"\xD5\xCB" => '&#36134;',
	"\xD5\xCC" => '&#20183;',
	"\xD5\xCD" => '&#32960;',
	"\xD5\xCE" => '&#30260;',
	"\xD5\xCF" => '&#38556;',
	"\xD5\xD0" => '&#25307;',
	"\xD5\xD1" => '&#26157;',
	"\xD5\xD2" => '&#25214;',
	"\xD5\xD3" => '&#27836;',
	"\xD5\xD4" => '&#36213;',
	"\xD5\xD5" => '&#29031;',
	"\xD5\xD6" => '&#32617;',
	"\xD5\xD7" => '&#20806;',
	"\xD5\xD8" => '&#32903;',
	"\xD5\xD9" => '&#21484;',
	"\xD5\xDA" => '&#36974;',
	"\xD5\xDB" => '&#25240;',
	"\xD5\xDC" => '&#21746;',
	"\xD5\xDD" => '&#34544;',
	"\xD5\xDE" => '&#36761;',
	"\xD5\xDF" => '&#32773;',
	"\xD5\xE0" => '&#38167;',
	"\xD5\xE1" => '&#34071;',
	"\xD5\xE2" => '&#36825;',
	"\xD5\xE3" => '&#27993;',
	"\xD5\xE4" => '&#29645;',
	"\xD5\xE5" => '&#26015;',
	"\xD5\xE6" => '&#30495;',
	"\xD5\xE7" => '&#29956;',
	"\xD5\xE8" => '&#30759;',
	"\xD5\xE9" => '&#33275;',
	"\xD5\xEA" => '&#36126;',
	"\xD5\xEB" => '&#38024;',
	"\xD5\xEC" => '&#20390;',
	"\xD5\xED" => '&#26517;',
	"\xD5\xEE" => '&#30137;',
	"\xD5\xEF" => '&#35786;',
	"\xD5\xF0" => '&#38663;',
	"\xD5\xF1" => '&#25391;',
	"\xD5\xF2" => '&#38215;',
	"\xD5\xF3" => '&#38453;',
	"\xD5\xF4" => '&#33976;',
	"\xD5\xF5" => '&#25379;',
	"\xD5\xF6" => '&#30529;',
	"\xD5\xF7" => '&#24449;',
	"\xD5\xF8" => '&#29424;',
	"\xD5\xF9" => '&#20105;',
	"\xD5\xFA" => '&#24596;',
	"\xD5\xFB" => '&#25972;',
	"\xD5\xFC" => '&#25327;',
	"\xD5\xFD" => '&#27491;',
	"\xD5\xFE" => '&#25919;',
	"\xD6\xA1" => '&#24103;',
	"\xD6\xA2" => '&#30151;',
	"\xD6\xA3" => '&#37073;',
	"\xD6\xA4" => '&#35777;',
	"\xD6\xA5" => '&#33437;',
	"\xD6\xA6" => '&#26525;',
	"\xD6\xA7" => '&#25903;',
	"\xD6\xA8" => '&#21553;',
	"\xD6\xA9" => '&#34584;',
	"\xD6\xAA" => '&#30693;',
	"\xD6\xAB" => '&#32930;',
	"\xD6\xAC" => '&#33026;',
	"\xD6\xAD" => '&#27713;',
	"\xD6\xAE" => '&#20043;',
	"\xD6\xAF" => '&#32455;',
	"\xD6\xB0" => '&#32844;',
	"\xD6\xB1" => '&#30452;',
	"\xD6\xB2" => '&#26893;',
	"\xD6\xB3" => '&#27542;',
	"\xD6\xB4" => '&#25191;',
	"\xD6\xB5" => '&#20540;',
	"\xD6\xB6" => '&#20356;',
	"\xD6\xB7" => '&#22336;',
	"\xD6\xB8" => '&#25351;',
	"\xD6\xB9" => '&#27490;',
	"\xD6\xBA" => '&#36286;',
	"\xD6\xBB" => '&#21482;',
	"\xD6\xBC" => '&#26088;',
	"\xD6\xBD" => '&#32440;',
	"\xD6\xBE" => '&#24535;',
	"\xD6\xBF" => '&#25370;',
	"\xD6\xC0" => '&#25527;',
	"\xD6\xC1" => '&#33267;',
	"\xD6\xC2" => '&#33268;',
	"\xD6\xC3" => '&#32622;',
	"\xD6\xC4" => '&#24092;',
	"\xD6\xC5" => '&#23769;',
	"\xD6\xC6" => '&#21046;',
	"\xD6\xC7" => '&#26234;',
	"\xD6\xC8" => '&#31209;',
	"\xD6\xC9" => '&#31258;',
	"\xD6\xCA" => '&#36136;',
	"\xD6\xCB" => '&#28825;',
	"\xD6\xCC" => '&#30164;',
	"\xD6\xCD" => '&#28382;',
	"\xD6\xCE" => '&#27835;',
	"\xD6\xCF" => '&#31378;',
	"\xD6\xD0" => '&#20013;',
	"\xD6\xD1" => '&#30405;',
	"\xD6\xD2" => '&#24544;',
	"\xD6\xD3" => '&#38047;',
	"\xD6\xD4" => '&#34935;',
	"\xD6\xD5" => '&#32456;',
	"\xD6\xD6" => '&#31181;',
	"\xD6\xD7" => '&#32959;',
	"\xD6\xD8" => '&#37325;',
	"\xD6\xD9" => '&#20210;',
	"\xD6\xDA" => '&#20247;',
	"\xD6\xDB" => '&#33311;',
	"\xD6\xDC" => '&#21608;',
	"\xD6\xDD" => '&#24030;',
	"\xD6\xDE" => '&#27954;',
	"\xD6\xDF" => '&#35788;',
	"\xD6\xE0" => '&#31909;',
	"\xD6\xE1" => '&#36724;',
	"\xD6\xE2" => '&#32920;',
	"\xD6\xE3" => '&#24090;',
	"\xD6\xE4" => '&#21650;',
	"\xD6\xE5" => '&#30385;',
	"\xD6\xE6" => '&#23449;',
	"\xD6\xE7" => '&#26172;',
	"\xD6\xE8" => '&#39588;',
	"\xD6\xE9" => '&#29664;',
	"\xD6\xEA" => '&#26666;',
	"\xD6\xEB" => '&#34523;',
	"\xD6\xEC" => '&#26417;',
	"\xD6\xED" => '&#29482;',
	"\xD6\xEE" => '&#35832;',
	"\xD6\xEF" => '&#35803;',
	"\xD6\xF0" => '&#36880;',
	"\xD6\xF1" => '&#31481;',
	"\xD6\xF2" => '&#28891;',
	"\xD6\xF3" => '&#29038;',
	"\xD6\xF4" => '&#25284;',
	"\xD6\xF5" => '&#30633;',
	"\xD6\xF6" => '&#22065;',
	"\xD6\xF7" => '&#20027;',
	"\xD6\xF8" => '&#33879;',
	"\xD6\xF9" => '&#26609;',
	"\xD6\xFA" => '&#21161;',
	"\xD6\xFB" => '&#34496;',
	"\xD6\xFC" => '&#36142;',
	"\xD6\xFD" => '&#38136;',
	"\xD6\xFE" => '&#31569;',
	"\xD7\xA1" => '&#20303;',
	"\xD7\xA2" => '&#27880;',
	"\xD7\xA3" => '&#31069;',
	"\xD7\xA4" => '&#39547;',
	"\xD7\xA5" => '&#25235;',
	"\xD7\xA6" => '&#29226;',
	"\xD7\xA7" => '&#25341;',
	"\xD7\xA8" => '&#19987;',
	"\xD7\xA9" => '&#30742;',
	"\xD7\xAA" => '&#36716;',
	"\xD7\xAB" => '&#25776;',
	"\xD7\xAC" => '&#36186;',
	"\xD7\xAD" => '&#31686;',
	"\xD7\xAE" => '&#26729;',
	"\xD7\xAF" => '&#24196;',
	"\xD7\xB0" => '&#35013;',
	"\xD7\xB1" => '&#22918;',
	"\xD7\xB2" => '&#25758;',
	"\xD7\xB3" => '&#22766;',
	"\xD7\xB4" => '&#29366;',
	"\xD7\xB5" => '&#26894;',
	"\xD7\xB6" => '&#38181;',
	"\xD7\xB7" => '&#36861;',
	"\xD7\xB8" => '&#36184;',
	"\xD7\xB9" => '&#22368;',
	"\xD7\xBA" => '&#32512;',
	"\xD7\xBB" => '&#35846;',
	"\xD7\xBC" => '&#20934;',
	"\xD7\xBD" => '&#25417;',
	"\xD7\xBE" => '&#25305;',
	"\xD7\xBF" => '&#21331;',
	"\xD7\xC0" => '&#26700;',
	"\xD7\xC1" => '&#29730;',
	"\xD7\xC2" => '&#33537;',
	"\xD7\xC3" => '&#37196;',
	"\xD7\xC4" => '&#21828;',
	"\xD7\xC5" => '&#30528;',
	"\xD7\xC6" => '&#28796;',
	"\xD7\xC7" => '&#27978;',
	"\xD7\xC8" => '&#20857;',
	"\xD7\xC9" => '&#21672;',
	"\xD7\xCA" => '&#36164;',
	"\xD7\xCB" => '&#23039;',
	"\xD7\xCC" => '&#28363;',
	"\xD7\xCD" => '&#28100;',
	"\xD7\xCE" => '&#23388;',
	"\xD7\xCF" => '&#32043;',
	"\xD7\xD0" => '&#20180;',
	"\xD7\xD1" => '&#31869;',
	"\xD7\xD2" => '&#28371;',
	"\xD7\xD3" => '&#23376;',
	"\xD7\xD4" => '&#33258;',
	"\xD7\xD5" => '&#28173;',
	"\xD7\xD6" => '&#23383;',
	"\xD7\xD7" => '&#39683;',
	"\xD7\xD8" => '&#26837;',
	"\xD7\xD9" => '&#36394;',
	"\xD7\xDA" => '&#23447;',
	"\xD7\xDB" => '&#32508;',
	"\xD7\xDC" => '&#24635;',
	"\xD7\xDD" => '&#32437;',
	"\xD7\xDE" => '&#37049;',
	"\xD7\xDF" => '&#36208;',
	"\xD7\xE0" => '&#22863;',
	"\xD7\xE1" => '&#25549;',
	"\xD7\xE2" => '&#31199;',
	"\xD7\xE3" => '&#36275;',
	"\xD7\xE4" => '&#21330;',
	"\xD7\xE5" => '&#26063;',
	"\xD7\xE6" => '&#31062;',
	"\xD7\xE7" => '&#35781;',
	"\xD7\xE8" => '&#38459;',
	"\xD7\xE9" => '&#32452;',
	"\xD7\xEA" => '&#38075;',
	"\xD7\xEB" => '&#32386;',
	"\xD7\xEC" => '&#22068;',
	"\xD7\xED" => '&#37257;',
	"\xD7\xEE" => '&#26368;',
	"\xD7\xEF" => '&#32618;',
	"\xD7\xF0" => '&#23562;',
	"\xD7\xF1" => '&#36981;',
	"\xD7\xF2" => '&#26152;',
	"\xD7\xF3" => '&#24038;',
	"\xD7\xF4" => '&#20304;',
	"\xD7\xF5" => '&#26590;',
	"\xD7\xF6" => '&#20570;',
	"\xD7\xF7" => '&#20316;',
	"\xD7\xF8" => '&#22352;',
	"\xD7\xF9" => '&#24231;',
	"\xD8\xA1" => '&#20109;',
	"\xD8\xA2" => '&#19980;',
	"\xD8\xA3" => '&#20800;',
	"\xD8\xA4" => '&#19984;',
	"\xD8\xA5" => '&#24319;',
	"\xD8\xA6" => '&#21317;',
	"\xD8\xA7" => '&#19989;',
	"\xD8\xA8" => '&#20120;',
	"\xD8\xA9" => '&#19998;',
	"\xD8\xAA" => '&#39730;',
	"\xD8\xAB" => '&#23404;',
	"\xD8\xAC" => '&#22121;',
	"\xD8\xAD" => '&#20008;',
	"\xD8\xAE" => '&#31162;',
	"\xD8\xAF" => '&#20031;',
	"\xD8\xB0" => '&#21269;',
	"\xD8\xB1" => '&#20039;',
	"\xD8\xB2" => '&#22829;',
	"\xD8\xB3" => '&#29243;',
	"\xD8\xB4" => '&#21358;',
	"\xD8\xB5" => '&#27664;',
	"\xD8\xB6" => '&#22239;',
	"\xD8\xB7" => '&#32996;',
	"\xD8\xB8" => '&#39319;',
	"\xD8\xB9" => '&#27603;',
	"\xD8\xBA" => '&#30590;',
	"\xD8\xBB" => '&#40727;',
	"\xD8\xBC" => '&#20022;',
	"\xD8\xBD" => '&#20127;',
	"\xD8\xBE" => '&#40720;',
	"\xD8\xBF" => '&#20060;',
	"\xD8\xC0" => '&#20073;',
	"\xD8\xC1" => '&#20115;',
	"\xD8\xC2" => '&#33416;',
	"\xD8\xC3" => '&#23387;',
	"\xD8\xC4" => '&#21868;',
	"\xD8\xC5" => '&#22031;',
	"\xD8\xC6" => '&#20164;',
	"\xD8\xC7" => '&#21389;',
	"\xD8\xC8" => '&#21405;',
	"\xD8\xC9" => '&#21411;',
	"\xD8\xCA" => '&#21413;',
	"\xD8\xCB" => '&#21422;',
	"\xD8\xCC" => '&#38757;',
	"\xD8\xCD" => '&#36189;',
	"\xD8\xCE" => '&#21274;',
	"\xD8\xCF" => '&#21493;',
	"\xD8\xD0" => '&#21286;',
	"\xD8\xD1" => '&#21294;',
	"\xD8\xD2" => '&#21310;',
	"\xD8\xD3" => '&#36188;',
	"\xD8\xD4" => '&#21350;',
	"\xD8\xD5" => '&#21347;',
	"\xD8\xD6" => '&#20994;',
	"\xD8\xD7" => '&#21000;',
	"\xD8\xD8" => '&#21006;',
	"\xD8\xD9" => '&#21037;',
	"\xD8\xDA" => '&#21043;',
	"\xD8\xDB" => '&#21055;',
	"\xD8\xDC" => '&#21056;',
	"\xD8\xDD" => '&#21068;',
	"\xD8\xDE" => '&#21086;',
	"\xD8\xDF" => '&#21089;',
	"\xD8\xE0" => '&#21084;',
	"\xD8\xE1" => '&#33967;',
	"\xD8\xE2" => '&#21117;',
	"\xD8\xE3" => '&#21122;',
	"\xD8\xE4" => '&#21121;',
	"\xD8\xE5" => '&#21136;',
	"\xD8\xE6" => '&#21139;',
	"\xD8\xE7" => '&#20866;',
	"\xD8\xE8" => '&#32596;',
	"\xD8\xE9" => '&#20155;',
	"\xD8\xEA" => '&#20163;',
	"\xD8\xEB" => '&#20169;',
	"\xD8\xEC" => '&#20162;',
	"\xD8\xED" => '&#20200;',
	"\xD8\xEE" => '&#20193;',
	"\xD8\xEF" => '&#20203;',
	"\xD8\xF0" => '&#20190;',
	"\xD8\xF1" => '&#20251;',
	"\xD8\xF2" => '&#20211;',
	"\xD8\xF3" => '&#20258;',
	"\xD8\xF4" => '&#20324;',
	"\xD8\xF5" => '&#20213;',
	"\xD8\xF6" => '&#20261;',
	"\xD8\xF7" => '&#20263;',
	"\xD8\xF8" => '&#20233;',
	"\xD8\xF9" => '&#20267;',
	"\xD8\xFA" => '&#20318;',
	"\xD8\xFB" => '&#20327;',
	"\xD8\xFC" => '&#25912;',
	"\xD8\xFD" => '&#20314;',
	"\xD8\xFE" => '&#20317;',
	"\xD9\xA1" => '&#20319;',
	"\xD9\xA2" => '&#20311;',
	"\xD9\xA3" => '&#20274;',
	"\xD9\xA4" => '&#20285;',
	"\xD9\xA5" => '&#20342;',
	"\xD9\xA6" => '&#20340;',
	"\xD9\xA7" => '&#20369;',
	"\xD9\xA8" => '&#20361;',
	"\xD9\xA9" => '&#20355;',
	"\xD9\xAA" => '&#20367;',
	"\xD9\xAB" => '&#20350;',
	"\xD9\xAC" => '&#20347;',
	"\xD9\xAD" => '&#20394;',
	"\xD9\xAE" => '&#20348;',
	"\xD9\xAF" => '&#20396;',
	"\xD9\xB0" => '&#20372;',
	"\xD9\xB1" => '&#20454;',
	"\xD9\xB2" => '&#20456;',
	"\xD9\xB3" => '&#20458;',
	"\xD9\xB4" => '&#20421;',
	"\xD9\xB5" => '&#20442;',
	"\xD9\xB6" => '&#20451;',
	"\xD9\xB7" => '&#20444;',
	"\xD9\xB8" => '&#20433;',
	"\xD9\xB9" => '&#20447;',
	"\xD9\xBA" => '&#20472;',
	"\xD9\xBB" => '&#20521;',
	"\xD9\xBC" => '&#20556;',
	"\xD9\xBD" => '&#20467;',
	"\xD9\xBE" => '&#20524;',
	"\xD9\xBF" => '&#20495;',
	"\xD9\xC0" => '&#20526;',
	"\xD9\xC1" => '&#20525;',
	"\xD9\xC2" => '&#20478;',
	"\xD9\xC3" => '&#20508;',
	"\xD9\xC4" => '&#20492;',
	"\xD9\xC5" => '&#20517;',
	"\xD9\xC6" => '&#20520;',
	"\xD9\xC7" => '&#20606;',
	"\xD9\xC8" => '&#20547;',
	"\xD9\xC9" => '&#20565;',
	"\xD9\xCA" => '&#20552;',
	"\xD9\xCB" => '&#20558;',
	"\xD9\xCC" => '&#20588;',
	"\xD9\xCD" => '&#20603;',
	"\xD9\xCE" => '&#20645;',
	"\xD9\xCF" => '&#20647;',
	"\xD9\xD0" => '&#20649;',
	"\xD9\xD1" => '&#20666;',
	"\xD9\xD2" => '&#20694;',
	"\xD9\xD3" => '&#20742;',
	"\xD9\xD4" => '&#20717;',
	"\xD9\xD5" => '&#20716;',
	"\xD9\xD6" => '&#20710;',
	"\xD9\xD7" => '&#20718;',
	"\xD9\xD8" => '&#20743;',
	"\xD9\xD9" => '&#20747;',
	"\xD9\xDA" => '&#20189;',
	"\xD9\xDB" => '&#27709;',
	"\xD9\xDC" => '&#20312;',
	"\xD9\xDD" => '&#20325;',
	"\xD9\xDE" => '&#20430;',
	"\xD9\xDF" => '&#40864;',
	"\xD9\xE0" => '&#27718;',
	"\xD9\xE1" => '&#31860;',
	"\xD9\xE2" => '&#20846;',
	"\xD9\xE3" => '&#24061;',
	"\xD9\xE4" => '&#40649;',
	"\xD9\xE5" => '&#39320;',
	"\xD9\xE6" => '&#20865;',
	"\xD9\xE7" => '&#22804;',
	"\xD9\xE8" => '&#21241;',
	"\xD9\xE9" => '&#21261;',
	"\xD9\xEA" => '&#35335;',
	"\xD9\xEB" => '&#21264;',
	"\xD9\xEC" => '&#20971;',
	"\xD9\xED" => '&#22809;',
	"\xD9\xEE" => '&#20821;',
	"\xD9\xEF" => '&#20128;',
	"\xD9\xF0" => '&#20822;',
	"\xD9\xF1" => '&#20147;',
	"\xD9\xF2" => '&#34926;',
	"\xD9\xF3" => '&#34980;',
	"\xD9\xF4" => '&#20149;',
	"\xD9\xF5" => '&#33044;',
	"\xD9\xF6" => '&#35026;',
	"\xD9\xF7" => '&#31104;',
	"\xD9\xF8" => '&#23348;',
	"\xD9\xF9" => '&#34819;',
	"\xD9\xFA" => '&#32696;',
	"\xD9\xFB" => '&#20907;',
	"\xD9\xFC" => '&#20913;',
	"\xD9\xFD" => '&#20925;',
	"\xD9\xFE" => '&#20924;',
	"\xDA\xA1" => '&#20935;',
	"\xDA\xA2" => '&#20886;',
	"\xDA\xA3" => '&#20898;',
	"\xDA\xA4" => '&#20901;',
	"\xDA\xA5" => '&#35744;',
	"\xDA\xA6" => '&#35750;',
	"\xDA\xA7" => '&#35751;',
	"\xDA\xA8" => '&#35754;',
	"\xDA\xA9" => '&#35764;',
	"\xDA\xAA" => '&#35765;',
	"\xDA\xAB" => '&#35767;',
	"\xDA\xAC" => '&#35778;',
	"\xDA\xAD" => '&#35779;',
	"\xDA\xAE" => '&#35787;',
	"\xDA\xAF" => '&#35791;',
	"\xDA\xB0" => '&#35790;',
	"\xDA\xB1" => '&#35794;',
	"\xDA\xB2" => '&#35795;',
	"\xDA\xB3" => '&#35796;',
	"\xDA\xB4" => '&#35798;',
	"\xDA\xB5" => '&#35800;',
	"\xDA\xB6" => '&#35801;',
	"\xDA\xB7" => '&#35804;',
	"\xDA\xB8" => '&#35807;',
	"\xDA\xB9" => '&#35808;',
	"\xDA\xBA" => '&#35812;',
	"\xDA\xBB" => '&#35816;',
	"\xDA\xBC" => '&#35817;',
	"\xDA\xBD" => '&#35822;',
	"\xDA\xBE" => '&#35824;',
	"\xDA\xBF" => '&#35827;',
	"\xDA\xC0" => '&#35830;',
	"\xDA\xC1" => '&#35833;',
	"\xDA\xC2" => '&#35836;',
	"\xDA\xC3" => '&#35839;',
	"\xDA\xC4" => '&#35840;',
	"\xDA\xC5" => '&#35842;',
	"\xDA\xC6" => '&#35844;',
	"\xDA\xC7" => '&#35847;',
	"\xDA\xC8" => '&#35852;',
	"\xDA\xC9" => '&#35855;',
	"\xDA\xCA" => '&#35857;',
	"\xDA\xCB" => '&#35858;',
	"\xDA\xCC" => '&#35860;',
	"\xDA\xCD" => '&#35861;',
	"\xDA\xCE" => '&#35862;',
	"\xDA\xCF" => '&#35865;',
	"\xDA\xD0" => '&#35867;',
	"\xDA\xD1" => '&#35864;',
	"\xDA\xD2" => '&#35869;',
	"\xDA\xD3" => '&#35871;',
	"\xDA\xD4" => '&#35872;',
	"\xDA\xD5" => '&#35873;',
	"\xDA\xD6" => '&#35877;',
	"\xDA\xD7" => '&#35879;',
	"\xDA\xD8" => '&#35882;',
	"\xDA\xD9" => '&#35883;',
	"\xDA\xDA" => '&#35886;',
	"\xDA\xDB" => '&#35887;',
	"\xDA\xDC" => '&#35890;',
	"\xDA\xDD" => '&#35891;',
	"\xDA\xDE" => '&#35893;',
	"\xDA\xDF" => '&#35894;',
	"\xDA\xE0" => '&#21353;',
	"\xDA\xE1" => '&#21370;',
	"\xDA\xE2" => '&#38429;',
	"\xDA\xE3" => '&#38434;',
	"\xDA\xE4" => '&#38433;',
	"\xDA\xE5" => '&#38449;',
	"\xDA\xE6" => '&#38442;',
	"\xDA\xE7" => '&#38461;',
	"\xDA\xE8" => '&#38460;',
	"\xDA\xE9" => '&#38466;',
	"\xDA\xEA" => '&#38473;',
	"\xDA\xEB" => '&#38484;',
	"\xDA\xEC" => '&#38495;',
	"\xDA\xED" => '&#38503;',
	"\xDA\xEE" => '&#38508;',
	"\xDA\xEF" => '&#38514;',
	"\xDA\xF0" => '&#38516;',
	"\xDA\xF1" => '&#38536;',
	"\xDA\xF2" => '&#38541;',
	"\xDA\xF3" => '&#38551;',
	"\xDA\xF4" => '&#38576;',
	"\xDA\xF5" => '&#37015;',
	"\xDA\xF6" => '&#37019;',
	"\xDA\xF7" => '&#37021;',
	"\xDA\xF8" => '&#37017;',
	"\xDA\xF9" => '&#37036;',
	"\xDA\xFA" => '&#37025;',
	"\xDA\xFB" => '&#37044;',
	"\xDA\xFC" => '&#37043;',
	"\xDA\xFD" => '&#37046;',
	"\xDA\xFE" => '&#37050;',
	"\xDB\xA1" => '&#37048;',
	"\xDB\xA2" => '&#37040;',
	"\xDB\xA3" => '&#37071;',
	"\xDB\xA4" => '&#37061;',
	"\xDB\xA5" => '&#37054;',
	"\xDB\xA6" => '&#37072;',
	"\xDB\xA7" => '&#37060;',
	"\xDB\xA8" => '&#37063;',
	"\xDB\xA9" => '&#37075;',
	"\xDB\xAA" => '&#37094;',
	"\xDB\xAB" => '&#37090;',
	"\xDB\xAC" => '&#37084;',
	"\xDB\xAD" => '&#37079;',
	"\xDB\xAE" => '&#37083;',
	"\xDB\xAF" => '&#37099;',
	"\xDB\xB0" => '&#37103;',
	"\xDB\xB1" => '&#37118;',
	"\xDB\xB2" => '&#37124;',
	"\xDB\xB3" => '&#37154;',
	"\xDB\xB4" => '&#37150;',
	"\xDB\xB5" => '&#37155;',
	"\xDB\xB6" => '&#37169;',
	"\xDB\xB7" => '&#37167;',
	"\xDB\xB8" => '&#37177;',
	"\xDB\xB9" => '&#37187;',
	"\xDB\xBA" => '&#37190;',
	"\xDB\xBB" => '&#21005;',
	"\xDB\xBC" => '&#22850;',
	"\xDB\xBD" => '&#21154;',
	"\xDB\xBE" => '&#21164;',
	"\xDB\xBF" => '&#21165;',
	"\xDB\xC0" => '&#21182;',
	"\xDB\xC1" => '&#21759;',
	"\xDB\xC2" => '&#21200;',
	"\xDB\xC3" => '&#21206;',
	"\xDB\xC4" => '&#21232;',
	"\xDB\xC5" => '&#21471;',
	"\xDB\xC6" => '&#29166;',
	"\xDB\xC7" => '&#30669;',
	"\xDB\xC8" => '&#24308;',
	"\xDB\xC9" => '&#20981;',
	"\xDB\xCA" => '&#20988;',
	"\xDB\xCB" => '&#39727;',
	"\xDB\xCC" => '&#21430;',
	"\xDB\xCD" => '&#24321;',
	"\xDB\xCE" => '&#30042;',
	"\xDB\xCF" => '&#24047;',
	"\xDB\xD0" => '&#22348;',
	"\xDB\xD1" => '&#22441;',
	"\xDB\xD2" => '&#22433;',
	"\xDB\xD3" => '&#22654;',
	"\xDB\xD4" => '&#22716;',
	"\xDB\xD5" => '&#22725;',
	"\xDB\xD6" => '&#22737;',
	"\xDB\xD7" => '&#22313;',
	"\xDB\xD8" => '&#22316;',
	"\xDB\xD9" => '&#22314;',
	"\xDB\xDA" => '&#22323;',
	"\xDB\xDB" => '&#22329;',
	"\xDB\xDC" => '&#22318;',
	"\xDB\xDD" => '&#22319;',
	"\xDB\xDE" => '&#22364;',
	"\xDB\xDF" => '&#22331;',
	"\xDB\xE0" => '&#22338;',
	"\xDB\xE1" => '&#22377;',
	"\xDB\xE2" => '&#22405;',
	"\xDB\xE3" => '&#22379;',
	"\xDB\xE4" => '&#22406;',
	"\xDB\xE5" => '&#22396;',
	"\xDB\xE6" => '&#22395;',
	"\xDB\xE7" => '&#22376;',
	"\xDB\xE8" => '&#22381;',
	"\xDB\xE9" => '&#22390;',
	"\xDB\xEA" => '&#22387;',
	"\xDB\xEB" => '&#22445;',
	"\xDB\xEC" => '&#22436;',
	"\xDB\xED" => '&#22412;',
	"\xDB\xEE" => '&#22450;',
	"\xDB\xEF" => '&#22479;',
	"\xDB\xF0" => '&#22439;',
	"\xDB\xF1" => '&#22452;',
	"\xDB\xF2" => '&#22419;',
	"\xDB\xF3" => '&#22432;',
	"\xDB\xF4" => '&#22485;',
	"\xDB\xF5" => '&#22488;',
	"\xDB\xF6" => '&#22490;',
	"\xDB\xF7" => '&#22489;',
	"\xDB\xF8" => '&#22482;',
	"\xDB\xF9" => '&#22456;',
	"\xDB\xFA" => '&#22516;',
	"\xDB\xFB" => '&#22511;',
	"\xDB\xFC" => '&#22520;',
	"\xDB\xFD" => '&#22500;',
	"\xDB\xFE" => '&#22493;',
	"\xDC\xA1" => '&#22539;',
	"\xDC\xA2" => '&#22541;',
	"\xDC\xA3" => '&#22525;',
	"\xDC\xA4" => '&#22509;',
	"\xDC\xA5" => '&#22528;',
	"\xDC\xA6" => '&#22558;',
	"\xDC\xA7" => '&#22553;',
	"\xDC\xA8" => '&#22596;',
	"\xDC\xA9" => '&#22560;',
	"\xDC\xAA" => '&#22629;',
	"\xDC\xAB" => '&#22636;',
	"\xDC\xAC" => '&#22657;',
	"\xDC\xAD" => '&#22665;',
	"\xDC\xAE" => '&#22682;',
	"\xDC\xAF" => '&#22656;',
	"\xDC\xB0" => '&#39336;',
	"\xDC\xB1" => '&#40729;',
	"\xDC\xB2" => '&#25087;',
	"\xDC\xB3" => '&#33401;',
	"\xDC\xB4" => '&#33405;',
	"\xDC\xB5" => '&#33407;',
	"\xDC\xB6" => '&#33423;',
	"\xDC\xB7" => '&#33418;',
	"\xDC\xB8" => '&#33448;',
	"\xDC\xB9" => '&#33412;',
	"\xDC\xBA" => '&#33422;',
	"\xDC\xBB" => '&#33425;',
	"\xDC\xBC" => '&#33431;',
	"\xDC\xBD" => '&#33433;',
	"\xDC\xBE" => '&#33451;',
	"\xDC\xBF" => '&#33464;',
	"\xDC\xC0" => '&#33470;',
	"\xDC\xC1" => '&#33456;',
	"\xDC\xC2" => '&#33480;',
	"\xDC\xC3" => '&#33482;',
	"\xDC\xC4" => '&#33507;',
	"\xDC\xC5" => '&#33432;',
	"\xDC\xC6" => '&#33463;',
	"\xDC\xC7" => '&#33454;',
	"\xDC\xC8" => '&#33483;',
	"\xDC\xC9" => '&#33484;',
	"\xDC\xCA" => '&#33473;',
	"\xDC\xCB" => '&#33449;',
	"\xDC\xCC" => '&#33460;',
	"\xDC\xCD" => '&#33441;',
	"\xDC\xCE" => '&#33450;',
	"\xDC\xCF" => '&#33439;',
	"\xDC\xD0" => '&#33476;',
	"\xDC\xD1" => '&#33486;',
	"\xDC\xD2" => '&#33444;',
	"\xDC\xD3" => '&#33505;',
	"\xDC\xD4" => '&#33545;',
	"\xDC\xD5" => '&#33527;',
	"\xDC\xD6" => '&#33508;',
	"\xDC\xD7" => '&#33551;',
	"\xDC\xD8" => '&#33543;',
	"\xDC\xD9" => '&#33500;',
	"\xDC\xDA" => '&#33524;',
	"\xDC\xDB" => '&#33490;',
	"\xDC\xDC" => '&#33496;',
	"\xDC\xDD" => '&#33548;',
	"\xDC\xDE" => '&#33531;',
	"\xDC\xDF" => '&#33491;',
	"\xDC\xE0" => '&#33553;',
	"\xDC\xE1" => '&#33562;',
	"\xDC\xE2" => '&#33542;',
	"\xDC\xE3" => '&#33556;',
	"\xDC\xE4" => '&#33557;',
	"\xDC\xE5" => '&#33504;',
	"\xDC\xE6" => '&#33493;',
	"\xDC\xE7" => '&#33564;',
	"\xDC\xE8" => '&#33617;',
	"\xDC\xE9" => '&#33627;',
	"\xDC\xEA" => '&#33628;',
	"\xDC\xEB" => '&#33544;',
	"\xDC\xEC" => '&#33682;',
	"\xDC\xED" => '&#33596;',
	"\xDC\xEE" => '&#33588;',
	"\xDC\xEF" => '&#33585;',
	"\xDC\xF0" => '&#33691;',
	"\xDC\xF1" => '&#33630;',
	"\xDC\xF2" => '&#33583;',
	"\xDC\xF3" => '&#33615;',
	"\xDC\xF4" => '&#33607;',
	"\xDC\xF5" => '&#33603;',
	"\xDC\xF6" => '&#33631;',
	"\xDC\xF7" => '&#33600;',
	"\xDC\xF8" => '&#33559;',
	"\xDC\xF9" => '&#33632;',
	"\xDC\xFA" => '&#33581;',
	"\xDC\xFB" => '&#33594;',
	"\xDC\xFC" => '&#33587;',
	"\xDC\xFD" => '&#33638;',
	"\xDC\xFE" => '&#33637;',
	"\xDD\xA1" => '&#33640;',
	"\xDD\xA2" => '&#33563;',
	"\xDD\xA3" => '&#33641;',
	"\xDD\xA4" => '&#33644;',
	"\xDD\xA5" => '&#33642;',
	"\xDD\xA6" => '&#33645;',
	"\xDD\xA7" => '&#33646;',
	"\xDD\xA8" => '&#33712;',
	"\xDD\xA9" => '&#33656;',
	"\xDD\xAA" => '&#33715;',
	"\xDD\xAB" => '&#33716;',
	"\xDD\xAC" => '&#33696;',
	"\xDD\xAD" => '&#33706;',
	"\xDD\xAE" => '&#33683;',
	"\xDD\xAF" => '&#33692;',
	"\xDD\xB0" => '&#33669;',
	"\xDD\xB1" => '&#33660;',
	"\xDD\xB2" => '&#33718;',
	"\xDD\xB3" => '&#33705;',
	"\xDD\xB4" => '&#33661;',
	"\xDD\xB5" => '&#33720;',
	"\xDD\xB6" => '&#33659;',
	"\xDD\xB7" => '&#33688;',
	"\xDD\xB8" => '&#33694;',
	"\xDD\xB9" => '&#33704;',
	"\xDD\xBA" => '&#33722;',
	"\xDD\xBB" => '&#33724;',
	"\xDD\xBC" => '&#33729;',
	"\xDD\xBD" => '&#33793;',
	"\xDD\xBE" => '&#33765;',
	"\xDD\xBF" => '&#33752;',
	"\xDD\xC0" => '&#22535;',
	"\xDD\xC1" => '&#33816;',
	"\xDD\xC2" => '&#33803;',
	"\xDD\xC3" => '&#33757;',
	"\xDD\xC4" => '&#33789;',
	"\xDD\xC5" => '&#33750;',
	"\xDD\xC6" => '&#33820;',
	"\xDD\xC7" => '&#33848;',
	"\xDD\xC8" => '&#33809;',
	"\xDD\xC9" => '&#33798;',
	"\xDD\xCA" => '&#33748;',
	"\xDD\xCB" => '&#33759;',
	"\xDD\xCC" => '&#33807;',
	"\xDD\xCD" => '&#33795;',
	"\xDD\xCE" => '&#33784;',
	"\xDD\xCF" => '&#33785;',
	"\xDD\xD0" => '&#33770;',
	"\xDD\xD1" => '&#33733;',
	"\xDD\xD2" => '&#33728;',
	"\xDD\xD3" => '&#33830;',
	"\xDD\xD4" => '&#33776;',
	"\xDD\xD5" => '&#33761;',
	"\xDD\xD6" => '&#33884;',
	"\xDD\xD7" => '&#33873;',
	"\xDD\xD8" => '&#33882;',
	"\xDD\xD9" => '&#33881;',
	"\xDD\xDA" => '&#33907;',
	"\xDD\xDB" => '&#33927;',
	"\xDD\xDC" => '&#33928;',
	"\xDD\xDD" => '&#33914;',
	"\xDD\xDE" => '&#33929;',
	"\xDD\xDF" => '&#33912;',
	"\xDD\xE0" => '&#33852;',
	"\xDD\xE1" => '&#33862;',
	"\xDD\xE2" => '&#33897;',
	"\xDD\xE3" => '&#33910;',
	"\xDD\xE4" => '&#33932;',
	"\xDD\xE5" => '&#33934;',
	"\xDD\xE6" => '&#33841;',
	"\xDD\xE7" => '&#33901;',
	"\xDD\xE8" => '&#33985;',
	"\xDD\xE9" => '&#33997;',
	"\xDD\xEA" => '&#34000;',
	"\xDD\xEB" => '&#34022;',
	"\xDD\xEC" => '&#33981;',
	"\xDD\xED" => '&#34003;',
	"\xDD\xEE" => '&#33994;',
	"\xDD\xEF" => '&#33983;',
	"\xDD\xF0" => '&#33978;',
	"\xDD\xF1" => '&#34016;',
	"\xDD\xF2" => '&#33953;',
	"\xDD\xF3" => '&#33977;',
	"\xDD\xF4" => '&#33972;',
	"\xDD\xF5" => '&#33943;',
	"\xDD\xF6" => '&#34021;',
	"\xDD\xF7" => '&#34019;',
	"\xDD\xF8" => '&#34060;',
	"\xDD\xF9" => '&#29965;',
	"\xDD\xFA" => '&#34104;',
	"\xDD\xFB" => '&#34032;',
	"\xDD\xFC" => '&#34105;',
	"\xDD\xFD" => '&#34079;',
	"\xDD\xFE" => '&#34106;',
	"\xDE\xA1" => '&#34134;',
	"\xDE\xA2" => '&#34107;',
	"\xDE\xA3" => '&#34047;',
	"\xDE\xA4" => '&#34044;',
	"\xDE\xA5" => '&#34137;',
	"\xDE\xA6" => '&#34120;',
	"\xDE\xA7" => '&#34152;',
	"\xDE\xA8" => '&#34148;',
	"\xDE\xA9" => '&#34142;',
	"\xDE\xAA" => '&#34170;',
	"\xDE\xAB" => '&#30626;',
	"\xDE\xAC" => '&#34115;',
	"\xDE\xAD" => '&#34162;',
	"\xDE\xAE" => '&#34171;',
	"\xDE\xAF" => '&#34212;',
	"\xDE\xB0" => '&#34216;',
	"\xDE\xB1" => '&#34183;',
	"\xDE\xB2" => '&#34191;',
	"\xDE\xB3" => '&#34169;',
	"\xDE\xB4" => '&#34222;',
	"\xDE\xB5" => '&#34204;',
	"\xDE\xB6" => '&#34181;',
	"\xDE\xB7" => '&#34233;',
	"\xDE\xB8" => '&#34231;',
	"\xDE\xB9" => '&#34224;',
	"\xDE\xBA" => '&#34259;',
	"\xDE\xBB" => '&#34241;',
	"\xDE\xBC" => '&#34268;',
	"\xDE\xBD" => '&#34303;',
	"\xDE\xBE" => '&#34343;',
	"\xDE\xBF" => '&#34309;',
	"\xDE\xC0" => '&#34345;',
	"\xDE\xC1" => '&#34326;',
	"\xDE\xC2" => '&#34364;',
	"\xDE\xC3" => '&#24318;',
	"\xDE\xC4" => '&#24328;',
	"\xDE\xC5" => '&#22844;',
	"\xDE\xC6" => '&#22849;',
	"\xDE\xC7" => '&#32823;',
	"\xDE\xC8" => '&#22869;',
	"\xDE\xC9" => '&#22874;',
	"\xDE\xCA" => '&#22872;',
	"\xDE\xCB" => '&#21263;',
	"\xDE\xCC" => '&#23586;',
	"\xDE\xCD" => '&#23589;',
	"\xDE\xCE" => '&#23596;',
	"\xDE\xCF" => '&#23604;',
	"\xDE\xD0" => '&#25164;',
	"\xDE\xD1" => '&#25194;',
	"\xDE\xD2" => '&#25247;',
	"\xDE\xD3" => '&#25275;',
	"\xDE\xD4" => '&#25290;',
	"\xDE\xD5" => '&#25306;',
	"\xDE\xD6" => '&#25303;',
	"\xDE\xD7" => '&#25326;',
	"\xDE\xD8" => '&#25378;',
	"\xDE\xD9" => '&#25334;',
	"\xDE\xDA" => '&#25401;',
	"\xDE\xDB" => '&#25419;',
	"\xDE\xDC" => '&#25411;',
	"\xDE\xDD" => '&#25517;',
	"\xDE\xDE" => '&#25590;',
	"\xDE\xDF" => '&#25457;',
	"\xDE\xE0" => '&#25466;',
	"\xDE\xE1" => '&#25486;',
	"\xDE\xE2" => '&#25524;',
	"\xDE\xE3" => '&#25453;',
	"\xDE\xE4" => '&#25516;',
	"\xDE\xE5" => '&#25482;',
	"\xDE\xE6" => '&#25449;',
	"\xDE\xE7" => '&#25518;',
	"\xDE\xE8" => '&#25532;',
	"\xDE\xE9" => '&#25586;',
	"\xDE\xEA" => '&#25592;',
	"\xDE\xEB" => '&#25568;',
	"\xDE\xEC" => '&#25599;',
	"\xDE\xED" => '&#25540;',
	"\xDE\xEE" => '&#25566;',
	"\xDE\xEF" => '&#25550;',
	"\xDE\xF0" => '&#25682;',
	"\xDE\xF1" => '&#25542;',
	"\xDE\xF2" => '&#25534;',
	"\xDE\xF3" => '&#25669;',
	"\xDE\xF4" => '&#25665;',
	"\xDE\xF5" => '&#25611;',
	"\xDE\xF6" => '&#25627;',
	"\xDE\xF7" => '&#25632;',
	"\xDE\xF8" => '&#25612;',
	"\xDE\xF9" => '&#25638;',
	"\xDE\xFA" => '&#25633;',
	"\xDE\xFB" => '&#25694;',
	"\xDE\xFC" => '&#25732;',
	"\xDE\xFD" => '&#25709;',
	"\xDE\xFE" => '&#25750;',
	"\xDF\xA1" => '&#25722;',
	"\xDF\xA2" => '&#25783;',
	"\xDF\xA3" => '&#25784;',
	"\xDF\xA4" => '&#25753;',
	"\xDF\xA5" => '&#25786;',
	"\xDF\xA6" => '&#25792;',
	"\xDF\xA7" => '&#25808;',
	"\xDF\xA8" => '&#25815;',
	"\xDF\xA9" => '&#25828;',
	"\xDF\xAA" => '&#25826;',
	"\xDF\xAB" => '&#25865;',
	"\xDF\xAC" => '&#25893;',
	"\xDF\xAD" => '&#25902;',
	"\xDF\xAE" => '&#24331;',
	"\xDF\xAF" => '&#24530;',
	"\xDF\xB0" => '&#29977;',
	"\xDF\xB1" => '&#24337;',
	"\xDF\xB2" => '&#21343;',
	"\xDF\xB3" => '&#21489;',
	"\xDF\xB4" => '&#21501;',
	"\xDF\xB5" => '&#21481;',
	"\xDF\xB6" => '&#21480;',
	"\xDF\xB7" => '&#21499;',
	"\xDF\xB8" => '&#21522;',
	"\xDF\xB9" => '&#21526;',
	"\xDF\xBA" => '&#21510;',
	"\xDF\xBB" => '&#21579;',
	"\xDF\xBC" => '&#21586;',
	"\xDF\xBD" => '&#21587;',
	"\xDF\xBE" => '&#21588;',
	"\xDF\xBF" => '&#21590;',
	"\xDF\xC0" => '&#21571;',
	"\xDF\xC1" => '&#21537;',
	"\xDF\xC2" => '&#21591;',
	"\xDF\xC3" => '&#21593;',
	"\xDF\xC4" => '&#21539;',
	"\xDF\xC5" => '&#21554;',
	"\xDF\xC6" => '&#21634;',
	"\xDF\xC7" => '&#21652;',
	"\xDF\xC8" => '&#21623;',
	"\xDF\xC9" => '&#21617;',
	"\xDF\xCA" => '&#21604;',
	"\xDF\xCB" => '&#21658;',
	"\xDF\xCC" => '&#21659;',
	"\xDF\xCD" => '&#21636;',
	"\xDF\xCE" => '&#21622;',
	"\xDF\xCF" => '&#21606;',
	"\xDF\xD0" => '&#21661;',
	"\xDF\xD1" => '&#21712;',
	"\xDF\xD2" => '&#21677;',
	"\xDF\xD3" => '&#21698;',
	"\xDF\xD4" => '&#21684;',
	"\xDF\xD5" => '&#21714;',
	"\xDF\xD6" => '&#21671;',
	"\xDF\xD7" => '&#21670;',
	"\xDF\xD8" => '&#21715;',
	"\xDF\xD9" => '&#21716;',
	"\xDF\xDA" => '&#21618;',
	"\xDF\xDB" => '&#21667;',
	"\xDF\xDC" => '&#21717;',
	"\xDF\xDD" => '&#21691;',
	"\xDF\xDE" => '&#21695;',
	"\xDF\xDF" => '&#21708;',
	"\xDF\xE0" => '&#21721;',
	"\xDF\xE1" => '&#21722;',
	"\xDF\xE2" => '&#21724;',
	"\xDF\xE3" => '&#21673;',
	"\xDF\xE4" => '&#21674;',
	"\xDF\xE5" => '&#21668;',
	"\xDF\xE6" => '&#21725;',
	"\xDF\xE7" => '&#21711;',
	"\xDF\xE8" => '&#21726;',
	"\xDF\xE9" => '&#21787;',
	"\xDF\xEA" => '&#21735;',
	"\xDF\xEB" => '&#21792;',
	"\xDF\xEC" => '&#21757;',
	"\xDF\xED" => '&#21780;',
	"\xDF\xEE" => '&#21747;',
	"\xDF\xEF" => '&#21794;',
	"\xDF\xF0" => '&#21795;',
	"\xDF\xF1" => '&#21775;',
	"\xDF\xF2" => '&#21777;',
	"\xDF\xF3" => '&#21799;',
	"\xDF\xF4" => '&#21802;',
	"\xDF\xF5" => '&#21863;',
	"\xDF\xF6" => '&#21903;',
	"\xDF\xF7" => '&#21941;',
	"\xDF\xF8" => '&#21833;',
	"\xDF\xF9" => '&#21869;',
	"\xDF\xFA" => '&#21825;',
	"\xDF\xFB" => '&#21845;',
	"\xDF\xFC" => '&#21823;',
	"\xDF\xFD" => '&#21840;',
	"\xDF\xFE" => '&#21820;',
	"\xE0\xA1" => '&#21815;',
	"\xE0\xA2" => '&#21846;',
	"\xE0\xA3" => '&#21877;',
	"\xE0\xA4" => '&#21878;',
	"\xE0\xA5" => '&#21879;',
	"\xE0\xA6" => '&#21811;',
	"\xE0\xA7" => '&#21808;',
	"\xE0\xA8" => '&#21852;',
	"\xE0\xA9" => '&#21899;',
	"\xE0\xAA" => '&#21970;',
	"\xE0\xAB" => '&#21891;',
	"\xE0\xAC" => '&#21937;',
	"\xE0\xAD" => '&#21945;',
	"\xE0\xAE" => '&#21896;',
	"\xE0\xAF" => '&#21889;',
	"\xE0\xB0" => '&#21919;',
	"\xE0\xB1" => '&#21886;',
	"\xE0\xB2" => '&#21974;',
	"\xE0\xB3" => '&#21905;',
	"\xE0\xB4" => '&#21883;',
	"\xE0\xB5" => '&#21983;',
	"\xE0\xB6" => '&#21949;',
	"\xE0\xB7" => '&#21950;',
	"\xE0\xB8" => '&#21908;',
	"\xE0\xB9" => '&#21913;',
	"\xE0\xBA" => '&#21994;',
	"\xE0\xBB" => '&#22007;',
	"\xE0\xBC" => '&#21961;',
	"\xE0\xBD" => '&#22047;',
	"\xE0\xBE" => '&#21969;',
	"\xE0\xBF" => '&#21995;',
	"\xE0\xC0" => '&#21996;',
	"\xE0\xC1" => '&#21972;',
	"\xE0\xC2" => '&#21990;',
	"\xE0\xC3" => '&#21981;',
	"\xE0\xC4" => '&#21956;',
	"\xE0\xC5" => '&#21999;',
	"\xE0\xC6" => '&#21989;',
	"\xE0\xC7" => '&#22002;',
	"\xE0\xC8" => '&#22003;',
	"\xE0\xC9" => '&#21964;',
	"\xE0\xCA" => '&#21965;',
	"\xE0\xCB" => '&#21992;',
	"\xE0\xCC" => '&#22005;',
	"\xE0\xCD" => '&#21988;',
	"\xE0\xCE" => '&#36756;',
	"\xE0\xCF" => '&#22046;',
	"\xE0\xD0" => '&#22024;',
	"\xE0\xD1" => '&#22028;',
	"\xE0\xD2" => '&#22017;',
	"\xE0\xD3" => '&#22052;',
	"\xE0\xD4" => '&#22051;',
	"\xE0\xD5" => '&#22014;',
	"\xE0\xD6" => '&#22016;',
	"\xE0\xD7" => '&#22055;',
	"\xE0\xD8" => '&#22061;',
	"\xE0\xD9" => '&#22104;',
	"\xE0\xDA" => '&#22073;',
	"\xE0\xDB" => '&#22103;',
	"\xE0\xDC" => '&#22060;',
	"\xE0\xDD" => '&#22093;',
	"\xE0\xDE" => '&#22114;',
	"\xE0\xDF" => '&#22105;',
	"\xE0\xE0" => '&#22108;',
	"\xE0\xE1" => '&#22092;',
	"\xE0\xE2" => '&#22100;',
	"\xE0\xE3" => '&#22150;',
	"\xE0\xE4" => '&#22116;',
	"\xE0\xE5" => '&#22129;',
	"\xE0\xE6" => '&#22123;',
	"\xE0\xE7" => '&#22139;',
	"\xE0\xE8" => '&#22140;',
	"\xE0\xE9" => '&#22149;',
	"\xE0\xEA" => '&#22163;',
	"\xE0\xEB" => '&#22191;',
	"\xE0\xEC" => '&#22228;',
	"\xE0\xED" => '&#22231;',
	"\xE0\xEE" => '&#22237;',
	"\xE0\xEF" => '&#22241;',
	"\xE0\xF0" => '&#22261;',
	"\xE0\xF1" => '&#22251;',
	"\xE0\xF2" => '&#22265;',
	"\xE0\xF3" => '&#22271;',
	"\xE0\xF4" => '&#22276;',
	"\xE0\xF5" => '&#22282;',
	"\xE0\xF6" => '&#22281;',
	"\xE0\xF7" => '&#22300;',
	"\xE0\xF8" => '&#24079;',
	"\xE0\xF9" => '&#24089;',
	"\xE0\xFA" => '&#24084;',
	"\xE0\xFB" => '&#24081;',
	"\xE0\xFC" => '&#24113;',
	"\xE0\xFD" => '&#24123;',
	"\xE0\xFE" => '&#24124;',
	"\xE1\xA1" => '&#24119;',
	"\xE1\xA2" => '&#24132;',
	"\xE1\xA3" => '&#24148;',
	"\xE1\xA4" => '&#24155;',
	"\xE1\xA5" => '&#24158;',
	"\xE1\xA6" => '&#24161;',
	"\xE1\xA7" => '&#23692;',
	"\xE1\xA8" => '&#23674;',
	"\xE1\xA9" => '&#23693;',
	"\xE1\xAA" => '&#23696;',
	"\xE1\xAB" => '&#23702;',
	"\xE1\xAC" => '&#23688;',
	"\xE1\xAD" => '&#23704;',
	"\xE1\xAE" => '&#23705;',
	"\xE1\xAF" => '&#23697;',
	"\xE1\xB0" => '&#23706;',
	"\xE1\xB1" => '&#23708;',
	"\xE1\xB2" => '&#23733;',
	"\xE1\xB3" => '&#23714;',
	"\xE1\xB4" => '&#23741;',
	"\xE1\xB5" => '&#23724;',
	"\xE1\xB6" => '&#23723;',
	"\xE1\xB7" => '&#23729;',
	"\xE1\xB8" => '&#23715;',
	"\xE1\xB9" => '&#23745;',
	"\xE1\xBA" => '&#23735;',
	"\xE1\xBB" => '&#23748;',
	"\xE1\xBC" => '&#23762;',
	"\xE1\xBD" => '&#23780;',
	"\xE1\xBE" => '&#23755;',
	"\xE1\xBF" => '&#23781;',
	"\xE1\xC0" => '&#23810;',
	"\xE1\xC1" => '&#23811;',
	"\xE1\xC2" => '&#23847;',
	"\xE1\xC3" => '&#23846;',
	"\xE1\xC4" => '&#23854;',
	"\xE1\xC5" => '&#23844;',
	"\xE1\xC6" => '&#23838;',
	"\xE1\xC7" => '&#23814;',
	"\xE1\xC8" => '&#23835;',
	"\xE1\xC9" => '&#23896;',
	"\xE1\xCA" => '&#23870;',
	"\xE1\xCB" => '&#23860;',
	"\xE1\xCC" => '&#23869;',
	"\xE1\xCD" => '&#23916;',
	"\xE1\xCE" => '&#23899;',
	"\xE1\xCF" => '&#23919;',
	"\xE1\xD0" => '&#23901;',
	"\xE1\xD1" => '&#23915;',
	"\xE1\xD2" => '&#23883;',
	"\xE1\xD3" => '&#23882;',
	"\xE1\xD4" => '&#23913;',
	"\xE1\xD5" => '&#23924;',
	"\xE1\xD6" => '&#23938;',
	"\xE1\xD7" => '&#23961;',
	"\xE1\xD8" => '&#23965;',
	"\xE1\xD9" => '&#35955;',
	"\xE1\xDA" => '&#23991;',
	"\xE1\xDB" => '&#24005;',
	"\xE1\xDC" => '&#24435;',
	"\xE1\xDD" => '&#24439;',
	"\xE1\xDE" => '&#24450;',
	"\xE1\xDF" => '&#24455;',
	"\xE1\xE0" => '&#24457;',
	"\xE1\xE1" => '&#24460;',
	"\xE1\xE2" => '&#24469;',
	"\xE1\xE3" => '&#24473;',
	"\xE1\xE4" => '&#24476;',
	"\xE1\xE5" => '&#24488;',
	"\xE1\xE6" => '&#24493;',
	"\xE1\xE7" => '&#24501;',
	"\xE1\xE8" => '&#24508;',
	"\xE1\xE9" => '&#34914;',
	"\xE1\xEA" => '&#24417;',
	"\xE1\xEB" => '&#29357;',
	"\xE1\xEC" => '&#29360;',
	"\xE1\xED" => '&#29364;',
	"\xE1\xEE" => '&#29367;',
	"\xE1\xEF" => '&#29368;',
	"\xE1\xF0" => '&#29379;',
	"\xE1\xF1" => '&#29377;',
	"\xE1\xF2" => '&#29390;',
	"\xE1\xF3" => '&#29389;',
	"\xE1\xF4" => '&#29394;',
	"\xE1\xF5" => '&#29416;',
	"\xE1\xF6" => '&#29423;',
	"\xE1\xF7" => '&#29417;',
	"\xE1\xF8" => '&#29426;',
	"\xE1\xF9" => '&#29428;',
	"\xE1\xFA" => '&#29431;',
	"\xE1\xFB" => '&#29441;',
	"\xE1\xFC" => '&#29427;',
	"\xE1\xFD" => '&#29443;',
	"\xE1\xFE" => '&#29434;',
	"\xE2\xA1" => '&#29435;',
	"\xE2\xA2" => '&#29463;',
	"\xE2\xA3" => '&#29459;',
	"\xE2\xA4" => '&#29473;',
	"\xE2\xA5" => '&#29450;',
	"\xE2\xA6" => '&#29470;',
	"\xE2\xA7" => '&#29469;',
	"\xE2\xA8" => '&#29461;',
	"\xE2\xA9" => '&#29474;',
	"\xE2\xAA" => '&#29497;',
	"\xE2\xAB" => '&#29477;',
	"\xE2\xAC" => '&#29484;',
	"\xE2\xAD" => '&#29496;',
	"\xE2\xAE" => '&#29489;',
	"\xE2\xAF" => '&#29520;',
	"\xE2\xB0" => '&#29517;',
	"\xE2\xB1" => '&#29527;',
	"\xE2\xB2" => '&#29536;',
	"\xE2\xB3" => '&#29548;',
	"\xE2\xB4" => '&#29551;',
	"\xE2\xB5" => '&#29566;',
	"\xE2\xB6" => '&#33307;',
	"\xE2\xB7" => '&#22821;',
	"\xE2\xB8" => '&#39143;',
	"\xE2\xB9" => '&#22820;',
	"\xE2\xBA" => '&#22786;',
	"\xE2\xBB" => '&#39267;',
	"\xE2\xBC" => '&#39271;',
	"\xE2\xBD" => '&#39272;',
	"\xE2\xBE" => '&#39273;',
	"\xE2\xBF" => '&#39274;',
	"\xE2\xC0" => '&#39275;',
	"\xE2\xC1" => '&#39276;',
	"\xE2\xC2" => '&#39284;',
	"\xE2\xC3" => '&#39287;',
	"\xE2\xC4" => '&#39293;',
	"\xE2\xC5" => '&#39296;',
	"\xE2\xC6" => '&#39300;',
	"\xE2\xC7" => '&#39303;',
	"\xE2\xC8" => '&#39306;',
	"\xE2\xC9" => '&#39309;',
	"\xE2\xCA" => '&#39312;',
	"\xE2\xCB" => '&#39313;',
	"\xE2\xCC" => '&#39315;',
	"\xE2\xCD" => '&#39316;',
	"\xE2\xCE" => '&#39317;',
	"\xE2\xCF" => '&#24192;',
	"\xE2\xD0" => '&#24209;',
	"\xE2\xD1" => '&#24203;',
	"\xE2\xD2" => '&#24214;',
	"\xE2\xD3" => '&#24229;',
	"\xE2\xD4" => '&#24224;',
	"\xE2\xD5" => '&#24249;',
	"\xE2\xD6" => '&#24245;',
	"\xE2\xD7" => '&#24254;',
	"\xE2\xD8" => '&#24243;',
	"\xE2\xD9" => '&#36179;',
	"\xE2\xDA" => '&#24274;',
	"\xE2\xDB" => '&#24273;',
	"\xE2\xDC" => '&#24283;',
	"\xE2\xDD" => '&#24296;',
	"\xE2\xDE" => '&#24298;',
	"\xE2\xDF" => '&#33210;',
	"\xE2\xE0" => '&#24516;',
	"\xE2\xE1" => '&#24521;',
	"\xE2\xE2" => '&#24534;',
	"\xE2\xE3" => '&#24527;',
	"\xE2\xE4" => '&#24579;',
	"\xE2\xE5" => '&#24558;',
	"\xE2\xE6" => '&#24580;',
	"\xE2\xE7" => '&#24545;',
	"\xE2\xE8" => '&#24548;',
	"\xE2\xE9" => '&#24574;',
	"\xE2\xEA" => '&#24581;',
	"\xE2\xEB" => '&#24582;',
	"\xE2\xEC" => '&#24554;',
	"\xE2\xED" => '&#24557;',
	"\xE2\xEE" => '&#24568;',
	"\xE2\xEF" => '&#24601;',
	"\xE2\xF0" => '&#24629;',
	"\xE2\xF1" => '&#24614;',
	"\xE2\xF2" => '&#24603;',
	"\xE2\xF3" => '&#24591;',
	"\xE2\xF4" => '&#24589;',
	"\xE2\xF5" => '&#24617;',
	"\xE2\xF6" => '&#24619;',
	"\xE2\xF7" => '&#24586;',
	"\xE2\xF8" => '&#24639;',
	"\xE2\xF9" => '&#24609;',
	"\xE2\xFA" => '&#24696;',
	"\xE2\xFB" => '&#24697;',
	"\xE2\xFC" => '&#24699;',
	"\xE2\xFD" => '&#24698;',
	"\xE2\xFE" => '&#24642;',
	"\xE3\xA1" => '&#24682;',
	"\xE3\xA2" => '&#24701;',
	"\xE3\xA3" => '&#24726;',
	"\xE3\xA4" => '&#24730;',
	"\xE3\xA5" => '&#24749;',
	"\xE3\xA6" => '&#24733;',
	"\xE3\xA7" => '&#24707;',
	"\xE3\xA8" => '&#24722;',
	"\xE3\xA9" => '&#24716;',
	"\xE3\xAA" => '&#24731;',
	"\xE3\xAB" => '&#24812;',
	"\xE3\xAC" => '&#24763;',
	"\xE3\xAD" => '&#24753;',
	"\xE3\xAE" => '&#24797;',
	"\xE3\xAF" => '&#24792;',
	"\xE3\xB0" => '&#24774;',
	"\xE3\xB1" => '&#24794;',
	"\xE3\xB2" => '&#24756;',
	"\xE3\xB3" => '&#24864;',
	"\xE3\xB4" => '&#24870;',
	"\xE3\xB5" => '&#24853;',
	"\xE3\xB6" => '&#24867;',
	"\xE3\xB7" => '&#24820;',
	"\xE3\xB8" => '&#24832;',
	"\xE3\xB9" => '&#24846;',
	"\xE3\xBA" => '&#24875;',
	"\xE3\xBB" => '&#24906;',
	"\xE3\xBC" => '&#24949;',
	"\xE3\xBD" => '&#25004;',
	"\xE3\xBE" => '&#24980;',
	"\xE3\xBF" => '&#24999;',
	"\xE3\xC0" => '&#25015;',
	"\xE3\xC1" => '&#25044;',
	"\xE3\xC2" => '&#25077;',
	"\xE3\xC3" => '&#24541;',
	"\xE3\xC4" => '&#38579;',
	"\xE3\xC5" => '&#38377;',
	"\xE3\xC6" => '&#38379;',
	"\xE3\xC7" => '&#38385;',
	"\xE3\xC8" => '&#38387;',
	"\xE3\xC9" => '&#38389;',
	"\xE3\xCA" => '&#38390;',
	"\xE3\xCB" => '&#38396;',
	"\xE3\xCC" => '&#38398;',
	"\xE3\xCD" => '&#38403;',
	"\xE3\xCE" => '&#38404;',
	"\xE3\xCF" => '&#38406;',
	"\xE3\xD0" => '&#38408;',
	"\xE3\xD1" => '&#38410;',
	"\xE3\xD2" => '&#38411;',
	"\xE3\xD3" => '&#38412;',
	"\xE3\xD4" => '&#38413;',
	"\xE3\xD5" => '&#38415;',
	"\xE3\xD6" => '&#38418;',
	"\xE3\xD7" => '&#38421;',
	"\xE3\xD8" => '&#38422;',
	"\xE3\xD9" => '&#38423;',
	"\xE3\xDA" => '&#38425;',
	"\xE3\xDB" => '&#38426;',
	"\xE3\xDC" => '&#20012;',
	"\xE3\xDD" => '&#29247;',
	"\xE3\xDE" => '&#25109;',
	"\xE3\xDF" => '&#27701;',
	"\xE3\xE0" => '&#27732;',
	"\xE3\xE1" => '&#27740;',
	"\xE3\xE2" => '&#27722;',
	"\xE3\xE3" => '&#27811;',
	"\xE3\xE4" => '&#27781;',
	"\xE3\xE5" => '&#27792;',
	"\xE3\xE6" => '&#27796;',
	"\xE3\xE7" => '&#27788;',
	"\xE3\xE8" => '&#27752;',
	"\xE3\xE9" => '&#27753;',
	"\xE3\xEA" => '&#27764;',
	"\xE3\xEB" => '&#27766;',
	"\xE3\xEC" => '&#27782;',
	"\xE3\xED" => '&#27817;',
	"\xE3\xEE" => '&#27856;',
	"\xE3\xEF" => '&#27860;',
	"\xE3\xF0" => '&#27821;',
	"\xE3\xF1" => '&#27895;',
	"\xE3\xF2" => '&#27896;',
	"\xE3\xF3" => '&#27889;',
	"\xE3\xF4" => '&#27863;',
	"\xE3\xF5" => '&#27826;',
	"\xE3\xF6" => '&#27872;',
	"\xE3\xF7" => '&#27862;',
	"\xE3\xF8" => '&#27898;',
	"\xE3\xF9" => '&#27883;',
	"\xE3\xFA" => '&#27886;',
	"\xE3\xFB" => '&#27825;',
	"\xE3\xFC" => '&#27859;',
	"\xE3\xFD" => '&#27887;',
	"\xE3\xFE" => '&#27902;',
	"\xE4\xA1" => '&#27961;',
	"\xE4\xA2" => '&#27943;',
	"\xE4\xA3" => '&#27916;',
	"\xE4\xA4" => '&#27971;',
	"\xE4\xA5" => '&#27976;',
	"\xE4\xA6" => '&#27911;',
	"\xE4\xA7" => '&#27908;',
	"\xE4\xA8" => '&#27929;',
	"\xE4\xA9" => '&#27918;',
	"\xE4\xAA" => '&#27947;',
	"\xE4\xAB" => '&#27981;',
	"\xE4\xAC" => '&#27950;',
	"\xE4\xAD" => '&#27957;',
	"\xE4\xAE" => '&#27930;',
	"\xE4\xAF" => '&#27983;',
	"\xE4\xB0" => '&#27986;',
	"\xE4\xB1" => '&#27988;',
	"\xE4\xB2" => '&#27955;',
	"\xE4\xB3" => '&#28049;',
	"\xE4\xB4" => '&#28015;',
	"\xE4\xB5" => '&#28062;',
	"\xE4\xB6" => '&#28064;',
	"\xE4\xB7" => '&#27998;',
	"\xE4\xB8" => '&#28051;',
	"\xE4\xB9" => '&#28052;',
	"\xE4\xBA" => '&#27996;',
	"\xE4\xBB" => '&#28000;',
	"\xE4\xBC" => '&#28028;',
	"\xE4\xBD" => '&#28003;',
	"\xE4\xBE" => '&#28186;',
	"\xE4\xBF" => '&#28103;',
	"\xE4\xC0" => '&#28101;',
	"\xE4\xC1" => '&#28126;',
	"\xE4\xC2" => '&#28174;',
	"\xE4\xC3" => '&#28095;',
	"\xE4\xC4" => '&#28128;',
	"\xE4\xC5" => '&#28177;',
	"\xE4\xC6" => '&#28134;',
	"\xE4\xC7" => '&#28125;',
	"\xE4\xC8" => '&#28121;',
	"\xE4\xC9" => '&#28182;',
	"\xE4\xCA" => '&#28075;',
	"\xE4\xCB" => '&#28172;',
	"\xE4\xCC" => '&#28078;',
	"\xE4\xCD" => '&#28203;',
	"\xE4\xCE" => '&#28270;',
	"\xE4\xCF" => '&#28238;',
	"\xE4\xD0" => '&#28267;',
	"\xE4\xD1" => '&#28338;',
	"\xE4\xD2" => '&#28255;',
	"\xE4\xD3" => '&#28294;',
	"\xE4\xD4" => '&#28243;',
	"\xE4\xD5" => '&#28244;',
	"\xE4\xD6" => '&#28210;',
	"\xE4\xD7" => '&#28197;',
	"\xE4\xD8" => '&#28228;',
	"\xE4\xD9" => '&#28383;',
	"\xE4\xDA" => '&#28337;',
	"\xE4\xDB" => '&#28312;',
	"\xE4\xDC" => '&#28384;',
	"\xE4\xDD" => '&#28461;',
	"\xE4\xDE" => '&#28386;',
	"\xE4\xDF" => '&#28325;',
	"\xE4\xE0" => '&#28327;',
	"\xE4\xE1" => '&#28349;',
	"\xE4\xE2" => '&#28347;',
	"\xE4\xE3" => '&#28343;',
	"\xE4\xE4" => '&#28375;',
	"\xE4\xE5" => '&#28340;',
	"\xE4\xE6" => '&#28367;',
	"\xE4\xE7" => '&#28303;',
	"\xE4\xE8" => '&#28354;',
	"\xE4\xE9" => '&#28319;',
	"\xE4\xEA" => '&#28514;',
	"\xE4\xEB" => '&#28486;',
	"\xE4\xEC" => '&#28487;',
	"\xE4\xED" => '&#28452;',
	"\xE4\xEE" => '&#28437;',
	"\xE4\xEF" => '&#28409;',
	"\xE4\xF0" => '&#28463;',
	"\xE4\xF1" => '&#28470;',
	"\xE4\xF2" => '&#28491;',
	"\xE4\xF3" => '&#28532;',
	"\xE4\xF4" => '&#28458;',
	"\xE4\xF5" => '&#28425;',
	"\xE4\xF6" => '&#28457;',
	"\xE4\xF7" => '&#28553;',
	"\xE4\xF8" => '&#28557;',
	"\xE4\xF9" => '&#28556;',
	"\xE4\xFA" => '&#28536;',
	"\xE4\xFB" => '&#28530;',
	"\xE4\xFC" => '&#28540;',
	"\xE4\xFD" => '&#28538;',
	"\xE4\xFE" => '&#28625;',
	"\xE5\xA1" => '&#28617;',
	"\xE5\xA2" => '&#28583;',
	"\xE5\xA3" => '&#28601;',
	"\xE5\xA4" => '&#28598;',
	"\xE5\xA5" => '&#28610;',
	"\xE5\xA6" => '&#28641;',
	"\xE5\xA7" => '&#28654;',
	"\xE5\xA8" => '&#28638;',
	"\xE5\xA9" => '&#28640;',
	"\xE5\xAA" => '&#28655;',
	"\xE5\xAB" => '&#28698;',
	"\xE5\xAC" => '&#28707;',
	"\xE5\xAD" => '&#28699;',
	"\xE5\xAE" => '&#28729;',
	"\xE5\xAF" => '&#28725;',
	"\xE5\xB0" => '&#28751;',
	"\xE5\xB1" => '&#28766;',
	"\xE5\xB2" => '&#23424;',
	"\xE5\xB3" => '&#23428;',
	"\xE5\xB4" => '&#23445;',
	"\xE5\xB5" => '&#23443;',
	"\xE5\xB6" => '&#23461;',
	"\xE5\xB7" => '&#23480;',
	"\xE5\xB8" => '&#29999;',
	"\xE5\xB9" => '&#39582;',
	"\xE5\xBA" => '&#25652;',
	"\xE5\xBB" => '&#23524;',
	"\xE5\xBC" => '&#23534;',
	"\xE5\xBD" => '&#35120;',
	"\xE5\xBE" => '&#23536;',
	"\xE5\xBF" => '&#36423;',
	"\xE5\xC0" => '&#35591;',
	"\xE5\xC1" => '&#36790;',
	"\xE5\xC2" => '&#36819;',
	"\xE5\xC3" => '&#36821;',
	"\xE5\xC4" => '&#36837;',
	"\xE5\xC5" => '&#36846;',
	"\xE5\xC6" => '&#36836;',
	"\xE5\xC7" => '&#36841;',
	"\xE5\xC8" => '&#36838;',
	"\xE5\xC9" => '&#36851;',
	"\xE5\xCA" => '&#36840;',
	"\xE5\xCB" => '&#36869;',
	"\xE5\xCC" => '&#36868;',
	"\xE5\xCD" => '&#36875;',
	"\xE5\xCE" => '&#36902;',
	"\xE5\xCF" => '&#36881;',
	"\xE5\xD0" => '&#36877;',
	"\xE5\xD1" => '&#36886;',
	"\xE5\xD2" => '&#36897;',
	"\xE5\xD3" => '&#36917;',
	"\xE5\xD4" => '&#36918;',
	"\xE5\xD5" => '&#36909;',
	"\xE5\xD6" => '&#36911;',
	"\xE5\xD7" => '&#36932;',
	"\xE5\xD8" => '&#36945;',
	"\xE5\xD9" => '&#36946;',
	"\xE5\xDA" => '&#36944;',
	"\xE5\xDB" => '&#36968;',
	"\xE5\xDC" => '&#36952;',
	"\xE5\xDD" => '&#36962;',
	"\xE5\xDE" => '&#36955;',
	"\xE5\xDF" => '&#26297;',
	"\xE5\xE0" => '&#36980;',
	"\xE5\xE1" => '&#36989;',
	"\xE5\xE2" => '&#36994;',
	"\xE5\xE3" => '&#37000;',
	"\xE5\xE4" => '&#36995;',
	"\xE5\xE5" => '&#37003;',
	"\xE5\xE6" => '&#24400;',
	"\xE5\xE7" => '&#24407;',
	"\xE5\xE8" => '&#24406;',
	"\xE5\xE9" => '&#24408;',
	"\xE5\xEA" => '&#23611;',
	"\xE5\xEB" => '&#21675;',
	"\xE5\xEC" => '&#23632;',
	"\xE5\xED" => '&#23641;',
	"\xE5\xEE" => '&#23409;',
	"\xE5\xEF" => '&#23651;',
	"\xE5\xF0" => '&#23654;',
	"\xE5\xF1" => '&#32700;',
	"\xE5\xF2" => '&#24362;',
	"\xE5\xF3" => '&#24361;',
	"\xE5\xF4" => '&#24365;',
	"\xE5\xF5" => '&#33396;',
	"\xE5\xF6" => '&#24380;',
	"\xE5\xF7" => '&#39739;',
	"\xE5\xF8" => '&#23662;',
	"\xE5\xF9" => '&#22913;',
	"\xE5\xFA" => '&#22915;',
	"\xE5\xFB" => '&#22925;',
	"\xE5\xFC" => '&#22953;',
	"\xE5\xFD" => '&#22954;',
	"\xE5\xFE" => '&#22947;',
	"\xE6\xA1" => '&#22935;',
	"\xE6\xA2" => '&#22986;',
	"\xE6\xA3" => '&#22955;',
	"\xE6\xA4" => '&#22942;',
	"\xE6\xA5" => '&#22948;',
	"\xE6\xA6" => '&#22994;',
	"\xE6\xA7" => '&#22962;',
	"\xE6\xA8" => '&#22959;',
	"\xE6\xA9" => '&#22999;',
	"\xE6\xAA" => '&#22974;',
	"\xE6\xAB" => '&#23045;',
	"\xE6\xAC" => '&#23046;',
	"\xE6\xAD" => '&#23005;',
	"\xE6\xAE" => '&#23048;',
	"\xE6\xAF" => '&#23011;',
	"\xE6\xB0" => '&#23000;',
	"\xE6\xB1" => '&#23033;',
	"\xE6\xB2" => '&#23052;',
	"\xE6\xB3" => '&#23049;',
	"\xE6\xB4" => '&#23090;',
	"\xE6\xB5" => '&#23092;',
	"\xE6\xB6" => '&#23057;',
	"\xE6\xB7" => '&#23075;',
	"\xE6\xB8" => '&#23059;',
	"\xE6\xB9" => '&#23104;',
	"\xE6\xBA" => '&#23143;',
	"\xE6\xBB" => '&#23114;',
	"\xE6\xBC" => '&#23125;',
	"\xE6\xBD" => '&#23100;',
	"\xE6\xBE" => '&#23138;',
	"\xE6\xBF" => '&#23157;',
	"\xE6\xC0" => '&#33004;',
	"\xE6\xC1" => '&#23210;',
	"\xE6\xC2" => '&#23195;',
	"\xE6\xC3" => '&#23159;',
	"\xE6\xC4" => '&#23162;',
	"\xE6\xC5" => '&#23230;',
	"\xE6\xC6" => '&#23275;',
	"\xE6\xC7" => '&#23218;',
	"\xE6\xC8" => '&#23250;',
	"\xE6\xC9" => '&#23252;',
	"\xE6\xCA" => '&#23224;',
	"\xE6\xCB" => '&#23264;',
	"\xE6\xCC" => '&#23267;',
	"\xE6\xCD" => '&#23281;',
	"\xE6\xCE" => '&#23254;',
	"\xE6\xCF" => '&#23270;',
	"\xE6\xD0" => '&#23256;',
	"\xE6\xD1" => '&#23260;',
	"\xE6\xD2" => '&#23305;',
	"\xE6\xD3" => '&#23319;',
	"\xE6\xD4" => '&#23318;',
	"\xE6\xD5" => '&#23346;',
	"\xE6\xD6" => '&#23351;',
	"\xE6\xD7" => '&#23360;',
	"\xE6\xD8" => '&#23573;',
	"\xE6\xD9" => '&#23580;',
	"\xE6\xDA" => '&#23386;',
	"\xE6\xDB" => '&#23397;',
	"\xE6\xDC" => '&#23411;',
	"\xE6\xDD" => '&#23377;',
	"\xE6\xDE" => '&#23379;',
	"\xE6\xDF" => '&#23394;',
	"\xE6\xE0" => '&#39541;',
	"\xE6\xE1" => '&#39543;',
	"\xE6\xE2" => '&#39544;',
	"\xE6\xE3" => '&#39546;',
	"\xE6\xE4" => '&#39551;',
	"\xE6\xE5" => '&#39549;',
	"\xE6\xE6" => '&#39552;',
	"\xE6\xE7" => '&#39553;',
	"\xE6\xE8" => '&#39557;',
	"\xE6\xE9" => '&#39560;',
	"\xE6\xEA" => '&#39562;',
	"\xE6\xEB" => '&#39568;',
	"\xE6\xEC" => '&#39570;',
	"\xE6\xED" => '&#39571;',
	"\xE6\xEE" => '&#39574;',
	"\xE6\xEF" => '&#39576;',
	"\xE6\xF0" => '&#39579;',
	"\xE6\xF1" => '&#39580;',
	"\xE6\xF2" => '&#39581;',
	"\xE6\xF3" => '&#39583;',
	"\xE6\xF4" => '&#39584;',
	"\xE6\xF5" => '&#39586;',
	"\xE6\xF6" => '&#39587;',
	"\xE6\xF7" => '&#39589;',
	"\xE6\xF8" => '&#39591;',
	"\xE6\xF9" => '&#32415;',
	"\xE6\xFA" => '&#32417;',
	"\xE6\xFB" => '&#32419;',
	"\xE6\xFC" => '&#32421;',
	"\xE6\xFD" => '&#32424;',
	"\xE6\xFE" => '&#32425;',
	"\xE7\xA1" => '&#32429;',
	"\xE7\xA2" => '&#32432;',
	"\xE7\xA3" => '&#32446;',
	"\xE7\xA4" => '&#32448;',
	"\xE7\xA5" => '&#32449;',
	"\xE7\xA6" => '&#32450;',
	"\xE7\xA7" => '&#32457;',
	"\xE7\xA8" => '&#32459;',
	"\xE7\xA9" => '&#32460;',
	"\xE7\xAA" => '&#32464;',
	"\xE7\xAB" => '&#32468;',
	"\xE7\xAC" => '&#32471;',
	"\xE7\xAD" => '&#32475;',
	"\xE7\xAE" => '&#32480;',
	"\xE7\xAF" => '&#32481;',
	"\xE7\xB0" => '&#32488;',
	"\xE7\xB1" => '&#32491;',
	"\xE7\xB2" => '&#32494;',
	"\xE7\xB3" => '&#32495;',
	"\xE7\xB4" => '&#32497;',
	"\xE7\xB5" => '&#32498;',
	"\xE7\xB6" => '&#32525;',
	"\xE7\xB7" => '&#32502;',
	"\xE7\xB8" => '&#32506;',
	"\xE7\xB9" => '&#32507;',
	"\xE7\xBA" => '&#32510;',
	"\xE7\xBB" => '&#32513;',
	"\xE7\xBC" => '&#32514;',
	"\xE7\xBD" => '&#32515;',
	"\xE7\xBE" => '&#32519;',
	"\xE7\xBF" => '&#32520;',
	"\xE7\xC0" => '&#32523;',
	"\xE7\xC1" => '&#32524;',
	"\xE7\xC2" => '&#32527;',
	"\xE7\xC3" => '&#32529;',
	"\xE7\xC4" => '&#32530;',
	"\xE7\xC5" => '&#32535;',
	"\xE7\xC6" => '&#32537;',
	"\xE7\xC7" => '&#32540;',
	"\xE7\xC8" => '&#32539;',
	"\xE7\xC9" => '&#32543;',
	"\xE7\xCA" => '&#32545;',
	"\xE7\xCB" => '&#32546;',
	"\xE7\xCC" => '&#32547;',
	"\xE7\xCD" => '&#32548;',
	"\xE7\xCE" => '&#32549;',
	"\xE7\xCF" => '&#32550;',
	"\xE7\xD0" => '&#32551;',
	"\xE7\xD1" => '&#32554;',
	"\xE7\xD2" => '&#32555;',
	"\xE7\xD3" => '&#32556;',
	"\xE7\xD4" => '&#32557;',
	"\xE7\xD5" => '&#32559;',
	"\xE7\xD6" => '&#32560;',
	"\xE7\xD7" => '&#32561;',
	"\xE7\xD8" => '&#32562;',
	"\xE7\xD9" => '&#32563;',
	"\xE7\xDA" => '&#32565;',
	"\xE7\xDB" => '&#24186;',
	"\xE7\xDC" => '&#30079;',
	"\xE7\xDD" => '&#24027;',
	"\xE7\xDE" => '&#30014;',
	"\xE7\xDF" => '&#37013;',
	"\xE7\xE0" => '&#29582;',
	"\xE7\xE1" => '&#29585;',
	"\xE7\xE2" => '&#29614;',
	"\xE7\xE3" => '&#29602;',
	"\xE7\xE4" => '&#29599;',
	"\xE7\xE5" => '&#29647;',
	"\xE7\xE6" => '&#29634;',
	"\xE7\xE7" => '&#29649;',
	"\xE7\xE8" => '&#29623;',
	"\xE7\xE9" => '&#29619;',
	"\xE7\xEA" => '&#29632;',
	"\xE7\xEB" => '&#29641;',
	"\xE7\xEC" => '&#29640;',
	"\xE7\xED" => '&#29669;',
	"\xE7\xEE" => '&#29657;',
	"\xE7\xEF" => '&#39036;',
	"\xE7\xF0" => '&#29706;',
	"\xE7\xF1" => '&#29673;',
	"\xE7\xF2" => '&#29671;',
	"\xE7\xF3" => '&#29662;',
	"\xE7\xF4" => '&#29626;',
	"\xE7\xF5" => '&#29682;',
	"\xE7\xF6" => '&#29711;',
	"\xE7\xF7" => '&#29738;',
	"\xE7\xF8" => '&#29787;',
	"\xE7\xF9" => '&#29734;',
	"\xE7\xFA" => '&#29733;',
	"\xE7\xFB" => '&#29736;',
	"\xE7\xFC" => '&#29744;',
	"\xE7\xFD" => '&#29742;',
	"\xE7\xFE" => '&#29740;',
	"\xE8\xA1" => '&#29723;',
	"\xE8\xA2" => '&#29722;',
	"\xE8\xA3" => '&#29761;',
	"\xE8\xA4" => '&#29788;',
	"\xE8\xA5" => '&#29783;',
	"\xE8\xA6" => '&#29781;',
	"\xE8\xA7" => '&#29785;',
	"\xE8\xA8" => '&#29815;',
	"\xE8\xA9" => '&#29805;',
	"\xE8\xAA" => '&#29822;',
	"\xE8\xAB" => '&#29852;',
	"\xE8\xAC" => '&#29838;',
	"\xE8\xAD" => '&#29824;',
	"\xE8\xAE" => '&#29825;',
	"\xE8\xAF" => '&#29831;',
	"\xE8\xB0" => '&#29835;',
	"\xE8\xB1" => '&#29854;',
	"\xE8\xB2" => '&#29864;',
	"\xE8\xB3" => '&#29865;',
	"\xE8\xB4" => '&#29840;',
	"\xE8\xB5" => '&#29863;',
	"\xE8\xB6" => '&#29906;',
	"\xE8\xB7" => '&#29882;',
	"\xE8\xB8" => '&#38890;',
	"\xE8\xB9" => '&#38891;',
	"\xE8\xBA" => '&#38892;',
	"\xE8\xBB" => '&#26444;',
	"\xE8\xBC" => '&#26451;',
	"\xE8\xBD" => '&#26462;',
	"\xE8\xBE" => '&#26440;',
	"\xE8\xBF" => '&#26473;',
	"\xE8\xC0" => '&#26533;',
	"\xE8\xC1" => '&#26503;',
	"\xE8\xC2" => '&#26474;',
	"\xE8\xC3" => '&#26483;',
	"\xE8\xC4" => '&#26520;',
	"\xE8\xC5" => '&#26535;',
	"\xE8\xC6" => '&#26485;',
	"\xE8\xC7" => '&#26536;',
	"\xE8\xC8" => '&#26526;',
	"\xE8\xC9" => '&#26541;',
	"\xE8\xCA" => '&#26507;',
	"\xE8\xCB" => '&#26487;',
	"\xE8\xCC" => '&#26492;',
	"\xE8\xCD" => '&#26608;',
	"\xE8\xCE" => '&#26633;',
	"\xE8\xCF" => '&#26584;',
	"\xE8\xD0" => '&#26634;',
	"\xE8\xD1" => '&#26601;',
	"\xE8\xD2" => '&#26544;',
	"\xE8\xD3" => '&#26636;',
	"\xE8\xD4" => '&#26585;',
	"\xE8\xD5" => '&#26549;',
	"\xE8\xD6" => '&#26586;',
	"\xE8\xD7" => '&#26547;',
	"\xE8\xD8" => '&#26589;',
	"\xE8\xD9" => '&#26624;',
	"\xE8\xDA" => '&#26563;',
	"\xE8\xDB" => '&#26552;',
	"\xE8\xDC" => '&#26594;',
	"\xE8\xDD" => '&#26638;',
	"\xE8\xDE" => '&#26561;',
	"\xE8\xDF" => '&#26621;',
	"\xE8\xE0" => '&#26674;',
	"\xE8\xE1" => '&#26675;',
	"\xE8\xE2" => '&#26720;',
	"\xE8\xE3" => '&#26721;',
	"\xE8\xE4" => '&#26702;',
	"\xE8\xE5" => '&#26722;',
	"\xE8\xE6" => '&#26692;',
	"\xE8\xE7" => '&#26724;',
	"\xE8\xE8" => '&#26755;',
	"\xE8\xE9" => '&#26653;',
	"\xE8\xEA" => '&#26709;',
	"\xE8\xEB" => '&#26726;',
	"\xE8\xEC" => '&#26689;',
	"\xE8\xED" => '&#26727;',
	"\xE8\xEE" => '&#26688;',
	"\xE8\xEF" => '&#26686;',
	"\xE8\xF0" => '&#26698;',
	"\xE8\xF1" => '&#26697;',
	"\xE8\xF2" => '&#26665;',
	"\xE8\xF3" => '&#26805;',
	"\xE8\xF4" => '&#26767;',
	"\xE8\xF5" => '&#26740;',
	"\xE8\xF6" => '&#26743;',
	"\xE8\xF7" => '&#26771;',
	"\xE8\xF8" => '&#26731;',
	"\xE8\xF9" => '&#26818;',
	"\xE8\xFA" => '&#26990;',
	"\xE8\xFB" => '&#26876;',
	"\xE8\xFC" => '&#26911;',
	"\xE8\xFD" => '&#26912;',
	"\xE8\xFE" => '&#26873;',
	"\xE9\xA1" => '&#26916;',
	"\xE9\xA2" => '&#26864;',
	"\xE9\xA3" => '&#26891;',
	"\xE9\xA4" => '&#26881;',
	"\xE9\xA5" => '&#26967;',
	"\xE9\xA6" => '&#26851;',
	"\xE9\xA7" => '&#26896;',
	"\xE9\xA8" => '&#26993;',
	"\xE9\xA9" => '&#26937;',
	"\xE9\xAA" => '&#26976;',
	"\xE9\xAB" => '&#26946;',
	"\xE9\xAC" => '&#26973;',
	"\xE9\xAD" => '&#27012;',
	"\xE9\xAE" => '&#26987;',
	"\xE9\xAF" => '&#27008;',
	"\xE9\xB0" => '&#27032;',
	"\xE9\xB1" => '&#27000;',
	"\xE9\xB2" => '&#26932;',
	"\xE9\xB3" => '&#27084;',
	"\xE9\xB4" => '&#27015;',
	"\xE9\xB5" => '&#27016;',
	"\xE9\xB6" => '&#27086;',
	"\xE9\xB7" => '&#27017;',
	"\xE9\xB8" => '&#26982;',
	"\xE9\xB9" => '&#26979;',
	"\xE9\xBA" => '&#27001;',
	"\xE9\xBB" => '&#27035;',
	"\xE9\xBC" => '&#27047;',
	"\xE9\xBD" => '&#27067;',
	"\xE9\xBE" => '&#27051;',
	"\xE9\xBF" => '&#27053;',
	"\xE9\xC0" => '&#27092;',
	"\xE9\xC1" => '&#27057;',
	"\xE9\xC2" => '&#27073;',
	"\xE9\xC3" => '&#27082;',
	"\xE9\xC4" => '&#27103;',
	"\xE9\xC5" => '&#27029;',
	"\xE9\xC6" => '&#27104;',
	"\xE9\xC7" => '&#27021;',
	"\xE9\xC8" => '&#27135;',
	"\xE9\xC9" => '&#27183;',
	"\xE9\xCA" => '&#27117;',
	"\xE9\xCB" => '&#27159;',
	"\xE9\xCC" => '&#27160;',
	"\xE9\xCD" => '&#27237;',
	"\xE9\xCE" => '&#27122;',
	"\xE9\xCF" => '&#27204;',
	"\xE9\xD0" => '&#27198;',
	"\xE9\xD1" => '&#27296;',
	"\xE9\xD2" => '&#27216;',
	"\xE9\xD3" => '&#27227;',
	"\xE9\xD4" => '&#27189;',
	"\xE9\xD5" => '&#27278;',
	"\xE9\xD6" => '&#27257;',
	"\xE9\xD7" => '&#27197;',
	"\xE9\xD8" => '&#27176;',
	"\xE9\xD9" => '&#27224;',
	"\xE9\xDA" => '&#27260;',
	"\xE9\xDB" => '&#27281;',
	"\xE9\xDC" => '&#27280;',
	"\xE9\xDD" => '&#27305;',
	"\xE9\xDE" => '&#27287;',
	"\xE9\xDF" => '&#27307;',
	"\xE9\xE0" => '&#29495;',
	"\xE9\xE1" => '&#29522;',
	"\xE9\xE2" => '&#27521;',
	"\xE9\xE3" => '&#27522;',
	"\xE9\xE4" => '&#27527;',
	"\xE9\xE5" => '&#27524;',
	"\xE9\xE6" => '&#27538;',
	"\xE9\xE7" => '&#27539;',
	"\xE9\xE8" => '&#27533;',
	"\xE9\xE9" => '&#27546;',
	"\xE9\xEA" => '&#27547;',
	"\xE9\xEB" => '&#27553;',
	"\xE9\xEC" => '&#27562;',
	"\xE9\xED" => '&#36715;',
	"\xE9\xEE" => '&#36717;',
	"\xE9\xEF" => '&#36721;',
	"\xE9\xF0" => '&#36722;',
	"\xE9\xF1" => '&#36723;',
	"\xE9\xF2" => '&#36725;',
	"\xE9\xF3" => '&#36726;',
	"\xE9\xF4" => '&#36728;',
	"\xE9\xF5" => '&#36727;',
	"\xE9\xF6" => '&#36729;',
	"\xE9\xF7" => '&#36730;',
	"\xE9\xF8" => '&#36732;',
	"\xE9\xF9" => '&#36734;',
	"\xE9\xFA" => '&#36737;',
	"\xE9\xFB" => '&#36738;',
	"\xE9\xFC" => '&#36740;',
	"\xE9\xFD" => '&#36743;',
	"\xE9\xFE" => '&#36747;',
	"\xEA\xA1" => '&#36749;',
	"\xEA\xA2" => '&#36750;',
	"\xEA\xA3" => '&#36751;',
	"\xEA\xA4" => '&#36760;',
	"\xEA\xA5" => '&#36762;',
	"\xEA\xA6" => '&#36558;',
	"\xEA\xA7" => '&#25099;',
	"\xEA\xA8" => '&#25111;',
	"\xEA\xA9" => '&#25115;',
	"\xEA\xAA" => '&#25119;',
	"\xEA\xAB" => '&#25122;',
	"\xEA\xAC" => '&#25121;',
	"\xEA\xAD" => '&#25125;',
	"\xEA\xAE" => '&#25124;',
	"\xEA\xAF" => '&#25132;',
	"\xEA\xB0" => '&#33255;',
	"\xEA\xB1" => '&#29935;',
	"\xEA\xB2" => '&#29940;',
	"\xEA\xB3" => '&#29951;',
	"\xEA\xB4" => '&#29967;',
	"\xEA\xB5" => '&#29969;',
	"\xEA\xB6" => '&#29971;',
	"\xEA\xB7" => '&#25908;',
	"\xEA\xB8" => '&#26094;',
	"\xEA\xB9" => '&#26095;',
	"\xEA\xBA" => '&#26096;',
	"\xEA\xBB" => '&#26122;',
	"\xEA\xBC" => '&#26137;',
	"\xEA\xBD" => '&#26482;',
	"\xEA\xBE" => '&#26115;',
	"\xEA\xBF" => '&#26133;',
	"\xEA\xC0" => '&#26112;',
	"\xEA\xC1" => '&#28805;',
	"\xEA\xC2" => '&#26359;',
	"\xEA\xC3" => '&#26141;',
	"\xEA\xC4" => '&#26164;',
	"\xEA\xC5" => '&#26161;',
	"\xEA\xC6" => '&#26166;',
	"\xEA\xC7" => '&#26165;',
	"\xEA\xC8" => '&#32774;',
	"\xEA\xC9" => '&#26207;',
	"\xEA\xCA" => '&#26196;',
	"\xEA\xCB" => '&#26177;',
	"\xEA\xCC" => '&#26191;',
	"\xEA\xCD" => '&#26198;',
	"\xEA\xCE" => '&#26209;',
	"\xEA\xCF" => '&#26199;',
	"\xEA\xD0" => '&#26231;',
	"\xEA\xD1" => '&#26244;',
	"\xEA\xD2" => '&#26252;',
	"\xEA\xD3" => '&#26279;',
	"\xEA\xD4" => '&#26269;',
	"\xEA\xD5" => '&#26302;',
	"\xEA\xD6" => '&#26331;',
	"\xEA\xD7" => '&#26332;',
	"\xEA\xD8" => '&#26342;',
	"\xEA\xD9" => '&#26345;',
	"\xEA\xDA" => '&#36146;',
	"\xEA\xDB" => '&#36147;',
	"\xEA\xDC" => '&#36150;',
	"\xEA\xDD" => '&#36155;',
	"\xEA\xDE" => '&#36157;',
	"\xEA\xDF" => '&#36160;',
	"\xEA\xE0" => '&#36165;',
	"\xEA\xE1" => '&#36166;',
	"\xEA\xE2" => '&#36168;',
	"\xEA\xE3" => '&#36169;',
	"\xEA\xE4" => '&#36167;',
	"\xEA\xE5" => '&#36173;',
	"\xEA\xE6" => '&#36181;',
	"\xEA\xE7" => '&#36185;',
	"\xEA\xE8" => '&#35271;',
	"\xEA\xE9" => '&#35274;',
	"\xEA\xEA" => '&#35275;',
	"\xEA\xEB" => '&#35276;',
	"\xEA\xEC" => '&#35278;',
	"\xEA\xED" => '&#35279;',
	"\xEA\xEE" => '&#35280;',
	"\xEA\xEF" => '&#35281;',
	"\xEA\xF0" => '&#29294;',
	"\xEA\xF1" => '&#29343;',
	"\xEA\xF2" => '&#29277;',
	"\xEA\xF3" => '&#29286;',
	"\xEA\xF4" => '&#29295;',
	"\xEA\xF5" => '&#29310;',
	"\xEA\xF6" => '&#29311;',
	"\xEA\xF7" => '&#29316;',
	"\xEA\xF8" => '&#29323;',
	"\xEA\xF9" => '&#29325;',
	"\xEA\xFA" => '&#29327;',
	"\xEA\xFB" => '&#29330;',
	"\xEA\xFC" => '&#25352;',
	"\xEA\xFD" => '&#25394;',
	"\xEA\xFE" => '&#25520;',
	"\xEB\xA1" => '&#25663;',
	"\xEB\xA2" => '&#25816;',
	"\xEB\xA3" => '&#32772;',
	"\xEB\xA4" => '&#27626;',
	"\xEB\xA5" => '&#27635;',
	"\xEB\xA6" => '&#27645;',
	"\xEB\xA7" => '&#27637;',
	"\xEB\xA8" => '&#27641;',
	"\xEB\xA9" => '&#27653;',
	"\xEB\xAA" => '&#27655;',
	"\xEB\xAB" => '&#27654;',
	"\xEB\xAC" => '&#27661;',
	"\xEB\xAD" => '&#27669;',
	"\xEB\xAE" => '&#27672;',
	"\xEB\xAF" => '&#27673;',
	"\xEB\xB0" => '&#27674;',
	"\xEB\xB1" => '&#27681;',
	"\xEB\xB2" => '&#27689;',
	"\xEB\xB3" => '&#27684;',
	"\xEB\xB4" => '&#27690;',
	"\xEB\xB5" => '&#27698;',
	"\xEB\xB6" => '&#25909;',
	"\xEB\xB7" => '&#25941;',
	"\xEB\xB8" => '&#25963;',
	"\xEB\xB9" => '&#29261;',
	"\xEB\xBA" => '&#29266;',
	"\xEB\xBB" => '&#29270;',
	"\xEB\xBC" => '&#29232;',
	"\xEB\xBD" => '&#34402;',
	"\xEB\xBE" => '&#21014;',
	"\xEB\xBF" => '&#32927;',
	"\xEB\xC0" => '&#32924;',
	"\xEB\xC1" => '&#32915;',
	"\xEB\xC2" => '&#32956;',
	"\xEB\xC3" => '&#26378;',
	"\xEB\xC4" => '&#32957;',
	"\xEB\xC5" => '&#32945;',
	"\xEB\xC6" => '&#32939;',
	"\xEB\xC7" => '&#32941;',
	"\xEB\xC8" => '&#32948;',
	"\xEB\xC9" => '&#32951;',
	"\xEB\xCA" => '&#32999;',
	"\xEB\xCB" => '&#33000;',
	"\xEB\xCC" => '&#33001;',
	"\xEB\xCD" => '&#33002;',
	"\xEB\xCE" => '&#32987;',
	"\xEB\xCF" => '&#32962;',
	"\xEB\xD0" => '&#32964;',
	"\xEB\xD1" => '&#32985;',
	"\xEB\xD2" => '&#32973;',
	"\xEB\xD3" => '&#32983;',
	"\xEB\xD4" => '&#26384;',
	"\xEB\xD5" => '&#32989;',
	"\xEB\xD6" => '&#33003;',
	"\xEB\xD7" => '&#33009;',
	"\xEB\xD8" => '&#33012;',
	"\xEB\xD9" => '&#33005;',
	"\xEB\xDA" => '&#33037;',
	"\xEB\xDB" => '&#33038;',
	"\xEB\xDC" => '&#33010;',
	"\xEB\xDD" => '&#33020;',
	"\xEB\xDE" => '&#26389;',
	"\xEB\xDF" => '&#33042;',
	"\xEB\xE0" => '&#35930;',
	"\xEB\xE1" => '&#33078;',
	"\xEB\xE2" => '&#33054;',
	"\xEB\xE3" => '&#33068;',
	"\xEB\xE4" => '&#33048;',
	"\xEB\xE5" => '&#33074;',
	"\xEB\xE6" => '&#33096;',
	"\xEB\xE7" => '&#33100;',
	"\xEB\xE8" => '&#33107;',
	"\xEB\xE9" => '&#33140;',
	"\xEB\xEA" => '&#33113;',
	"\xEB\xEB" => '&#33114;',
	"\xEB\xEC" => '&#33137;',
	"\xEB\xED" => '&#33120;',
	"\xEB\xEE" => '&#33129;',
	"\xEB\xEF" => '&#33148;',
	"\xEB\xF0" => '&#33149;',
	"\xEB\xF1" => '&#33133;',
	"\xEB\xF2" => '&#33127;',
	"\xEB\xF3" => '&#22605;',
	"\xEB\xF4" => '&#23221;',
	"\xEB\xF5" => '&#33160;',
	"\xEB\xF6" => '&#33154;',
	"\xEB\xF7" => '&#33169;',
	"\xEB\xF8" => '&#28373;',
	"\xEB\xF9" => '&#33187;',
	"\xEB\xFA" => '&#33194;',
	"\xEB\xFB" => '&#33228;',
	"\xEB\xFC" => '&#26406;',
	"\xEB\xFD" => '&#33226;',
	"\xEB\xFE" => '&#33211;',
	"\xEC\xA1" => '&#33217;',
	"\xEC\xA2" => '&#33190;',
	"\xEC\xA3" => '&#27428;',
	"\xEC\xA4" => '&#27447;',
	"\xEC\xA5" => '&#27449;',
	"\xEC\xA6" => '&#27459;',
	"\xEC\xA7" => '&#27462;',
	"\xEC\xA8" => '&#27481;',
	"\xEC\xA9" => '&#39121;',
	"\xEC\xAA" => '&#39122;',
	"\xEC\xAB" => '&#39123;',
	"\xEC\xAC" => '&#39125;',
	"\xEC\xAD" => '&#39129;',
	"\xEC\xAE" => '&#39130;',
	"\xEC\xAF" => '&#27571;',
	"\xEC\xB0" => '&#24384;',
	"\xEC\xB1" => '&#27586;',
	"\xEC\xB2" => '&#35315;',
	"\xEC\xB3" => '&#26000;',
	"\xEC\xB4" => '&#40785;',
	"\xEC\xB5" => '&#26003;',
	"\xEC\xB6" => '&#26044;',
	"\xEC\xB7" => '&#26054;',
	"\xEC\xB8" => '&#26052;',
	"\xEC\xB9" => '&#26051;',
	"\xEC\xBA" => '&#26060;',
	"\xEC\xBB" => '&#26062;',
	"\xEC\xBC" => '&#26066;',
	"\xEC\xBD" => '&#26070;',
	"\xEC\xBE" => '&#28800;',
	"\xEC\xBF" => '&#28828;',
	"\xEC\xC0" => '&#28822;',
	"\xEC\xC1" => '&#28829;',
	"\xEC\xC2" => '&#28859;',
	"\xEC\xC3" => '&#28864;',
	"\xEC\xC4" => '&#28855;',
	"\xEC\xC5" => '&#28843;',
	"\xEC\xC6" => '&#28849;',
	"\xEC\xC7" => '&#28904;',
	"\xEC\xC8" => '&#28874;',
	"\xEC\xC9" => '&#28944;',
	"\xEC\xCA" => '&#28947;',
	"\xEC\xCB" => '&#28950;',
	"\xEC\xCC" => '&#28975;',
	"\xEC\xCD" => '&#28977;',
	"\xEC\xCE" => '&#29043;',
	"\xEC\xCF" => '&#29020;',
	"\xEC\xD0" => '&#29032;',
	"\xEC\xD1" => '&#28997;',
	"\xEC\xD2" => '&#29042;',
	"\xEC\xD3" => '&#29002;',
	"\xEC\xD4" => '&#29048;',
	"\xEC\xD5" => '&#29050;',
	"\xEC\xD6" => '&#29080;',
	"\xEC\xD7" => '&#29107;',
	"\xEC\xD8" => '&#29109;',
	"\xEC\xD9" => '&#29096;',
	"\xEC\xDA" => '&#29088;',
	"\xEC\xDB" => '&#29152;',
	"\xEC\xDC" => '&#29140;',
	"\xEC\xDD" => '&#29159;',
	"\xEC\xDE" => '&#29177;',
	"\xEC\xDF" => '&#29213;',
	"\xEC\xE0" => '&#29224;',
	"\xEC\xE1" => '&#28780;',
	"\xEC\xE2" => '&#28952;',
	"\xEC\xE3" => '&#29030;',
	"\xEC\xE4" => '&#29113;',
	"\xEC\xE5" => '&#25150;',
	"\xEC\xE6" => '&#25149;',
	"\xEC\xE7" => '&#25155;',
	"\xEC\xE8" => '&#25160;',
	"\xEC\xE9" => '&#25161;',
	"\xEC\xEA" => '&#31035;',
	"\xEC\xEB" => '&#31040;',
	"\xEC\xEC" => '&#31046;',
	"\xEC\xED" => '&#31049;',
	"\xEC\xEE" => '&#31067;',
	"\xEC\xEF" => '&#31068;',
	"\xEC\xF0" => '&#31059;',
	"\xEC\xF1" => '&#31066;',
	"\xEC\xF2" => '&#31074;',
	"\xEC\xF3" => '&#31063;',
	"\xEC\xF4" => '&#31072;',
	"\xEC\xF5" => '&#31087;',
	"\xEC\xF6" => '&#31079;',
	"\xEC\xF7" => '&#31098;',
	"\xEC\xF8" => '&#31109;',
	"\xEC\xF9" => '&#31114;',
	"\xEC\xFA" => '&#31130;',
	"\xEC\xFB" => '&#31143;',
	"\xEC\xFC" => '&#31155;',
	"\xEC\xFD" => '&#24529;',
	"\xEC\xFE" => '&#24528;',
	"\xED\xA1" => '&#24636;',
	"\xED\xA2" => '&#24669;',
	"\xED\xA3" => '&#24666;',
	"\xED\xA4" => '&#24679;',
	"\xED\xA5" => '&#24641;',
	"\xED\xA6" => '&#24665;',
	"\xED\xA7" => '&#24675;',
	"\xED\xA8" => '&#24747;',
	"\xED\xA9" => '&#24838;',
	"\xED\xAA" => '&#24845;',
	"\xED\xAB" => '&#24925;',
	"\xED\xAC" => '&#25001;',
	"\xED\xAD" => '&#24989;',
	"\xED\xAE" => '&#25035;',
	"\xED\xAF" => '&#25041;',
	"\xED\xB0" => '&#25094;',
	"\xED\xB1" => '&#32896;',
	"\xED\xB2" => '&#32895;',
	"\xED\xB3" => '&#27795;',
	"\xED\xB4" => '&#27894;',
	"\xED\xB5" => '&#28156;',
	"\xED\xB6" => '&#30710;',
	"\xED\xB7" => '&#30712;',
	"\xED\xB8" => '&#30720;',
	"\xED\xB9" => '&#30729;',
	"\xED\xBA" => '&#30743;',
	"\xED\xBB" => '&#30744;',
	"\xED\xBC" => '&#30737;',
	"\xED\xBD" => '&#26027;',
	"\xED\xBE" => '&#30765;',
	"\xED\xBF" => '&#30748;',
	"\xED\xC0" => '&#30749;',
	"\xED\xC1" => '&#30777;',
	"\xED\xC2" => '&#30778;',
	"\xED\xC3" => '&#30779;',
	"\xED\xC4" => '&#30751;',
	"\xED\xC5" => '&#30780;',
	"\xED\xC6" => '&#30757;',
	"\xED\xC7" => '&#30764;',
	"\xED\xC8" => '&#30755;',
	"\xED\xC9" => '&#30761;',
	"\xED\xCA" => '&#30798;',
	"\xED\xCB" => '&#30829;',
	"\xED\xCC" => '&#30806;',
	"\xED\xCD" => '&#30807;',
	"\xED\xCE" => '&#30758;',
	"\xED\xCF" => '&#30800;',
	"\xED\xD0" => '&#30791;',
	"\xED\xD1" => '&#30796;',
	"\xED\xD2" => '&#30826;',
	"\xED\xD3" => '&#30875;',
	"\xED\xD4" => '&#30867;',
	"\xED\xD5" => '&#30874;',
	"\xED\xD6" => '&#30855;',
	"\xED\xD7" => '&#30876;',
	"\xED\xD8" => '&#30881;',
	"\xED\xD9" => '&#30883;',
	"\xED\xDA" => '&#30898;',
	"\xED\xDB" => '&#30905;',
	"\xED\xDC" => '&#30885;',
	"\xED\xDD" => '&#30932;',
	"\xED\xDE" => '&#30937;',
	"\xED\xDF" => '&#30921;',
	"\xED\xE0" => '&#30956;',
	"\xED\xE1" => '&#30962;',
	"\xED\xE2" => '&#30981;',
	"\xED\xE3" => '&#30964;',
	"\xED\xE4" => '&#30995;',
	"\xED\xE5" => '&#31012;',
	"\xED\xE6" => '&#31006;',
	"\xED\xE7" => '&#31028;',
	"\xED\xE8" => '&#40859;',
	"\xED\xE9" => '&#40697;',
	"\xED\xEA" => '&#40699;',
	"\xED\xEB" => '&#40700;',
	"\xED\xEC" => '&#30449;',
	"\xED\xED" => '&#30468;',
	"\xED\xEE" => '&#30477;',
	"\xED\xEF" => '&#30457;',
	"\xED\xF0" => '&#30471;',
	"\xED\xF1" => '&#30472;',
	"\xED\xF2" => '&#30490;',
	"\xED\xF3" => '&#30498;',
	"\xED\xF4" => '&#30489;',
	"\xED\xF5" => '&#30509;',
	"\xED\xF6" => '&#30502;',
	"\xED\xF7" => '&#30517;',
	"\xED\xF8" => '&#30520;',
	"\xED\xF9" => '&#30544;',
	"\xED\xFA" => '&#30545;',
	"\xED\xFB" => '&#30535;',
	"\xED\xFC" => '&#30531;',
	"\xED\xFD" => '&#30554;',
	"\xED\xFE" => '&#30568;',
	"\xEE\xA1" => '&#30562;',
	"\xEE\xA2" => '&#30565;',
	"\xEE\xA3" => '&#30591;',
	"\xEE\xA4" => '&#30605;',
	"\xEE\xA5" => '&#30589;',
	"\xEE\xA6" => '&#30592;',
	"\xEE\xA7" => '&#30604;',
	"\xEE\xA8" => '&#30609;',
	"\xEE\xA9" => '&#30623;',
	"\xEE\xAA" => '&#30624;',
	"\xEE\xAB" => '&#30640;',
	"\xEE\xAC" => '&#30645;',
	"\xEE\xAD" => '&#30653;',
	"\xEE\xAE" => '&#30010;',
	"\xEE\xAF" => '&#30016;',
	"\xEE\xB0" => '&#30030;',
	"\xEE\xB1" => '&#30027;',
	"\xEE\xB2" => '&#30024;',
	"\xEE\xB3" => '&#30043;',
	"\xEE\xB4" => '&#30066;',
	"\xEE\xB5" => '&#30073;',
	"\xEE\xB6" => '&#30083;',
	"\xEE\xB7" => '&#32600;',
	"\xEE\xB8" => '&#32609;',
	"\xEE\xB9" => '&#32607;',
	"\xEE\xBA" => '&#35400;',
	"\xEE\xBB" => '&#32616;',
	"\xEE\xBC" => '&#32628;',
	"\xEE\xBD" => '&#32625;',
	"\xEE\xBE" => '&#32633;',
	"\xEE\xBF" => '&#32641;',
	"\xEE\xC0" => '&#32638;',
	"\xEE\xC1" => '&#30413;',
	"\xEE\xC2" => '&#30437;',
	"\xEE\xC3" => '&#34866;',
	"\xEE\xC4" => '&#38021;',
	"\xEE\xC5" => '&#38022;',
	"\xEE\xC6" => '&#38023;',
	"\xEE\xC7" => '&#38027;',
	"\xEE\xC8" => '&#38026;',
	"\xEE\xC9" => '&#38028;',
	"\xEE\xCA" => '&#38029;',
	"\xEE\xCB" => '&#38031;',
	"\xEE\xCC" => '&#38032;',
	"\xEE\xCD" => '&#38036;',
	"\xEE\xCE" => '&#38039;',
	"\xEE\xCF" => '&#38037;',
	"\xEE\xD0" => '&#38042;',
	"\xEE\xD1" => '&#38043;',
	"\xEE\xD2" => '&#38044;',
	"\xEE\xD3" => '&#38051;',
	"\xEE\xD4" => '&#38052;',
	"\xEE\xD5" => '&#38059;',
	"\xEE\xD6" => '&#38058;',
	"\xEE\xD7" => '&#38061;',
	"\xEE\xD8" => '&#38060;',
	"\xEE\xD9" => '&#38063;',
	"\xEE\xDA" => '&#38064;',
	"\xEE\xDB" => '&#38066;',
	"\xEE\xDC" => '&#38068;',
	"\xEE\xDD" => '&#38070;',
	"\xEE\xDE" => '&#38071;',
	"\xEE\xDF" => '&#38072;',
	"\xEE\xE0" => '&#38073;',
	"\xEE\xE1" => '&#38074;',
	"\xEE\xE2" => '&#38076;',
	"\xEE\xE3" => '&#38077;',
	"\xEE\xE4" => '&#38079;',
	"\xEE\xE5" => '&#38084;',
	"\xEE\xE6" => '&#38088;',
	"\xEE\xE7" => '&#38089;',
	"\xEE\xE8" => '&#38090;',
	"\xEE\xE9" => '&#38091;',
	"\xEE\xEA" => '&#38092;',
	"\xEE\xEB" => '&#38093;',
	"\xEE\xEC" => '&#38094;',
	"\xEE\xED" => '&#38096;',
	"\xEE\xEE" => '&#38097;',
	"\xEE\xEF" => '&#38098;',
	"\xEE\xF0" => '&#38101;',
	"\xEE\xF1" => '&#38102;',
	"\xEE\xF2" => '&#38103;',
	"\xEE\xF3" => '&#38105;',
	"\xEE\xF4" => '&#38104;',
	"\xEE\xF5" => '&#38107;',
	"\xEE\xF6" => '&#38110;',
	"\xEE\xF7" => '&#38111;',
	"\xEE\xF8" => '&#38112;',
	"\xEE\xF9" => '&#38114;',
	"\xEE\xFA" => '&#38116;',
	"\xEE\xFB" => '&#38117;',
	"\xEE\xFC" => '&#38119;',
	"\xEE\xFD" => '&#38120;',
	"\xEE\xFE" => '&#38122;',
	"\xEF\xA1" => '&#38121;',
	"\xEF\xA2" => '&#38123;',
	"\xEF\xA3" => '&#38126;',
	"\xEF\xA4" => '&#38127;',
	"\xEF\xA5" => '&#38131;',
	"\xEF\xA6" => '&#38132;',
	"\xEF\xA7" => '&#38133;',
	"\xEF\xA8" => '&#38135;',
	"\xEF\xA9" => '&#38137;',
	"\xEF\xAA" => '&#38140;',
	"\xEF\xAB" => '&#38141;',
	"\xEF\xAC" => '&#38143;',
	"\xEF\xAD" => '&#38147;',
	"\xEF\xAE" => '&#38146;',
	"\xEF\xAF" => '&#38150;',
	"\xEF\xB0" => '&#38151;',
	"\xEF\xB1" => '&#38153;',
	"\xEF\xB2" => '&#38154;',
	"\xEF\xB3" => '&#38157;',
	"\xEF\xB4" => '&#38158;',
	"\xEF\xB5" => '&#38159;',
	"\xEF\xB6" => '&#38162;',
	"\xEF\xB7" => '&#38163;',
	"\xEF\xB8" => '&#38164;',
	"\xEF\xB9" => '&#38165;',
	"\xEF\xBA" => '&#38166;',
	"\xEF\xBB" => '&#38168;',
	"\xEF\xBC" => '&#38171;',
	"\xEF\xBD" => '&#38173;',
	"\xEF\xBE" => '&#38174;',
	"\xEF\xBF" => '&#38175;',
	"\xEF\xC0" => '&#38178;',
	"\xEF\xC1" => '&#38186;',
	"\xEF\xC2" => '&#38187;',
	"\xEF\xC3" => '&#38185;',
	"\xEF\xC4" => '&#38188;',
	"\xEF\xC5" => '&#38193;',
	"\xEF\xC6" => '&#38194;',
	"\xEF\xC7" => '&#38196;',
	"\xEF\xC8" => '&#38198;',
	"\xEF\xC9" => '&#38199;',
	"\xEF\xCA" => '&#38200;',
	"\xEF\xCB" => '&#38204;',
	"\xEF\xCC" => '&#38206;',
	"\xEF\xCD" => '&#38207;',
	"\xEF\xCE" => '&#38210;',
	"\xEF\xCF" => '&#38197;',
	"\xEF\xD0" => '&#38212;',
	"\xEF\xD1" => '&#38213;',
	"\xEF\xD2" => '&#38214;',
	"\xEF\xD3" => '&#38217;',
	"\xEF\xD4" => '&#38220;',
	"\xEF\xD5" => '&#38222;',
	"\xEF\xD6" => '&#38223;',
	"\xEF\xD7" => '&#38226;',
	"\xEF\xD8" => '&#38227;',
	"\xEF\xD9" => '&#38228;',
	"\xEF\xDA" => '&#38230;',
	"\xEF\xDB" => '&#38231;',
	"\xEF\xDC" => '&#38232;',
	"\xEF\xDD" => '&#38233;',
	"\xEF\xDE" => '&#38235;',
	"\xEF\xDF" => '&#38238;',
	"\xEF\xE0" => '&#38239;',
	"\xEF\xE1" => '&#38237;',
	"\xEF\xE2" => '&#38241;',
	"\xEF\xE3" => '&#38242;',
	"\xEF\xE4" => '&#38244;',
	"\xEF\xE5" => '&#38245;',
	"\xEF\xE6" => '&#38246;',
	"\xEF\xE7" => '&#38247;',
	"\xEF\xE8" => '&#38248;',
	"\xEF\xE9" => '&#38249;',
	"\xEF\xEA" => '&#38250;',
	"\xEF\xEB" => '&#38251;',
	"\xEF\xEC" => '&#38252;',
	"\xEF\xED" => '&#38255;',
	"\xEF\xEE" => '&#38257;',
	"\xEF\xEF" => '&#38258;',
	"\xEF\xF0" => '&#38259;',
	"\xEF\xF1" => '&#38202;',
	"\xEF\xF2" => '&#30695;',
	"\xEF\xF3" => '&#30700;',
	"\xEF\xF4" => '&#38601;',
	"\xEF\xF5" => '&#31189;',
	"\xEF\xF6" => '&#31213;',
	"\xEF\xF7" => '&#31203;',
	"\xEF\xF8" => '&#31211;',
	"\xEF\xF9" => '&#31238;',
	"\xEF\xFA" => '&#23879;',
	"\xEF\xFB" => '&#31235;',
	"\xEF\xFC" => '&#31234;',
	"\xEF\xFD" => '&#31262;',
	"\xEF\xFE" => '&#31252;',
	"\xF0\xA1" => '&#31289;',
	"\xF0\xA2" => '&#31287;',
	"\xF0\xA3" => '&#31313;',
	"\xF0\xA4" => '&#40655;',
	"\xF0\xA5" => '&#39333;',
	"\xF0\xA6" => '&#31344;',
	"\xF0\xA7" => '&#30344;',
	"\xF0\xA8" => '&#30350;',
	"\xF0\xA9" => '&#30355;',
	"\xF0\xAA" => '&#30361;',
	"\xF0\xAB" => '&#30372;',
	"\xF0\xAC" => '&#29918;',
	"\xF0\xAD" => '&#29920;',
	"\xF0\xAE" => '&#29996;',
	"\xF0\xAF" => '&#40480;',
	"\xF0\xB0" => '&#40482;',
	"\xF0\xB1" => '&#40488;',
	"\xF0\xB2" => '&#40489;',
	"\xF0\xB3" => '&#40490;',
	"\xF0\xB4" => '&#40491;',
	"\xF0\xB5" => '&#40492;',
	"\xF0\xB6" => '&#40498;',
	"\xF0\xB7" => '&#40497;',
	"\xF0\xB8" => '&#40502;',
	"\xF0\xB9" => '&#40504;',
	"\xF0\xBA" => '&#40503;',
	"\xF0\xBB" => '&#40505;',
	"\xF0\xBC" => '&#40506;',
	"\xF0\xBD" => '&#40510;',
	"\xF0\xBE" => '&#40513;',
	"\xF0\xBF" => '&#40514;',
	"\xF0\xC0" => '&#40516;',
	"\xF0\xC1" => '&#40518;',
	"\xF0\xC2" => '&#40519;',
	"\xF0\xC3" => '&#40520;',
	"\xF0\xC4" => '&#40521;',
	"\xF0\xC5" => '&#40523;',
	"\xF0\xC6" => '&#40524;',
	"\xF0\xC7" => '&#40526;',
	"\xF0\xC8" => '&#40529;',
	"\xF0\xC9" => '&#40533;',
	"\xF0\xCA" => '&#40535;',
	"\xF0\xCB" => '&#40538;',
	"\xF0\xCC" => '&#40539;',
	"\xF0\xCD" => '&#40540;',
	"\xF0\xCE" => '&#40542;',
	"\xF0\xCF" => '&#40547;',
	"\xF0\xD0" => '&#40550;',
	"\xF0\xD1" => '&#40551;',
	"\xF0\xD2" => '&#40552;',
	"\xF0\xD3" => '&#40553;',
	"\xF0\xD4" => '&#40554;',
	"\xF0\xD5" => '&#40555;',
	"\xF0\xD6" => '&#40556;',
	"\xF0\xD7" => '&#40561;',
	"\xF0\xD8" => '&#40557;',
	"\xF0\xD9" => '&#40563;',
	"\xF0\xDA" => '&#30098;',
	"\xF0\xDB" => '&#30100;',
	"\xF0\xDC" => '&#30102;',
	"\xF0\xDD" => '&#30112;',
	"\xF0\xDE" => '&#30109;',
	"\xF0\xDF" => '&#30124;',
	"\xF0\xE0" => '&#30115;',
	"\xF0\xE1" => '&#30131;',
	"\xF0\xE2" => '&#30132;',
	"\xF0\xE3" => '&#30136;',
	"\xF0\xE4" => '&#30148;',
	"\xF0\xE5" => '&#30129;',
	"\xF0\xE6" => '&#30128;',
	"\xF0\xE7" => '&#30147;',
	"\xF0\xE8" => '&#30146;',
	"\xF0\xE9" => '&#30166;',
	"\xF0\xEA" => '&#30157;',
	"\xF0\xEB" => '&#30179;',
	"\xF0\xEC" => '&#30184;',
	"\xF0\xED" => '&#30182;',
	"\xF0\xEE" => '&#30180;',
	"\xF0\xEF" => '&#30187;',
	"\xF0\xF0" => '&#30183;',
	"\xF0\xF1" => '&#30211;',
	"\xF0\xF2" => '&#30193;',
	"\xF0\xF3" => '&#30204;',
	"\xF0\xF4" => '&#30207;',
	"\xF0\xF5" => '&#30224;',
	"\xF0\xF6" => '&#30208;',
	"\xF0\xF7" => '&#30213;',
	"\xF0\xF8" => '&#30220;',
	"\xF0\xF9" => '&#30231;',
	"\xF0\xFA" => '&#30218;',
	"\xF0\xFB" => '&#30245;',
	"\xF0\xFC" => '&#30232;',
	"\xF0\xFD" => '&#30229;',
	"\xF0\xFE" => '&#30233;',
	"\xF1\xA1" => '&#30235;',
	"\xF1\xA2" => '&#30268;',
	"\xF1\xA3" => '&#30242;',
	"\xF1\xA4" => '&#30240;',
	"\xF1\xA5" => '&#30272;',
	"\xF1\xA6" => '&#30253;',
	"\xF1\xA7" => '&#30256;',
	"\xF1\xA8" => '&#30271;',
	"\xF1\xA9" => '&#30261;',
	"\xF1\xAA" => '&#30275;',
	"\xF1\xAB" => '&#30270;',
	"\xF1\xAC" => '&#30259;',
	"\xF1\xAD" => '&#30285;',
	"\xF1\xAE" => '&#30302;',
	"\xF1\xAF" => '&#30292;',
	"\xF1\xB0" => '&#30300;',
	"\xF1\xB1" => '&#30294;',
	"\xF1\xB2" => '&#30315;',
	"\xF1\xB3" => '&#30319;',
	"\xF1\xB4" => '&#32714;',
	"\xF1\xB5" => '&#31462;',
	"\xF1\xB6" => '&#31352;',
	"\xF1\xB7" => '&#31353;',
	"\xF1\xB8" => '&#31360;',
	"\xF1\xB9" => '&#31366;',
	"\xF1\xBA" => '&#31368;',
	"\xF1\xBB" => '&#31381;',
	"\xF1\xBC" => '&#31398;',
	"\xF1\xBD" => '&#31392;',
	"\xF1\xBE" => '&#31404;',
	"\xF1\xBF" => '&#31400;',
	"\xF1\xC0" => '&#31405;',
	"\xF1\xC1" => '&#31411;',
	"\xF1\xC2" => '&#34916;',
	"\xF1\xC3" => '&#34921;',
	"\xF1\xC4" => '&#34930;',
	"\xF1\xC5" => '&#34941;',
	"\xF1\xC6" => '&#34943;',
	"\xF1\xC7" => '&#34946;',
	"\xF1\xC8" => '&#34978;',
	"\xF1\xC9" => '&#35014;',
	"\xF1\xCA" => '&#34999;',
	"\xF1\xCB" => '&#35004;',
	"\xF1\xCC" => '&#35017;',
	"\xF1\xCD" => '&#35042;',
	"\xF1\xCE" => '&#35022;',
	"\xF1\xCF" => '&#35043;',
	"\xF1\xD0" => '&#35045;',
	"\xF1\xD1" => '&#35057;',
	"\xF1\xD2" => '&#35098;',
	"\xF1\xD3" => '&#35068;',
	"\xF1\xD4" => '&#35048;',
	"\xF1\xD5" => '&#35070;',
	"\xF1\xD6" => '&#35056;',
	"\xF1\xD7" => '&#35105;',
	"\xF1\xD8" => '&#35097;',
	"\xF1\xD9" => '&#35091;',
	"\xF1\xDA" => '&#35099;',
	"\xF1\xDB" => '&#35082;',
	"\xF1\xDC" => '&#35124;',
	"\xF1\xDD" => '&#35115;',
	"\xF1\xDE" => '&#35126;',
	"\xF1\xDF" => '&#35137;',
	"\xF1\xE0" => '&#35174;',
	"\xF1\xE1" => '&#35195;',
	"\xF1\xE2" => '&#30091;',
	"\xF1\xE3" => '&#32997;',
	"\xF1\xE4" => '&#30386;',
	"\xF1\xE5" => '&#30388;',
	"\xF1\xE6" => '&#30684;',
	"\xF1\xE7" => '&#32786;',
	"\xF1\xE8" => '&#32788;',
	"\xF1\xE9" => '&#32790;',
	"\xF1\xEA" => '&#32796;',
	"\xF1\xEB" => '&#32800;',
	"\xF1\xEC" => '&#32802;',
	"\xF1\xED" => '&#32805;',
	"\xF1\xEE" => '&#32806;',
	"\xF1\xEF" => '&#32807;',
	"\xF1\xF0" => '&#32809;',
	"\xF1\xF1" => '&#32808;',
	"\xF1\xF2" => '&#32817;',
	"\xF1\xF3" => '&#32779;',
	"\xF1\xF4" => '&#32821;',
	"\xF1\xF5" => '&#32835;',
	"\xF1\xF6" => '&#32838;',
	"\xF1\xF7" => '&#32845;',
	"\xF1\xF8" => '&#32850;',
	"\xF1\xF9" => '&#32873;',
	"\xF1\xFA" => '&#32881;',
	"\xF1\xFB" => '&#35203;',
	"\xF1\xFC" => '&#39032;',
	"\xF1\xFD" => '&#39040;',
	"\xF1\xFE" => '&#39043;',
	"\xF2\xA1" => '&#39049;',
	"\xF2\xA2" => '&#39052;',
	"\xF2\xA3" => '&#39053;',
	"\xF2\xA4" => '&#39055;',
	"\xF2\xA5" => '&#39060;',
	"\xF2\xA6" => '&#39066;',
	"\xF2\xA7" => '&#39067;',
	"\xF2\xA8" => '&#39070;',
	"\xF2\xA9" => '&#39071;',
	"\xF2\xAA" => '&#39073;',
	"\xF2\xAB" => '&#39074;',
	"\xF2\xAC" => '&#39077;',
	"\xF2\xAD" => '&#39078;',
	"\xF2\xAE" => '&#34381;',
	"\xF2\xAF" => '&#34388;',
	"\xF2\xB0" => '&#34412;',
	"\xF2\xB1" => '&#34414;',
	"\xF2\xB2" => '&#34431;',
	"\xF2\xB3" => '&#34426;',
	"\xF2\xB4" => '&#34428;',
	"\xF2\xB5" => '&#34427;',
	"\xF2\xB6" => '&#34472;',
	"\xF2\xB7" => '&#34445;',
	"\xF2\xB8" => '&#34443;',
	"\xF2\xB9" => '&#34476;',
	"\xF2\xBA" => '&#34461;',
	"\xF2\xBB" => '&#34471;',
	"\xF2\xBC" => '&#34467;',
	"\xF2\xBD" => '&#34474;',
	"\xF2\xBE" => '&#34451;',
	"\xF2\xBF" => '&#34473;',
	"\xF2\xC0" => '&#34486;',
	"\xF2\xC1" => '&#34500;',
	"\xF2\xC2" => '&#34485;',
	"\xF2\xC3" => '&#34510;',
	"\xF2\xC4" => '&#34480;',
	"\xF2\xC5" => '&#34490;',
	"\xF2\xC6" => '&#34481;',
	"\xF2\xC7" => '&#34479;',
	"\xF2\xC8" => '&#34505;',
	"\xF2\xC9" => '&#34511;',
	"\xF2\xCA" => '&#34484;',
	"\xF2\xCB" => '&#34537;',
	"\xF2\xCC" => '&#34545;',
	"\xF2\xCD" => '&#34546;',
	"\xF2\xCE" => '&#34541;',
	"\xF2\xCF" => '&#34547;',
	"\xF2\xD0" => '&#34512;',
	"\xF2\xD1" => '&#34579;',
	"\xF2\xD2" => '&#34526;',
	"\xF2\xD3" => '&#34548;',
	"\xF2\xD4" => '&#34527;',
	"\xF2\xD5" => '&#34520;',
	"\xF2\xD6" => '&#34513;',
	"\xF2\xD7" => '&#34563;',
	"\xF2\xD8" => '&#34567;',
	"\xF2\xD9" => '&#34552;',
	"\xF2\xDA" => '&#34568;',
	"\xF2\xDB" => '&#34570;',
	"\xF2\xDC" => '&#34573;',
	"\xF2\xDD" => '&#34569;',
	"\xF2\xDE" => '&#34595;',
	"\xF2\xDF" => '&#34619;',
	"\xF2\xE0" => '&#34590;',
	"\xF2\xE1" => '&#34597;',
	"\xF2\xE2" => '&#34606;',
	"\xF2\xE3" => '&#34586;',
	"\xF2\xE4" => '&#34622;',
	"\xF2\xE5" => '&#34632;',
	"\xF2\xE6" => '&#34612;',
	"\xF2\xE7" => '&#34609;',
	"\xF2\xE8" => '&#34601;',
	"\xF2\xE9" => '&#34615;',
	"\xF2\xEA" => '&#34623;',
	"\xF2\xEB" => '&#34690;',
	"\xF2\xEC" => '&#34594;',
	"\xF2\xED" => '&#34685;',
	"\xF2\xEE" => '&#34686;',
	"\xF2\xEF" => '&#34683;',
	"\xF2\xF0" => '&#34656;',
	"\xF2\xF1" => '&#34672;',
	"\xF2\xF2" => '&#34636;',
	"\xF2\xF3" => '&#34670;',
	"\xF2\xF4" => '&#34699;',
	"\xF2\xF5" => '&#34643;',
	"\xF2\xF6" => '&#34659;',
	"\xF2\xF7" => '&#34684;',
	"\xF2\xF8" => '&#34660;',
	"\xF2\xF9" => '&#34649;',
	"\xF2\xFA" => '&#34661;',
	"\xF2\xFB" => '&#34707;',
	"\xF2\xFC" => '&#34735;',
	"\xF2\xFD" => '&#34728;',
	"\xF2\xFE" => '&#34770;',
	"\xF3\xA1" => '&#34758;',
	"\xF3\xA2" => '&#34696;',
	"\xF3\xA3" => '&#34693;',
	"\xF3\xA4" => '&#34733;',
	"\xF3\xA5" => '&#34711;',
	"\xF3\xA6" => '&#34691;',
	"\xF3\xA7" => '&#34731;',
	"\xF3\xA8" => '&#34789;',
	"\xF3\xA9" => '&#34732;',
	"\xF3\xAA" => '&#34741;',
	"\xF3\xAB" => '&#34739;',
	"\xF3\xAC" => '&#34763;',
	"\xF3\xAD" => '&#34771;',
	"\xF3\xAE" => '&#34749;',
	"\xF3\xAF" => '&#34769;',
	"\xF3\xB0" => '&#34752;',
	"\xF3\xB1" => '&#34762;',
	"\xF3\xB2" => '&#34779;',
	"\xF3\xB3" => '&#34794;',
	"\xF3\xB4" => '&#34784;',
	"\xF3\xB5" => '&#34798;',
	"\xF3\xB6" => '&#34838;',
	"\xF3\xB7" => '&#34835;',
	"\xF3\xB8" => '&#34814;',
	"\xF3\xB9" => '&#34826;',
	"\xF3\xBA" => '&#34843;',
	"\xF3\xBB" => '&#34849;',
	"\xF3\xBC" => '&#34873;',
	"\xF3\xBD" => '&#34876;',
	"\xF3\xBE" => '&#32566;',
	"\xF3\xBF" => '&#32578;',
	"\xF3\xC0" => '&#32580;',
	"\xF3\xC1" => '&#32581;',
	"\xF3\xC2" => '&#33296;',
	"\xF3\xC3" => '&#31482;',
	"\xF3\xC4" => '&#31485;',
	"\xF3\xC5" => '&#31496;',
	"\xF3\xC6" => '&#31491;',
	"\xF3\xC7" => '&#31492;',
	"\xF3\xC8" => '&#31509;',
	"\xF3\xC9" => '&#31498;',
	"\xF3\xCA" => '&#31531;',
	"\xF3\xCB" => '&#31503;',
	"\xF3\xCC" => '&#31559;',
	"\xF3\xCD" => '&#31544;',
	"\xF3\xCE" => '&#31530;',
	"\xF3\xCF" => '&#31513;',
	"\xF3\xD0" => '&#31534;',
	"\xF3\xD1" => '&#31537;',
	"\xF3\xD2" => '&#31520;',
	"\xF3\xD3" => '&#31525;',
	"\xF3\xD4" => '&#31524;',
	"\xF3\xD5" => '&#31539;',
	"\xF3\xD6" => '&#31550;',
	"\xF3\xD7" => '&#31518;',
	"\xF3\xD8" => '&#31576;',
	"\xF3\xD9" => '&#31578;',
	"\xF3\xDA" => '&#31557;',
	"\xF3\xDB" => '&#31605;',
	"\xF3\xDC" => '&#31564;',
	"\xF3\xDD" => '&#31581;',
	"\xF3\xDE" => '&#31584;',
	"\xF3\xDF" => '&#31598;',
	"\xF3\xE0" => '&#31611;',
	"\xF3\xE1" => '&#31586;',
	"\xF3\xE2" => '&#31602;',
	"\xF3\xE3" => '&#31601;',
	"\xF3\xE4" => '&#31632;',
	"\xF3\xE5" => '&#31654;',
	"\xF3\xE6" => '&#31655;',
	"\xF3\xE7" => '&#31672;',
	"\xF3\xE8" => '&#31660;',
	"\xF3\xE9" => '&#31645;',
	"\xF3\xEA" => '&#31656;',
	"\xF3\xEB" => '&#31621;',
	"\xF3\xEC" => '&#31658;',
	"\xF3\xED" => '&#31644;',
	"\xF3\xEE" => '&#31650;',
	"\xF3\xEF" => '&#31659;',
	"\xF3\xF0" => '&#31668;',
	"\xF3\xF1" => '&#31697;',
	"\xF3\xF2" => '&#31681;',
	"\xF3\xF3" => '&#31692;',
	"\xF3\xF4" => '&#31709;',
	"\xF3\xF5" => '&#31706;',
	"\xF3\xF6" => '&#31717;',
	"\xF3\xF7" => '&#31718;',
	"\xF3\xF8" => '&#31722;',
	"\xF3\xF9" => '&#31756;',
	"\xF3\xFA" => '&#31742;',
	"\xF3\xFB" => '&#31740;',
	"\xF3\xFC" => '&#31759;',
	"\xF3\xFD" => '&#31766;',
	"\xF3\xFE" => '&#31755;',
	"\xF4\xA1" => '&#31775;',
	"\xF4\xA2" => '&#31786;',
	"\xF4\xA3" => '&#31782;',
	"\xF4\xA4" => '&#31800;',
	"\xF4\xA5" => '&#31809;',
	"\xF4\xA6" => '&#31808;',
	"\xF4\xA7" => '&#33278;',
	"\xF4\xA8" => '&#33281;',
	"\xF4\xA9" => '&#33282;',
	"\xF4\xAA" => '&#33284;',
	"\xF4\xAB" => '&#33260;',
	"\xF4\xAC" => '&#34884;',
	"\xF4\xAD" => '&#33313;',
	"\xF4\xAE" => '&#33314;',
	"\xF4\xAF" => '&#33315;',
	"\xF4\xB0" => '&#33325;',
	"\xF4\xB1" => '&#33327;',
	"\xF4\xB2" => '&#33320;',
	"\xF4\xB3" => '&#33323;',
	"\xF4\xB4" => '&#33336;',
	"\xF4\xB5" => '&#33339;',
	"\xF4\xB6" => '&#33331;',
	"\xF4\xB7" => '&#33332;',
	"\xF4\xB8" => '&#33342;',
	"\xF4\xB9" => '&#33348;',
	"\xF4\xBA" => '&#33353;',
	"\xF4\xBB" => '&#33355;',
	"\xF4\xBC" => '&#33359;',
	"\xF4\xBD" => '&#33370;',
	"\xF4\xBE" => '&#33375;',
	"\xF4\xBF" => '&#33384;',
	"\xF4\xC0" => '&#34942;',
	"\xF4\xC1" => '&#34949;',
	"\xF4\xC2" => '&#34952;',
	"\xF4\xC3" => '&#35032;',
	"\xF4\xC4" => '&#35039;',
	"\xF4\xC5" => '&#35166;',
	"\xF4\xC6" => '&#32669;',
	"\xF4\xC7" => '&#32671;',
	"\xF4\xC8" => '&#32679;',
	"\xF4\xC9" => '&#32687;',
	"\xF4\xCA" => '&#32688;',
	"\xF4\xCB" => '&#32690;',
	"\xF4\xCC" => '&#31868;',
	"\xF4\xCD" => '&#25929;',
	"\xF4\xCE" => '&#31889;',
	"\xF4\xCF" => '&#31901;',
	"\xF4\xD0" => '&#31900;',
	"\xF4\xD1" => '&#31902;',
	"\xF4\xD2" => '&#31906;',
	"\xF4\xD3" => '&#31922;',
	"\xF4\xD4" => '&#31932;',
	"\xF4\xD5" => '&#31933;',
	"\xF4\xD6" => '&#31937;',
	"\xF4\xD7" => '&#31943;',
	"\xF4\xD8" => '&#31948;',
	"\xF4\xD9" => '&#31949;',
	"\xF4\xDA" => '&#31944;',
	"\xF4\xDB" => '&#31941;',
	"\xF4\xDC" => '&#31959;',
	"\xF4\xDD" => '&#31976;',
	"\xF4\xDE" => '&#33390;',
	"\xF4\xDF" => '&#26280;',
	"\xF4\xE0" => '&#32703;',
	"\xF4\xE1" => '&#32718;',
	"\xF4\xE2" => '&#32725;',
	"\xF4\xE3" => '&#32741;',
	"\xF4\xE4" => '&#32737;',
	"\xF4\xE5" => '&#32742;',
	"\xF4\xE6" => '&#32745;',
	"\xF4\xE7" => '&#32750;',
	"\xF4\xE8" => '&#32755;',
	"\xF4\xE9" => '&#31992;',
	"\xF4\xEA" => '&#32119;',
	"\xF4\xEB" => '&#32166;',
	"\xF4\xEC" => '&#32174;',
	"\xF4\xED" => '&#32327;',
	"\xF4\xEE" => '&#32411;',
	"\xF4\xEF" => '&#40632;',
	"\xF4\xF0" => '&#40628;',
	"\xF4\xF1" => '&#36211;',
	"\xF4\xF2" => '&#36228;',
	"\xF4\xF3" => '&#36244;',
	"\xF4\xF4" => '&#36241;',
	"\xF4\xF5" => '&#36273;',
	"\xF4\xF6" => '&#36199;',
	"\xF4\xF7" => '&#36205;',
	"\xF4\xF8" => '&#35911;',
	"\xF4\xF9" => '&#35913;',
	"\xF4\xFA" => '&#37194;',
	"\xF4\xFB" => '&#37200;',
	"\xF4\xFC" => '&#37198;',
	"\xF4\xFD" => '&#37199;',
	"\xF4\xFE" => '&#37220;',
	"\xF5\xA1" => '&#37218;',
	"\xF5\xA2" => '&#37217;',
	"\xF5\xA3" => '&#37232;',
	"\xF5\xA4" => '&#37225;',
	"\xF5\xA5" => '&#37231;',
	"\xF5\xA6" => '&#37245;',
	"\xF5\xA7" => '&#37246;',
	"\xF5\xA8" => '&#37234;',
	"\xF5\xA9" => '&#37236;',
	"\xF5\xAA" => '&#37241;',
	"\xF5\xAB" => '&#37260;',
	"\xF5\xAC" => '&#37253;',
	"\xF5\xAD" => '&#37264;',
	"\xF5\xAE" => '&#37261;',
	"\xF5\xAF" => '&#37265;',
	"\xF5\xB0" => '&#37282;',
	"\xF5\xB1" => '&#37283;',
	"\xF5\xB2" => '&#37290;',
	"\xF5\xB3" => '&#37293;',
	"\xF5\xB4" => '&#37294;',
	"\xF5\xB5" => '&#37295;',
	"\xF5\xB6" => '&#37301;',
	"\xF5\xB7" => '&#37300;',
	"\xF5\xB8" => '&#37306;',
	"\xF5\xB9" => '&#35925;',
	"\xF5\xBA" => '&#40574;',
	"\xF5\xBB" => '&#36280;',
	"\xF5\xBC" => '&#36331;',
	"\xF5\xBD" => '&#36357;',
	"\xF5\xBE" => '&#36441;',
	"\xF5\xBF" => '&#36457;',
	"\xF5\xC0" => '&#36277;',
	"\xF5\xC1" => '&#36287;',
	"\xF5\xC2" => '&#36284;',
	"\xF5\xC3" => '&#36282;',
	"\xF5\xC4" => '&#36292;',
	"\xF5\xC5" => '&#36310;',
	"\xF5\xC6" => '&#36311;',
	"\xF5\xC7" => '&#36314;',
	"\xF5\xC8" => '&#36318;',
	"\xF5\xC9" => '&#36302;',
	"\xF5\xCA" => '&#36303;',
	"\xF5\xCB" => '&#36315;',
	"\xF5\xCC" => '&#36294;',
	"\xF5\xCD" => '&#36332;',
	"\xF5\xCE" => '&#36343;',
	"\xF5\xCF" => '&#36344;',
	"\xF5\xD0" => '&#36323;',
	"\xF5\xD1" => '&#36345;',
	"\xF5\xD2" => '&#36347;',
	"\xF5\xD3" => '&#36324;',
	"\xF5\xD4" => '&#36361;',
	"\xF5\xD5" => '&#36349;',
	"\xF5\xD6" => '&#36372;',
	"\xF5\xD7" => '&#36381;',
	"\xF5\xD8" => '&#36383;',
	"\xF5\xD9" => '&#36396;',
	"\xF5\xDA" => '&#36398;',
	"\xF5\xDB" => '&#36387;',
	"\xF5\xDC" => '&#36399;',
	"\xF5\xDD" => '&#36410;',
	"\xF5\xDE" => '&#36416;',
	"\xF5\xDF" => '&#36409;',
	"\xF5\xE0" => '&#36405;',
	"\xF5\xE1" => '&#36413;',
	"\xF5\xE2" => '&#36401;',
	"\xF5\xE3" => '&#36425;',
	"\xF5\xE4" => '&#36417;',
	"\xF5\xE5" => '&#36418;',
	"\xF5\xE6" => '&#36433;',
	"\xF5\xE7" => '&#36434;',
	"\xF5\xE8" => '&#36426;',
	"\xF5\xE9" => '&#36464;',
	"\xF5\xEA" => '&#36470;',
	"\xF5\xEB" => '&#36476;',
	"\xF5\xEC" => '&#36463;',
	"\xF5\xED" => '&#36468;',
	"\xF5\xEE" => '&#36485;',
	"\xF5\xEF" => '&#36495;',
	"\xF5\xF0" => '&#36500;',
	"\xF5\xF1" => '&#36496;',
	"\xF5\xF2" => '&#36508;',
	"\xF5\xF3" => '&#36510;',
	"\xF5\xF4" => '&#35960;',
	"\xF5\xF5" => '&#35970;',
	"\xF5\xF6" => '&#35978;',
	"\xF5\xF7" => '&#35973;',
	"\xF5\xF8" => '&#35992;',
	"\xF5\xF9" => '&#35988;',
	"\xF5\xFA" => '&#26011;',
	"\xF5\xFB" => '&#35286;',
	"\xF5\xFC" => '&#35294;',
	"\xF5\xFD" => '&#35290;',
	"\xF5\xFE" => '&#35292;',
	"\xF6\xA1" => '&#35301;',
	"\xF6\xA2" => '&#35307;',
	"\xF6\xA3" => '&#35311;',
	"\xF6\xA4" => '&#35390;',
	"\xF6\xA5" => '&#35622;',
	"\xF6\xA6" => '&#38739;',
	"\xF6\xA7" => '&#38633;',
	"\xF6\xA8" => '&#38643;',
	"\xF6\xA9" => '&#38639;',
	"\xF6\xAA" => '&#38662;',
	"\xF6\xAB" => '&#38657;',
	"\xF6\xAC" => '&#38664;',
	"\xF6\xAD" => '&#38671;',
	"\xF6\xAE" => '&#38670;',
	"\xF6\xAF" => '&#38698;',
	"\xF6\xB0" => '&#38701;',
	"\xF6\xB1" => '&#38704;',
	"\xF6\xB2" => '&#38718;',
	"\xF6\xB3" => '&#40832;',
	"\xF6\xB4" => '&#40835;',
	"\xF6\xB5" => '&#40837;',
	"\xF6\xB6" => '&#40838;',
	"\xF6\xB7" => '&#40839;',
	"\xF6\xB8" => '&#40840;',
	"\xF6\xB9" => '&#40841;',
	"\xF6\xBA" => '&#40842;',
	"\xF6\xBB" => '&#40844;',
	"\xF6\xBC" => '&#40702;',
	"\xF6\xBD" => '&#40715;',
	"\xF6\xBE" => '&#40717;',
	"\xF6\xBF" => '&#38585;',
	"\xF6\xC0" => '&#38588;',
	"\xF6\xC1" => '&#38589;',
	"\xF6\xC2" => '&#38606;',
	"\xF6\xC3" => '&#38610;',
	"\xF6\xC4" => '&#30655;',
	"\xF6\xC5" => '&#38624;',
	"\xF6\xC6" => '&#37518;',
	"\xF6\xC7" => '&#37550;',
	"\xF6\xC8" => '&#37576;',
	"\xF6\xC9" => '&#37694;',
	"\xF6\xCA" => '&#37738;',
	"\xF6\xCB" => '&#37834;',
	"\xF6\xCC" => '&#37775;',
	"\xF6\xCD" => '&#37950;',
	"\xF6\xCE" => '&#37995;',
	"\xF6\xCF" => '&#40063;',
	"\xF6\xD0" => '&#40066;',
	"\xF6\xD1" => '&#40069;',
	"\xF6\xD2" => '&#40070;',
	"\xF6\xD3" => '&#40071;',
	"\xF6\xD4" => '&#40072;',
	"\xF6\xD5" => '&#31267;',
	"\xF6\xD6" => '&#40075;',
	"\xF6\xD7" => '&#40078;',
	"\xF6\xD8" => '&#40080;',
	"\xF6\xD9" => '&#40081;',
	"\xF6\xDA" => '&#40082;',
	"\xF6\xDB" => '&#40084;',
	"\xF6\xDC" => '&#40085;',
	"\xF6\xDD" => '&#40090;',
	"\xF6\xDE" => '&#40091;',
	"\xF6\xDF" => '&#40094;',
	"\xF6\xE0" => '&#40095;',
	"\xF6\xE1" => '&#40096;',
	"\xF6\xE2" => '&#40097;',
	"\xF6\xE3" => '&#40098;',
	"\xF6\xE4" => '&#40099;',
	"\xF6\xE5" => '&#40101;',
	"\xF6\xE6" => '&#40102;',
	"\xF6\xE7" => '&#40103;',
	"\xF6\xE8" => '&#40104;',
	"\xF6\xE9" => '&#40105;',
	"\xF6\xEA" => '&#40107;',
	"\xF6\xEB" => '&#40109;',
	"\xF6\xEC" => '&#40110;',
	"\xF6\xED" => '&#40112;',
	"\xF6\xEE" => '&#40113;',
	"\xF6\xEF" => '&#40114;',
	"\xF6\xF0" => '&#40115;',
	"\xF6\xF1" => '&#40116;',
	"\xF6\xF2" => '&#40117;',
	"\xF6\xF3" => '&#40118;',
	"\xF6\xF4" => '&#40119;',
	"\xF6\xF5" => '&#40122;',
	"\xF6\xF6" => '&#40123;',
	"\xF6\xF7" => '&#40124;',
	"\xF6\xF8" => '&#40125;',
	"\xF6\xF9" => '&#40132;',
	"\xF6\xFA" => '&#40133;',
	"\xF6\xFB" => '&#40134;',
	"\xF6\xFC" => '&#40135;',
	"\xF6\xFD" => '&#40138;',
	"\xF6\xFE" => '&#40139;',
	"\xF7\xA1" => '&#40140;',
	"\xF7\xA2" => '&#40141;',
	"\xF7\xA3" => '&#40142;',
	"\xF7\xA4" => '&#40143;',
	"\xF7\xA5" => '&#40144;',
	"\xF7\xA6" => '&#40147;',
	"\xF7\xA7" => '&#40148;',
	"\xF7\xA8" => '&#40149;',
	"\xF7\xA9" => '&#40151;',
	"\xF7\xAA" => '&#40152;',
	"\xF7\xAB" => '&#40153;',
	"\xF7\xAC" => '&#40156;',
	"\xF7\xAD" => '&#40157;',
	"\xF7\xAE" => '&#40159;',
	"\xF7\xAF" => '&#40162;',
	"\xF7\xB0" => '&#38780;',
	"\xF7\xB1" => '&#38789;',
	"\xF7\xB2" => '&#38801;',
	"\xF7\xB3" => '&#38802;',
	"\xF7\xB4" => '&#38804;',
	"\xF7\xB5" => '&#38831;',
	"\xF7\xB6" => '&#38827;',
	"\xF7\xB7" => '&#38819;',
	"\xF7\xB8" => '&#38834;',
	"\xF7\xB9" => '&#38836;',
	"\xF7\xBA" => '&#39601;',
	"\xF7\xBB" => '&#39600;',
	"\xF7\xBC" => '&#39607;',
	"\xF7\xBD" => '&#40536;',
	"\xF7\xBE" => '&#39606;',
	"\xF7\xBF" => '&#39610;',
	"\xF7\xC0" => '&#39612;',
	"\xF7\xC1" => '&#39617;',
	"\xF7\xC2" => '&#39616;',
	"\xF7\xC3" => '&#39621;',
	"\xF7\xC4" => '&#39618;',
	"\xF7\xC5" => '&#39627;',
	"\xF7\xC6" => '&#39628;',
	"\xF7\xC7" => '&#39633;',
	"\xF7\xC8" => '&#39749;',
	"\xF7\xC9" => '&#39747;',
	"\xF7\xCA" => '&#39751;',
	"\xF7\xCB" => '&#39753;',
	"\xF7\xCC" => '&#39752;',
	"\xF7\xCD" => '&#39757;',
	"\xF7\xCE" => '&#39761;',
	"\xF7\xCF" => '&#39144;',
	"\xF7\xD0" => '&#39181;',
	"\xF7\xD1" => '&#39214;',
	"\xF7\xD2" => '&#39253;',
	"\xF7\xD3" => '&#39252;',
	"\xF7\xD4" => '&#39647;',
	"\xF7\xD5" => '&#39649;',
	"\xF7\xD6" => '&#39654;',
	"\xF7\xD7" => '&#39663;',
	"\xF7\xD8" => '&#39659;',
	"\xF7\xD9" => '&#39675;',
	"\xF7\xDA" => '&#39661;',
	"\xF7\xDB" => '&#39673;',
	"\xF7\xDC" => '&#39688;',
	"\xF7\xDD" => '&#39695;',
	"\xF7\xDE" => '&#39699;',
	"\xF7\xDF" => '&#39711;',
	"\xF7\xE0" => '&#39715;',
	"\xF7\xE1" => '&#40637;',
	"\xF7\xE2" => '&#40638;',
	"\xF7\xE3" => '&#32315;',
	"\xF7\xE4" => '&#40578;',
	"\xF7\xE5" => '&#40583;',
	"\xF7\xE6" => '&#40584;',
	"\xF7\xE7" => '&#40587;',
	"\xF7\xE8" => '&#40594;',
	"\xF7\xE9" => '&#37846;',
	"\xF7\xEA" => '&#40605;',
	"\xF7\xEB" => '&#40607;',
	"\xF7\xEC" => '&#40667;',
	"\xF7\xED" => '&#40668;',
	"\xF7\xEE" => '&#40669;',
	"\xF7\xEF" => '&#40672;',
	"\xF7\xF0" => '&#40671;',
	"\xF7\xF1" => '&#40674;',
	"\xF7\xF2" => '&#40681;',
	"\xF7\xF3" => '&#40679;',
	"\xF7\xF4" => '&#40677;',
	"\xF7\xF5" => '&#40682;',
	"\xF7\xF6" => '&#40687;',
	"\xF7\xF7" => '&#40738;',
	"\xF7\xF8" => '&#40748;',
	"\xF7\xF9" => '&#40751;',
	"\xF7\xFA" => '&#40761;',
	"\xF7\xFB" => '&#40759;',
	"\xF7\xFC" => '&#40765;',
	"\xF7\xFD" => '&#40766;',
	"\xF7\xFE" => '&#40772;'
    );

    $index=0;
    $ret="";
    
    while ( $index < strlen($string)) {
      if ( ereg("[\241-\367]", $string[$index])) {
        $ret.= str_replace(array_keys($gb2312), array_values($gb2312), $string[$index] . $string[$index+1]);
        $index=$index+2;
      } else {
        $ret.= $string[$index];
        $index=$index+1;
      }
    }

    return $ret;
}
?>
