#include <pthread.h>
#include <glib.h>
#include "patch.h"
#include "specimen.h"
#include "driver.h"
#include "sample.h"
#include "maths.h"
#include "ticks.h"
#include "lfo.h"

/* magic numbers */
enum
{
     EVENTMAX = 1024,
};

/* event codes */
typedef enum
{
     MIXER_NOTEON = 1,
     MIXER_NOTEOFF,
     MIXER_NOTEON_WITH_ID,
     MIXER_NOTEOFF_WITH_ID,
     MIXER_CONTROL
}
MixerEventType;

typedef struct {
     int chan;
     int note;
     int vel;
} MixerNoteEvent;

typedef struct {
     int id;
     int note;
     int vel;
} MixerIdNoteEvent;

typedef struct {
     int chan;
     PatchParamType param;
     float value;
} MixerControlEvent;

/* type for ringbuffer of incoming events */
typedef struct _Event
{
     MixerEventType type;	/* type of event */
     Tick ticks;		/* time of event's occurence */
     union {
	  MixerNoteEvent    note;
	  MixerIdNoteEvent  id_note;
	  MixerControlEvent control;
     };
}
Event;

/* special structure for previewing samples */
typedef struct _MixerPreview
{
     Atomic active;
     Sample* sample;
     int next_frame;
     pthread_mutex_t mutex;
}
MixerPreview;

/* general variables */
static float volume = 0.5;	/* master volume */
static MixerPreview preview;	/* current preview sample */
static Event events[EVENTMAX];
static Event* writer = events;
static Event* reader = events;
static int samplerate = DRIVER_DEFAULT_SAMPLERATE;

/* advance reader event vector */
inline static void advance_reader ( )
{
     reader = (reader + 1 >= events + EVENTMAX) ? events : reader + 1;
}

/* advance writer event vector */
inline static void advance_writer ( )
{
     writer = (writer + 1 >= events + EVENTMAX) ? events : writer + 1;
}

/* mixdown current preview sample (if any) into buf */
inline static void preview_render (float *buf, int frames)
{
     int i, j;

     /* if there is a sample currently being previewed let's throw it into the mix */
     if (preview.active && pthread_mutex_trylock (&preview.mutex) == 0)
     {
	  if (preview.sample->sp != NULL)
	  {
	       /* fill the buf with samples from preview (two samples per frame) */
	       for (i = 0, j = preview.next_frame * 2;
		    i < frames * 2 && j < preview.sample->frames * 2; i++, j++)
		    buf[i] += preview.sample->sp[j];

	       /* increment next frame indicator (half as much as the number of samples written) */
	       if ((preview.next_frame = j / 2) >= preview.sample->frames)
	       {
		    preview.active = 0;
		    sample_free_file (preview.sample);
	       }
	  }
	  else
	  {
	       preview.active = 0;
	  }
	  pthread_mutex_unlock (&preview.mutex);
     }
}

/* flush all events currently queued */
void mixer_flush ( )
{
     /* skip any queued ringbuffer events */
     reader = writer;

     patch_flush_all ( );
}

/* constructor */
void mixer_init ( )
{
     debug ("initializing...\n");
     pthread_mutex_init (&preview.mutex, NULL);
     preview.sample = sample_new ( );
     debug ("done\n");
}

/* mix current soundscape into buf */
void mixer_mixdown (float *buf, int frames)
{
     Tick curticks = ticks_get_ticks ( );
     int offset;
     int i;

     /* clear the buffer */
     for (i = 0; i < frames * 2; i++)
	  buf[i] = 0.0;

     /* get events */
     while (reader != writer)
     {
	  if (reader->ticks > curticks)
	       break;

	  offset = reader->ticks - (curticks - frames);
	  if (offset < 0)
	       offset = 0;

	  switch (reader->type)
	  {
	  case MIXER_NOTEON:
	       patch_trigger (reader->note.chan, reader->note.note, reader->note.vel / 127.0,
			      reader->ticks, offset);
	       break;
	  case MIXER_NOTEON_WITH_ID:
	       patch_trigger_with_id (reader->id_note.id, reader->id_note.note,
				      reader->id_note.vel / 127.0, reader->ticks,
				      offset);
	       break;
	  case MIXER_NOTEOFF:
	       patch_release (reader->note.chan, reader->note.note, offset);
	       break;
	  case MIXER_NOTEOFF_WITH_ID:
	       patch_release_with_id (reader->id_note.id, reader->id_note.note, offset);
	       break;
	  case MIXER_CONTROL:
	       patch_control(reader->control.chan, reader->control.param, reader->control.value);
	  }

	  advance_reader ( );
     }

     patch_render (buf, frames);
     preview_render (buf, frames);

     /* scale to master volume */
     for (i = 0; i < frames * 2; i++)
	  buf[i] *= volume;
}

/* queue a note-off event */
void mixer_note_off (int chan, int note)
{
     writer->type = MIXER_NOTEOFF;
     writer->ticks = ticks_get_ticks ( );
     writer->note.chan = chan;
     writer->note.note = note;
     advance_writer ( );
}

/* queue a note-off event by patch id */
void mixer_note_off_with_id (int id, int note)
{
     writer->type = MIXER_NOTEOFF_WITH_ID;
     writer->ticks = ticks_get_ticks ( );
     writer->id_note.id = id;
     writer->id_note.note = note;
     advance_writer ( );
}

/* queue a note-on event */
void mixer_note_on (int chan, int note, int vel)
{
     writer->type = MIXER_NOTEON;
     writer->ticks = ticks_get_ticks ( );
     writer->note.chan = chan;
     writer->note.note = note;
     writer->note.vel = vel;
     advance_writer ( );
}

/* queue a note-on event by patch id */
void mixer_note_on_with_id (int id, int note, int vel)
{
     writer->type = MIXER_NOTEON_WITH_ID;
     writer->ticks = ticks_get_ticks ( );
     writer->id_note.id = id;
     writer->id_note.note = note;
     writer->id_note.vel = vel;
     advance_writer ( );
}

/* queue control change event */
void mixer_control(int chan, ControlParamType param, float value)
{
     writer->type = MIXER_CONTROL;
     writer->ticks = ticks_get_ticks ( );
     writer->control.chan = chan;
     writer->control.param = param;
     writer->control.value = value;
     advance_writer ( );
}

/* preview a sample file */
void mixer_preview (char *name)
{
     pthread_mutex_lock (&preview.mutex);
     preview.active = 0;
     sample_load_file (preview.sample, name, samplerate);

     preview.next_frame = 0;
     preview.active = 1;
     pthread_mutex_unlock (&preview.mutex);
}

/* set the master volume */
int mixer_set_volume (float vol)
{
     if (vol < 0.0 || vol > 1.0)
	  return -1;

     volume = vol;
     return 0;
}

/* return the master volume */
float mixer_get_volume(void)
{
    return volume;
}

/* set internally assumed samplerate */
void mixer_set_samplerate (int rate)
{
     samplerate = rate;
}

/* destructor */
void mixer_shutdown ( )
{
     debug ("shutting down...\n");
     sample_free (preview.sample);
     debug ("done\n");
}
