#!/usr/bin/python
# -*- coding: utf-8 -*-

from gi.repository import Gtk, GObject

import atexit
import dbus
import logging
import glob
import os
import signal
import subprocess
import sys
import time
import unittest

sys.path.insert(0, "../")
from softwareproperties.dbus.SoftwarePropertiesDBus import (
    SoftwarePropertiesDBus)
from softwareproperties import (
    UPDATE_INST_SEC, UPDATE_DOWNLOAD, UPDATE_NOTIFY)

def get_test_source_line():
    distro_release = get_distro_release()
    return "deb http://archive.ubuntu.com/ubuntu %s main restricted #"\
           " comment with unicode äöü" % distro_release

def get_distro_release():
    return subprocess.check_output(["lsb_release", "-c", "-s"]).strip()

def create_sources_list():
    s = get_test_source_line() + "\n"
    f=open("./aptroot/etc/apt/sources.list", "w")
    f.write(s)
    f.close()

def start_software_properties_dbus_in_session_bus():
    create_sources_list()
    pid = os.fork()
    if pid == 0:
        bus = dbus.SessionBus()
        try:
            spd = SoftwarePropertiesDBus(bus, rootdir="./aptroot")
            spd.enforce_polkit = False
            Gtk.main()
        except:
            logging.exception("dbus service failed")
            sys.exit(1)
    else:
        print "starting", pid
        time.sleep(0.5)
        atexit.register(stop_software_properties_dbus, pid)
def stop_software_properties_dbus(pid):
    print "stopping", pid
    os.kill(pid, signal.SIGTERM)
    #os.kill(self.pid, signal.SIGKILL)
    os.waitpid(pid, 0)
pid = start_software_properties_dbus_in_session_bus()

class TestDBus(unittest.TestCase):

    def setUp(self):
        # create sources.list file
        self.distro_release = get_distro_release()
        create_sources_list()
        # create the client proxy
        bus = dbus.SessionBus()
        proxy = bus.get_object("com.ubuntu.SoftwareProperties", "/")
        iface = dbus.Interface(proxy, "com.ubuntu.SoftwareProperties")
        self.iface = iface
        self._signal_id = iface.connect_to_signal(
            "SourcesListModified", self._on_sources_list_modified)
        self.loop = GObject.MainLoop(GObject.main_context_default())

    def tearDown(self):
        # ensure we remove the "modified" signal again
        self._signal_id.remove()

    def _on_sources_list_modified(self):
        #print "_on_modified_sources_list"
        self.loop.quit()

    def _debug_sourceslist(self, text=""):
        sourceslist = open("./aptroot/etc/apt/sources.list").read()
        logging.debug("sourceslist: %s '%s'" % (text, sourceslist))

    def test_enable_disable_component(self):
        # ensure its not there
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self.assertFalse("universe" in sourceslist)
        # enable
        res = self.iface.EnableComponent("universe")
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("2")
        self.assertTrue("universe" in sourceslist)
        # disable again
        res = self.iface.DisableComponent("universe")
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("3")
        self.assertFalse("universe" in sourceslist)
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()

    def test_enable_enable_disable_source_code_sources(self):
        # ensure its not there
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("4")
        self.assertFalse("deb-src" in sourceslist)
        # enable
        res = self.iface.EnableSourceCodeSources()
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("5")
        self.assertTrue("deb-src" in sourceslist)
        # disable again
        res = self.iface.DisableSourceCodeSources()
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("6")
        self.assertFalse("deb-src" in sourceslist)
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()

    def test_enable_child_source(self):
        child_source = "%s-updates" % self.distro_release
        # ensure its not there
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("7")
        self.assertFalse(child_source in sourceslist)
        # enable
        res = self.iface.EnableChildSource(child_source)
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("8")
        self.assertTrue(child_source in sourceslist)
        # disable again
        res = self.iface.DisableChildSource(child_source)
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("9")
        self.assertFalse(child_source in sourceslist)
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()
        
    def test_toggle_source(self):
        # test toggle
        source = get_test_source_line()
        res = self.iface.ToggleSourceUse(source)
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("10")
        self.assertTrue("# deb http://archive.ubuntu.com/ubuntu" in sourceslist)
        # to disable the line again, we need to match the new "#"
        source = "# " + source
        res = self.iface.ToggleSourceUse(source)
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("11")
        self.assertFalse("# deb http://archive.ubuntu.com/ubuntu" in sourceslist)
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()

    def test_replace_entry(self):
        # test toggle
        source = get_test_source_line()
        source_new = "deb http://xxx/ %s" % self.distro_release
        res = self.iface.ReplaceSourceEntry(source, source_new)
        sourceslist =  open("./aptroot/etc/apt/sources.list").read()
        self._debug_sourceslist("11")
        self.assertTrue(source_new in sourceslist)
        self.assertFalse(source in sourceslist)
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()
        res = self.iface.ReplaceSourceEntry(source_new, source)
        self.loop.run()

    def test_popcon(self):
        # ensure its set to no
        popcon = open("./aptroot/etc/popularity-contest.conf").read()
        self.assertTrue('PARTICIPATE="no"' in popcon)
        # toggle
        res = self.iface.SetPopconPariticipation(True)
        popcon = open("./aptroot/etc/popularity-contest.conf").read()
        self.assertTrue('PARTICIPATE="yes"' in popcon)
        self.assertFalse('PARTICIPATE="no"' in popcon)
        # and back
        res = self.iface.SetPopconPariticipation(False)
        popcon = open("./aptroot/etc/popularity-contest.conf").read()
        self.assertFalse('PARTICIPATE="yes"' in popcon)
        self.assertTrue('PARTICIPATE="no"' in popcon)

    def test_updates_automation(self):
        states = [UPDATE_INST_SEC, UPDATE_DOWNLOAD, UPDATE_NOTIFY]
        # security
        res = self.iface.SetUpdateAutomationLevel(states[0])
        self.assertTrue('APT::Periodic::Unattended-Upgrade "1";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        # download
        res = self.iface.SetUpdateAutomationLevel(states[1])
        self.assertTrue('APT::Periodic::Unattended-Upgrade "0";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        self.assertTrue('APT::Periodic::Download-Upgradeable-Packages "1";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        # notify
        res = self.iface.SetUpdateAutomationLevel(states[2])
        self.assertTrue('APT::Periodic::Update-Package-Lists "1";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        self.assertTrue('APT::Periodic::Unattended-Upgrade "0";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        self.assertTrue('APT::Periodic::Download-Upgradeable-Packages "0";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())

    def test_updates_interval(self):
        # interval
        res = self.iface.SetUpdateInterval(0)
        self.assertTrue('APT::Periodic::Update-Package-Lists "0";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())
        res = self.iface.SetUpdateInterval(1)
        self.assertTrue('APT::Periodic::Update-Package-Lists "1";' in
                        open("./aptroot/etc/apt/apt.conf.d/10periodic").read())

    def test_add_remove_source_by_line(self):
        # add invalid
        res = self.iface.AddSourceFromLine("xxx")
        self.assertFalse(res)
        # add real
        s = "deb http//ppa.launchpad.net/ foo bar"
        res = self.iface.AddSourceFromLine(s)
        self.assertTrue(s in open("./aptroot/etc/apt/sources.list").read())
        # remove again, needs to remove both deb and deb-src lines
        self.iface.RemoveSource(s)
        self.iface.RemoveSource(s.replace("deb", "deb-src"))
        self.assertFalse(s in open("./aptroot/etc/apt/sources.list").read())
        # wait for the _on_modified_sources_list signal to arrive"
        self.loop.run()

    def test_add_gpg_key(self):
        # clean
        for f in glob.glob("./aptroot/etc/apt/*.gpg"):
            os.remove(f)
        self.assertTrue(len(glob.glob("./aptroot/etc/apt/*.gpg")) == 0)
        # add key from file
        res = self.iface.AddKey("./data/testkey.gpg")
        self.assertTrue(res)
        self.assertEqual(len(glob.glob("./aptroot/etc/apt/*.gpg")), 1)
        self.assertNotEqual(os.path.getsize("./aptroot/etc/apt/trusted.gpg"), 0)
        # remove the key 
        res = self.iface.RemoveKey("D732CA59")
        self.assertTrue(res)
        self.assertEqual(os.path.getsize("./aptroot/etc/apt/trusted.gpg"), 0)
        # add from data
        data = open("./data/testkey.gpg").read()
        res = self.iface.AddKeyFromData(data)
        self.assertTrue(res)
        self.assertEqual(len(glob.glob("./aptroot/etc/apt/*.gpg")), 1)
        self.assertNotEqual(os.path.getsize("./aptroot/etc/apt/trusted.gpg"), 0)
        # remove the key 
        res = self.iface.RemoveKey("D732CA59")
        self.assertTrue(res)
        self.assertEqual(os.path.getsize("./aptroot/etc/apt/trusted.gpg"), 0)
        # test nonsense
        res = self.iface.AddKeyFromData("nonsens")
        self.assertFalse(res)
        # test apt-key update
        res = self.iface.UpdateKeys()
        self.assertTrue(res)
        

if __name__ == "__main__":
    if "-d" in sys.argv:
        logging.basicConfig(level=logging.DEBUG)
    else:
        logging.basicConfig(level=logging.INFO)
    unittest.main()
