/*
** icat 
** The Sleuth Kit 
**
** $Date: 2006/08/30 21:09:12 $
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2006 Brian Carrier, Basis Technology.  All Rights reserved
** Copyright (c) 2003-2005 Brian Carrier.  All rights reserved 
**
** TASK
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
** 
** Copyright (c) 1997,1998,1999, International Business Machines          
** Corporation and others. All Rights Reserved.

 * LICENSE
 *	This software is distributed under the IBM Public License.
 * AUTHOR(S)
 *	Wietse Venema
 *	IBM T.J. Watson Research
 *	P.O. Box 704
 *	Yorktown Heights, NY 10598, USA
 --*/
#include <locale.h>
#include "fs_tools.h"


/* usage - explain and terminate */

static TCHAR *progname;

static void
usage()
{
    TFPRINTF(stderr,
	_T
	("usage: %s [-hHsvV] [-f fstype] [-i imgtype] [-o imgoffset] image [images] inum[-typ[-id]]\n"),
	progname);
    tsk_fprintf(stderr, "\t-h: Do not display holes in sparse files\n");
    tsk_fprintf(stderr, "\t-r: Recover deleted file\n");
    tsk_fprintf(stderr,
	"\t-R: Recover deleted file and suppress recovery errors\n");
    tsk_fprintf(stderr, "\t-s: Display slack space at end of file\n");
    tsk_fprintf(stderr,
	"\t-i imgtype: The format of the image file (use '-i list' for supported types)\n");
    tsk_fprintf(stderr,
	"\t-f fstype: File system type (use '-f list' for supported types)\n");
    tsk_fprintf(stderr,
	"\t-o imgoffset: The offset of the file system in the image (in sectors)\n");
    tsk_fprintf(stderr, "\t-v: verbose to stderr\n");
    tsk_fprintf(stderr, "\t-V: Print version\n");

    exit(1);
}

int
MAIN(int argc, TCHAR ** argv)
{
    TCHAR *fstype = NULL;
    TCHAR *imgtype = NULL;
    FS_INFO *fs;
    IMG_INFO *img;
    INUM_T inum;
    int flags = 0;
    int ch;
    uint32_t type = 0;
    uint16_t id = 0;
    int id_used = 0;
    int retval;
    SSIZE_T imgoff = 0;
    int suppress_recover_error = 0;

    progname = argv[0];
    setlocale(LC_ALL, "");

    while ((ch = getopt(argc, argv, _T("f:hi:o:rRsvV"))) > 0) {
	switch (ch) {
	case _T('?'):
	default:
	    TFPRINTF(stderr, _T("Invalid argument: %s\n"), argv[optind]);
	    usage();
	case _T('f'):
	    fstype = optarg;
	    if (TSTRCMP(fstype, _T("list")) == 0) {
		fs_print_types(stderr);
		exit(1);
	    }
	    break;
	case _T('h'):
	    flags |= FS_FLAG_FILE_NOSPARSE;
	    break;
	case _T('i'):
	    imgtype = optarg;
	    if (TSTRCMP(imgtype, _T("list")) == 0) {
		img_print_types(stderr);
		exit(1);
	    }
	    break;
	case _T('o'):
	    if ((imgoff = parse_offset(optarg)) == -1) {
		tsk_error_print(stderr);
		exit(1);
	    }
	    break;
	case _T('r'):
	    flags |= FS_FLAG_FILE_RECOVER;
	    break;
	case _T('R'):
	    flags |= FS_FLAG_FILE_RECOVER;
	    suppress_recover_error = 1;
	    break;
	case _T('s'):
	    flags |= FS_FLAG_FILE_SLACK;
	    break;
	case _T('v'):
	    verbose++;
	    break;
	case _T('V'):
	    print_version(stdout);
	    exit(0);
	}
    }

    /* We need at least two more argument */
    if (optind + 1 >= argc) {
	tsk_fprintf(stderr, "Missing image name and/or address\n");
	usage();
    }

    /* Get the inode address */
    if (parse_inum(argv[argc - 1], &inum, &type, &id, &id_used)) {
	TFPRINTF(stderr, _T("Invalid inode address: %s\n"),
	    argv[argc - 1]);
	usage();
    }

    if ((img =
	    img_open(imgtype, argc - optind - 1,
		(const TCHAR **) &argv[optind])) == NULL) {
	tsk_error_print(stderr);
	exit(1);
    }
    if ((fs = fs_open(img, imgoff, fstype)) == NULL) {
	tsk_error_print(stderr);
	if (tsk_errno == TSK_ERR_FS_UNSUPTYPE)
	    fs_print_types(stderr);
	img->close(img);
	exit(1);
    }

    if (inum > fs->last_inum) {
	tsk_fprintf(stderr,
	    "Metadata address too large for image (%" PRIuINUM ")\n",
	    fs->last_inum);
	fs->close(fs);
	img->close(img);
	exit(1);
    }
    if (inum < fs->first_inum) {
	tsk_fprintf(stderr,
	    "Metadata address too small for image (%" PRIuINUM ")\n",
	    fs->first_inum);
	fs->close(fs);
	img->close(img);
	exit(1);
    }

    if (id_used)
	retval = fs_icat(fs, 0, inum, type, id, flags);
    /* If the id value was not used, then set the flag accordingly so the '0' value is ignored */
    else
	retval = fs_icat(fs, 0, inum, type, id, flags | FS_FLAG_FILE_NOID);

    if (retval) {
	if ((suppress_recover_error == 1)
	    && (tsk_errno == TSK_ERR_FS_RECOVER)) {
	    tsk_error_reset();
	}
	else {
	    tsk_error_print(stderr);
	    fs->close(fs);
	    img->close(img);
	    exit(1);
	}
    }
    fs->close(fs);
    img->close(img);
    exit(0);
}
