/*
 * __BEGIN_COPYRIGHT
 * SimpleDB API
 * 
 * Copyright (C) 2005 Eminence Technology Pty Ltd
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You can view the GNU Lesser General Public Licence at
 * http://www.gnu.org/licenses/lgpl.html or you can write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Eminence Technology Pty Ltd can be contacted by writing to
 * Eminence Technology, PO Box 118, Moorooka QLD 4105, Australia.
 * Alternatively, you may email opensource [at] eminence [dot] com [dot] au
 * __END_COPYRIGHT
 */

#include "SimpleDB.h"

SimpleDB::Query::Query(SQLHDBC connectionHandle) {
  long return_value=SQLAllocHandle(SQL_HANDLE_STMT, connectionHandle,
                              &statementHandle);
  if ((return_value != SQL_SUCCESS) && 
      (return_value != SQL_SUCCESS_WITH_INFO)){
    throw Exception(std::string("Error allocating statement handle"));
  } 
  this->connectionHandle = connectionHandle;
}

SimpleDB::Query::~Query() {
  SQLFreeHandle(SQL_HANDLE_STMT,statementHandle);
}

void SimpleDB::Query::bind(Column* columns[], int numberColumns) {
  for(int i = 0; i < numberColumns; i++) {
    columns[i]->bind(statementHandle, i+1);
  }
}

void SimpleDB::Query::execute(const std::string& sqlQuery) {
  
  /*
   * Close any open cursors before executing this query
   */
  SQLCloseCursor(statementHandle);

  long return_value;
  return_value=SQLExecDirect(statementHandle,
                             (SQLCHAR *)sqlQuery.c_str(),
                             SQL_NTS);
  if ((return_value != SQL_SUCCESS) && 
      (return_value != SQL_SUCCESS_WITH_INFO)){
    char sqlState[10];
    SQLSMALLINT V_OD_mlen;
    SQLINTEGER V_OD_err;
    char messageBuffer[400];
    std::string errorMessage = "Error ExecDirect: ";

    switch(return_value) {
    case SQL_NEED_DATA:
      errorMessage += "SQL need data";
      break;
    case SQL_STILL_EXECUTING:
      errorMessage += "SQL still executing";
      break;
    case SQL_ERROR:
      errorMessage += "SQL error: ";
      if(SQLGetDiagRec(SQL_HANDLE_STMT, statementHandle,1,
		       (SQLCHAR *)sqlState,&V_OD_err,
		       (SQLCHAR *) messageBuffer,
		       400,&V_OD_mlen) == SQL_SUCCESS) {
	
	errorMessage += sqlState;
	errorMessage += ": ";
	errorMessage += messageBuffer;
      } else
	errorMessage += "Unknown error";
      break;
    case SQL_INVALID_HANDLE:
      errorMessage += "SQL invalid handle";
      break;
    default:
      errorMessage += "Unknown error";
    }

    SQLFreeHandle(SQL_HANDLE_STMT,statementHandle);
    
    /* 
     * The NoDataException is thrown here because if it
     * was thrown in the above swtich then it would cause
     * Memory to be leaned : SQLFreeHandle() call
     */
    if (return_value == SQL_NO_DATA) {
      throw SimpleDB::Database::NoDataException();
    } 
    throw Exception(errorMessage);
  }
}

bool SimpleDB::Query::fetchRow() {
  if(SQLFetch(statementHandle)== SQL_NO_DATA)
    return false;
  else
    return true;
}
