#!/usr/bin/env ruby
#
# Samizdat RDF storage tests
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'test/unit'
require 'samizdat/engine'

include Samizdat

class TC_Storage < Test::Unit::TestCase

  def setup
    @ns = config['ns']
  end

  def test_query_select
    # initialize
    query_text = %{
SELECT ?msg, ?title, ?name, ?date, ?rating
WHERE (dc::title ?msg ?title)
      (dc::creator ?msg ?creator)
      (s::fullName ?creator ?name)
      (dc::date ?msg ?date)
      (rdf::subject ?stmt ?msg)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt s::Quality)
      (s::rating ?stmt ?rating)
LITERAL ?rating >= -1
ORDER BY ?rating DESC
USING PRESET NS}
    begin
      query = SquishQuery.new(query_text, @ns)
    rescue
      assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
    end

    # query parser
    assert_equal 'SELECT', query.key
    assert_equal :select, query.type
    assert_equal %w[?msg ?title ?name ?date ?rating], query.nodes
    assert query.pattern.include?(["#{@ns['dc']}title", "?msg", "?title", nil])
    assert_equal '?rating >= -1', query.literal
    assert_equal '?rating', query.order
    assert_equal 'DESC', query.order_dir
    assert_equal @ns['s'], query.ns['s']

    # transform_pattern
    sm = query.sql_mapper
    map = sm.clauses[0][:object]
    assert_equal config['map']['dc::title'], { map[:table] => map[:field] }
    assert_equal [ { :clause => 0, :role => :object } ], sm.nodes['?title'][:positions]
    # todo: check jc
    assert sm.nodes['?title'][:bindings].first

    # query result
    begin
      sql1 = rdf.select(query)
    rescue
      assert false, "select with pre-parsed query raised #{$!.class}: #{$!}"
    end
    begin
      sql2 = rdf.select(query_text)
    rescue
      assert false, "select with query text raised #{$!.class}: #{$!}"
    end
    assert sql1 == sql2
  end

  def test_query_assert
    # initialize
    query_text = %{
INSERT ?msg
UPDATE ?title = 'Test Message', ?content = 'Some text.'
WHERE (dc::creator ?msg 1)
      (dc::title ?msg ?title)
      (s::content ?msg ?content)
USING dc FOR #{@ns['dc']}
      s FOR #{@ns['s']}}
    begin
      query = SquishQuery.new(query_text)
    rescue
      assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
    end

    # query parser
    assert_equal 'INSERT', query.key
    assert_equal :assert, query.type
    assert_equal [['?msg'], {'?title' => "'0'", '?content' => "'1'"}],
      query.nodes
    assert query.pattern.include?(["#{@ns['dc']}title", "?msg", "?title", nil])
    assert_equal @ns['s'], query.ns['s']
    assert_equal "'Test Message'", query.substitute_literals("'0'")
    assert_equal "'Some text.'", query.substitute_literals("'1'")

    # transform_pattern
    sm = query.sql_mapper
    map = sm.clauses[0][:object]
    assert_equal config['map']['dc::creator'], { map[:table] => map[:field] }
    assert_equal [ { :clause => 0, :role => :object } ], sm.nodes['1'][:positions]
    # todo: check jc
    assert sm.nodes['1'][:bindings].first

    # todo: check assert
  end

  def test_dangling_blank_node
    # initialize
    query_text = %{
SELECT ?msg
WHERE (s::inReplyTo ?msg ?parent)
USING s FOR #{@ns['s']}}
    begin
      query = SquishQuery.new(query_text)
    rescue
      assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
    end

    # query parser
    assert_equal 'SELECT', query.key
    assert_equal :select, query.type
    assert_equal %w[?msg], query.nodes
    assert query.pattern.include?(["#{@ns['s']}inReplyTo", "?msg", "?parent", nil])
    assert_equal @ns['s'], query.ns['s']

    # transform_pattern
    sm = query.sql_mapper
    map = sm.clauses[0][:object]
    assert_equal config['map']['s::inReplyTo'], { map[:table] => map[:field] }
    assert_equal [ { :clause => 0, :role => :object } ], sm.nodes['?parent'][:positions]
    # todo: check jc
    assert sm.nodes['?parent'][:bindings].first

    # dangling blank node
    assert sm.where =~ /\b#{sm.nodes['?parent'][:bindings].first} IS NOT NULL\b/

    # query result
    begin
      sql1 = rdf.select(query)
    rescue
      assert false, "select with pre-parsed query raised #{$!.class}: #{$!}"
    end
    begin
      sql2 = rdf.select(query_text)
    rescue
      assert false, "select with query text raised #{$!.class}: #{$!}"
    end
    assert sql1 == sql2
  end

  def test_external_resource_no_self_join
    # initialize
    query_text = %{SELECT ?id WHERE (s::id focus::Translation ?id)}
    begin
      query = SquishQuery.new(query_text, @ns)
    rescue
      assert false, "SquishQuery initialization raised #{$!.class}: #{$!}"
    end

    # query parser
    assert_equal 'SELECT', query.key
    assert_equal :select, query.type
    assert_equal %w[?id], query.nodes
    assert query.pattern.include?(["#{@ns['s']}id", "#{@ns['focus']}Translation", "?id", nil])
    assert_equal @ns['s'], query.ns['s']

    # transform_pattern
    sm = query.sql_mapper
    map = sm.clauses[0][:object]
    assert_equal config['map']['s::id'], { map[:table] => map[:field] }
    assert_equal [ { :clause => 0, :role => :object } ], sm.nodes['?id'][:positions]
    assert sm.nodes['?id'][:bindings].first

    # no self join
    assert_equal false, sm.from.include?('JOIN')

    # query result
    begin
      sql1 = rdf.select(query)
    rescue
      assert false, "select with pre-parsed query raised #{$!.class}: #{$!}"
    end
    begin
      sql2 = rdf.select(query_text)
    rescue
      assert false, "select with query text raised #{$!.class}: #{$!}"
    end
    assert sql1 == sql2
  end

  #def test_internal_resource
  #end

  #def test_external_subject_internal_property
  #end
end
