require 'active_support/core_ext/hash/keys'
require 'active_support/core_ext/class/attribute'
require 'active_support/deprecation'

module ActiveModel
  class MissingAttributeError < NoMethodError
  end
  # == Active Model Attribute Methods
  #
  # <tt>ActiveModel::AttributeMethods</tt> provides a way to add prefixes and suffixes
  # to your methods as well as handling the creation of Active Record like class methods
  # such as +table_name+.
  #
  # The requirements to implement ActiveModel::AttributeMethods are to:
  #
  # * <tt>include ActiveModel::AttributeMethods</tt> in your object
  # * Call each Attribute Method module method you want to add, such as
  #   attribute_method_suffix or attribute_method_prefix
  # * Call <tt>define_attribute_methods</tt> after the other methods are
  #   called.
  # * Define the various generic +_attribute+ methods that you have declared
  #
  # A minimal implementation could be:
  #
  #   class Person
  #     include ActiveModel::AttributeMethods
  #
  #     attribute_method_affix  :prefix => 'reset_', :suffix => '_to_default!'
  #     attribute_method_suffix '_contrived?'
  #     attribute_method_prefix 'clear_'
  #     define_attribute_methods ['name']
  #
  #     attr_accessor :name
  #
  #     private
  #
  #     def attribute_contrived?(attr)
  #       true
  #     end
  #
  #     def clear_attribute(attr)
  #       send("#{attr}=", nil)
  #     end
  #
  #     def reset_attribute_to_default!(attr)
  #       send("#{attr}=", "Default Name")
  #     end
  #   end
  #
  # Note that whenever you include ActiveModel::AttributeMethods in your class,
  # it requires you to implement an <tt>attributes</tt> method which returns a hash
  # with each attribute name in your model as hash key and the attribute value as
  # hash value.
  #
  # Hash keys must be strings.
  #
  module AttributeMethods
    extend ActiveSupport::Concern

    NAME_COMPILABLE_REGEXP = /\A[a-zA-Z_]\w*[!?=]?\z/
    CALL_COMPILABLE_REGEXP = /\A[a-zA-Z_]\w*[!?]?\z/

    included do
      class_attribute :attribute_method_matchers, :instance_writer => false
      self.attribute_method_matchers = [ClassMethods::AttributeMethodMatcher.new]
    end

    module ClassMethods
      def define_attr_method(name, value=nil, deprecation_warning = true, &block) #:nodoc:
        # This deprecation_warning param is for internal use so that we can silence
        # the warning from Active Record, because we are implementing more specific
        # messages there instead.
        #
        # It doesn't apply to the original_#{name} method as we want to warn if
        # people are calling that regardless.
        if deprecation_warning
          ActiveSupport::Deprecation.warn("define_attr_method is deprecated and will be removed without replacement.")
        end

        sing = singleton_class
        sing.class_eval <<-eorb, __FILE__, __LINE__ + 1
          remove_possible_method :'original_#{name}'
          remove_possible_method :'_original_#{name}'
          alias_method :'_original_#{name}', :'#{name}'
          define_method :'original_#{name}' do
            ActiveSupport::Deprecation.warn(
              "This method is generated by ActiveModel::AttributeMethods::ClassMethods#define_attr_method, " \
              "which is deprecated and will be removed."
            )
            send(:'_original_#{name}')
          end
        eorb
        if block_given?
          sing.send :define_method, name, &block
        else
          # If we can compile the method name, do it. Otherwise use define_method.
          # This is an important *optimization*, please don't change it. define_method
          # has slower dispatch and consumes more memory.
          if name =~ NAME_COMPILABLE_REGEXP
            sing.class_eval <<-RUBY, __FILE__, __LINE__ + 1
              def #{name}; #{value.nil? ? 'nil' : value.to_s.inspect}; end
            RUBY
          else
            value = value.to_s if value
            sing.send(:define_method, name) { value }
          end
        end
      end

      # Declares a method available for all attributes with the given prefix.
      # Uses +method_missing+ and <tt>respond_to?</tt> to rewrite the method.
      #
      #   #{prefix}#{attr}(*args, &block)
      #
      # to
      #
      #   #{prefix}attribute(#{attr}, *args, &block)
      #
      # An instance method <tt>#{prefix}attribute</tt> must exist and accept
      # at least the +attr+ argument.
      #
      # For example:
      #
      #   class Person
      #
      #     include ActiveModel::AttributeMethods
      #     attr_accessor :name
      #     attribute_method_prefix 'clear_'
      #     define_attribute_methods [:name]
      #
      #     private
      #
      #     def clear_attribute(attr)
      #       send("#{attr}=", nil)
      #     end
      #   end
      #
      #   person = Person.new
      #   person.name = "Bob"
      #   person.name          # => "Bob"
      #   person.clear_name
      #   person.name          # => nil
      def attribute_method_prefix(*prefixes)
        self.attribute_method_matchers += prefixes.map { |prefix| AttributeMethodMatcher.new :prefix => prefix }
        undefine_attribute_methods
      end

      # Declares a method available for all attributes with the given suffix.
      # Uses +method_missing+ and <tt>respond_to?</tt> to rewrite the method.
      #
      #   #{attr}#{suffix}(*args, &block)
      #
      # to
      #
      #   attribute#{suffix}(#{attr}, *args, &block)
      #
      # An <tt>attribute#{suffix}</tt> instance method must exist and accept at least
      # the +attr+ argument.
      #
      # For example:
      #
      #   class Person
      #
      #     include ActiveModel::AttributeMethods
      #     attr_accessor :name
      #     attribute_method_suffix '_short?'
      #     define_attribute_methods [:name]
      #
      #     private
      #
      #     def attribute_short?(attr)
      #       send(attr).length < 5
      #     end
      #   end
      #
      #   person = Person.new
      #   person.name = "Bob"
      #   person.name          # => "Bob"
      #   person.name_short?   # => true
      def attribute_method_suffix(*suffixes)
        self.attribute_method_matchers += suffixes.map { |suffix| AttributeMethodMatcher.new :suffix => suffix }
        undefine_attribute_methods
      end

      # Declares a method available for all attributes with the given prefix
      # and suffix. Uses +method_missing+ and <tt>respond_to?</tt> to rewrite
      # the method.
      #
      #   #{prefix}#{attr}#{suffix}(*args, &block)
      #
      # to
      #
      #   #{prefix}attribute#{suffix}(#{attr}, *args, &block)
      #
      # An <tt>#{prefix}attribute#{suffix}</tt> instance method must exist and
      # accept at least the +attr+ argument.
      #
      # For example:
      #
      #   class Person
      #
      #     include ActiveModel::AttributeMethods
      #     attr_accessor :name
      #     attribute_method_affix :prefix => 'reset_', :suffix => '_to_default!'
      #     define_attribute_methods [:name]
      #
      #     private
      #
      #     def reset_attribute_to_default!(attr)
      #       ...
      #     end
      #   end
      #
      #   person = Person.new
      #   person.name                         # => 'Gem'
      #   person.reset_name_to_default!
      #   person.name                         # => 'Gemma'
      def attribute_method_affix(*affixes)
        self.attribute_method_matchers += affixes.map { |affix| AttributeMethodMatcher.new :prefix => affix[:prefix], :suffix => affix[:suffix] }
        undefine_attribute_methods
      end

      def alias_attribute(new_name, old_name)
        attribute_method_matchers.each do |matcher|
          matcher_new = matcher.method_name(new_name).to_s
          matcher_old = matcher.method_name(old_name).to_s
          define_optimized_call self, matcher_new, matcher_old
        end
      end

      # Declares the attributes that should be prefixed and suffixed by
      # ActiveModel::AttributeMethods.
      #
      # To use, pass in an array of attribute names (as strings or symbols),
      # be sure to declare +define_attribute_methods+ after you define any
      # prefix, suffix or affix methods, or they will not hook in.
      #
      #   class Person
      #
      #     include ActiveModel::AttributeMethods
      #     attr_accessor :name, :age, :address
      #     attribute_method_prefix 'clear_'
      #
      #     # Call to define_attribute_methods must appear after the
      #     # attribute_method_prefix, attribute_method_suffix or
      #     # attribute_method_affix declares.
      #     define_attribute_methods [:name, :age, :address]
      #
      #     private
      #
      #     def clear_attribute(attr)
      #       ...
      #     end
      #   end
      def define_attribute_methods(attr_names)
        attr_names.each { |attr_name| define_attribute_method(attr_name) }
      end

      def define_attribute_method(attr_name)
        attribute_method_matchers.each do |matcher|
          method_name = matcher.method_name(attr_name)

          unless instance_method_already_implemented?(method_name)
            generate_method = "define_method_#{matcher.method_missing_target}"

            if respond_to?(generate_method, true)
              send(generate_method, attr_name)
            else
              define_optimized_call generated_attribute_methods, method_name, matcher.method_missing_target, attr_name.to_s
            end
          end
        end
        attribute_method_matchers_cache.clear
      end

      # Removes all the previously dynamically defined methods from the class
      def undefine_attribute_methods
        generated_attribute_methods.module_eval do
          instance_methods.each { |m| undef_method(m) }
        end
        attribute_method_matchers_cache.clear
      end

      # Returns true if the attribute methods defined have been generated.
      def generated_attribute_methods #:nodoc:
        @generated_attribute_methods ||= begin
          mod = Module.new
          include mod
          mod
        end
      end

      protected
        def instance_method_already_implemented?(method_name)
          generated_attribute_methods.method_defined?(method_name)
        end

      private
        # The methods +method_missing+ and +respond_to?+ of this module are
        # invoked often in a typical rails, both of which invoke the method
        # +match_attribute_method?+. The latter method iterates through an
        # array doing regular expression matches, which results in a lot of
        # object creations. Most of the times it returns a +nil+ match. As the
        # match result is always the same given a +method_name+, this cache is
        # used to alleviate the GC, which ultimately also speeds up the app
        # significantly (in our case our test suite finishes 10% faster with
        # this cache).
        def attribute_method_matchers_cache #:nodoc:
          @attribute_method_matchers_cache ||= {}
        end

        def attribute_method_matcher(method_name) #:nodoc:
          if attribute_method_matchers_cache.key?(method_name)
            attribute_method_matchers_cache[method_name]
          else
            # Must try to match prefixes/suffixes first, or else the matcher with no prefix/suffix
            # will match every time.
            matchers = attribute_method_matchers.partition(&:plain?).reverse.flatten(1)
            match = nil
            matchers.detect { |method| match = method.match(method_name) }
            attribute_method_matchers_cache[method_name] = match
          end
        end

        # Define a method `name` in `mod` that dispatches to `send`
        # using the given `extra` args. This fallbacks `define_method`
        # and `send` if the given names cannot be compiled.
        def define_optimized_call(mod, name, send, *extra) #:nodoc:
          if name =~ NAME_COMPILABLE_REGEXP
            defn = "def #{name}(*args)"
          else
            defn = "define_method(:'#{name}') do |*args|"
          end

          extra = (extra.map(&:inspect) << "*args").join(", ")

          if send =~ CALL_COMPILABLE_REGEXP
            target = "#{send}(#{extra})"
          else
            target = "send(:'#{send}', #{extra})"
          end

          mod.module_eval <<-RUBY, __FILE__, __LINE__ + 1
            #{defn}
              #{target}
            end
          RUBY
        end

        class AttributeMethodMatcher
          attr_reader :prefix, :suffix, :method_missing_target

          AttributeMethodMatch = Struct.new(:target, :attr_name, :method_name)

          def initialize(options = {})
            options.symbolize_keys!

            if options[:prefix] == '' || options[:suffix] == ''
              ActiveSupport::Deprecation.warn(
                "Specifying an empty prefix/suffix for an attribute method is no longer " \
                "necessary. If the un-prefixed/suffixed version of the method has not been " \
                "defined when `define_attribute_methods` is called, it will be defined " \
                "automatically."
              )
            end

            @prefix, @suffix = options[:prefix] || '', options[:suffix] || ''
            @regex = /^(#{Regexp.escape(@prefix)})(.+?)(#{Regexp.escape(@suffix)})$/
            @method_missing_target = "#{@prefix}attribute#{@suffix}"
            @method_name = "#{prefix}%s#{suffix}"
          end

          def match(method_name)
            if @regex =~ method_name
              AttributeMethodMatch.new(method_missing_target, $2, method_name)
            else
              nil
            end
          end

          def method_name(attr_name)
            @method_name % attr_name
          end

          def plain?
            prefix.empty? && suffix.empty?
          end
        end
    end

    # Allows access to the object attributes, which are held in the
    # <tt>@attributes</tt> hash, as though they were first-class methods. So a
    # Person class with a name attribute can use Person#name and Person#name=
    # and never directly use the attributes hash -- except for multiple assigns
    # with ActiveRecord#attributes=. A Milestone class can also ask
    # Milestone#completed? to test that the completed attribute is not +nil+
    # or 0.
    #
    # It's also possible to instantiate related objects, so a Client class
    # belonging to the clients table with a +master_id+ foreign key can
    # instantiate master through Client#master.
    def method_missing(method, *args, &block)
      if respond_to_without_attributes?(method, true)
        super
      else
        match = match_attribute_method?(method.to_s)
        match ? attribute_missing(match, *args, &block) : super
      end
    end

    # attribute_missing is like method_missing, but for attributes. When method_missing is
    # called we check to see if there is a matching attribute method. If so, we call
    # attribute_missing to dispatch the attribute. This method can be overloaded to
    # customise the behaviour.
    def attribute_missing(match, *args, &block)
      __send__(match.target, match.attr_name, *args, &block)
    end

    # A Person object with a name attribute can ask <tt>person.respond_to?(:name)</tt>,
    # <tt>person.respond_to?(:name=)</tt>, and <tt>person.respond_to?(:name?)</tt>
    # which will all return +true+.
    alias :respond_to_without_attributes? :respond_to?
    def respond_to?(method, include_private_methods = false)
      if super
        true
      elsif !include_private_methods && super(method, true)
        # If we're here then we haven't found among non-private methods
        # but found among all methods. Which means that the given method is private.
        false
      else
        !match_attribute_method?(method.to_s).nil?
      end
    end

    protected
      def attribute_method?(attr_name)
        respond_to_without_attributes?(:attributes) && attributes.include?(attr_name)
      end

    private
      # Returns a struct representing the matching attribute method.
      # The struct's attributes are prefix, base and suffix.
      def match_attribute_method?(method_name)
        match = self.class.send(:attribute_method_matcher, method_name)
        match && attribute_method?(match.attr_name) ? match : nil
      end

      def missing_attribute(attr_name, stack)
        raise ActiveModel::MissingAttributeError, "missing attribute: #{attr_name}", stack
      end
  end
end
