/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: decreq.c,v 1.15 2000/08/16 10:08:58 mbp Exp $
 * 
 * Copyright (C) 2000 by Martin Pool <mbp@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* decreq.c -- stuff that happens when we decide to decode a request
 * for a client. */

#include "config.h"
#include "sysheaders.h"
#include "rproxy.h"

#include "cache.h"
#include "util.h"
#include "urllib.h"
#include "header.h"
#include "msgpage.h"
#include "rsglue.h"
#include "myname.h"
#include "error.h"
#include "decreq.h"
#include "trace.h"

int
will_decode_hsync(request_t const *req)
{
    if (req->resp_rsync_encoded) {
	if (req->status != HTTPX_DELTA) {
	    trace(LOGAREA_PROTO,
		  "oops!  response is encoded, "
		  "but status is %d rather than %d Delta",
		  req->status, HTTPX_DELTA);
	    return 0;
	} else if (req->client_does_hsync) {
	    trace(LOGAREA_PROTO,
		  "response is encoded but the client can handle itself");
	    return 0;
	} else {
	    return 1;
	}
    } else {
	trace(LOGAREA_PROTO, "response is not encoded");
	return 0;
    }
}


int
want_to_decode(request_t const *req)
{
    return !req->client_does_hsync &&
	!req->client_wants_direct &&
	(req->method_id == METHOD_GET || req->method_id == METHOD_POST);
    /* TODO: If there are request headers that make it likely that encoding
       will fail, then don't bother sending the header.  For example, if
       there's a Range request. */
}


static int
do_decode(request_t * req)
{
    hs_filebuf_t   *out_fb, *diff_fb, *newsig_fb;
    int             ret;

    assert(req->f_old_cache);

    /* TODO: Check we don't clash with stdio buffering.  Which one
     * should we use? */
    out_fb = hs_filebuf_from_fd(fileno(req->f_to_client));
    hs_filebuf_add_cache(out_fb, fileno(req->f_new_cache));
    diff_fb = hs_filebuf_from_fd(fileno(req->f_upstream));
    newsig_fb = hs_filebuf_from_fd(req->fd_new_sig);

    ret = hs_decode(fileno(req->f_old_cache),
		    hs_filebuf_write, out_fb,
		    hs_filebuf_read, diff_fb,
		    hs_filebuf_write, newsig_fb, &req->hs_stats);
    if (ret < 0) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
		       "error decoding response from upstream");
    }

    return ret;
}


int
decode_hsync_request(request_t * req)
{
    int             ret;

    req->action_str = "decode";
    rp_process_name("%s %s", req->action_str, req->url);

    if (!config.no_cache) {
	req->f_new_cache = cache_open_tmp(req->url);
	req->fd_new_sig = cache_open_sig_tmp(req->url);
    }

    header_remove(req->headers, "Content-Length");
    header_remove_list(req->headers, "Content-Encoding",
		       HSYNC_ENCODING_TOKEN);

    if (req->status == HTTPX_DELTA)
	set_request_status(req, HTTP_OK, "OK");

    req->state = sending_response_headers;
    rp_send_response_line(req);
    if (header_send(req->headers, req->f_to_client) < 0)
	return -1;

    req->state = sending_body;

    ret = do_decode(req);

    if (!config.no_cache) {
	if (ret >= 0)
	    cache_tmp_commit();

	fclose(req->f_new_cache);
	hs_file_close(req->fd_new_sig);
    }

    if (req->f_old_cache)
	fclose(req->f_old_cache);
    if (req->f_old_sig)
	fclose(req->f_old_sig);

    return 0;
}



/* Add the base64-encoded version of the old signature to the request
 * headers. */
int
add_old_sig_header(request_t * req)
{
    ssize_t         size;
    char           *b64, *raw;

    assert(req->f_old_sig);
    size = read_and_alloc(req->f_old_sig, &raw);
    if (size < 0) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
		       "couldn't read old signature file");
    }

    /* FIXME: We urgently need to make this split the signature into
     * several headers to keep the size reasonable. */

    b64 = xmalloc(size * 4 / 3 + 1);
    base64_encode(raw, size, b64);

    header_add(req->headers, SIGNATURE_HEADER, b64);
    /* XXX: header_add_split */
    free(raw);

    return 1;
}
