########################################################################
#
# File Name:            VariableElement.py
#
# Documentation:        http://docs.4suite.org/4XSLT/VariableElement.py.html
#
"""
Implementation of the XSLT Spec variable stylesheet element.
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import xml.dom.ext
import xml.xslt
from xml.xslt import XsltElement, XsltException, Error, XSL_NAMESPACE
from xml.xpath import CoreFunctions, Util

class VariableElement(XsltElement):
    legalAttrs = ('name', 'select')

    def __init__(self, doc, uri=xml.xslt.XSL_NAMESPACE, localName='variable', prefix='xsl', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)
        return

    def setup(self):
        self._nss = xml.dom.ext.GetAllNs(self)
        name_attr = self.getAttributeNS('', 'name')
        self._name = Util.ExpandQName(name_attr, namespaces=self._nss)
        self._select = self.getAttributeNS('', 'select')
        if self._select:
            self._expr = self.parseExpression(self._select)
        else:
            self._expr = None
        self._register()
        return

    def _register(self):
        node = self
        last_visible_var = None
        top = self.ownerDocument.documentElement
        while (not last_visible_var) and ((node.namespaceURI, node.localName) != (XSL_NAMESPACE, 'template')) and (node.parentNode != top):
            psib = node.previousSibling
            while psib and not last_visible_var:
                if psib.namespaceURI == XSL_NAMESPACE and psib.localName in ['param', 'variable']:
                    last_visible_var = psib
                psib = psib.previousSibling
            if (not last_visible_var) and (node != self):
                if node.namespaceURI == XSL_NAMESPACE and node.localName in ['param', 'variable']:
                    last_visible_var = node
            node = node.parentNode

        if last_visible_var:
            if self._name in last_visible_var._visibleLocalVarNames:
                raise XsltException(Error.ILLEGAL_SHADOWING, self._name)
            self._visibleLocalVarNames = last_visible_var._visibleLocalVarNames + [self._name]
        else:
            self._visibleLocalVarNames = [self._name]
                
    def instantiate(self, context, processor):
        #Note all we want to do is change the varBindings
        origState = context.copy()
        context.setNamespaces(self._nss)
        context.currentInstruction = self

        text = ''
        if self._select:
            result = self._expr.evaluate(context)
        else:
            processor.pushResult()
            try:
                for child in self.childNodes:
                    context = child.instantiate(context, processor)[0]
            finally:
                result = processor.popResult()
            context.rtfs.append(result)

        context.set(origState)
        #Because of the in-place mutability of dicts, we have to copy off
        #to avoid illegal side-effects
        new_bindings = context.varBindings.copy()
        new_bindings[self._name] = result
        context.varBindings = new_bindings

        return (context,)

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix, self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self._nss, self._name, self._select, self._expr)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self._nss = state[1]
        self._name = state[2]
        self._select = state[3]
        self._expr = state[4]
        return

