########################################################################
#
# File Name:            Stylesheet.py
#
# Documentation:        http://docs.4suite.org/4XSLT/Stylesheet.py.py.html
#
"""
Implement all the stylesheet internals
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import string, types
import urlparse, urllib
import xml.dom.ext
from xml import xpath, xslt
from xml.dom import Node
from xml.dom.NodeFilter import NodeFilter
from xml.xslt import XsltElement, XsltException, Error
from xml.xslt import XPatternParser, XsltContext
from xml.xslt import XSL_NAMESPACE, OutputParameters
from xml.xpath import CoreFunctions, Util, Conversions
from Ft.Lib import pDomlette


class PatternInfo:
    """Indexes into the tuple for pattern information"""
    PATTERN = 0
    AXIS_TYPE = 1
    PRIORITY = 2
    MODE = 3
    NSS = 4
    TEMPLATE = 5

SPECIAL_RE_CHARS = ['.', '^', '$', '*', '+', '?']


def MatchTree(patterns, context):
    '''Select all nodes from node on down that match the pattern'''
    matched = map(lambda x, c=context, n=context.node: [n]*x.match(c,n),
                  patterns)
    counter = 1
    size = len(context.node.childNodes)
    origState = context.copyNodePosSize()
    for child in context.node.childNodes:
        context.setNodePosSize((child, counter, size))
        map(lambda x, y: x.extend(y), matched, MatchTree(patterns, context))
        context.setNodePosSize(origState)
        counter = counter + 1
    if context.node.nodeType == Node.ELEMENT_NODE:
        counter = 1
        size = len(context.node.attributes)
        for attr in context.node.attributes.values():
            context.setNodePosSize((attr, counter, size))
            map(lambda x, y: x.extend(y),
                matched, MatchTree(patterns, context))
            context.setNodePosSize(origState)
            counter = counter + 1
    return matched


def TemplateCmp(t1, t2):
    if t1[PatternInfo.TEMPLATE].importIndex == t2[PatternInfo.TEMPLATE].importIndex:
        return cmp(t2[PatternInfo.PRIORITY], t1[PatternInfo.PRIORITY])
    else:
        return cmp(t1[PatternInfo.TEMPLATE].importIndex, t2[PatternInfo.TEMPLATE].importIndex)


class StylesheetElement(XsltElement):
    legalAttrs = ('id', 'extension-element-prefixes',
                  'exclude-result-prefixes', 'version')

    def __init__(self, doc, uri=xml.xslt.XSL_NAMESPACE, localName='stylesheet',
                 prefix='xsl', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)
        self._imports = []
        self.extensionNss = []
        self._primedContext = None
        self._lres = []
        self.importingSty = None
        self._patterns = {}
        self._importedTemplates = []
        return

    def _updateKeys(self, doc, processor):
        processor.keys[doc] = {}
        context = XsltContext.XsltContext(doc, 1, 1,
                                          processorNss=self.namespaces,
                                          processor=processor)
        patterns = map(lambda x: x[1], self._kelems)
        if not patterns:
            return
        match_lists = MatchTree(patterns, context)
        ctr = 0
        for (name, match_pattern, use_expr) in self._kelems:
            match_list = match_lists[ctr]
            if not processor.keys[doc].has_key(name):
                processor.keys[doc][name] = {}
            for node in match_list:
                #context.stylesheet = self
                origState = context.copy()
                context.node = node
                key_value_list = use_expr.evaluate(context)
                #NOTE: use attrs can't contain var refs, so result can't
                #be RTF So use CoreFunc StringValue, not ExtFunc version
                if type(key_value_list) != type([]):
                    key_value_list = [key_value_list]
                for obj in key_value_list:
                    keystr = Conversions.StringValue(obj)
                    if not processor.keys[doc][name].has_key(keystr):
                        processor.keys[doc][name][keystr] = []
                    processor.keys[doc][name][keystr].append(node)
                context.set(origState)
            ctr = ctr + 1
        context.release()
        return

    def setup(self):
        '''
        Called only once, at the first initialization
        '''
        if self.parentNode != self.ownerDocument:
            raise XsltException(Error.XSL_STYLESHEET_NOT_DOCELEM)
        self.namespaces = xml.dom.ext.GetAllNs(self)
        self.spaceRules = {}
        self._topLevelVarNodes = {}
        self.namespaceAliases = {}
        self.decimalFormats = {'': ('.', ',', 'Infinity', '-', 'NaN', '%', '?', '0', '#', ';')}
        self.outputParams = OutputParameters()
        excluded_prefixes = self.getAttributeNS('', 'exclude-result-prefixes')
        self.excludedNss = []
        if excluded_prefixes:
            excluded_prefixes = string.splitfields(excluded_prefixes)
            for prefix in excluded_prefixes:
                if prefix == '#default': prefix = ''
                self.excludedNss.append(self.namespaces[prefix])
        self._setupNamespaceAliases()
        self._setupChildNodes()
        self._setupDecimalFormats()
        self._setupWhitespaceRules()
        self._setupOutput()
        self._setupTemplates()
        self._setupKeys()
        self._setupTopLevelVarParams()
        return

    def _setupNamespaceAliases(self):
        #Namespace aliases
        ns_aliases = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) == (XSL_NAMESPACE, 'namespace-alias'), self.childNodes)
        for nsa in ns_aliases:
            stylesheet_prefix = nsa.getAttributeNS('', 'stylesheet-prefix')
            result_prefix = nsa.getAttributeNS('', 'result-prefix')
            if not (stylesheet_prefix and result_prefix):
                raise XsltException(Error.INVALID_NAMESPACE_ALIAS)
            if stylesheet_prefix == '#default':
                stylesheet_prefix == ''
            if result_prefix == '#default':
                result_prefix == ''
            sty_ns = self.namespaces[stylesheet_prefix]
            res_ns = self.namespaces[result_prefix]
            update = 1
            if self.namespaceAliases.has_key(sty_ns):
                if self.namespaceAliases[sty_ns].importIndex > nsa.importIndex:
                    update = 0
                elif self.namespaceAliases[sty_ns].importIndex == nsa.importIndex:
                    raise XsltException(Error.DUPLICATE_NAMESPACE_ALIAS, sty_ns)
            if update:
                self.namespaceAliases[sty_ns] = (res_ns, stylesheet_prefix,
                                                 result_prefix,
                                                 nsa.importIndex)
        return

    def _setupChildNodes(self):
        snit = self.ownerDocument.createNodeIterator(
            self, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT, None, 0
            )
        curr_node = snit.nextNode()
        curr_node = snit.nextNode()
        while curr_node:
            try:
                curr_node.setup()
            except (xpath.RuntimeException,
                    xpath.CompiletimeException,
                    XsltException), error:
                #import traceback
                #traceback.print_exc(1000)
                if not hasattr(error, 'stylesheetUri'):
                    error.stylesheetUri = curr_node.baseUri
                raise error
            curr_node = snit.nextNode()
        return

    def _setupDecimalFormats(self):
        #FIXME: clash rules for decimal-format suck, so it's not fully conformant for now.  See XSLT spec 12.3
        dec_formats = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) == (XSL_NAMESPACE, 'decimal-format'), self.childNodes)
        for dc in dec_formats:
            format_settings = (
                (dc.getAttributeNS('', 'decimal-separator') or '.'),
                (dc.getAttributeNS('', 'grouping-separator') or ','),
                (dc.getAttributeNS('', 'infinity') or 'Infinity'),
                (dc.getAttributeNS('', 'minus-sign') or '-'),
                (dc.getAttributeNS('', 'NaN') or 'NaN'),
                (dc.getAttributeNS('', 'percent') or '%'),
                (dc.getAttributeNS('', 'per-mille') or '?'),
                (dc.getAttributeNS('', 'zero-digit') or '0'),
                (dc.getAttributeNS('', 'digit') or '#'),
                (dc.getAttributeNS('', 'pattern-separator') or ';')
                )
            nfs = []
            for fc in format_settings:
                if fc in SPECIAL_RE_CHARS:
                    nfs.append('\\'+fc)
                else:
                    nfs.append(fc)
            name = dc.getAttributeNS('', 'name')
            name = name and Util.ExpandQName(name, dc)
            self.decimalFormats[name] = tuple(nfs)
        return

    def _setupWhitespaceRules(self):
        #Whitespace rules
        space_rules = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) in [(XSL_NAMESPACE, 'preserve-space'), (XSL_NAMESPACE, 'strip-space')], self.childNodes)
        for sr in space_rules:
            args = string.splitfields(sr.getAttributeNS('', 'elements'))
            for an_arg in args:
                #FIXME: watch out!  ExpandQName doesn't handle ns defaulting
                split_name = Util.ExpandQName(an_arg, sr)
                #FIXME: Add support for import precedence checks by remembering the import depth in the self.spaceRules dict
                self.spaceRules[split_name] = string.splitfields(sr.localName, '-')[0]
        return

    def _setupOutput(self):
        #Output
        #FIXME: add import precedence rules
        output = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) == (XSL_NAMESPACE, 'output'), self.childNodes)
        for out in output:
            self.outputParams.parse(out, self.namespaces)
        return

    def _setupTemplates(self):
        templates = filter(lambda n:
                           n.nodeType == Node.ELEMENT_NODE and
                           (n.namespaceURI, n.localName) == (XSL_NAMESPACE, 'template'),
                           self.childNodes)

        # Preprocess all of the templates with names
        # NOTE: the template name is a tuple of (namespaceURI, localName)
        named_tpls = filter(lambda t: t._name[1] != '', templates)
        named_tpls.sort(lambda a, b: cmp(a.importIndex, b.importIndex))
        self._call_templates = tpls = {}
        for t in named_tpls:
            if not tpls.has_key(t._name):
                tpls[t._name] = t

        #Preprocess the patterns from all templates
        patterns = []
        for tpl in templates:
            (patternInfo, mode, nss) = tpl.getMatchInfo()
            for pi in patternInfo:
                patterns.append(pi+(mode, nss, tpl))

        patterns.reverse()
        patterns.sort(TemplateCmp)
        self._patterns = patts = {}
        for p in patterns:
            m = p[PatternInfo.MODE]
            if not patts.has_key(m):
                patts[m] = []
            patts[m].append(p)
        return

    def _setupKeys(self):
        self._kelems = []
        kelems = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) == (XSL_NAMESPACE, 'key'), self.childNodes)
        for kelem in kelems:
            name = Util.ExpandQName(kelem.getAttributeNS('', 'name'), kelem)
            match = kelem.getAttributeNS('', 'match')
            match_pattern = self.parsePattern(match)
            use = kelem.getAttributeNS('', 'use')
            use_expr = self.parseExpression(use)
            self._kelems.append((name, match_pattern, use_expr))

        self.reset()
        return

    def _setupTopLevelVarParams(self):
        #Is there a more efficient way to zip two sequences into a dict?
        vars = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and x.namespaceURI == XSL_NAMESPACE and x.localName == 'variable', self.childNodes)
        self._topVariables = ({},[])
        for var in vars:
            update = 1
            if self._topVariables[0].has_key(var._name):
                if var.importIndex > self._topVariables[0][var._name].importIndex:
                    update = 0
                elif var.importIndex == self._topVariables[0][var._name].importIndex:
                    raise XsltException(Error.DUPLICATE_TOP_LEVEL_VAR, var._name)
            if update:
                self._topVariables[0][var._name] = var
                self._topVariables[1].append(var)

        params = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and x.namespaceURI == XSL_NAMESPACE and x.localName == 'param', self.childNodes)
        for param in params:
            update = 1
            if self._topVariables[0].has_key(param._name):
                if param.importIndex > self._topVariables[0][param._name].importIndex:
                    update = 0
                elif param.importIndex == self._topVariables[0][param._name].importIndex:
                    raise XsltException(Error.DUPLICATE_TOP_LEVEL_VAR, param._name)
            if update:
                self._topVariables[0][param._name] = param
                self._topVariables[1].append(param)
        return

    def newSource(self, doc, processor):
        """
        Called whenever there's a new source document registed to the processor
        """
        self._updateKeys(doc, processor)
        return

    def reset(self):
        """
        Called whenever the processor is reset, i.e. after each run
        """
        if self._primedContext:
            self._primedContext.release()
            self._primedContext = None
        return

    def _computeVar(self, vname, context, processed, deferred,
                    overriddenParams, topLevelParams, processor):
        vnode = self._topVariables[0][vname]
        if vnode in deferred:
            raise XsltException(Error.CIRCULAR_VAR, vname[0], vname[1])
        if vnode in processed:
            return
        if vnode.localName[0] == 'p':
            if overriddenParams.has_key(vname):
                context.varBindings[vname] = overriddenParams[vname]
            else:
                try:
                    context = vnode.instantiate(context, processor)[0]
                except xpath.RuntimeException, e:
                    if e.errorCode == xpath.RuntimeException.UNDEFINED_VARIABLE:
                        deferred.append(vnode)
                        self._computeVar((e.params[0], e.params[1]), context,
                                         processed, deferred, overriddenParams,
                                         topLevelParams, processor)
                        deferred.remove(vnode)
                        context = vnode.instantiate(context, processor)[0]
                    else:
                        raise e
                #Set up so that later stylesheets will get overridden by
                #parameter values set in higher-priority stylesheets
                topLevelParams[vname] = context.varBindings[vname]
        else:
            try:
                context = vnode.instantiate(context, processor)[0]
            except xpath.RuntimeException, e:
                if e.errorCode == xpath.RuntimeException.UNDEFINED_VARIABLE:
                    deferred.append(vnode)
                    self._computeVar((e.params[0], e.params[1]), context,
                                     processed, deferred, overriddenParams,
                                     topLevelParams, processor)
                    deferred.remove(vnode)
                    context = vnode.instantiate(context, processor)[0]
                else:
                    raise e
        processed.append(vnode)
        return

    def idle(self, contextNode, processor, baseUri=None):
        snit = self.ownerDocument.createNodeIterator(self, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT, None,0)
        curr_node = snit.nextNode()
        curr_node = snit.nextNode()
        while curr_node:
            curr_node.idle(processor)
            curr_node = snit.nextNode()
        return

    def prime(self, contextNode, processor, topLevelParams, baseUri=None):
        doc = contextNode.ownerDocument or contextNode
        self.newSource(doc, processor)
        context = XsltContext.XsltContext(
            contextNode.ownerDocument or contextNode, 1, 1,
            processorNss=self.namespaces, stylesheet=self,
            processor=processor
            )
        baseUri = baseUri or getattr(context.node, 'refUri', None)
        if baseUri: context.documents[baseUri] = context.node
        self._primedContext = context
        self._docReader = processor._docReader

        overridden_params = {}
        for k in topLevelParams.keys():
            if type(k) != types.TupleType:
                try:
                    split_name = Util.ExpandQName(k, namespaces=context.processorNss)
                except KeyError:
                    continue
            else:
                split_name = k
            overridden_params[split_name] = topLevelParams[k]
        for vnode in self._topVariables[1]:
            self._computeVar(vnode._name, context, [], [], overridden_params,
                             topLevelParams, processor)
        self._primedContext = context
        #Note: key expressions can't have var refs, so we needn't worry about imports
        self._updateKeys(contextNode, processor)

        #Attribute sets
        attribute_sets = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and (x.namespaceURI, x.localName) == (XSL_NAMESPACE, 'attribute-set'), self.childNodes)
        for as in attribute_sets:
            as.instantiate(context, processor)

        snit = self.ownerDocument.createNodeIterator(self, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT, None,0)
        curr_node = snit.nextNode()
        curr_node = snit.nextNode()
        while curr_node:
            curr_node.prime(processor, self._primedContext)
            curr_node = snit.nextNode()
        return topLevelParams


    ########################## Run-time methods ########################
    
    def getNamedTemplates(self):
        templates = {}
        for name,tpl in self._call_templates.items():
            templates[name] = (self, tpl)
        return templates

    def getTopLevelVariables(self):
        return self._primedContext.varBindings.copy()

    def applyTemplates(self, context, mode, processor, params=None):
        params = params or {}
        origState = context.copyStylesheet()
        context.setStylesheet((self._primedContext.varBindings,
                               self.namespaces, self))
        #Set the current node for this template application
        context.currentNode = context.node

        matched = 0
        for patternInfo in self._patterns.get(mode, []):
            context.processorNss = patternInfo[PatternInfo.NSS]
            pattern = patternInfo[PatternInfo.PATTERN]
            template = patternInfo[PatternInfo.TEMPLATE]
            if pattern.match(context, context.node, patternInfo[PatternInfo.AXIS_TYPE]):
                template.instantiate(context, processor, params)
                matched = 1
                break
        context.setStylesheet(origState)
        return matched

    def applyImports(self, context, mode, processor, params=None):
        #FIXME: Actually, in XSLT 1.0 apply-imports can't have params
        params = params or {}
        pat_info_list = self._patterns.get(mode, [])
        #FIXME: very slow.  Cache the appropriate patterns in stylesheet setup
        pat_info_list = filter(lambda x, s=self: x[PatternInfo.TEMPLATE] in s._importedTemplates,
                               pat_info_list)
        matched = 0
        for patternInfo in pat_info_list:
            context.processorNss = patternInfo[PatternInfo.NSS]
            pattern = patternInfo[PatternInfo.PATTERN]
            if pattern.match(context, context.node, patternInfo[PatternInfo.AXIS_TYPE]):
                template = patternInfo[PatternInfo.TEMPLATE]
                patternInfo[PatternInfo.TEMPLATE].instantiate(context, processor, params)
                matched = 1
                break
        if not matched:
            for imported in self._imports:
                if imported.applyImports(context, mode, processor, params):
                    matched = 1
                    break
        return matched

    def reclaim(self):
        self.__dict__['_primedContext'] = None
        self.importingSty = None
        for imp in self._imports:
            imp.reclaim()
            #pDomlette.ReleaseNode(imp)
        self._imports = []
        self.namespaces = xml.dom.ext.GetAllNs(self)
        self.spaceRules = {}
        self.namespaceAliases = ({}, {})
        self.decimalFormats = {'': ('.', ',', 'Infinity', '-', 'NaN', '%', '?', '0', '#', ';')}
        self.outputParams = OutputParameters()
        excluded_prefixes = self.getAttributeNS('', 'exclude-result-prefixes')
        self.excludedNss = []
        self._lres = []
        for key, patternInfo in self._patterns.items():
            try:
                patternInfo[PatternInfo.TEMPLATE].stylesheet = None
            except (IndexError, TypeError):
                pass
        return

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self.namespaces, self.spaceRules,
                      self.namespaceAliases, self.decimalFormats,
                      self.outputParams, self.excludedNss,
                      self._patterns, self._call_templates,
                      self._kelems, self.extensionNss,
                      self._topVariables, self._importedTemplates,
                      self.importingSty)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self.namespaces = state[1]
        self.spaceRules = state[2]
        self.namespaceAliases = state[3]
        self.decimalFormats = state[4]
        self.outputParams = state[5]
        self.excludedNss = state[6]
        self._patterns = state[7]
        self._call_templates = state[8]
        self._kelems = state[9]
        self.extensionNss = state[10]
        self._topVariables = state[11]
        self._importedTemplates = state[12]
        self.importingSty = state[13]
        return

