#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "string_pool.h"

/* should always be a prime number */
#define MAX_BUCKETS 511

unsigned hash_key(const char *key)
{
  unsigned hash = 0;
  char *p = (char *)key;
  int i;

  while (*p) {
    i = (int)*p;
    hash ^= i;
    hash <<= 1;
    p++;
  }
  return hash % MAX_BUCKETS;
}


StringPool *string_pool_create(void)
{
  int i;
  /* By using calloc() everything is initialized already */
  StringPool *pool = (StringPool *)calloc(1, sizeof(StringPool));

  /* create the buckets */
  pool->buckets = (PoolBucket **)malloc(MAX_BUCKETS*sizeof(PoolBucket));
  for (i = 0; i < MAX_BUCKETS; i++)
    pool->buckets[i] = (PoolBucket *)calloc(1, sizeof(PoolBucket));
  return pool;
}


void string_pool_destroy(StringPool *pool)
{
  int i;
  PoolElement *element;

  for (i = 0; i < MAX_BUCKETS; i++) {
    for (element = pool->buckets[i]->head; element; element = element->next) {
      Py_DECREF(element->data);
    }
    free(pool->buckets[i]);
  }
  free(pool->buckets);
  free(pool);
}


/* returns the actual element added, or NULL on failure */
PoolElement *string_pool_insert(StringPool *pool, const char *key, const PyObject *data)
{
  PoolBucket *bucket;
  PoolElement *element = (PoolElement *)malloc(sizeof(PoolElement));

  if (!element) return NULL;

  element->key = (char *)key;
  element->data = (PyObject *)data;

  /* get the bucket for this item */
  bucket = pool->buckets[hash_key(key)];

  /* insert it at the beginning */
  if (bucket->size == 0) bucket->tail = element;
  element->next = bucket->head;
  bucket->head = element;
  bucket->size++;

  pool->size++;
  return element;
}


PyObject *string_pool_lookup(StringPool *pool, const char *key)
{
  PoolElement *element;
  unsigned hash = hash_key(key);

  /* search the bucket for an element with this key */
  for (element = pool->buckets[hash]->head; element; element = element->next) {
    if (!strcmp(key, element->key)) {
      return element->data;
    }
  }
  return NULL;
}


PyObject *pystring_from_pool(StringPool *pool, const char *cstr) {
  PyObject *pstr = string_pool_lookup(pool, cstr);
  if (!pstr) {
    pstr = PyString_FromString(cstr);
    /* Use the PyString data pointer for the key to ensure proper
     * memory allocation and cleanup.*/
    string_pool_insert(pool, PyString_AS_STRING(pstr), pstr);
  }
  Py_INCREF(pstr);
  return pstr;
}


