/*
    UserPipePreferences.m

    Implementation of the UserPipePreferences class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "UserPipePreferences.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSUserDefaults.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSTableView.h>
#import <AppKit/NSTableColumn.h>
#import <AppKit/NSNibLoading.h>

static inline void
SetDefaultUserPipes(NSDictionary * userPipes)
{
  [[NSUserDefaults standardUserDefaults]
    setObject: userPipes forKey: @"UserPipes"];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: NSUserDefaultsDidChangeNotification
                  object: [NSUserDefaults standardUserDefaults]];
}

@implementation UserPipePreferences

+ (NSString *) name
{
  return _(@"User Defined Pipes");
}

- (void) dealloc
{
  TEST_RELEASE(view);

  [super dealloc];
}

- (void) awakeFromNib
{
  [view retain];
  [view removeFromSuperview];
  DESTROY(bogusWindow);
}

- (NSView *) view
{
  if (view == nil)
    {
      ASSIGN(userPipes, [NSMutableDictionary
        dictionaryWithDictionary: [[NSUserDefaults standardUserDefaults]
        objectForKey: @"UserPipes"]]);
      ASSIGN(sortedKeys, [[userPipes allKeys] sortedArrayUsingSelector:
        @selector(caseInsensitiveCompare:)]);

      [NSBundle loadNibNamed: @"UserPipes" owner: self];
    }

  return view;
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [userPipes count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  NSString * identifier = [aTableColumn identifier];

  if ([identifier isEqualToString: @"Name"])
    {
      return [sortedKeys objectAtIndex: rowIndex];
    }
  else if ([identifier isEqualToString: @"Command"])
    {
      return [[userPipes objectForKey: [sortedKeys objectAtIndex: rowIndex]]
        objectForKey: @"Command"];
    }
  else if ([identifier isEqualToString: @"KeyEquivalent"])
    {
      return [[userPipes objectForKey: [sortedKeys objectAtIndex: rowIndex]]
        objectForKey: @"KeyEquivalent"];
    }
  else
    return nil;
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  NSString * identifier = [aTableColumn identifier];
  NSString * oldName = [sortedKeys objectAtIndex: rowIndex];
  NSDictionary * pipeDescription = [userPipes objectForKey: oldName];

  if ([identifier isEqualToString: @"Name"])
    {
      if (![oldName isEqualToString: anObject])
        {
          [userPipes setObject: pipeDescription
                        forKey: anObject];
          [userPipes removeObjectForKey: oldName];
          ASSIGN(sortedKeys, [[userPipes allKeys] sortedArrayUsingSelector:
            @selector(caseInsensitiveCompare:)]);
          [table reloadData];
        }
    }
  else
    {
      NSMutableDictionary * newPipeDescription = [[pipeDescription
        mutableCopy] autorelease];

      [newPipeDescription setObject: anObject
                             forKey: identifier];
      [userPipes setObject: [[newPipeDescription copy] autorelease]
                    forKey: oldName];
    }

  SetDefaultUserPipes(userPipes);
}

- (BOOL)tableView: (NSTableView *)aTableView 
shouldEditTableColumn: (NSTableColumn *)aTableColumn 
              row: (int)rowIndex
{
  return YES;
}

- (void) add: sender
{
  NSString * newName;
  unsigned int i;

  for (newName = _(@"New Pipe"), i = 1;
       [userPipes objectForKey: newName] != nil;
       newName = [NSString stringWithFormat: _(@"New Name %i"), i], i++);

  [userPipes setObject: [NSDictionary dictionaryWithObjectsAndKeys:
    @"", @"Command",
    @"", @"KeyEquivalent", nil]
                forKey: newName];
  ASSIGN(sortedKeys, [[userPipes allKeys] sortedArrayUsingSelector:
        @selector(caseInsensitiveCompare:)]);
  [table reloadData];
  SetDefaultUserPipes(userPipes);
}

- (void) remove: sender
{
  int row = [table selectedRow];

  if (row >= 0)
    {
      NSString * name = [sortedKeys objectAtIndex: row];

      [userPipes removeObjectForKey: name];
      ASSIGN(sortedKeys, [[userPipes allKeys] sortedArrayUsingSelector:
        @selector(caseInsensitiveCompare:)]);

      [table reloadData];
      SetDefaultUserPipes(userPipes);
    }
}

@end
