#include <stdio.h>
#include <strings.h>
#include <stdarg.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>

#include <config.h>
#include _PATH_CURSES_H

#include <support.h>
#include "pform.h"

/*#define	DEBUG*/

#ifdef	USE_ACS
# define	FRAME_HLINE	ACS_HLINE
# define	FRAME_VLINE	ACS_VLINE
# define	FRAME_TLEFT	ACS_ULCORNER
# define	FRAME_TRIGHT	ACS_URCORNER
# define	FRAME_BLEFT	ACS_LLCORNER
# define	FRAME_BRIGHT	ACS_LRCORNER
# define	FRAME_RTEE	ACS_RTEE
# define	FRAME_LTEE	ACS_LTEE
#else
# define	FRAME_HLINE	'-'
# define	FRAME_VLINE	'|'
# define	FRAME_TLEFT	'+'
# define	FRAME_TRIGHT	'+'
# define	FRAME_BLEFT	'+'
# define	FRAME_BRIGHT	'+'
# define	FRAME_RTEE	'|'
# define	FRAME_LTEE	'|'
/*
# define	FRAME_RTEE	'['
# define	FRAME_LTEE	']'
*/
#endif

#define	KEY_BS		0x7f

typedef enum {
    PFRET_NONE=0,
    PFRET_NEXT,
    PFRET_CALLBACK,
} pfret_t;

struct pobj_s {
    struct pobj_s *next, *prev;
    char *name;
    int (*cb)();
    pform_t type;
    pfret_t ret;
    bool_t keyin;
    int x, y, l, w, jkey;
    union {
	struct {
	    int (*filter)();
	    char *is;
	    bool_t hide;
	    int ip, im;
	} input;
	struct {
	    int mc, mp;
	    char **mv;
	} menu;
	struct {
	    bool_t c;
	} check;
    } u;
};

struct pwin_s {
    struct pwin_s *next;
    struct pobj_s *ohp, *ocp;
    WINDOW *wp, *fwp;
    char *name;
    int x, y, w, h;
};

static struct pwin_s *pwinHp;

static struct pobj_s *PrevKeyObj(), *NextKeyObj();

static struct pobj_s *
InputKey(struct pobj_s *obj, int c)
{
    switch (c) {
    case KEY_CTRL('U'):
	obj->u.input.is[0] = '\0';
	obj->w = obj->u.input.ip = 0;
	break;
    case KEY_CTRL('A'):
	obj->u.input.ip = 0;
	break;
    case KEY_CTRL('E'):
	obj->u.input.ip = obj->w;
	break;
    case KEY_CTRL('F'):
#ifdef	KEY_RIGHT
    case KEY_RIGHT:
#endif
	if (obj->u.input.ip < obj->w) obj->u.input.ip ++;
	break;
#ifdef	KEY_BACKSPACE
    case KEY_BACKSPACE:
	c = KEY_BS;
#endif
    case KEY_BS:
    case KEY_CTRL('B'):
#ifdef	KEY_LEFT
    case KEY_LEFT:
#endif
	if (obj->u.input.ip > 0) {
	    obj->u.input.ip --;
	    if (c == KEY_BS) { /* BS */
		memmove(&obj->u.input.is[obj->u.input.ip],
			&obj->u.input.is[obj->u.input.ip + 1],
			obj->u.input.im - obj->u.input.ip);
		obj->w --;
		obj->u.input.is[obj->w + 1] = '\0';
	    }
	}
	break;
    default:
	if (obj->u.input.filter) c = obj->u.input.filter(obj, c);
	if (!isprint(c)) break;
	if (obj->u.input.im <= obj->w) {
	    obj->u.input.im ++;

	    obj->u.input.is = Realloc(obj->u.input.is,
				      obj->u.input.im + 1);
	}
	if (obj->u.input.ip < obj->u.input.im - 1) {
	    memmove(&obj->u.input.is[obj->u.input.ip + 1],
		    &obj->u.input.is[obj->u.input.ip],
		    obj->u.input.im - obj->u.input.ip);
	}
	obj->u.input.is[obj->u.input.ip] = c;
	obj->u.input.is[obj->w + 1] = '\0';
	obj->w ++;
	obj->u.input.ip ++;
    }
#ifdef DEBUG
    mvprintw(0, 0, "%04o/%04x)", c, c);
#endif
    return(obj);
}

static void
InputDestroy(struct pobj_s *obj)
{
    if (obj->u.input.is) Free(obj->u.input.is);
}

static struct pobj_s *
MenuKey(struct pobj_s *obj, int c)
{
    switch (c) {
    case KEY_CTRL('B'):
#ifdef	KEY_LEFT
    case KEY_LEFT:
#endif
    case ' ':
	obj->u.menu.mp ++;
	if (obj->u.menu.mp >= obj->u.menu.mc) obj->u.menu.mp = 0;
	break;
#ifdef	KEY_BACKSPACE
    case KEY_BACKSPACE:
#endif
    case KEY_BS:
    case KEY_CTRL('F'):
#ifdef	KEY_RIGHT
    case KEY_RIGHT:
#endif
	obj->u.menu.mp --;
	if (obj->u.menu.mp < 0) obj->u.menu.mp = obj->u.menu.mc - 1;
	break;
    }
    return(obj);
}

static void
MenuDisplay(struct pobj_s *obj, bool_t sw)
{
    if (!obj->u.menu.mc) return;
    mvwprintw(pwinHp->wp, obj->y, obj->x + obj->l - 1, "{%-*s ",
	      obj->w, obj->u.menu.mv[obj->u.menu.mp]);
    mvwprintw(pwinHp->wp, obj->y, obj->x + obj->l
	      + strlen(obj->u.menu.mv[obj->u.menu.mp]), "}");
}

static void
MenuDestroy(struct pobj_s *obj)
{
    char **mv;
    int mc;

    if ((mv = obj->u.menu.mv) == NULL) return;
    for (mc = 0; mc < obj->u.menu.mc; mc ++) Free(mv[mc]);
    Free(obj->u.menu.mv);
}

static struct pobj_s *
CheckKey(struct pobj_s *obj, int c)
{
    switch (c) {
    case ' ':
	obj->u.check.c = obj->u.check.c ? FALSE: TRUE;
	break;
    case KEY_CTRL('B'):
#ifdef	KEY_LEFT
    case KEY_LEFT:
#endif
	obj->u.check.c = FALSE;
	break;
    case KEY_CTRL('F'):
#ifdef	KEY_RIGHT
    case KEY_RIGHT:
#endif
	obj->u.check.c = TRUE;
	break;
    }
    return(obj);
}

static void
CheckDisplay(struct pobj_s *obj, bool_t sw)
{
    mvwprintw(pwinHp->wp, obj->y, obj->x + obj->l - 1,
	      "(%c)", obj->u.check.c ? '*': ' ');
    wmove(pwinHp->wp, obj->y, obj->x + obj->l);
}

static void
InputDisplay(struct pobj_s *obj, bool_t sw)
{
    int len;
    char *ds=NULL, *is=obj->u.input.is;

    if (obj->u.input.hide) {
	if (is && *is) {
	    if (sw) {
		int n;

		len = strlen(is);
		n = len + 1;
		if (n < (int)sizeof("<^_^>")) n = (int)sizeof("<^_^>");
		ds = Malloc(n);
		sprintf(ds, "%-*s", n - 1, is);
		is = ds;
		for (n = 0; n < len; n ++, ds ++) *ds = '*';
		ds = is;
	    } else {
		is = "<^_^>";
		len = strlen(is);
	    }
	} else {
	    is = "     ";
	    len = 0;
	}
    } else {
	if (!is) is = "";
	len = strlen(is);
    }
    mvwprintw(pwinHp->wp, obj->y, obj->x + obj->l - 1, "[%-*s ",
	      obj->u.input.im, is);
    mvwprintw(pwinHp->wp, obj->y, obj->x + obj->l + len, "]");
    if (obj->u.input.hide)
	wmove(pwinHp->wp, obj->y, obj->x + obj->l);
    else
	wmove(pwinHp->wp, obj->y, obj->x + obj->l + obj->u.input.ip);
    if (ds) Free(ds);
}

static struct pobj_s *
ButtonKey(struct pobj_s *obj, int c)
{
    switch (c) {
    case KEY_CTRL('B'):
#ifdef	KEY_LEFT
    case KEY_LEFT:
#endif
	obj = PrevKeyObj(obj);
	break;
    case KEY_CTRL('F'):
#ifdef	KEY_RIGHT
    case KEY_RIGHT:
#endif
	obj = NextKeyObj(obj);
	break;
    }
    return(obj);
}

static void
ButtonDisplay(struct pobj_s *obj, bool_t sw)
{
    mvwprintw(pwinHp->wp, obj->y, obj->x - 1, "<");
    mvwprintw(pwinHp->wp, obj->y, obj->x + strlen(obj->name), ">");
    wmove(pwinHp->wp, obj->y, obj->x - 1);
}

static struct {
    struct pobj_s *(*key)();
    void (*display)();
    void (*destroy)();
    bool_t keyin, curs;
} pformProcs[]={
    {InputKey, InputDisplay, InputDestroy, TRUE, TRUE},
    {MenuKey, MenuDisplay, MenuDestroy, TRUE, FALSE},
    {CheckKey, CheckDisplay, NULL, TRUE, FALSE},
    {ButtonKey, ButtonDisplay, NULL, TRUE, FALSE},
    {NULL, NULL, NULL, 0, 0}
};

static struct pobj_s *
NextKeyObj(struct pobj_s *obj)
{
    struct pobj_s *ob1;

    ob1 = obj = obj->next;
    while (!obj->keyin) {
	obj = obj->next;
	if (ob1 == obj) break;
    }
    return(obj);
}

static struct pobj_s *
PrevKeyObj(struct pobj_s *obj)
{
    struct pobj_s *ob1;

    ob1 = obj = obj->prev;
    while (!obj->keyin) {
	obj = obj->prev;
	if (ob1 == obj) break;
    }
    return(obj);
}

void
PformInputHide(struct pobj_s *obj, bool_t hide)
{
    obj->u.input.hide = hide;
}

void
PformInputFilter(struct pobj_s *obj, int (*filter)())
{
    obj->u.input.filter = filter;
}

void
PformInputSet(struct pobj_s *obj, char *str)
{
    obj->w = obj->u.input.ip = obj->u.input.im = strlen(str);
    obj->u.input.im ++;
    obj->u.input.is = Realloc(obj->u.input.is, obj->u.input.im + 1);
    strcpy(obj->u.input.is, str);
}

char *
PformInputGet(struct pobj_s *obj)
{
    return(obj->u.input.is);
}

void
PformCheckSet(struct pobj_s *obj, bool_t sw)
{
    obj->u.check.c = sw;
}

bool_t
PformCheckGet(struct pobj_s *obj)
{
    return(obj->u.check.c);
}

int
PformMenuAdd(struct pobj_s *obj, char *item)
{
    char **mv;
    int w;

    mv = Malloc((obj->u.menu.mc + 1) * sizeof(char *));
    if (obj->u.menu.mc)
	memcpy(mv, obj->u.menu.mv, obj->u.menu.mc * sizeof(char *));
    mv[obj->u.menu.mc] = Strdup(item);
    obj->u.menu.mc ++;
    Free(obj->u.menu.mv);
    obj->u.menu.mv = mv;
    w = strlen(item);
    if (obj->w < w) obj->w = w;
    return(obj->u.menu.mc);
}

int
PformMenuGet(struct pobj_s *obj)
{
    return(obj->u.menu.mp);
}

void
PformMenuSet(struct pobj_s *obj, int n)
{
    if (n < obj->u.menu.mc) obj->u.menu.mp = n;
}

struct pobj_s *
PformNew(pform_t type, int x, int y, char *name)
{
    struct pobj_s *obj;

    if (!pwinHp) return(NULL);
    obj = TCALLOC(struct pobj_s);
    obj->type = type;
    obj->name = Strdup(name);
    obj->l = strlen(name) + 2;
    obj->keyin = pformProcs[obj->type].keyin;
    if (pwinHp->ohp) {
	struct pobj_s *ob1;

	ob1 = pwinHp->ohp;
	do {
	    if (ob1->next == pwinHp->ohp) {
		ob1->next = obj;
		obj->prev = ob1;
		break;
	    }
	    ob1 = ob1->next;
	} while(ob1);
    } else pwinHp->ohp = obj;
    obj->next = pwinHp->ohp;
    pwinHp->ohp->prev = obj;
    obj->x = x;
    obj->y = y;
    return(obj);
}

void
PformActivate(struct pobj_s *obj, bool_t sw)
{
    obj->keyin = sw ? pformProcs[obj->type].keyin: FALSE;
}

void
PformHighlight(struct pobj_s *obj, bool_t sw)
{
    if (sw) {
	if (obj->keyin) {
	    pwinHp->ocp = obj;
	    wstandout(pwinHp->wp);
	}
    }
    mvwprintw(pwinHp->wp, obj->y, obj->x, obj->name);
    wstandend(pwinHp->wp);
    wprintw(pwinHp->wp, " ");
    if (pformProcs[obj->type].display)
	pformProcs[obj->type].display(obj, sw);
}


static void
DrawFrame(struct pwin_s *win)
{
    int i;

    wmove(win->fwp, 0, 0);
    waddch(win->fwp, FRAME_TLEFT);
    for (i = 0; i < win->w; i ++) waddch(win->fwp, FRAME_HLINE);
    waddch(win->fwp, FRAME_TRIGHT);
    wmove(win->fwp, win->h + 1, 0);
    waddch(win->fwp, FRAME_BLEFT);
    for (i = 0; i < win->w; i ++) waddch(win->fwp, FRAME_HLINE);
    waddch(win->fwp, FRAME_BRIGHT);
    for (i = 1; i < win->h + 1; i ++) {
	wmove(win->fwp, i, 0);
	waddch(win->fwp, FRAME_VLINE);
	wmove(win->fwp, i, win->w + 1);
	waddch(win->fwp, FRAME_VLINE);
    }

    wmove(win->fwp, 0, (win->w - strlen(win->name) - 2) / 2);
/*    wstandout(win->fwp);*/
    waddch(win->fwp, FRAME_RTEE);
    wprintw(win->fwp, win->name);
    waddch(win->fwp, FRAME_LTEE);
/*    wstandend(win->fwp);*/
    wrefresh(win->fwp);
}

void
PformRealize(struct pobj_s *obj)
{
    if (!obj) {
	obj = pwinHp->ohp;
	do {
	    PformHighlight(obj, FALSE);
	    obj = obj->next;
	} while (obj != pwinHp->ohp);
	obj = pwinHp->ocp;
    }
    if (obj) PformHighlight(obj, TRUE);
    DrawFrame(pwinHp);
    wrefresh(pwinHp->wp);
}

void
PformReturnCallback(struct pobj_s *obj, int (*cb)())
{
    obj->cb = cb;
    obj->ret = PFRET_CALLBACK;
}

void
PformReturnNext(struct pobj_s *obj)
{
    obj->ret = PFRET_NEXT;
}

void
PformJumpKey(struct pobj_s *obj, int key)
{
    obj->jkey = key;
}

struct pobj_s *
PformLoop()
{
    struct pobj_s *obj, *ob1, *ob2, *robj=NULL;
    bool_t rewrite=FALSE;
    int c;

    if (!pwinHp->ocp) {
	obj = pwinHp->ohp;
	while (obj->next != pwinHp->ohp) {
	    if (obj->keyin) break;
	    obj = obj->next;
	}
	pwinHp->ocp = obj;
    }
    ob1 = obj = pwinHp->ocp;
    while(1) {
	if (!pformProcs[obj->type].curs) {
	    wmove(pwinHp->fwp, pwinHp->h + 1, pwinHp->w + 1);
	    wrefresh(pwinHp->fwp);
	}
	if ((c = getch()) == ERR) break;
	ob2 = pwinHp->ohp;
	do {
	    if (ob2->keyin && ob2->jkey && ob2->jkey == c) {
		if (obj->ret == PFRET_NEXT) robj = obj;
		obj = ob2;
		goto refresh_pform;
	    }
	    ob2 = ob2->next;
	} while (ob2 != pwinHp->ohp);
	switch(c) {
	case '\t':
	case KEY_CTRL('N'):
#ifdef	KEY_DOWN
	case KEY_DOWN:
#endif
	    if (obj->ret == PFRET_NEXT) robj = obj;
	    obj = NextKeyObj(obj);
	    break;
	case KEY_CTRL('P'):
#ifdef	KEY_UP
	case KEY_UP:
#endif
	    if (obj->ret == PFRET_NEXT) robj = obj;
	    obj = PrevKeyObj(obj);
	    break;
	case '\r':
	case '\n':
	    robj = obj;
	    if (obj->ret == PFRET_NEXT) obj = NextKeyObj(obj);
	    else if (obj->ret == PFRET_CALLBACK && obj->cb) {
		obj->cb(obj);
		robj = NULL;
	    }
	    break;
	default:
	    if (obj->keyin) {
		struct pobj_s *ob2;

		ob2 = pformProcs[obj->type].key(obj, c);
  		if (ob2) obj = ob2;
		rewrite = TRUE;
	    }
	}
      refresh_pform:
	if (obj != ob1 || rewrite) {
	    PformHighlight(ob1, FALSE);
	    PformHighlight(obj, TRUE);
	    ob1 = obj;
	}
	wrefresh(pwinHp->wp);
	if (robj) return(robj);
    }
    return(NULL);
}

WINDOW *
PwinGet(struct pwin_s *win)
{
    return(win->wp);
}

struct pwin_s *
PwinOpen(char *name, int x, int y, int *rw, int *rh)
{
    struct pwin_s *win;
    WINDOW *wp, *fwp;

    if (*rw == 0 || *rw > COLS - 2) *rw = COLS - 2;
    if (*rh == 0 || *rh > LINES - 2) *rh = LINES - 2;
    if ((fwp = newwin(*rh + 2, *rw + 2, y, x)) == NULL) return(NULL);
    if ((wp = subwin(fwp, *rh, *rw, x + 1, y + 1)) == NULL) {
	delwin(fwp);
	return(NULL);
    }
    win = TCALLOC(struct pwin_s);
    win->name = Strdup(name);
    win->wp = wp;
    win->fwp = fwp;
    win->next = pwinHp;
    win->x = x;
    win->y = y;
    win->w = *rw;
    win->h = *rh;
    pwinHp = win;
    return(win);
}

void
PwinClose(struct pwin_s *win)
{
    struct pobj_s *obj, *ob1;
    struct pwin_s *win1, *wp;

    if ((obj = win->ohp) != NULL) do {
	if (pformProcs[obj->type].destroy)
	    pformProcs[obj->type].destroy(obj);
	ob1 = obj->next;
	Free(obj->name);
	Free(obj);
	obj = ob1;
    } while (obj != win->ohp);

    win1 = NULL;
    wp = pwinHp;
    while (wp) {
	if (wp == win) {
	    if (win1) win1->next = win->next;
	    else pwinHp = win->next;
	    break;
	}
	win1 = wp;
	wp = wp->next;
    }
    werase(win->fwp);
    wrefresh(win->fwp);
    delwin(win->wp);
    delwin(win->fwp);
    Free(win->name);
    Free(win);

    if (pwinHp) {
	werase(pwinHp->fwp);
	PformRealize(NULL);
    }
}

void
PformOpen()
{
    initscr();
    cbreak();
    noecho();
#ifdef	KEY_MAX
    keypad(stdscr, TRUE);
#endif
    refresh();
}

void
PformClose()
{
    struct pwin_s *win, *win1;

    win = pwinHp;
    while (win) {
	win1 = win->next;
	PwinClose(win);
	win = win1;
    }
    pwinHp = NULL;
    nocbreak();
    echo();
#ifdef	KEY_MAX
    keypad(stdscr, FALSE);
#endif
    clear();
    refresh();
    endwin();
    printf("\r\n");
}
