/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  system-mon.c: The 'motherboard' logic which connects the entire
 *    PC system.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"

#if InstrumentIO
// xxx Move this to monitor.h
extern unsigned ioAccessCount[];
#endif


  Bit32u
sysIOIn(vm_t *vm, Bit32u port, unsigned len)
{
  unsigned handleID;

  port &= 0x0000ffff;
#if InstrumentIO
  ioAccessCount[port]++;
#endif
  handleID = vm->ioHandlers.readHandlerID[port];
  if (handleID < MaxIOHandlers) {
    if (vm->ioHandlers.read[handleID].space == MonitorSpace) {
      ioReadHandler_t callback;
      callback = vm->ioHandlers.read[handleID].callback;
      return( callback(vm, port, len) );
      }
    else {
      IO_msg_t *io_msg = (IO_msg_t *) vm->mon_msgs.msg;
      Bit32u data;
  
      CLI();
      vm->mon_msgs.header.msg_type = VMMessageIOInRequest;
      vm->mon_msgs.header.msg_len  = sizeof(IO_msg_t);
      io_msg->port = port;
      io_msg->len  = len;
      io_msg->op   = IO_IN;
      io_msg->thisPtr  = vm->ioHandlers.read[handleID].thisPtr;
      io_msg->callback = vm->ioHandlers.read[handleID].callback;
  
      vm->mon_request = MON_REQ_RESPONSE_PENDING;
      vm->guest.__mon2host();
  
      data = io_msg->data;
      ClearMonMessageQ(vm);
      STI();
  
      return data;
      }
    }
  else {
    if (handleID == NoIOHandlerID) {
      monprint(vm, "sysIOIn: No handler defined for port 0x%x.\n", port);
      /* Tag this port, so we only report the unhandled access once. */
      vm->ioHandlers.readHandlerID[port] = UnmappedIOSeen;
      }
    return( 0xffffffff );
    }
}

  unsigned
sysIOInBatch(vm_t *vm, Bit32u port, unsigned len, unsigned n, Bit32u paddr)
{
  unsigned handleID;

  port &= 0x0000ffff;
#if InstrumentIO
  ioAccessCount[port]++;
#endif
  handleID = vm->ioHandlers.readHandlerID[port];
  if (handleID < MaxIOHandlers) {
    if (vm->ioHandlers.read[handleID].space == MonitorSpace) {
      monpanic(vm, "sysIOInBatch: Monitor space IO!\n");
      }
    else {
      IOBatch_msg_t *msg = (IOBatch_msg_t *) vm->mon_msgs.msg;
   
      CLI();
      vm->mon_msgs.header.msg_type = VMMessageIOBatchRequest;
      vm->mon_msgs.header.msg_len  = sizeof(IOBatch_msg_t);
      msg->port  = port;
      msg->len   = len;
      msg->op    = IO_IN;
      msg->n     = n;
      msg->paddr = paddr;
      msg->thisPtr  = vm->ioHandlers.read[handleID].thisPtr;
      msg->callback = vm->ioHandlers.read[handleID].callback;
   
      vm->mon_request = MON_REQ_RESPONSE_PENDING;
      vm->guest.__mon2host();
      n = msg->n;
   
      ClearMonMessageQ(vm);
      STI();
  
      return n;
      }
    }
  else {
    monpanic(vm, "sysIOInBatch: No handler defined for port 0x%x!.\n",
             port);
    }
}

  void
sysIOOut(vm_t *vm, Bit32u port, unsigned len, Bit32u data)
{
  unsigned handleID;

  port &= 0x0000ffff;
#if InstrumentIO
  ioAccessCount[port]++;
#endif
  handleID = vm->ioHandlers.writeHandlerID[port];
  if (handleID < MaxIOHandlers) {
    if (vm->ioHandlers.write[handleID].space == MonitorSpace) {
      ioWriteHandler_t callback;
      callback = vm->ioHandlers.write[handleID].callback;
      callback(vm, port, data, len);
      }
    else {
      IO_msg_t *io_msg = (IO_msg_t *) vm->mon_msgs.msg;
      CLI();
      vm->mon_msgs.header.msg_type = VMMessageIOOutRequest;
      vm->mon_msgs.header.msg_len  = sizeof(IO_msg_t);
      io_msg->port = port;
      io_msg->len  = len;
      io_msg->op   = IO_OUT;
      io_msg->data = data;
      io_msg->thisPtr  = vm->ioHandlers.write[handleID].thisPtr;
      io_msg->callback = vm->ioHandlers.write[handleID].callback;
    
      vm->mon_request = MON_REQ_NO_RESPONSE;
      vm->guest.__mon2host();
    
      ClearMonMessageQ(vm);
      STI();
      }
    }
  else {
    if (handleID == NoIOHandlerID) {
      monprint(vm, "sysIOOut: No handler defined for port 0x%x!.\n", port);
      /* Tag this port, so we only report the unhandled access once. */
      vm->ioHandlers.writeHandlerID[port] = UnmappedIOSeen;
      }
    }

#ifdef DoInstrEmulateOpcode
{
static unsigned ioTriggerCnt = 0;
void instrReset(vm_t *);
void instrPrint(vm_t *);
// Delete this IO trigger
if ( (port == 0x70) && (data == 0x4) ) {
  monprint(vm, "sysIOOut(0x70, 4)\n");
  ioTriggerCnt++;
  if (ioTriggerCnt == 3)
    instrReset(vm);
  if (ioTriggerCnt == 4)
    instrPrint(vm);
  }
}
#endif

}

  unsigned
sysIOOutBatch(vm_t *vm, Bit32u port, unsigned len, unsigned n, Bit32u paddr)
{
  unsigned handleID;

  port &= 0x0000ffff;
#if InstrumentIO
  ioAccessCount[port]++;
#endif
  handleID = vm->ioHandlers.writeHandlerID[port];
  if (handleID < MaxIOHandlers) {
    if (vm->ioHandlers.write[handleID].space == MonitorSpace) {
      monpanic(vm, "sysIOOutBatch: Monitor space IO!\n");
      }
    else {
      IOBatch_msg_t *msg = (IOBatch_msg_t *) vm->mon_msgs.msg;

      CLI();
      vm->mon_msgs.header.msg_type = VMMessageIOBatchRequest;
      vm->mon_msgs.header.msg_len  = sizeof(IOBatch_msg_t);
      msg->port  = port;
      msg->len   = len;
      msg->op    = IO_OUT;
      msg->n     = n;
      msg->paddr = paddr;
      msg->thisPtr  = vm->ioHandlers.write[handleID].thisPtr;
      msg->callback = vm->ioHandlers.write[handleID].callback;
    
      vm->mon_request = MON_REQ_RESPONSE_PENDING;
      vm->guest.__mon2host();
      n = msg->n;
    
      ClearMonMessageQ(vm);
      STI();

      return( n );
      }
    }
  else {
    monpanic(vm, "sysIOOutBatch: No handler defined for port 0x%x!.\n",
             port);
    }
}

  void
sysMemMapIOWrite(vm_t *vm, Bit32u addr, unsigned len, Bit32u data)
{
    memMapIO_msg_t *memMapIO_msg = (memMapIO_msg_t *) vm->mon_msgs.msg;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageMemMapIOWriteRequest;
    vm->mon_msgs.header.msg_len  = sizeof(memMapIO_msg_t);
    memMapIO_msg->addr = addr;
    memMapIO_msg->len  = len;
    memMapIO_msg->op   = IO_OUT;
    memMapIO_msg->data = data;

    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();

    ClearMonMessageQ(vm);
    STI();
}

  Bit32u
sysMemMapIORead(vm_t *vm, Bit32u addr, unsigned len)
{
    memMapIO_msg_t *memMapIO_msg = (memMapIO_msg_t *) vm->mon_msgs.msg;
    Bit32u data;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageMemMapIOReadRequest;
    vm->mon_msgs.header.msg_len  = sizeof(memMapIO_msg_t);

    memMapIO_msg->addr = addr;
    memMapIO_msg->len  = len;
    memMapIO_msg->op   = IO_IN;

    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    data = memMapIO_msg->data;
    ClearMonMessageQ(vm);
    STI();

    return data;
}


  unsigned
sysIAC(vm_t *vm)
{
  return( picIAC(vm) ); // xxx Eliminate this extra level of indirection.
}

  void
sysRaiseHLDA(vm_t *vm)
{
    /* IF handling ??? */
    monpanic(vm, "sysRaiseHLDA:\n");
}

  unsigned
sysReflectInt(vm_t *vm, unsigned vector)
{
    INT_msg_t *int_msg = (INT_msg_t *) vm->mon_msgs.msg;
    unsigned reflect;

monpanic(vm, "sysReflectInt:\n");
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageIntRequest;
    vm->mon_msgs.header.msg_len  = sizeof(INT_msg_t);
    int_msg->vector = vector;

    vm->mon_request = MON_REQ_RESPONSE_PENDING;
    vm->guest.__mon2host();

    reflect = int_msg->reflect;
    ClearMonMessageQ(vm);
    STI();

    return reflect;
}


  void
sysDisasm(vm_t *vm)
{
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageDisasm;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysFlushPrintBuf(vm_t *vm)
{
#if 0
    CLI();
    vm->mon_request = MON_REQ_FLUSH_PRINT_BUF;
    vm->guest.__mon2host();
    STI();
#endif

    CLI();
    vm->mon_msgs.header.msg_type = VMMessagePrintBuf;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysEOICount(vm_t *vm)
{
    EOICount_t *msg = (EOICount_t *) vm->mon_msgs.msg;

    CLI();
    vm->mon_msgs.header.msg_type = VMMessageEOICount;
    vm->mon_msgs.header.msg_len  = sizeof(EOICount_t);
    cache_sreg(vm, SRegCS);
    msg->cs = vm->guest_cpu.selector[SRegCS].raw;
    msg->eip = G_EIP(vm);
    msg->laddr = vm->guest_cpu.desc_cache[SRegCS].base + G_EIP(vm);
    msg->seg32 = vm->guest_cpu.desc_cache[SRegCS].desc.d_b;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysReqComplete(vm_t *vm)
{
    CLI();
    vm->mon_msgs.header.msg_type = VMMessageReqComplete;
    vm->mon_msgs.header.msg_len  = 0;
    vm->mon_request = MON_REQ_NO_RESPONSE;
    vm->guest.__mon2host();
 
    ClearMonMessageQ(vm);
    STI();
}

  void
sysRemapMonitor(vm_t *vm)
{
    CLI();
    vm->mon_request = MON_REQ_REMAP_MONITOR;
    vm->guest.__mon2host();
    STI();
}

  void
sysNOP(vm_t *vm)
{
    CLI();
    vm->mon_request = MON_REQ_NOP;
    vm->guest.__mon2host();
    STI();
}

  void
sysTimerCallback(vm_t *vm, void *thisPtr, void *callback, unsigned id)
{
  timerCallback_t *timerCallback = (timerCallback_t *) vm->mon_msgs.msg;

  CLI();
  vm->mon_msgs.header.msg_type = VMMessageTimeElapsed;
  vm->mon_msgs.header.msg_len  = sizeof(timerCallback_t);
  timerCallback->thisPtr    = thisPtr;
  timerCallback->callback   = callback;
  timerCallback->callbackID = id;

  vm->mon_request = MON_REQ_NO_RESPONSE;
  vm->guest.__mon2host();

  ClearMonMessageQ(vm);
  STI();
}

  int
handleTimers(vm_t *vm)
{
  Bit64u t0, t1, cyclesInPeriod;
  unsigned id;
  plex86_timer_t *timer = vm->system.timer;
 
  t0 = 0; /* Beginning of time window. */

  while ( t0 < vm->system.cyclesElapsed ) {
    Bit64u minCycles;
    unsigned minI;

    t1 = t0 + vm->system.cyclesInPeriod;
    for (id=0; id < vm->system.num_timers; id++) {
      timer[id].triggered = 0; /* Mark not triggered intially */

      /* If timer was active during this window of time... */
      if (timer[id].active &&
          (timer[id].inServiceElapsed < t1) ) {
        if (timer[id].inServiceElapsed <= t0)
          /* Timer was active during the entire time window. */
          cyclesInPeriod = vm->system.cyclesInPeriod;
        else {
          /* Timer was active only in _part_ of the time window. */
          cyclesInPeriod = (t1 - timer[id].inServiceElapsed);
          }
        if (cyclesInPeriod > timer[id].remaining)
          return( -1 ); /* Error. */
        timer[id].remaining -= cyclesInPeriod;
        if (timer[id].remaining == 0) {
          /* This timer has triggered. */
          timer[id].triggered = 1;
          timer[id].remaining = timer[id].period;
          /* If triggered timer is one-shot, deactive. */
          if (timer[id].continuous==0) {
            timer[id].active = 0;
            }
          }
        }
      }

    /* Advance time0 to next time window, for the next iteration. */
    t0 = t1;

    /* Flag the cyclesRemaining as 0 for this time period, since we
     * handled it already.  This is useful, so that when we invoke the
     * callbacks, any timer requests that result (like registerTimer),
     * will not both recalculating cyclesInPeriod/cyclesRemaining.  These
     * are recalculated below after all callbacks are invoked.
     */
    vm->system.cyclesRemaining = 0;

    /* Now invoke all callbacks that were triggered at this time. */
    for (id=0; id < vm->system.num_timers; id++) {
      if (timer[id].triggered) {
        if (timer[id].space == MonitorSpace) {
          timer[id].callback(vm, id);
          }
        else {
          /* Guest space */
          sysTimerCallback(vm, timer[id].thisPtr, timer[id].callback, id);
          }
        }
      }
 
    /* Now that we've marked the trigger flags for appropriate timers,
     * compute the next minimum remaining time of all active timers for
     * the next period.
     */
    minI = vm->system.num_timers; /* Start with illegal value. */
    minCycles = (Bit64u) -1; /* max number in Bit64u range */

    for (id=0; id < vm->system.num_timers; id++) {
      if (timer[id].active) {
        Bit64u cyclesToNextT0;
        if (timer[id].inServiceElapsed > t0) {
          /* If this timer was activated (placed in service) after the
           * beginning of this time window, calculate the amount of cycles
           * to the end of its expiration, in case it ends up being the
           * soonest event.
           */
          cyclesToNextT0 = timer[id].remaining +
                           (timer[id].inServiceElapsed - t0);
          }
        else {
          /* Timer was active during entire current time window. */
          cyclesToNextT0 = timer[id].remaining;
          }
        if (cyclesToNextT0 < minCycles) {
          minI = id;
          minCycles = cyclesToNextT0;
          }
        }
      }
    if (minI >= vm->system.num_timers)
      return( -1 ); /* Error. */
    vm->system.cyclesInPeriod = minCycles;
    vm->system.cyclesRemaining = minCycles;
    }
 
  /* Reset cyclesElapsed time.  All elapsed time has been accounted for. */
  vm->system.cyclesElapsed = 0;

  /* Reset in-service flags.  These were based on the cyclesElapsed value
   * at the time of timer activation.  All timers are now established,
   * and no longer need this.
   */
  for (id=0; id < vm->system.num_timers; id++) {
    timer[id].inServiceElapsed = 0;
    }
  return( 0 ); /* OK */
}
