/*  Copyright (C) 2000-2001  MandrakeSoft S.A.
 *
 *    MandrakeSoft S.A.
 *    43, rue d'Aboukir
 *    75002 Paris - France
 *    http://www.linux-mandrake.com/
 *    http://www.mandrakesoft.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"


#ifdef OUT
#  undef OUT
#endif





static Bit8u   pitReadCounter(vm_t *, unsigned timerid);
static void    pitWriteCountReg(vm_t *, Bit8u value, unsigned timerid);
static void    pitLatch(vm_t *, unsigned timerid);
static void    pitSetGATE(vm_t *, unsigned pit_id, unsigned value);
static void    pitStart(vm_t *, unsigned timerid);
static Boolean pitPeriodic(vm_t *vm, Bit32u usec_delta);


  Bit32u
pitIORead(vm_t *vm, Bit32u address, unsigned io_len)
{
  if (io_len > 1)
    monpanic(vm, "pit: io read from port %04x, len=%u\n", (unsigned) address,
             (unsigned) io_len);

  switch (address) {
    case 0x40: /* timer 0 - system ticks */
      return( pitReadCounter(vm, 0) );
      break;

    case 0x42: /* timer 2 read */
      return( pitReadCounter(vm, 2) );
      break;

    case 0x61:
      /* AT, port 61h */
      vm->pit.s.refresh_clock_div2 = !vm->pit.s.refresh_clock_div2;
      return( (vm->pit.s.timer[2].OUT<<5) |
              (vm->pit.s.refresh_clock_div2<<4) |
              (vm->pit.s.speaker_data_on<<1) |
              (vm->pit.s.timer[2].GATE) );
      break;

    default:
      monpanic(vm, "pit: unsupported io read from port %04x\n", address);
    }
}

  void
pitIOWrite(vm_t *vm, Bit32u address, Bit32u dvalue, unsigned io_len)
{
  Bit8u    command, mode, bcd_mode;
  Bit8u   value;

  value = (Bit8u) dvalue;

  if (io_len > 1)
    monpanic(vm, "pit: io write to port %04x, len=%u\n", (unsigned) address,
             (unsigned) io_len);

  switch (address) {
    case 0x40: /* timer 0: write count register */
      pitWriteCountReg(vm, value, 0 );
      break;

    case 0x41: /* timer 1: write count register */
      pitWriteCountReg(vm, value, 1 );
      break;

    case 0x42: /* timer 2: write count register */
      pitWriteCountReg(vm, value, 2 );
      break;

    case 0x43: /* timer 0-2 mode control */
      /* |7 6 5 4|3 2 1|0|
       * |-------|-----|-|
       * |command|mode |bcd/binary|
       */
      command  = value >> 4;
      mode     = (value >> 1) & 0x07;
      bcd_mode = value & 0x01;
#if 0
bx_printf("timer 0-2 mode control: comm:%02x mode:%02x bcd_mode:%u\n",
  (unsigned) command, (unsigned) mode, (unsigned) bcd_mode);
#endif

      if ( (mode > 5) || (command > 0x0e) )
        monpanic(vm, "pit: outp(43h)=%02xh out of range\n", (unsigned) value);
      if (bcd_mode)
        monpanic(vm, "pit: outp(43h)=%02xh: bcd mode unhandled\n",
          (unsigned) bcd_mode);

      switch (command) {
        case 0x0: /* timer 0: counter latch */
          pitLatch(vm, 0);
          break;

        case 0x1: /* timer 0: LSB mode */
        case 0x2: /* timer 0: MSB mode */
          monpanic(vm, "pit: outp(43h): command %02xh unhandled\n",
            (unsigned) command);
          break;
        case 0x3: /* timer 0: 16-bit mode */
          vm->pit.s.timer[0].mode = mode;
          vm->pit.s.timer[0].latch_mode   = BX_PIT_LATCH_MODE_16BIT;
          vm->pit.s.timer[0].input_latch_value = 0;
          vm->pit.s.timer[0].input_latch_toggle = 0;
          vm->pit.s.timer[0].bcd_mode    = bcd_mode;
          if ( (mode!=3 && mode!=2 && mode!=0) || bcd_mode!=0 )
            monpanic(vm, "pit: outp(43h): comm 3, mode %02x, bcd %02x unhandled\n",
              (unsigned) mode, bcd_mode);
          break;
        case 0x4: /* timer 1: counter latch */
          pitLatch(vm, 1);
          break;

        case 0x5: /* timer 1: LSB mode */
        case 0x6: /* timer 1: MSB mode */
          monprint(vm, "pit: outp(43h): command %02xh unhandled (ignored)\n",
            (unsigned) command);
          break;
        case 0x7: /* timer 1: 16-bit mode */
          vm->pit.s.timer[1].mode = mode;
          vm->pit.s.timer[1].latch_mode   = BX_PIT_LATCH_MODE_16BIT;
          vm->pit.s.timer[1].input_latch_value = 0;
          vm->pit.s.timer[1].input_latch_toggle = 0;
          vm->pit.s.timer[1].bcd_mode    = bcd_mode;
          if ( mode!=2 || bcd_mode!=0 )
            monpanic(vm, "pit: outp(43h): comm 7, mode %02x, bcd %02x unhandled\n",
              (unsigned) mode, bcd_mode);
          break;
        case 0x8: /* timer 2: counter latch */
          pitLatch(vm, 2);
          break;

        case 0x9: /* timer 2: LSB mode */
        case 0xa: /* timer 2: MSB mode */
          monpanic(vm, "pit: outp(43h): command %02xh unhandled\n",
            (unsigned) command);
          break;
        case 0xb: /* timer 2: 16-bit mode */
          vm->pit.s.timer[2].mode = mode;
          vm->pit.s.timer[2].latch_mode   = BX_PIT_LATCH_MODE_16BIT;
          vm->pit.s.timer[2].input_latch_value = 0;
          vm->pit.s.timer[2].input_latch_toggle = 0;
          vm->pit.s.timer[2].bcd_mode    = bcd_mode;
          if ( (mode!=3 && mode!=2) || bcd_mode!=0 )
            monpanic(vm, "pit: outp(43h): comm Bh, mode %02x, bcd %02x unhandled\n",
              (unsigned) mode, bcd_mode);
          break;
#if 0
        case 0xd: /* general counter latch */
          if (value & 0x08) /* select counter 2 */
            pitLatch(vm, 2);
          if (value & 0x04) /* select counter 1 */
            pitLatch(vm, 1);
          if (value & 0x02) /* select counter 0 */
            pitLatch(vm, 0);
          break;

        case 0xe: /* latch status of timers */
          monpanic(vm, "pit: outp(43h): command %02xh unhandled\n",
            (unsigned) command);
          break;
#endif
        case 0x0c: case 0x0d: case 0x0e: case 0x0f:
          monprint(vm, "pit: ignoring 8254 command %u\n", (unsigned) command);
          break;

        default: /* 0xc & 0xf */
          monpanic(vm, "pit: outp(43h) command %1xh unhandled\n",
            (unsigned) command);
          break;
        }
      break;

    case 0x61:
      vm->pit.s.speaker_data_on = (value >> 1) & 0x01;
      pitSetGATE(vm, 2, value & 0x01);
      break;

    default:
      monpanic(vm, "pit: unsupported io write to port %04x = %02x\n",
        (unsigned) address, (unsigned) value);
    }
}




  void
pitWriteCountReg(vm_t *vm, Bit8u   value, unsigned timerid)
{
  Boolean xfer_complete;

  switch ( vm->pit.s.timer[timerid].latch_mode ) {
    case BX_PIT_LATCH_MODE_16BIT: /* write1=LSB, write2=MSB */
      if (vm->pit.s.timer[timerid].input_latch_toggle==0) {
        vm->pit.s.timer[timerid].input_latch_value = value;
        vm->pit.s.timer[timerid].input_latch_toggle = 1;
        xfer_complete = 0;
        }
      else {
        vm->pit.s.timer[timerid].input_latch_value |= (value << 8);
        vm->pit.s.timer[timerid].input_latch_toggle = 0;
        xfer_complete = 1;
        }
      break;

    case BX_PIT_LATCH_MODE_MSB: /* write1=MSB, LSB=0 */
      vm->pit.s.timer[timerid].input_latch_value = (value << 8);
      xfer_complete = 1;
      break;

    case BX_PIT_LATCH_MODE_LSB: /* write1=LSB, MSB=0 */
      vm->pit.s.timer[timerid].input_latch_value = value;
      xfer_complete = 1;
      break;

    default:
      monpanic(vm, "write_count_reg: latch_mode unknown\n");
      xfer_complete = 0;
    }

  if (xfer_complete) {
    vm->pit.s.timer[timerid].counter_max = vm->pit.s.timer[timerid].input_latch_value;

    // reprogramming counter clears latch
    vm->pit.s.timer[timerid].output_latch_full = 0;

    // counter bounds
    // mode      minimum    maximum
    //  0           1          0
    //  1           1          0
    //  2           2          0
    //  3           2          0
    //  4           1          0
    //  5           1          0
    switch (vm->pit.s.timer[timerid].mode) {
      case 0:
        vm->pit.s.timer[timerid].counter = vm->pit.s.timer[timerid].counter_max;
        vm->pit.s.timer[timerid].active = 1;
        if (vm->pit.s.timer[timerid].GATE) {
          vm->pit.s.timer[timerid].OUT = 0; // OUT pin starts low
          pitStart(vm, timerid);
          }
        break;
      case 1:
        monpanic(vm, "pit:write_count_reg(%u): mode1 unsupported\n",
                 timerid);
        break;
      case 2:
        if ( vm->pit.s.timer[timerid].counter_max == 1 )
          monpanic(vm, "pit:write_count_reg(%u): mode %u counter_max=1\n",
                   timerid, (unsigned) vm->pit.s.timer[timerid].mode);
        if ( vm->pit.s.timer[timerid].GATE && !vm->pit.s.timer[timerid].active ) {
          // software triggered
          vm->pit.s.timer[timerid].counter = vm->pit.s.timer[timerid].counter_max;
          vm->pit.s.timer[timerid].active  = 1;
          vm->pit.s.timer[timerid].OUT     = 1; // initially set high
          pitStart(vm, timerid);
          }
        break;
      case 3:
        if ( vm->pit.s.timer[timerid].counter_max == 1 )
          monpanic(vm, "pit:write_count_reg(%u): mode %u counter_max=1\n",
                   timerid, (unsigned) vm->pit.s.timer[timerid].mode);
        vm->pit.s.timer[timerid].counter_max = vm->pit.s.timer[timerid].counter_max & 0xfffe;
        if ( vm->pit.s.timer[timerid].GATE && !vm->pit.s.timer[timerid].active ) {
          // software triggered
          vm->pit.s.timer[timerid].counter = vm->pit.s.timer[timerid].counter_max;
          vm->pit.s.timer[timerid].active  = 1;
          vm->pit.s.timer[timerid].OUT     = 1; // initially set high
          pitStart(vm, timerid);
          }
        break;
      case 4:
        monpanic(vm, "pit:write_count_reg(%u): mode4 unsupported\n",
                 timerid);
        break;
      case 5:
        monpanic(vm, "pit:write_count_reg(%u): mode5 unsupported\n",
                 timerid);
        break;
      }
    }
}


  Bit8u
pitReadCounter(vm_t *vm, unsigned timerid)
{
  Bit16u  counter_value;
  Bit8u    retval;

  if (vm->pit.s.timer[timerid].output_latch_full) { /* latched read */
    counter_value = vm->pit.s.timer[timerid].output_latch_value;
    }
  else { /* direct unlatched read */
    counter_value = vm->pit.s.timer[timerid].counter;
monprint(vm, "CV=%04x\n", (unsigned) vm->pit.s.timer[timerid].counter);
    }

  switch (vm->pit.s.timer[timerid].latch_mode) {
    case BX_PIT_LATCH_MODE_LSB:
      retval = (Bit8u  ) counter_value;
      vm->pit.s.timer[timerid].output_latch_full = 0;
      break;
    case BX_PIT_LATCH_MODE_MSB:
      retval = (Bit8u  ) ( counter_value >> 8 );
      vm->pit.s.timer[timerid].output_latch_full = 0;
      break;
    case BX_PIT_LATCH_MODE_16BIT:
      if (vm->pit.s.timer[timerid].output_latch_toggle==0) { /* LSB 1st */
        retval = (Bit8u  ) counter_value;
        }
      else { /* MSB 2nd */
        retval = (Bit8u  ) ( counter_value >> 8 );
        }
      vm->pit.s.timer[timerid].output_latch_toggle = !vm->pit.s.timer[timerid].output_latch_toggle;
      if (vm->pit.s.timer[timerid].output_latch_toggle == 0)
        vm->pit.s.timer[timerid].output_latch_full = 0;
      break;
    default:
      monpanic(vm, "pit: io read from port 40h: unknown latch mode\n");
      retval = 0; /* keep compiler happy */
    }
  return( retval );
}

  void
pitLatch(vm_t *vm, unsigned timerid)
{
  /* subsequent counter latch commands are ignored until value read out */
  if (vm->pit.s.timer[timerid].output_latch_full) {
    monprint(vm, "pit: pit(%u) latch: output latch full, ignoring\n",
             timerid);
    return;
    }

  vm->pit.s.timer[timerid].output_latch_value = vm->pit.s.timer[timerid].counter;

  vm->pit.s.timer[timerid].output_latch_toggle = 0;
  vm->pit.s.timer[timerid].output_latch_full   = 1;
}

  void
pitSetGATE(vm_t *vm, unsigned pit_id, unsigned value)
{
  // GATE's for Timer 0 & Timer 1 are tied high.
  if (pit_id != 2)
    monpanic(vm, "pit:set_GATE: pit_id != 2\n");

  value = (value > 0);

  /* if no transition of GATE input line, then nothing to do */
  if (value == vm->pit.s.timer[2].GATE)
    return;

  if (value) { /* PIT2: GATE transition from 0 to 1 */
    vm->pit.s.timer[2].GATE  = 1;
    switch ( vm->pit.s.timer[2].mode ) {
      case 0:
        vm->pit.s.timer[2].counter = vm->pit.s.timer[2].counter_max;
        if (vm->pit.s.timer[2].active) {
          vm->pit.s.timer[2].OUT = 0;
          }
        pitStart(vm, 2);
        break;
      case 2:
        // begin counting, reload counter
        vm->pit.s.timer[2].active = 1;
        vm->pit.s.timer[2].OUT = 1;
        vm->pit.s.timer[2].counter = vm->pit.s.timer[2].counter_max;
        pitStart(vm, 2);
        break;
      case 3:
        // begin counting, reload counter
        vm->pit.s.timer[2].active = 1;
        vm->pit.s.timer[2].OUT = 1;
        vm->pit.s.timer[2].counter = vm->pit.s.timer[2].counter_max;
        pitStart(vm, 2);
        break;
      case 1:
      case 4:
      case 5:
      default:
        monpanic(vm, "bx_pit_c::set_GATE: unhandled timer2 mode %u\n",
                 (unsigned) vm->pit.s.timer[2].mode);
      }
    }
  else {       // PIT2: GATE transition from 1 to 0, deactivate
    vm->pit.s.timer[2].GATE  = 0;
    switch ( vm->pit.s.timer[2].mode ) {
      case 0:
        break;
      case 2:
        // 1) stops count, 2) OUT goes immediately high
        vm->pit.s.timer[2].active = 0;
        vm->pit.s.timer[2].OUT = 1;
        break;
      case 3:
        // 1) stops count, 2) OUT goes immediately high
        vm->pit.s.timer[2].active = 0;
        vm->pit.s.timer[2].OUT = 1;
        break;
      case 1:
      case 4:
      case 5:
      default:
        monpanic(vm, "bx_pit_c::set_GATE: unhandled timer2 mode %u\n",
                 (unsigned) vm->pit.s.timer[2].mode);
      }
    }
}


  void
pitStart(vm_t *vm, unsigned timerid)
{
  unsigned long period_hz;

  if (vm->pit.s.timer[timerid].counter_max == 0x0000) {
    period_hz   = 1193182 / 65536;
    }
  else {
    period_hz = 1193182 / vm->pit.s.timer[timerid].counter_max;
    }
  //monprint(vm, "timer%u period set to %lu hz\n", timerid, period_hz);


  switch (vm->pit.s.timer[timerid].mode) {
    case 0: /* single timeout */
      break;
    case 1: /* retriggerable one-shot */
      monpanic(vm, "start: mode %u unhandled\n",
               (unsigned) vm->pit.s.timer[timerid].mode);
      break;
    case 2: /* rate generator */
      break;
    case 3: /* square wave mode */
      break;
    case 4: /* software triggered strobe */
      monpanic(vm, "start: mode %u unhandled\n",
               (unsigned) vm->pit.s.timer[timerid].mode);
      break;
    case 5: /* hardware retriggerable strobe */
      monpanic(vm, "start: mode %u unhandled\n",
               (unsigned) vm->pit.s.timer[timerid].mode);
      break;
    default:
      monpanic(vm, "start: timer%u has bad mode\n",
               (unsigned) vm->pit.s.timer[timerid].mode);
    }
}


  void
pitTimerCallback(vm_t *vm, unsigned i)
{
  // xxx Fix this
  if ( pitPeriodic(vm, PitPeriodicDelta) ) {
    picTriggerIRQ(vm, 0);
    }
}

  Boolean
pitPeriodic(vm_t *vm, Bit32u usec_delta)
{
  Boolean prev_timer0_out;
  unsigned i;

  prev_timer0_out = vm->pit.s.timer[0].OUT;

  for (i = 0; i < 3; i++) {
    // is timer enabled and active?
    if ( vm->pit.s.timer[i].GATE && vm->pit.s.timer[i].active ) {
      switch ( vm->pit.s.timer[i].mode ) {
        case 0: // Mode 0: Single Timeout
          // wraps after count expires
          if ( vm->pit.s.timer[i].counter == 0 ) {
            // counter previously expired, wrap counter
            vm->pit.s.timer[i].counter = 0xffff;
            }
          else if ( usec_delta >= vm->pit.s.timer[i].counter ) {
            // counter expired
            vm->pit.s.timer[i].counter = 0;
            vm->pit.s.timer[i].OUT     = 1;
            }
          else {
            // decrement counter by elapsed useconds
            vm->pit.s.timer[i].counter -= (Bit16u ) usec_delta;
            }
          break;

        case 1: // Mode 1: Retriggerable One-Shot
          // wraps after count expires
          monpanic(vm, "bx_pit_c::periodic: bad mode: timer[%u], mode %u\n",
                        i, (unsigned) vm->pit.s.timer[i].mode);
          break;

        case 2: // Mode 2: Rate Generator
          // reloads after count expires
          // OUT is low when counter=1, high otherwise
          // min count=2, max count=0
          if ( vm->pit.s.timer[i].counter == 0 ) {
            // max counter val, just wrap
            vm->pit.s.timer[i].counter = 0xffff;
            vm->pit.s.timer[i].OUT     = 1;
            }
          else if ( vm->pit.s.timer[i].counter == 1 ) {
            // counter previously expired, reload
            vm->pit.s.timer[i].counter = vm->pit.s.timer[i].counter_max;
            vm->pit.s.timer[i].OUT     = 1;
            }
          else if ( (vm->pit.s.timer[i].counter == 2) ||
                    (usec_delta >= (((Bit32u)vm->pit.s.timer[i].counter) - 1))
                  ) {
            // in either case, counter will reach 1
            vm->pit.s.timer[i].counter = 1;
            vm->pit.s.timer[i].OUT = 0;
            }
          else {
            // decrement counter by elapsed useconds
            vm->pit.s.timer[i].counter -= (Bit16u ) usec_delta;
            }
          break;

        case 3: // Mode 3: Square Wave Mode
          // reloads after count expires
          // min count=2, max count=0
          if ( vm->pit.s.timer[i].counter == 0 ) {
            // max count, dec by 2
            vm->pit.s.timer[i].counter = 0xfffe;
            }
          else if ( (vm->pit.s.timer[i].counter <= 2) ||
                    ( (usec_delta*2) >= vm->pit.s.timer[i].counter ) ) {
            // counter expired, reload
            vm->pit.s.timer[i].counter = vm->pit.s.timer[i].counter_max;
            vm->pit.s.timer[i].OUT     = !vm->pit.s.timer[i].OUT;
            //monprint(vm, "CV: reload t%u to %04x\n", (unsigned) i, (unsigned)
            //  vm->pit.s.timer[i].counter);
            }
          else {
            // decrement counter by elapsed useconds
            vm->pit.s.timer[i].counter -= (Bit16u ) ( 2*usec_delta );
            //monprint(vm, "CV: dec count to %04x\n",
            //          (unsigned) vm->pit.s.timer[i].counter);
            }
          break;

        case 4: // Mode 4: Software Triggered Strobe
          // wraps after count expires
          monpanic(vm, "bx_pit_c::periodic: bad mode: timer[%u], mode %u\n",
                        i, (unsigned) vm->pit.s.timer[i].mode);
          break;

        case 5: // Mode 5: Hardware Retriggerable Strobe
          // wraps after count expires
          monpanic(vm, "bx_pit_c::periodic: bad mode: timer[%u], mode %u\n",
                        i, (unsigned) vm->pit.s.timer[i].mode);
          break;
        default:
          monpanic(vm, "bx_pit_c::periodic: bad mode: timer[%u], mode %u\n",
                        i, (unsigned) vm->pit.s.timer[i].mode);
          break;
        } // switch ( vm->pit.s.tim...
      } // if ( vm->pit.s.timer[i]...
    } // for (unsigned i...

  // see if there's a rising edge on timer0's output to trigger an IRQ0.
  if ( (prev_timer0_out==0) && (vm->pit.s.timer[0].OUT==1) )
    return(1); // request IRQ 0
  else
    return(0);
}
