/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  stack.c:  stack oriented instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
PUSH_Ed(vm_t *vm)
{
  Bit32u op1;

  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  push32(vm, op1);
}

  void
POP_Ed(vm_t *vm)
{
  Bit32u val32;
 
  pop32(vm, &val32);
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, val32);
    }
  else {
    /* Note: there is one little weirdism here.  When 32bit addressing */
    /* is used, it is possible to use ESP in the modrm addressing. */
    /* If used, the value of ESP after the pop is used to calculate */
    /* the address. */
    if (vm->i.as_32 && (vm->i.mod!=0xc0) &&
        (vm->i.rm==4) && (vm->i.base==4)) {
      vm->i.ResolveModrm(&vm->i, vm->guest.addr.guest_context);
      }
    write_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &val32);
    }
}

  void
POP_Ew(vm_t *vm)
{
  Bit16u val16;
 
  pop16(vm, &val16);
 
  if (vm->i.mod == 0xc0) {
    WriteReg16(vm, vm->i.rm, val16);
    }
  else {
    /* Note: there is one little weirdism here.  When 32bit addressing */
    /* is used, it is possible to use ESP in the modrm addressing. */
    /* If used, the value of ESP after the pop is used to calculate */
    /* the address. */
    if (vm->i.as_32 && (vm->i.mod!=0xc0) &&
        (vm->i.rm==4) && (vm->i.base==4)) {
      vm->i.ResolveModrm(&vm->i, vm->guest.addr.guest_context);
      }
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &val16);
    }
}

  void
PUSH_Iv(vm_t *vm)
{
  if (vm->i.os_32) {
    push32(vm, vm->i.Id);
    }
  else {
    push16(vm, vm->i.Iw);
    }
}

  void
PUSH_Ew(vm_t *vm)
{
  Bit16u op1;

  if (vm->i.mod == 0xc0) {
    op1 = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  push16(vm, op1);
}

  void
PUSH_SREG(vm_t *vm, unsigned sreg)
{
  cache_selector(vm, sreg);
  if (vm->i.os_32)
    push32(vm, vm->guest_cpu.selector[sreg].raw);
  else
    push16(vm, vm->guest_cpu.selector[sreg].raw);
}

  void
LEAVE(vm_t *vm)
{
  Bit32u temp_EBP;
  unsigned d_b;

  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;
  if (d_b)
    temp_EBP = G_EBP(vm);
  else
    temp_EBP = G_BP(vm);

  if ( ProtectedMode(vm) ) {
    if (vm->guest_cpu.desc_cache[SRegSS].desc.type & D_EXDOWN) {
      if (temp_EBP <= vm->guest_cpu.desc_cache[SRegSS].limit_scaled) {
        monpanic(vm, "LEAVE: eBP > SS.limit\n");
        exception(vm, ExceptionSS, 0);
        }
      }
    else { /* normal */
      if (temp_EBP > vm->guest_cpu.desc_cache[SRegSS].limit_scaled) {
        monpanic(vm, "LEAVE: eBP > SS.limit\n");
        exception(vm, ExceptionSS, 0);
        }
      }
    }


  /* delete frame */
  if (d_b)
    G_ESP(vm) = G_EBP(vm);
  else
    G_SP(vm) = G_BP(vm);

  /* restore frame pointer */
  if (vm->i.os_32) {
    Bit32u temp32;

    pop32(vm, &temp32);
    G_EBP(vm) = temp32;
    }
  else {
    Bit16u temp16;

    pop16(vm, &temp16);
    G_BP(vm) = temp16;
    }
}

  void
PUSHAD32(vm_t *vm)
{
  Bit32u esp;
 
  esp = G_ESP(vm);
 
  /* +++ other checks here */
  push32(vm, G_EAX(vm));
  push32(vm, G_ECX(vm));
  push32(vm, G_EDX(vm));
  push32(vm, G_EBX(vm));
  push32(vm, esp);
  push32(vm, G_EBP(vm));
  push32(vm, G_ESI(vm));
  push32(vm, G_EDI(vm));
}

  void
PUSHAD16(vm_t *vm)
{
  Bit16u sp;
 
  sp = G_SP(vm);
 
  /* +++ other checks here */
  push16(vm, G_AX(vm));
  push16(vm, G_CX(vm));
  push16(vm, G_DX(vm));
  push16(vm, G_BX(vm));
  push16(vm, sp);
  push16(vm, G_BP(vm));
  push16(vm, G_SI(vm));
  push16(vm, G_DI(vm));
}

  void
POPAD32(vm_t *vm)
{
  Bit32u edi, esi, ebp, etmp, ebx, edx, ecx, eax;
 
  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 32) ) {
      monpanic(vm, "pop_ad: not enough bytes on stack\n");
      exception(vm, ExceptionSS, 0);
      }
    }
 
  /* ??? optimize this */
  pop32(vm, &edi);
  pop32(vm, &esi);
  pop32(vm, &ebp);
  pop32(vm, &etmp); /* value for SP discarded */
  pop32(vm, &ebx);
  pop32(vm, &edx);
  pop32(vm, &ecx);
  pop32(vm, &eax);
 
  G_EDI(vm) = edi;
  G_ESI(vm) = esi;
  G_EBP(vm) = ebp;
  G_EBX(vm) = ebx;
  G_EDX(vm) = edx;
  G_ECX(vm) = ecx;
  G_EAX(vm) = eax;
}

  void
POPAD16(vm_t *vm)
{
  Bit16u di, si, bp, tmp, bx, dx, cx, ax;
 
  if (ProtectedMode(vm)) {
    if ( !can_pop(vm, 16) ) {
      monpanic(vm, "pop_a: not enough bytes on stack\n");
      exception(vm, ExceptionSS, 0);
      }
    }
 
  /* ??? optimize this */
  pop16(vm, &di);
  pop16(vm, &si);
  pop16(vm, &bp);
  pop16(vm, &tmp); /* value for SP discarded */
  pop16(vm, &bx);
  pop16(vm, &dx);
  pop16(vm, &cx);
  pop16(vm, &ax);
 
  G_DI(vm) = di;
  G_SI(vm) = si;
  G_BP(vm) = bp;
  G_BX(vm) = bx;
  G_DX(vm) = dx;
  G_CX(vm) = cx;
  G_AX(vm) = ax;
}


  void
ENTER_IwIb(vm_t *vm)
{
  Bit32u frame_ptr32;
  /* Bit16u frame_ptr16; */
  Bit8u level;
  unsigned d_b;

  invalidate_prefetch_q();

  level = vm->i.Ib2;
  level %= 32;

#if 0
if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b && vm->i.os_32==0) {
  bx_printf("enter(): stacksize!=opsize: I'm unsure of the code for this\n");
  bx_panic("         The Intel manuals are a mess on this one!\n");
  }
#endif


  cache_sreg(vm, SRegSS);
  d_b = vm->guest_cpu.desc_cache[SRegSS].desc.d_b;

  if ( ProtectedMode(vm) ) {
    Bit32u bytes_to_push, temp_ESP;

    if (level == 0) {
      if (vm->i.os_32)
        bytes_to_push = 4 + vm->i.Iw;
      else
        bytes_to_push = 2 + vm->i.Iw;
      }
    else { /* level > 0 */
      if (vm->i.os_32)
        bytes_to_push = 4 + (level-1)*4 + 4 + vm->i.Iw;
      else
        bytes_to_push = 2 + (level-1)*2 + 2 + vm->i.Iw;
      }
    if (d_b)
      temp_ESP = G_ESP(vm);
    else
      temp_ESP = G_SP(vm);
    if ( !can_push(vm, &vm->guest_cpu.desc_cache[SRegSS],
                   temp_ESP, bytes_to_push) ) {
      monpanic(vm, "ENTER: not enough room on stack!\n");
      exception(vm, ExceptionSS, 0);
      }
    }

  if (vm->i.os_32)
    push32(vm, G_EBP(vm));
  else
    push16(vm, G_BP(vm));

  /* can just do frame_ptr32 = ESP for either case ??? */
  if (d_b)
    frame_ptr32 = G_ESP(vm);
  else
    frame_ptr32 = G_SP(vm);

  if (level > 0) {
    monpanic(vm, "ENTER: level > 0\n");

#if 0
    /* do level-1 times */
    while (--level) {
      if (vm->i.os_32) {
        Bit32u temp32;

        if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b) { /* 32bit stacksize */
          EBP -= 4;
          read_virtual_dword(BX_SEG_REG_SS, EBP, &temp32);
          ESP -= 4;
          write_virtual_dword(BX_SEG_REG_SS, ESP, &temp32);
          }
        else { /* 16bit stacksize */
          BP -= 4;
          read_virtual_dword(BX_SEG_REG_SS, BP, &temp32);
          SP -= 4;
          write_virtual_dword(BX_SEG_REG_SS, SP, &temp32);
          }
        }
      else { /* 16bit opsize */
        Bit16u temp16;

        if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b) { /* 32bit stacksize */
          EBP -= 2;
          read_virtual_word(BX_SEG_REG_SS, EBP, &temp16);
          ESP -= 2;
          write_virtual_word(BX_SEG_REG_SS, ESP, &temp16);
          }
        else { /* 16bit stacksize */
          BP -= 2;
          read_virtual_word(BX_SEG_REG_SS, BP, &temp16);
          SP -= 2;
          write_virtual_word(BX_SEG_REG_SS, SP, &temp16);
          }
        }
      } /* while (--level) */

    /* push(frame pointer) */
    if (vm->i.os_32) {
      if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b) { /* 32bit stacksize */
        ESP -= 4;
        write_virtual_dword(BX_SEG_REG_SS, ESP, &frame_ptr32);
        }
      else {
        SP -= 4;
        write_virtual_dword(BX_SEG_REG_SS, SP, &frame_ptr32);
        }
      }
    else { /* 16bit opsize */
      if (BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.u.segment.d_b) { /* 32bit stacksize */
        frame_ptr16 = frame_ptr32;
        ESP -= 2;
        write_virtual_word(BX_SEG_REG_SS, ESP, &frame_ptr16);
        }
      else {
        frame_ptr16 = frame_ptr32;
        SP -= 2;
        write_virtual_word(BX_SEG_REG_SS, SP, &frame_ptr16);
        }
      }
#endif
    } /* if (level > 0) ... */

  if (vm->i.os_32)
    G_EBP(vm) = frame_ptr32;
  else
    G_BP(vm) = frame_ptr32;

  if (d_b)
    G_ESP(vm) -= vm->i.Iw;
  else
    G_SP(vm) -= vm->i.Iw;
}
