/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  handlers-r3.c: Ring3 DT assist routines.  Rather than inline
 *    similar code in a lot of DT sequences, common code is centralized
 *    in special ring3 handlers which are called from the DT code.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


/*
 * xxx Make __r3h_branch_dynamic asm code compliant with end-of-row rules.
 * xxx   Make sure not to backpatch.  If use the static code, must pass
 * xxx   an extra flag to turn off backpatching.
 * xxx Make sure all functions here are tagged with __attribute__ clause
 */

#include "plex86.h"
#define IN_R3_SPACE
#include "monitor.h"


/* Notes:
 *   - This code executes at ring3 and is called by the translated
 *     guest instructions (tcode), without need for a context switch.
 *   - Do not make calls to any functions outside of ones offered in
 *     the "r3hSection" (this module), because only pages comprising
 *     that section are mapped with readable permissions by ring3.
 *   - We do not have access to monitor structures/function calls here.
 *     Only use the "r3hData" structure pointer, and values passed to
 *     these functions on the stack.  No print/panic functions exist
 *     in this space currently.
 */


/* Put this special ring3 handler code in its own section.  Start
 * the section on a page boundary, and end on one too.  This way
 * we can map this section of code into the monitor with read access
 * even for the guest (ring3), without giving the guest any access
 * to other monitor code/data pages.
 */

asm (
  ".section  r3hSection,\"ax\",@progbits \n\t"
  ".globl __r3hSectionStart \n\t"
  "__r3hSectionStart: \n\t"
  );


/* xxx Fix this comment. */

/*  This represents the C component of the dynamic branch convenience
 *  routine.  I just pretend the targets of the branches are already
 *  translated in the cache, and use a simple switch statement to
 *  get the translated code address.  Real table lookup logic would
 *  go here.  This should be coded extremely efficiently, as it will
 *  be executed a lot.
 *
 *  If we opened the code cache pages and related tables up for write
 *  permissions from ring3 (and thus the ring3 convenience handlers),
 *  conceivably we could code the translation here, kicking in upon
 *  cache misses.  Otherwise, we would defer to the monitor at ring0.
 */

  void
r3hBranchStatic(void)
{
  r3hData_t *r3hData = (r3hData_t *) &__r3hStubsDataStart;
  unsigned hashRow, hashCol;
  dtG2THash_t  *dtG2THash = r3hData->dtG2THash;
  Bit32u unsigned gOff, *r3hBranchDataPtr;

  /*
   *  Search the G2T table first, ideally the instruction will have
   *  been translated already, and the translation address is in
   *  there.
   */
  r3hBranchDataPtr = (Bit32u*) __r3hBranchDataAddr;
  gOff = r3hBranchDataPtr[0];

  /* similar code as in dtTranslateG2T() */
  hashRow = DT_G2THashSelect(gOff);

  if ((*dtG2THash)[hashRow][0].tOff == TcodeOffsetNone) {
    /* First entry in Row was marked empty.  This is our signal to mark the
     * rest of the entries in the row as empty.
     */
    for (hashCol = 0; hashCol < DT_G2THashWidth; hashCol++) {
      (*dtG2THash)[hashRow][hashCol].tOff = TcodeOffsetNone;
      }
    hashCol = 0; /* Signal that 0th entry is available */
    }
  else {
    for (hashCol = 0; hashCol < DT_G2THashWidth; hashCol++) {
      /* If the guest offset matches, and the tcode address is valid, then
       * return the tcode address.
       */
      if ( ((*dtG2THash)[hashRow][hashCol].gOff == gOff) &&
           ((*dtG2THash)[hashRow][hashCol].tOff != TcodeOffsetNone) ) {
        /* xxx It might be smart to reorder this entry to the beginning
         * of the list for performance.
         */
        r3hData->r3hTargetEIP = (*dtG2THash)[hashRow][hashCol].tOff;
        InstrG2THit(hashCol);
        goto done;
        }
      if ( (*dtG2THash)[hashRow][hashCol].tOff == TcodeOffsetNone ) {
        // xxx Should we bother with this check?
        break;
        }
      }
    }

  InstrG2TMiss();

  /* Guest to Tcode address pair not in G2T hash table. */
  /* xxx Should look in L2M hash table first here. */
  r3hData->r3hTargetEIP = (Bit32u) __r3hNewEIP;
  r3hData->r3hData = (Bit32u) gOff;
  return;

done:

#if DT_UseBackpatch
  /*
   *  We can now backpatch the current context switch ID and
   *  direct tcode branch into the branch instruction.
   */

  r3hBranchDataPtr[1] = r3hData->hashID;
  r3hBranchDataPtr[2] = r3hData->r3hTargetEIP;
#endif
}



#if 0
  void
r3hBranchDynamic(gc_t gc, Bit32u tcodeRetOff, unsigned gOff)
{
  r3hData_t *r3hData = (r3hData_t *) &__r3hStubsDataStart;
  //unsigned hashRow, hashCol;
  //dtG2THash_t  *dtG2THash = r3hData->dtG2THash;

  /* The asm stubs look in the G2T hash table.  No match was found. */

  InstrG2TMiss();

  /* Guest to Tcode address pair not in G2T hash table. */
  /* xxx Should look in L2M hash table first here */
  r3hData->r3hTargetEIP = (Bit32u) __r3hNewEIP;
  r3hData->r3hData = gOff;
  return;
}
#endif



asm (
  ".section  r3hSection,\"ax\",@progbits \n\t"
  ".p2align 12 \n\t"
  ".globl __r3hSectionEnd \n\t"
  "__r3hSectionEnd: \n\t"
  );
