/* Some examples of how the DT engine would generate code for
 * various non pass-through instructions.  The real code would be
 * generated in binary, of course.
 */

/* This preamble is some common code which switches over to the
 * guest stack, and saves the EIP of the instruction in question,
 * for potential exception handling.
 */
#define PREAMBLE(guestSrcEIP) \
  cs; mov  handler_DS, %ss ;         /* Load handler SS. */ \
  ss; movl %esp,  guest_ESP ;        /* Save guest ESP */ \
  ss; movl $guestSrcEIP, guest_EIP ; /* Save guest source EIP */ \
  ss; movl handler_ESP, %esp         /* Fully loaded handler SS:ESP */

/* And some postamble code for switching back to the guest stack. */
#define POSTAMBLE() \
  cs; mov guest_SS,  %ss ; \
  cs; mov guest_ESP, %esp


/* Perhaps a special data page for use by the handler routines. */
.data
__target_ptr: ; .skip 4,0
__temp16:     ; .skip 2,0


/* The psuedo translated code sequences, actually generated by hand. */
.text

/*
 * =====================
 * jmp static-oop-target
 * =====================
 *
 * Notes: This sequence concludes the one we are currently building.
 *   It is just a standard preamble to switch to the handler stack,
 *   and save the source EIP value, a push of the target EIP value,
 *   and a jump to the ring3 branch handler.
 */


.globl __seq0000
__seq0000:
  PREAMBLE(0xa0000000)
  pushl $0xa0000001         /* Guest target offset */
  jmp  __r3h_branch         /* Transfer to convenience handler */

/*
 * =====================
 * jc  static-oop-target
 * =====================
 *
 * Notes: For conditional branches, an approach would be to negate
 *   the condition, generate code to handle the branch inline for the
 *   branch-taken case, then route the negated branch to the end of the
 *   generated sequence.  This would yield one sequence.  However, we
 *   should observe the original structure of the conditional branch
 *   in case the compiler that generated it, did so for optimization
 *   reasons.  We can simply build a separate sequence which handles
 *   the branch-taken case, and generate a conditional branch to that
 *   sequence, preserving the structure of the original branch.  Since
 *   this 2nd sequence is a terminal, we could emit it to the translation
 *   buffer (and thus know its address) before emitting the code sequence
 *   we are currently building, which may be a long sequence possibly
 *   containing many conditionals.
 */

__seq0001_branch:
  /* This terminal sequence is generated specifically to handle the
   * jump taken case for __seq0001.  It is built as a separate
   * sequence.  Since it is a terminal, it could be emitted to the
   * translation buffer prior to the sequence we are currently
   * building directly below at seq0001.
   */
  PREAMBLE(0xa0000001)
  pushl $0xa0000002                /* Guest target offset */
  jmp  __r3h_branch                /* Transfer to convenience handler */

.globl __seq0001
__seq0001:
  /* Prime the eflags.cf bit with a sample value causing the branch */
  stc

  /* This would be the actual generated code */
  jc   __seq0001_branch
  /* ... control flows to next translated instruction */

  /* should not get here for our specific case */
  hlt


/*
 * =====================
 * jmp ds:[edi]
 * =====================
 *
 * Notes: Dynamic (calculated) branch target.  As long as the
 *   calcuations don't use the stack (SS or ESP), then this is
 *   handled much like the static OOP case above.  Simply push
 *   the target address on the handler stack using the native
 *   address generation/operand fetch as would the jmp.
 */

.globl __seq0002
__seq0002:
  /* Prime the sequence with sample guest values */
  movl $0xa0000003, __target_ptr
  movl $__target_ptr, %edi

  /* This would be the actual generated code */
  PREAMBLE(0xa0000002)
  pushl (%edi)              /* Push using native addressing/operand fetch */
  jmp  __r3h_branch         /* Transfer to convenience handler */


/*
 * =====================
 * jmp ss:[edi]
 * =====================
 *
 * Notes: Dynamic (calculated) branch target, operand addressing
 *   uses stack.  Since we switch to the handler stack, we can't use
 *   the operand as-is.  But we can save/restore the guest DS and
 *   use it for the operand instead.
 */

.globl __seq0003
__seq0003:
  /* Prime the sequence with sample guest values */
  movl $0xa0000004, __target_ptr
  movl $__target_ptr, %edi

  /* This would be the actual generated code */
  PREAMBLE(0xa0000003)

  /* Code to use the guest DS to access the operand instead of SS */
  ss; movw %ds, __temp16    /* Temporarily save guest DS */
  ss; movw guest_SS, %ds    /* Move guest SS -> DS */
  ds; pushl (%edi)          /* Push using native addressing/operand fetch */
  ss; movw __temp16, %ds    /* Restore guest DS */

  jmp  __r3h_branch         /* Transfer to convenience handler */


/*
 * =====================
 * movw %ds, %ax
 * =====================
 *
 * Notes: Handling reads of selectors.  Assume for this example, that
 *   we are translating ring0 guest code, which we will run at ring3,
 *   but use the expected selector values.  We need to modify the
 *   selector RPL bits to 0, as seen by the read instruction.
 */

.globl __seq0004
__seq0004:
  /* This would be the actual generated code */
  PREAMBLE(0xa0000004)

  pushfl                /* Save guest eflags */
  movw   %ds, %ax       /* Do selector to register move */
  andw   $0xfffc, %ax   /* Zero out RPL bits for ring0 */
  popfl                 /* Restore guest eflags */

  POSTAMBLE()

  /* End of simulated tests */
  call __exit_ok


/*
 * ======================================================
 * Handling of translating code from 16-bit code segments
 * ======================================================
 *
 * Notes: I need to fill out some examples here.  But in short,
 *   just use a 32-bit segment for all translated code sequences,
 *   and insert OPSIZE and ADDRSIZE prefixes when not negated by
 *   such prefixes actually in the guest code.  This way we can
 *   easily use an arbitrarily sized translation buffer, and 32-bit
 *   offset branches to handler routines.
 */
