/***********************************************************************************
* Adjustable Clock: Plasmoid to show date and time in adjustable format.
* Copyright (C) 2008 - 2010 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "AdjustableClock.h"

#include <QRegExp>
#include <QWebPage>
#include <QWebFrame>
#include <QClipboard>

#include <KMenu>
#include <KLocale>
#include <KMessageBox>
#include <KColorDialog>
#include <KInputDialog>
#include <KConfigDialog>
#include <KCalendarSystem>
#include <KSystemTimeZones>

#include <Plasma/Theme>
#include <Plasma/Containment>


K_EXPORT_PLASMA_APPLET(adjustableclock, AdjustableClock)

AdjustableClock::AdjustableClock(QObject *parent, const QVariantList &args) : ClockApplet(parent, args),
        m_clipboardAction(NULL)
{
    KGlobal::locale()->insertCatalog("libplasmaclock");
    KGlobal::locale()->insertCatalog("timezones4");
    KGlobal::locale()->insertCatalog("adjustableclock");

    setHasConfigurationInterface(true);
    resize(150, 80);

    m_timeFormatStrings << "<div style=\"text-align:center; margin:5px; white-space:pre;\"><big>%H:%M:%S</big>\n<small>%d.%m.%Y</small></div>"
    << "<div style=\"text-align:center; margin:5px; white-space:pre;\"><big style=\"font-family:'Nimbus Sans L Condensed';\">%H:%M:%S</big>\n<span style=\"font-size:small; font-family:'Nimbus Sans L';\">%d.%m.%Y</small></div>"
    << "<div style=\"text-align:center; white-space:pre; font-size:25px; margin:5px;\">%H:%M</div>"
    << "<div style=\"text-align:center; white-space:pre; opacity:0.85; background:none;\"><span style=\"font-size:30px;\">%H:%M:%S</span><br><span style=\"font-size:12px;\">%A, %d.%m.%Y</span></div>"
    << "<div style=\"text-align:center; white-space:pre; font-size:25px; margin:0 0 5px 5px; background:none;\">%H:%M<span style=\"font-size:30px; position:relative; left:-8px; top:4px; z-index:-1; opacity:0.5;\">%S</span></div>"
    << "<div style=\"height:50px; background:none;\"><div style=\"text-align:center; white-space:pre; font-size:25px; margin:-10px 0 5px 5px; -webkit-box-reflect:below -5px -webkit-gradient(linear, left top, left bottom, from(transparent), color-stop(0.5, transparent), to(white));\">%H:%M<span style=\"font-size:30px; position:relative; left:-8px; top:4px; z-index:-1; opacity:0.5;\">%S</span></div></div>";

    m_timeFormatNames << i18n("Default")
    << i18n("Flat")
    << i18n("Simple")
    << i18n("Verbose")
    << i18n("dbClock")
    << i18n("dbClock with reflection");

    m_clipboardFormats << "%x"
    << "%f"
    << "%H:%M:%S"
    << QString()
    << "%X"
    << "%F"
    << QString()
    << "%c"
    << "%C"
    << "%Y-%m-%d %H:%M:%S"
    << QString()
    << "%t";
}

void AdjustableClock::init()
{
    ClockApplet::init();

    m_timeFormat = config().readEntry("timeFormat", m_timeFormatStrings.at(0));

    m_page.mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);
    m_page.mainFrame()->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);

    updateTheme();
    connectSource(currentTimezone());
    setText(formatDateTime(currentDateTime(), m_timeFormat));
    constraintsEvent(Plasma::SizeConstraint);

    connect(this, SIGNAL(activate()), this, SLOT(copyToClipboard()));
    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), this, SLOT(updateTheme()));
}

void AdjustableClock::dataUpdated(const QString &source, const Plasma::DataEngine::Data &data)
{
    Q_UNUSED(source)

    m_dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());
    m_sunrise = data["Sunrise"].toTime();
    m_sunset = data["Sunset"].toTime();

    setText(formatDateTime(m_dateTime, m_timeFormat));

    if (Plasma::ToolTipManager::self()->isVisible(this)) {
        updateToolTipContent();
    }
}

void AdjustableClock::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::SizeConstraint) {
        updateSize();
    }

    setBackgroundHints(QRegExp("<[a-z].*\\sstyle=('|\").*background(-image)?\\s*:\\s*none.*('|\").*>", Qt::CaseInsensitive).exactMatch(m_timeFormat) ? NoBackground : DefaultBackground);
}

void AdjustableClock::paintInterface(QPainter *painter, const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED(option)
    Q_UNUSED(contentsRect)

    painter->setRenderHint(QPainter::SmoothPixmapTransform);

    m_page.mainFrame()->render(painter);
}

void AdjustableClock::createClockConfigurationInterface(KConfigDialog *parent)
{
    KConfigGroup configuration = config();
    QStringList timeFormatStrings = m_timeFormatStrings;
    QStringList timeFormatNames = m_timeFormatNames;
    QStringList clipboardFormats = configuration.readEntry("clipboardFormats", m_clipboardFormats);
    QString preview;
    int row;

    QWidget *appearanceConfiguration = new QWidget;
    m_appearanceUi.setupUi(appearanceConfiguration);

    QWidget *clipboardActions = new QWidget;
    m_clipboardUi.setupUi(clipboardActions);

    KMenu *placeholdersMenu = new KMenu(m_appearanceUi.placeholdersButton);

    QMap<QChar, QString> placeholders;

    placeholders['H'] = i18n("The hour in 24h format with two digits");
    placeholders['k'] = i18n("The hour in 24h format");
    placeholders['I'] = i18n("The hour in 12h format with two digits");
    placeholders['l'] = i18n("The hour in 12h format");
    placeholders['M'] = i18n("The minute number with two digits");
    placeholders['S'] = i18n("The seconds number with two digits");
    placeholders['s'] = i18n("The seconds number");
    placeholders['p'] = i18n("The pm or am string");
    placeholders['Y'] = i18n("The whole year");
    placeholders['y'] = i18n("The lower two digits of the year");
    placeholders['n'] = i18n("The month number");
    placeholders['m'] = i18n("The month number with two digits");
    placeholders['e'] = i18n("The day of the month number");
    placeholders['d'] = i18n("The day of the month number with two digits");
    placeholders['b'] = i18n("The short form of the month name");
    placeholders['B'] = i18n("The long form of the month name");
    placeholders['a'] = i18n("The short form of the weekday name");
    placeholders['A'] = i18n("The long form of the weekday name");
    placeholders['g'] = i18n("The timezone city string");
    placeholders['z'] = i18n("The timezone offset to UTC");
    placeholders['Z'] = i18n("The timezone abbreviation string");
    placeholders['j'] = i18n("The day of year number");
    placeholders['w'] = i18n("The weekday number");
    placeholders['W'] = i18n("The week number");
    placeholders['c'] = i18n("The default short date and time format");
    placeholders['C'] = i18n("The default long date and time format");
    placeholders['x'] = i18n("The default long date format");
    placeholders['X'] = i18n("The default long time format");
    placeholders['f'] = i18n("The default short date format");
    placeholders['F'] = i18n("The default short time format");
    placeholders['t'] = i18n("The UNIX timestamp");
    placeholders['o'] = i18n("The sunrise time");
    placeholders['O'] = i18n("The sunset time");

    QMap<QChar, QString>::const_iterator iterator = placeholders.constBegin();

    while (iterator != placeholders.constEnd()) {
        QAction *action = placeholdersMenu->addAction(QString("%1 (%%2)\t%3").arg(iterator.value()).arg(iterator.key()).arg(formatDateTime(m_dateTime, (QString('%').append(iterator.key())))));
        action->setData(iterator.key());

        ++iterator;
    }

    timeFormatStrings.append(configuration.readEntry("timeFormatStrings", QStringList()));

    timeFormatNames.append(configuration.readEntry("timeFormatNames", QStringList()));

    for (int i = 0; i < timeFormatStrings.count(); ++i) {
        if (i == m_timeFormatNames.count()) {
            m_appearanceUi.timeFormatComboBox->insertSeparator(i);
        }

        m_appearanceUi.timeFormatComboBox->addItem(timeFormatNames.at(i), timeFormatStrings.at(i));
    }

    QPalette webViewPalette = m_appearanceUi.webView->page()->palette();
    webViewPalette.setBrush(QPalette::Base, Qt::transparent);

    m_appearanceUi.webView->setAttribute(Qt::WA_OpaquePaintEvent, false);
    m_appearanceUi.webView->page()->setPalette(webViewPalette);
    m_appearanceUi.webView->page()->setContentEditable(true);
    m_appearanceUi.addButton->setIcon(KIcon("list-add"));
    m_appearanceUi.removeButton->setIcon(KIcon("list-remove"));
    m_appearanceUi.placeholdersButton->setIcon(KIcon("chronometer"));
    m_appearanceUi.placeholdersButton->setMenu(placeholdersMenu);
    m_appearanceUi.boldButton->setIcon(KIcon("format-text-bold"));
    m_appearanceUi.italicButton->setIcon(KIcon("format-text-italic"));
    m_appearanceUi.underlineButton->setIcon(KIcon("format-text-underline"));
    m_appearanceUi.justifyLeftButton->setIcon(KIcon("format-justify-left"));
    m_appearanceUi.justifyCenterButton->setIcon(KIcon("format-justify-center"));
    m_appearanceUi.justifyRightButton->setIcon(KIcon("format-justify-right"));

    m_clipboardUi.moveUpButton->setIcon(KIcon("arrow-up"));
    m_clipboardUi.moveDownButton->setIcon(KIcon("arrow-down"));
    m_clipboardUi.fastCopyFormat->setText(config().readEntry("fastCopyFormat", "%Y-%m-%d %H:%M:%S"));

    for (int i = 0; i < clipboardFormats.count(); ++i) {
        row = m_clipboardUi.clipboardActionsTable->rowCount();
        m_clipboardUi.clipboardActionsTable->insertRow(row);
        m_clipboardUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(clipboardFormats.at(i)));

        preview = formatDateTime(m_dateTime, clipboardFormats.at(i));

        QTableWidgetItem *item = new QTableWidgetItem(preview);
        item->setFlags(0);
        item->setToolTip(preview);

        m_clipboardUi.clipboardActionsTable->setItem(row, 1, item);
    }

    updateControls();
    itemSelectionChanged();

    QPalette buttonPalette = m_appearanceUi.colorButton->palette();
    buttonPalette.setBrush(QPalette::Button, Qt::black);

    m_appearanceUi.colorButton->setPalette(buttonPalette);

    parent->addPage(appearanceConfiguration, i18n("Appearance"), "preferences-desktop-theme");
    parent->addPage(clipboardActions, i18n("Clipboard actions"), "edit-copy");

    m_controlsTimer = new QTimer(this);
    m_controlsTimer->setInterval(250);
    m_controlsTimer->setSingleShot(true);

    connect(m_controlsTimer, SIGNAL(timeout()), this, SLOT(updateControls()));
    connect(parent, SIGNAL(finished()), m_controlsTimer, SLOT(stop()));
    connect(placeholdersMenu, SIGNAL(triggered(QAction*)), this, SLOT(insertPlaceholder(QAction*)));
    connect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadFormat(int)));
    connect(m_appearanceUi.addButton, SIGNAL(clicked()), this, SLOT(addFormat()));
    connect(m_appearanceUi.removeButton, SIGNAL(clicked()), this, SLOT(removeFormat()));
    connect(m_appearanceUi.webView->page(), SIGNAL(selectionChanged()), this, SLOT(selectionChanged()));
    connect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()), this, SLOT(changeFormat()));
    connect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(changeFormat()));
    connect(m_appearanceUi.boldButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.italicButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.underlineButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyLeftButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyCenterButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyRightButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.colorButton, SIGNAL(clicked()), this, SLOT(selectColor()));
    connect(m_appearanceUi.fontSizeComboBox, SIGNAL(editTextChanged(QString)), this, SLOT(selectFontSize(QString)));
    connect(m_appearanceUi.fontFamilyComboBox, SIGNAL(currentFontChanged(QFont)), this, SLOT(selectFontFamily(QFont)));
    connect(m_clipboardUi.addButton, SIGNAL(clicked()), this, SLOT(insertRow()));
    connect(m_clipboardUi.deleteButton, SIGNAL(clicked()), this, SLOT(deleteRow()));
    connect(m_clipboardUi.moveUpButton, SIGNAL(clicked()), this, SLOT(moveRowUp()));
    connect(m_clipboardUi.moveDownButton, SIGNAL(clicked()), this, SLOT(moveRowDown()));
    connect(m_clipboardUi.clipboardActionsTable, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
    connect(m_clipboardUi.clipboardActionsTable, SIGNAL(cellChanged(int, int)), this, SLOT(updateRow(int, int)));

    int currentFormat = m_appearanceUi.timeFormatComboBox->findData(m_timeFormat);

    if (currentFormat < 0) {
        m_appearanceUi.timeFormatComboBox->insertItem(0, i18n("Custom"), m_timeFormat);

        currentFormat = 0;
    }

    m_appearanceUi.timeFormatComboBox->setCurrentIndex(currentFormat);

    loadFormat(currentFormat);
}

void AdjustableClock::clockConfigAccepted()
{
    KConfigGroup configuration = config();
    QStringList timeFormatStrings;
    QStringList timeFormatNames;
    QStringList clipboardFormats;

    m_controlsTimer->deleteLater();

    dataEngine("time")->disconnectSource((currentTimezone() + "|Solar"), this);

    for (int i = 0; i < m_appearanceUi.timeFormatComboBox->count(); ++i) {
        if (m_timeFormatNames.contains(m_appearanceUi.timeFormatComboBox->itemText(i)) || m_appearanceUi.timeFormatComboBox->itemText(i).isEmpty()) {
            continue;
        }

        timeFormatStrings.append(m_appearanceUi.timeFormatComboBox->itemData(i).toString());

        timeFormatNames.append(m_appearanceUi.timeFormatComboBox->itemText(i));
    }

    for (int i = 0; i < m_clipboardUi.clipboardActionsTable->rowCount(); ++i) {
        clipboardFormats.append(m_clipboardUi.clipboardActionsTable->item(i, 0)->text());
    }

    m_timeFormat = m_appearanceUi.timeFormat->toPlainText();

    configuration.writeEntry("timeFormat", m_timeFormat);
    configuration.writeEntry("timeFormatStrings", timeFormatStrings);
    configuration.writeEntry("timeFormatNames", timeFormatNames);
    configuration.writeEntry("clipboardFormats", clipboardFormats);
    configuration.writeEntry("fastCopyFormat", m_clipboardUi.fastCopyFormat->text());

    connectSource(currentTimezone());
    updateSize();

    emit configNeedsSaving();
}

void AdjustableClock::connectSource(const QString &timezone)
{
    QRegExp formatWithSeconds = QRegExp("%\\d*(S|c|C|t|F|X)");
    const bool alignToSeconds = (m_timeFormat.contains(formatWithSeconds) || config().readEntry("toolTipFormat", "<div style=\"text-align:center;\">%Y-%m-%d<br />%H:%M:%S</div>").contains(formatWithSeconds));

    dataEngine("time")->connectSource((timezone + "|Solar"), this, (alignToSeconds ? 1000 : 60000), (alignToSeconds ? Plasma::NoAlignment : Plasma::AlignToMinute));

    m_timeZoneAbbreviation = KSystemTimeZones::zone(timezone).abbreviation(QDateTime::currentDateTime().toUTC());

    if (m_timeZoneAbbreviation.isEmpty()) {
        m_timeZoneAbbreviation = i18n("UTC");
    }

    int seconds = KSystemTimeZones::zone(currentTimezone()).currentOffset();
    int minutes = abs(seconds / 60);
    int hours = abs(minutes / 60);

    minutes = (minutes - (hours * 60));

    m_timeZoneOffset = QString::number(hours);

    if (minutes) {
        m_timeZoneOffset.append(':');

        if (minutes < 10) {
            m_timeZoneOffset.append('0');
        }

        m_timeZoneOffset.append(QString::number(minutes));
    }

    m_timeZoneOffset = (QChar((seconds >= 0) ? '+' : '-') + m_timeZoneOffset);

    updateSize();
}

void AdjustableClock::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->buttons() == Qt::MidButton) {
        copyToClipboard();
    }

    ClockApplet::mousePressEvent(event);
}

void AdjustableClock::copyToClipboard()
{
    QApplication::clipboard()->setText(formatDateTime(currentDateTime(), config().readEntry("fastCopyFormat", "%Y-%m-%d %H:%M:%S")));
}

void AdjustableClock::insertPlaceholder(QAction *action)
{
    QString placeholder = QString('%').append(action->data().toChar());

    if (m_appearanceUi.tabWidget->currentIndex() > 0) {
        m_appearanceUi.timeFormat->insertPlainText(placeholder);
    } else {
        m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('inserthtml', false, '" + placeholder + "')");
    }
}

void AdjustableClock::loadFormat(int index)
{
    m_appearanceUi.timeFormat->setPlainText(m_appearanceUi.timeFormatComboBox->itemData(index).toString());
    m_appearanceUi.removeButton->setEnabled(index >= m_timeFormatNames.count());
}

void AdjustableClock::changeFormat()
{
    QString text;

    if (sender() == m_appearanceUi.webView->page()) {
        QRegExp fontSize = QRegExp(" class=\"Apple-style-span\"");
        QRegExp fontColor = QRegExp("<font color=\"(#?[\\w\\s]+)\">(.+)</font>");
        fontColor.setMinimal(true);

        QRegExp fontFamily = QRegExp("<font face=\"'?([\\w\\s]+)'?\">(.+)</font>");
        fontFamily.setMinimal(true);

        text = m_appearanceUi.webView->page()->mainFrame()->toHtml().remove("<html><body>").remove("</body></html>").remove(fontSize).replace(fontColor, "<span style=\"color:\\1;\">\\2</span>").replace(fontFamily, "<span style=\"font-family:'\\1';\">\\2</span>");
    } else {
        text = m_appearanceUi.timeFormat->toPlainText();
    }

    const QString toolTip = formatDateTime(m_dateTime, text);

    disconnect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()), this, SLOT(changeFormat()));
    disconnect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(changeFormat()));

    m_appearanceUi.timeFormat->setToolTip(toolTip);
    m_appearanceUi.webView->setToolTip(toolTip);

    if (sender() == m_appearanceUi.webView->page()) {
        m_appearanceUi.timeFormat->setPlainText(text);
    } else {
        m_appearanceUi.webView->page()->mainFrame()->setHtml(text);
        m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("boldButton", m_appearanceUi.boldButton);
        m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("italicButton", m_appearanceUi.italicButton);
        m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("underlineButton", m_appearanceUi.underlineButton);
        m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("designModeEditor", this);
    }

    if (m_appearanceUi.timeFormatComboBox->currentIndex() < m_timeFormatNames.count() && m_appearanceUi.timeFormatComboBox->itemData(m_appearanceUi.timeFormatComboBox->currentIndex(), Qt::UserRole).toString() != text) {
        addFormat(true);
    }

    m_appearanceUi.timeFormatComboBox->setItemData(m_appearanceUi.timeFormatComboBox->currentIndex(), text);

    connect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()), this, SLOT(changeFormat()));
    connect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(changeFormat()));
}

void AdjustableClock::addFormat(bool automatically)
{
    QString formatName = m_appearanceUi.timeFormatComboBox->itemText(m_appearanceUi.timeFormatComboBox->currentIndex());

    if (automatically) {
        int i = 2;

        while (m_appearanceUi.timeFormatComboBox->findText(QString("%1 %2").arg(formatName).arg(i)) >= 0) {
            ++i;
        }

        formatName = QString("%1 %2").arg(formatName).arg(i);
    } else {
        formatName = KInputDialog::getText(i18n("Add new format"), i18n("Format name:"), formatName);
    }

    if (m_appearanceUi.timeFormatComboBox->findText(formatName) >= 0) {
        KMessageBox::error(m_appearanceUi.timeFormatComboBox, i18n("A format with this name already exists."));
    } else if (!formatName.isEmpty()) {
        int index = (m_appearanceUi.timeFormatComboBox->currentIndex() + 1);

        if (index <= m_timeFormatNames.count()) {
            index = m_appearanceUi.timeFormatComboBox->count();
        }

        disconnect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadFormat(int)));

        if (index == m_timeFormatNames.count() && m_timeFormatNames.count() == m_appearanceUi.timeFormatComboBox->count())
        {
            m_appearanceUi.timeFormatComboBox->insertSeparator(index);

            ++index;
        }

        m_appearanceUi.timeFormatComboBox->insertItem(index, formatName, m_appearanceUi.timeFormat->toPlainText());
        m_appearanceUi.timeFormatComboBox->setCurrentIndex(index);
        m_appearanceUi.removeButton->setEnabled(true);

        connect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadFormat(int)));
    }
}

void AdjustableClock::removeFormat()
{
    if (m_appearanceUi.timeFormatComboBox->currentIndex() > m_timeFormatNames.count()) {
        m_appearanceUi.timeFormatComboBox->removeItem(m_appearanceUi.timeFormatComboBox->currentIndex());

        if (m_appearanceUi.timeFormatComboBox->itemData((m_appearanceUi.timeFormatComboBox->count() - 1), Qt::DisplayRole).toString().isEmpty()) {
            m_appearanceUi.timeFormatComboBox->removeItem(m_appearanceUi.timeFormatComboBox->count() - 1);
        }
    }
}

void AdjustableClock::updateControls()
{
    disconnect(m_appearanceUi.fontSizeComboBox, SIGNAL(editTextChanged(QString)), this, SLOT(selectFontSize(QString)));

    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("boldButton.setChecked(document.queryCommandState('bold'));"
            "italicButton.setChecked(document.queryCommandState('italic'));"
            "underlineButton.setChecked(document.queryCommandState('underline'));"
            "designModeEditor.setColor(document.queryCommandValue('forecolor'));"
            "designModeEditor.setFontSize(document.queryCommandValue('fontsize').replace('px', ''));"
            "designModeEditor.setFontFamily(document.queryCommandValue('fontname'))");

    connect(m_appearanceUi.fontSizeComboBox, SIGNAL(editTextChanged(QString)), this, SLOT(selectFontSize(QString)));
}

void AdjustableClock::toggleState()
{
    QString actionName = sender()->objectName().remove("Button").toLower();
    QHash<QString, QWebPage::WebAction> actions;
    actions["bold"] = QWebPage::ToggleBold;
    actions["italic"] = QWebPage::ToggleItalic;
    actions["underline"] = QWebPage::ToggleUnderline;
    actions["justifyLeft"] = QWebPage::AlignLeft;
    actions["justifyCenter"] = QWebPage::AlignCenter;
    actions["justifyRight"] = QWebPage::AlignRight;

    if (actions.contains(actionName))
    {
        m_appearanceUi.webView->page()->triggerAction(actions[actionName]);
    }
}

void AdjustableClock::selectColor()
{
    KColorDialog colorDialog;
    colorDialog.setAlphaChannelEnabled(true);
    colorDialog.setColor(m_appearanceUi.colorButton->palette().button().color());
    colorDialog.setButtons(KDialog::Ok | KDialog::Cancel);

    if (colorDialog.exec() == QDialog::Accepted) {
        QPalette palette = m_appearanceUi.colorButton->palette();
        palette.setBrush(QPalette::Button, colorDialog.color());

        m_appearanceUi.colorButton->setPalette(palette);
        m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('forecolor', false, '" + colorDialog.color().name() + "')");
    }
}

void AdjustableClock::selectFontSize(const QString &size)
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('fontsizedelta', false, " + QString::number(size.toInt() - m_fontSize) + ")");

    m_fontSize = size.toInt();
}

void AdjustableClock::selectFontFamily(const QFont &font)
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('fontname', false, '" + font.family() + "')");
}

void AdjustableClock::setColor(const QString &color)
{
    if (color != "false") {
        QRegExp expression = QRegExp("rgb\\((\\d+), (\\d+), (\\d+)\\)");
        expression.indexIn(color);

        const QStringList rgb = expression.capturedTexts();

        QPalette palette = m_appearanceUi.colorButton->palette();
        palette.setBrush(QPalette::Button, QColor(rgb.at(1).toInt(), rgb.at(2).toInt(), rgb.at(3).toInt()));

        m_appearanceUi.colorButton->setPalette(palette);
    }
}

void AdjustableClock::setFontSize(const QString &size)
{
    if (!m_appearanceUi.fontSizeComboBox->hasFocus()) {
        m_appearanceUi.fontSizeComboBox->setEditText(size);
    }

    m_fontSize = size.toInt();
}

void AdjustableClock::setFontFamily(const QString &font)
{
    m_appearanceUi.fontFamilyComboBox->setCurrentFont(QFont(font));
}

void AdjustableClock::selectionChanged()
{
    m_controlsTimer->start();

    if (m_appearanceUi.webView->page()->selectedText().endsWith('%')) {
        m_appearanceUi.webView->page()->triggerAction(QWebPage::SelectNextChar);
    }
}

void AdjustableClock::itemSelectionChanged()
{
    QList<QTableWidgetItem*> selectedItems = m_clipboardUi.clipboardActionsTable->selectedItems();

    m_clipboardUi.moveUpButton->setEnabled(!selectedItems.isEmpty() && m_clipboardUi.clipboardActionsTable->row(selectedItems.first()) != 0);
    m_clipboardUi.moveDownButton->setEnabled(!selectedItems.isEmpty() && m_clipboardUi.clipboardActionsTable->row(selectedItems.last()) != (m_clipboardUi.clipboardActionsTable->rowCount() - 1));
    m_clipboardUi.deleteButton->setEnabled(!selectedItems.isEmpty());
}

void AdjustableClock::insertRow()
{
    const int row = ((m_clipboardUi.clipboardActionsTable->rowCount() && m_clipboardUi.clipboardActionsTable->currentRow() >= 0) ? m_clipboardUi.clipboardActionsTable->currentRow() : 0);

    m_clipboardUi.clipboardActionsTable->insertRow(row);
    m_clipboardUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(QString()));

    QTableWidgetItem *item = new QTableWidgetItem(QString());
    item->setFlags(0);

    m_clipboardUi.clipboardActionsTable->setItem(row, 1, item);
    m_clipboardUi.clipboardActionsTable->setCurrentCell(row, 0);
}

void AdjustableClock::deleteRow()
{
    m_clipboardUi.clipboardActionsTable->removeRow(m_clipboardUi.clipboardActionsTable->row(m_clipboardUi.clipboardActionsTable->selectedItems().at(0)));
}

void AdjustableClock::moveRow(bool up)
{
    int sourceRow = m_clipboardUi.clipboardActionsTable->row(m_clipboardUi.clipboardActionsTable->selectedItems().at(0));
    int destinationRow = (up ? (sourceRow - 1) : (sourceRow + 1));

    QList<QTableWidgetItem*> sourceItems;
    QList<QTableWidgetItem*> destinationItems;

    for (int i = 0; i < 2; ++i) {
        sourceItems.append(m_clipboardUi.clipboardActionsTable->takeItem(sourceRow, i));

        destinationItems.append(m_clipboardUi.clipboardActionsTable->takeItem(destinationRow, i));
    }

    for (int i = 0; i < 2; ++i) {
        m_clipboardUi.clipboardActionsTable->setItem(sourceRow, i, destinationItems.at(i));
        m_clipboardUi.clipboardActionsTable->setItem(destinationRow, i, sourceItems.at(i));
    }

    m_clipboardUi.clipboardActionsTable->setCurrentCell(destinationRow, 0);
}

void AdjustableClock::moveRowUp()
{
    moveRow(true);
}

void AdjustableClock::moveRowDown()
{
    moveRow(false);
}

void AdjustableClock::updateRow(int row, int column)
{
    Q_UNUSED(column)

    if (!m_clipboardUi.clipboardActionsTable->item(row, 1)) {
        return;
    }

    const QString preview = formatDateTime(m_dateTime, m_clipboardUi.clipboardActionsTable->item(row, 0)->text());

    m_clipboardUi.clipboardActionsTable->item(row, 1)->setText(preview);
    m_clipboardUi.clipboardActionsTable->item(row, 1)->setToolTip(preview);
}

void AdjustableClock::toolTipAboutToShow()
{
    updateToolTipContent();
}

void AdjustableClock::toolTipHidden()
{
    Plasma::ToolTipManager::self()->clearContent(this);
}

void AdjustableClock::setText(const QString &text)
{
    if (text != m_currentText) {
        m_page.mainFrame()->setHtml("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"><html><head><style type=\"text/css\">html, body, table, td {margin:0; padding:0; height:100%; width:100%; vertical-align:middle;}</style></head><body><table><tr><td id=\"clock\">" + text + "</td></tr></table></body></html>");

        m_currentText = text;

        update();
    }
}

void AdjustableClock::copyToClipboard(QAction* action)
{
    QApplication::clipboard()->setText(action->text());
}

void AdjustableClock::updateClipboardMenu()
{
    const QDateTime dateTime = currentDateTime();
    const QStringList clipboardFormats = config().readEntry("clipboardFormats", m_clipboardFormats);

    qDeleteAll(m_clipboardAction->menu()->actions());

    m_clipboardAction->menu()->clear();

    for (int i = 0; i < clipboardFormats.count(); ++i) {
        if (clipboardFormats.at(i).isEmpty()) {
            m_clipboardAction->menu()->addSeparator();
        } else {
            m_clipboardAction->menu()->addAction(formatDateTime(dateTime, clipboardFormats.at(i)));
        }
    }
}

void AdjustableClock::changeEngineTimezone(const QString &oldTimezone, const QString &newTimezone)
{
    dataEngine("time")->disconnectSource((oldTimezone + "|Solar"), this);

    connectSource(newTimezone);
}

void AdjustableClock::updateToolTipContent()
{
    Plasma::ToolTipContent toolTipData;
    const QString toolTipFormat = config().readEntry("toolTipFormat", "<div style=\"text-align:center;\">%Y-%m-%d<br />%H:%M:%S</div>");

    if (!toolTipFormat.isEmpty()) {
        toolTipData.setImage(KIcon("chronometer").pixmap(IconSize(KIconLoader::Desktop)));
        toolTipData.setMainText(formatDateTime(m_dateTime, toolTipFormat));
        toolTipData.setAutohide(false);
    }

    Plasma::ToolTipManager::self()->setContent(this, toolTipData);
}

void AdjustableClock::updateSize()
{
    QSizeF size;
    QString string;
    QString longest;
    QString temporary;
    int placeholder;
    const int length = (m_timeFormat.length() - 1);
    int number = 0;

    for (int index = 0; index < length; ++index) {
        if (m_timeFormat.at(index) == '%' && m_timeFormat.at(index + 1) != '%') {
            ++index;

            if (m_timeFormat.at(index).isDigit()) {
                while (m_timeFormat.at(index).isDigit()) {
                    ++index;
                }

                ++index;

                string.append('W');

                continue;
            }

            placeholder = m_timeFormat.at(index).unicode();

            longest.clear();

            switch (placeholder) {
            case 'a':
            case 'A':
                number = KGlobal::locale()->calendar()->daysInWeek(m_dateTime.date());

                for (int i = 0; i <= number; ++i) {
                    temporary = KGlobal::locale()->calendar()->weekDayName(i, ((placeholder == 'a') ? KCalendarSystem::ShortDayName : KCalendarSystem::LongDayName));

                    if (temporary.length() > longest.length()) {
                        longest = temporary;
                    }
                }

                string.append(longest);
                break;
            case 'b':
            case 'B':
                number = KGlobal::locale()->calendar()->monthsInYear(m_dateTime.date());

                for (int i = 0; i < number; ++i) {
                    temporary = KGlobal::locale()->calendar()->monthName(i, KGlobal::locale()->calendar()->year(m_dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive()) ? ((placeholder == 'b') ? KCalendarSystem::ShortNamePossessive : KCalendarSystem::LongNamePossessive) : ((placeholder == 'b') ? KCalendarSystem::ShortName : KCalendarSystem::LongName)));

                    if (temporary.length() > longest.length()) {
                        longest = temporary;
                    }
                }

                string.append(longest);
                break;
            case 'c':
                string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::LongDate));
                break;
            case 'C':
                string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::ShortDate));
                break;
            case 'd':
            case 'e':
            case 'H':
            case 'I':
            case 'k':
            case 'l':
            case 'm':
            case 'M':
            case 'n':
            case 'S':
            case 'W':
            case 'U':
            case 'y':
                string.append("00");
                break;
            case 'f':
                string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::ShortDate));
                break;
            case 'x':
                string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::LongDate));
                break;
            case 'o':
                string.append(KGlobal::locale()->formatTime(m_sunrise, false));
                break;
            case 'O':
                string.append(KGlobal::locale()->formatTime(m_sunset, false));
                break;
            case 'F':
                string.append(KGlobal::locale()->formatTime(m_dateTime.time(), false));
                break;
            case 'X':
                string.append(KGlobal::locale()->formatTime(m_dateTime.time(), true));
                break;
            case 'g':
                string.append(prettyTimezone());
                break;
            case 'j':
                string.append("000");
                break;
            case 'p':
                string.append((i18n("pm").length() > i18n("am").length()) ? i18n("pm") : i18n("am"));
                break;
            case 't':
                string.append(QString::number(m_dateTime.toTime_t()));
                break;
            case 'w':
                string.append("0");
                break;
            case 'Y':
                string.append("0000");
                break;
            case 'Z':
                string.append(m_timeZoneAbbreviation);
                break;
            case 'z':
                string.append(m_timeZoneOffset);
            default:
                string.append(m_timeFormat.at(index));
                break;
            }
        } else {
            string.append(m_timeFormat.at(index));
        }
    }

    if (m_timeFormat.at(length - 1) != '%') {
        string.append(m_timeFormat.at(length));
    }

    setText(string);

    m_page.setViewportSize(QSize(0, 0));
    m_page.mainFrame()->setZoomFactor(1);

    if (formFactor() == Plasma::Horizontal) {
        size = QSizeF(containment()->boundingRect().width(), boundingRect().height());
    } else if (formFactor() == Plasma::Vertical) {
        size = QSizeF(boundingRect().width(), containment()->boundingRect().height());
    } else {
        size = boundingRect().size();
    }

    qreal widthFactor = (size.width() / m_page.mainFrame()->contentsSize().width());
    qreal heightFactor = (size.height() / m_page.mainFrame()->contentsSize().height());

    m_page.mainFrame()->setZoomFactor((widthFactor > heightFactor) ? heightFactor : widthFactor);

    if (formFactor() == Plasma::Horizontal) {
        setMinimumWidth(m_page.mainFrame()->contentsSize().width());
        setMinimumHeight(0);
    } else if (formFactor() == Plasma::Vertical) {
        setMinimumHeight(m_page.mainFrame()->contentsSize().height());
        setMinimumWidth(0);
    }

    m_page.setViewportSize(boundingRect().size().toSize());

    setText(formatDateTime(m_dateTime, m_timeFormat));
}

void AdjustableClock::updateTheme()
{
    QPalette palette = m_page.palette();
    palette.setBrush(QPalette::Base, Qt::transparent);

    m_page.setPalette(palette);
    m_page.mainFrame()->evaluateJavaScript("document.fgColor = '" + Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor).name() + '\'');

    update();
}

QDateTime AdjustableClock::currentDateTime() const
{
    Plasma::DataEngine::Data data = dataEngine("time")->query(currentTimezone());
    QDateTime dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());

    return dateTime;
}

QString AdjustableClock::formatDateTime(const QDateTime dateTime, const QString &format) const
{
    if (format.isEmpty()) {
        return QString();
    }

    QString string;
    const int length = (format.length() - 1);

    for (int index = 0; index < length; ++index) {
        if (format.at(index) == '%' && format.at(index + 1) != '%') {
            QString placeholderString;
            int charNumber = -1;

            ++index;

            if (format.at(index).isDigit()) {
                QString numberString;

                while (format.at(index).isDigit()) {
                    numberString.append(format.at(index));

                    ++index;
                }

                charNumber = numberString.toInt();
            }

            switch (format.at(index).unicode()) {
            case 'a': // weekday, short form
                placeholderString.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::ShortDayName));
                break;
            case 'A': // weekday, long form
                placeholderString.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::LongDayName));
                break;
            case 'b': // month, short form
                placeholderString.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive()) ? KCalendarSystem::ShortNamePossessive : KCalendarSystem::ShortName)));
                break;
            case 'B': // month, long form
                placeholderString.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), KCalendarSystem::LongName));
                break;
            case 'c': // date and time format, short
                placeholderString.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::LongDate));
                break;
            case 'C': // date and time format, long
                placeholderString.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::ShortDate));
                break;
            case 'd': // day of the month, two digits
                placeholderString.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
            case 'e': // day of the month, one digit
                placeholderString.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
            case 'f': // date format, short
                placeholderString.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::ShortDate));
                break;
            case 'F': // time format, short
                placeholderString.append(KGlobal::locale()->formatTime(dateTime.time(), false));
                break;
            case 'g': // timezone city
                placeholderString.append(prettyTimezone());
                break;
            case 'H': // hour, 24h format
                if (dateTime.time().hour() < 10) {
                    placeholderString.append('0');
                }

                placeholderString.append(QString::number(dateTime.time().hour()));
                break;
            case 'I': // hour, 12h format
                if ((((dateTime.time().hour() + 11) % 12) + 1) < 10) {
                    placeholderString.append('0');
                }

                placeholderString.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
            case 'j': // day of the year
                placeholderString.append(QString::number(KGlobal::locale()->calendar()->dayOfYear(dateTime.date())));
                break;
            case 'k': // hour, 24h format, one digit
                placeholderString.append(QString::number(dateTime.time().hour()));
                break;
            case 'l': // hour, 12h format, one digit
                placeholderString.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
            case 'm': // month, two digits
                placeholderString.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
            case 'M': // minute, two digits
                if (dateTime.time().minute() < 10) {
                    placeholderString.append('0');
                }

                placeholderString.append(QString::number(dateTime.time().minute()));
                break;
            case 'n': // month, one digit
                placeholderString.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
            case 'o': // sunrise time
                placeholderString.append(KGlobal::locale()->formatTime(m_sunrise, false));
                break;
            case 'O': // sunset time
                placeholderString.append(KGlobal::locale()->formatTime(m_sunset, false));
                break;
            case 'p': // pm or am
                placeholderString.append((dateTime.time().hour() >= 12) ? i18n("pm") : i18n("am"));
                break;
            case 's': // second, one digit
                placeholderString.append(QString::number(dateTime.time().second()));
                break;
            case 'S': // second, two digits
                if (dateTime.time().second() < 10) {
                    placeholderString.append('0');
                }

                placeholderString.append(QString::number(dateTime.time().second()));
                break;
            case 't': // UNIX timestamp
                placeholderString.append(QString::number(dateTime.toTime_t()));
                break;
            case 'w': // day of week
                placeholderString.append(QString::number(KGlobal::locale()->calendar()->dayOfWeek(dateTime.date())));
                break;
            case 'W': // week number
            case 'U':
                placeholderString.append(QString::number(KGlobal::locale()->calendar()->weekNumber(dateTime.date())));
                break;
            case 'x': // date format, long
                placeholderString.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::LongDate));
                break;
            case 'X': // time format, long
                placeholderString.append(KGlobal::locale()->formatTime(dateTime.time(), true));
                break;
            case 'Y': // year, four digits
                placeholderString.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
            case 'y': // year, two digits
                placeholderString.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
            case 'Z': // timezone abbreviation
                placeholderString.append(m_timeZoneAbbreviation);
                break;
            case 'z': // timezone offset
                placeholderString.append(m_timeZoneOffset);
                break;
            default:
                placeholderString.append(format.at(index));
                break;
            }

            if (charNumber >= 0) {
                string.append(placeholderString.at((charNumber >= placeholderString.count())?(placeholderString.count() - 1):charNumber));
            } else {
                string.append(placeholderString);
            }
        } else {
            string.append(format.at(index));
        }
    }

    if (format.at(length - 1) != '%') {
        string.append(format.at(length));
    }

    return string;
}

QList<QAction*> AdjustableClock::contextualActions()
{
    QList<QAction*> actions = ClockApplet::contextualActions();

    if (!m_clipboardAction) {
        m_clipboardAction = new QAction(SmallIcon("edit-copy"), i18n("C&opy to Clipboard"), this);
        m_clipboardAction->setMenu(new KMenu);

        connect(this, SIGNAL(destroyed()), m_clipboardAction->menu(), SLOT(deleteLater()));
        connect(m_clipboardAction->menu(), SIGNAL(aboutToShow()), this, SLOT(updateClipboardMenu()));
        connect(m_clipboardAction->menu(), SIGNAL(triggered(QAction*)), this, SLOT(copyToClipboard(QAction*)));
    }

    for (int i = 0; i < actions.count(); ++i) {
        if (actions.at(i)->text() == i18n("C&opy to Clipboard")) {
            actions.removeAt(i);
            actions.insert(i, m_clipboardAction);

            m_clipboardAction->setVisible(!config().readEntry("clipboardFormats", m_clipboardFormats).isEmpty());
        }
    }

    return actions;

}

#include "AdjustableClock.moc"
