/*
**  Folder.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/Folder.h>

#import <Pantomime/Message.h>
#import <Pantomime/Constants.h>
#import <Pantomime/Store.h>

@implementation Folder 

//
//
//
- (id) initWithName: (NSString *) theName
{
  self = [super init];

  allMessages = [[NSMutableArray alloc] init];
  allVisibleMessages = nil;

  [self setName: theName];
  [self setShowDeleted: NO];

  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(name);
  RELEASE(allMessages);
  TEST_RELEASE(allVisibleMessages);

  [super dealloc];
}


//
//
//
- (NSString *) name
{
  return name;
}


//
//
//
- (void) setName: (NSString *) theName
{
  RETAIN(theName);
  RELEASE(name);
  name = theName;
}

//
// This method is used to append a message to our current folder.
//
- (void) appendMessage: (Message *) theMessage
{
  if (theMessage)
    {
      [allMessages addObject: theMessage];
      
      if ( allVisibleMessages )
	{
	  [allVisibleMessages addObject: theMessage];
	}
    }
}


//
//
//
- (void) appendMessageFromRawSource: (NSData *) theData
{
  [self subclassResponsibility: _cmd];
  return;
}


//
//
//
- (void) appendMessageFromRawSource: (NSData *) theData
                              flags: (Flags *) theFlags
{
  Message *aMessage;

  [self appendMessageFromRawSource: theData];

  if ( theFlags )
    {
      aMessage = [allMessages lastObject];
      [aMessage setFlags: theFlags];
    }
}

//
// This method is used to return an array containing all the messages
// of the current folder. Not that the messages MIGHT NOT been all
// completely initialized.
//
- (NSArray *) allMessages
{
  int i;
 
  if ( allVisibleMessages == nil )
    {
      allVisibleMessages = [[NSMutableArray alloc] initWithCapacity: [allMessages count]];
      
      for (i = 0; i < [allMessages count]; i++)
	{
	  Message *aMessage;
	  
	  aMessage = [allMessages objectAtIndex: i];
	  
	  if ( [[aMessage flags] contain: TRANSFERRED] )
	    {
	      // Do nothing
	      continue;
	    }
      
	  if ( [self showDeleted] )
	    {
	      [allVisibleMessages addObject: aMessage];
	    }
	  else
	    {
	      if ( [[aMessage flags] contain: DELETED] )
		{
		  // Do nothing
		  continue;
		}
	      else
		{
		  [allVisibleMessages addObject: aMessage];
		}
	    }
	}
    }

  return allVisibleMessages;
}


//
//
//
- (void) setMessages: (NSArray *) theMessages
{
  if ( theMessages )
    {
      RELEASE(allMessages);
      allMessages = [[NSMutableArray alloc] initWithArray: theMessages];
    }
  else
    {
      RELEASE(allMessages);
      allMessages = nil;
    }

  TEST_RELEASE(allVisibleMessages);
  allVisibleMessages = nil;
}

//
// This method is used to obtain a message using an index of the
// message in the folder.
//
// This method might return nil if it doesn't make sense to obtain
// a message by using an index in a context of a Folder.
//
- (Message *) messageAtIndex: (int) theIndex
{
  if (theIndex < 0 || theIndex >= [self count])
    {
      return nil;
    }
  
  return [[self allMessages] objectAtIndex: theIndex];
}


//
// This method is used to return the number of message present in this
// folder. The subclasses of Folder MUST implement this method.
//
- (int) count
{
  return [[self allMessages] count];
}


//
// This method is used to close the Folder. The subclasses of Folder MUST 
// implement this method.
//
- (void) close
{
  [self subclassResponsibility: _cmd];
  return;
}


//
// This method is used to remove permanently all the messages that have been marked as
// deleted in this folder. ALL THE RETURNED MESSAGES ARE IN RAW SOURCE.
//
- (NSArray *) expunge: (BOOL) returnDeletedMessages
{
  [self subclassResponsibility: _cmd];
  return nil;
}

- (Store *) store
{
  return store;
}

- (void) setStore: (Store *) theStore
{
  store = theStore;
}

//
// This method removes permenantly a message from
// this folder. To folder must be close (by calling -close) in
// order to make the changes on the disk.
//
// This method is used when transferring message between folders
// in order to update the view or when expunge delete messages
// from a view.
//
- (void) removeMessage: (Message *) theMessage
{
  if ( theMessage )
    {
      [allMessages removeObject: theMessage];
      
      if ( allVisibleMessages )
	{
	  [allVisibleMessages removeObject: theMessage];
	}
    }
}


//
//
//
- (BOOL) showDeleted
{
  return showDeleted;
}

- (void) setShowDeleted: (BOOL) theBOOL
{
  if (theBOOL != showDeleted)
    {
      showDeleted = theBOOL;

      TEST_RELEASE(allVisibleMessages);
      allVisibleMessages = nil;
    }
}


//
//
//
- (int) numberOfDeletedMessages
{
  int i, count;
  
  count = 0;

  for (i = 0; i < [allMessages count]; i++)
    {
      if ( [[[allMessages objectAtIndex: i] flags] contain: DELETED] )
	{
	  count = count + 1;
	}
    }

  return count;
}


//
// This method is used to update our cache (allVisibleMessages).
// Applications can call this method if they set the TRANSFERRED flags to
// messages inside this folder. If not called, the cache won't be updated
// the messages having the flag TRANSFERRED will still be visible.
//
- (void) updateCache
{
  TEST_RELEASE(allVisibleMessages);
  allVisibleMessages = nil;
}

@end



