/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "SlowEventQueue.h"
#include "SlowEvent.h"
#include <iostream>
using std::cout; using std::cerr; using std::endl;


namespace Om {


SlowEventQueue::SlowEventQueue(uint size)
: m_back(0),
  m_prepared_back(0),
  m_front(0),
  m_end_of_queue(size),
  m_size(size+1),
  m_thread_exists(false),
  m_prepare_thread_exit_flag(false),
  m_blocking(false)
{
	m_events = (SlowEvent**)calloc(m_size, sizeof(SlowEvent*));

	pthread_mutex_init(&m_prepare_mutex, NULL);
	pthread_cond_init(&m_prepare_cond, NULL);
	pthread_mutex_init(&m_blocking_mutex, NULL);
	pthread_cond_init(&m_blocking_cond, NULL);
}


SlowEventQueue::~SlowEventQueue()
{
	stop();
	
	free(m_events);
	pthread_mutex_destroy(&m_prepare_mutex);
	pthread_cond_destroy(&m_prepare_cond);
	pthread_mutex_destroy(&m_blocking_mutex);
	pthread_cond_destroy(&m_blocking_cond);
}


void
SlowEventQueue::push(SlowEvent* const ev)
{
	//std::cerr << "Pushing at index " << m_back << "\n";
	pthread_mutex_lock(&m_prepare_mutex);
	
	if (m_back == m_end_of_queue) {
		cerr << "[SlowEventQueue] Error: Slow Event Queue is full!  Event is lost, this is BAD - please report." << endl;
		delete ev;
	} else {
		m_events[m_back] = ev;
		if (m_back < (m_size-1))
			++m_back;
		else
			m_back = 0;
		
		pthread_cond_signal(&m_prepare_cond);
	}

	pthread_mutex_unlock(&m_prepare_mutex);
}


/** Pops the prepared event at the front of the queue, if it exists.
 *
 * This method will only pop events that have been prepared, and are
 * stamped before the time passed.  In other words, it may return NULL
 * even if there are events pending in the queue.
 */
SlowEvent* const
SlowEventQueue::pop_before(const samplecount time)
{
	//std::cerr << "Popping at index " << m_front << "\n";

	SlowEvent* r = NULL;
	
	if (m_prepared_back != m_front && m_events[m_front]->time_stamp() < time) {
		r = m_events[m_front];
		m_end_of_queue = m_front;
		m_events[m_front] = NULL;
		if (m_front < (m_size-1))
			m_front++;
		else
			m_front = 0;
	}
	
	return r;
}


/** Start the prepare thread.
 */
void
SlowEventQueue::start()
{
	cout << "[SlowEventQueue] Starting." << endl;
	m_prepare_thread_exit_flag = false;
	pthread_create(&m_prepare_thread, NULL, &SlowEventQueue::prepare_loop, this);
	m_thread_exists = true;
}


/** Destroy the prepare thread.
 */
void
SlowEventQueue::stop()
{
	if (m_thread_exists) {
		m_prepare_thread_exit_flag = true;
		pthread_mutex_lock(&m_prepare_mutex);
		pthread_cond_signal(&m_prepare_cond);
		pthread_mutex_unlock(&m_prepare_mutex);
		pthread_cancel(m_prepare_thread);
		pthread_join(m_prepare_thread, NULL);
		m_thread_exists = false;
	}
}



// Private //



/** Peeks any event
 *
 * See comments to pop(), same applies here.
 */
SlowEvent*
SlowEventQueue::peek_any()
{
	if (m_front != m_back) {
		return m_events[m_front];
	} else {
		return NULL;
	}
}


/** Signal that the blocking event is finished.
 *
 * When this is called preparing will resume.  This will be called by
 * blocking events in their post_process() method.
 */
void
SlowEventQueue::signal_blocking_event_finished()
{
	pthread_mutex_lock(&m_blocking_mutex);
	m_blocking = false;
	pthread_cond_signal(&m_blocking_cond);
	pthread_mutex_unlock(&m_blocking_mutex);
}


void*
SlowEventQueue::m_prepare_loop()
{
	pthread_mutex_lock(&m_prepare_mutex);
	
	while ( ! m_prepare_thread_exit_flag) {
		//std::cerr << "Prepare loop waiting for mutex/signal\n";
		
		pthread_cond_wait(&m_prepare_cond, &m_prepare_mutex);
		//cout << "Prepare loop caught signal.\n";

		for (uint i=m_prepared_back; i != m_back; ++i) {
			if (i == m_size) i = 0;
			if (i == m_back) break;
			
			if (! m_events[i]->is_prepared()) {
				m_blocking = m_events[i]->is_blocking();
				if (m_blocking)
					pthread_mutex_lock(&m_blocking_mutex);
				
				m_events[i]->prepare();
				if (m_prepared_back == m_size-1)
					m_prepared_back = 0;
				else
					++m_prepared_back;

				if (m_blocking) {
					pthread_cond_wait(&m_blocking_cond, &m_blocking_mutex);
					pthread_mutex_unlock(&m_blocking_mutex);
				}
			}	
		}
	}
	
	pthread_mutex_unlock(&m_prepare_mutex);
	
	cout << "[SlowEventQueue] Exiting slow event queue thread." << endl;
	//pthread_exit(NULL);
	return NULL;
}


} // namespace Om

