/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OSCListener.h"
#include "DummyClientHooks.h"
#include "PatchModel.h"
#include "ConnectionModel.h"
#include "MetadataModel.h"
#include "PresetModel.h"
#include "ControlModel.h"
#include "NodeModel.h"
#include "PluginModel.h"
#include <list>
#include <cassert>
#include <cstring>
#include <iostream>
using std::cerr; using std::cout; using std::endl;

namespace LibOmClient {

	
/** Construct a OSCListener with a user-provided ClientHooks object for notification
 *  of engine events.
 */
OSCListener::OSCListener(lo_server_thread st, ClientHooks* ch)
: m_client_hooks(ch),
  m_dummy_client_hooks(false),
  m_st(st),
  m_receiving_node(false),
  m_receiving_node_model(NULL)
{
	/* If NULL is passed for ClientHooks, the user will receive no notification
	 * of anything that happens in the engine.  (If this seems silly, it's
	 * because I don't want to have "if (m_client_hooks != NULL)" before every
	 * single call to a ClientHooks method in this code). */
	if (m_client_hooks == NULL) {
		m_dummy_client_hooks = true;
		m_client_hooks = new DummyClientHooks();
	}

	char* url_cstr = lo_server_thread_get_url(st);
	m_listen_url = url_cstr;
	free(url_cstr);
}


OSCListener::~OSCListener()
{
	if (m_dummy_client_hooks)
		delete m_client_hooks;
}


void
OSCListener::setup_callbacks()
{
	lo_server_thread_add_method(m_st, "/om/plugin", "ssss", plugin_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine_enabled", "", engine_enabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine_disabled", "", engine_disabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_patch", "si", new_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_destruction", "s", patch_destruction_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_enabled", "s", patch_enabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_disabled", "s", patch_disabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/object_renamed", "ss", object_renamed_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_connection", "ss", connection_cb, this);
	lo_server_thread_add_method(m_st, "/om/disconnection", "ss", disconnection_cb, this);
	lo_server_thread_add_method(m_st, "/om/node_removal", "s", node_removal_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_node", "sisss", new_node_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_node_end", "", new_node_end_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_port", "ssssfff", new_port_cb, this);
	lo_server_thread_add_method(m_st, "/om/port_removal", "s", port_removal_cb, this);
	lo_server_thread_add_method(m_st, "/om/metadata/update", "sss", metadata_update_cb, this);
	lo_server_thread_add_method(m_st, "/om/control_change", "sf", control_change_cb, this);
}


/** Catches errors that aren't a direct result of a client request.
 */
int
OSCListener::m_om_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->error((char*)argv[0]);
	return 0;
}
 

int
OSCListener::m_engine_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->engine_enabled();
	return 0;
}


int
OSCListener::m_engine_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->engine_disabled();
	return 0;
}


int
OSCListener::m_new_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	PatchModel* pm = new PatchModel(&argv[0]->s, argv[1]->i);
	PluginModel* pi = new PluginModel();
	pi->type(PluginModel::Patch);
	pm->plugin_model(pi);

	m_client_hooks->new_patch(pm);
	return 0;
}


int
OSCListener::m_patch_destruction_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_destruction((const char*)&argv[0]->s);
	return 0;
}


int
OSCListener::m_patch_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_enabled((const char*)&argv[0]->s);
	return 0;
}


int
OSCListener::m_patch_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_disabled((const char*)&argv[0]->s);
	return 0;
}


int
OSCListener::m_object_renamed_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->object_renamed((const char*)&argv[0]->s, (const char*)&argv[1]->s);
	return 0;
}


int
OSCListener::m_connection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* src_port_path = &argv[0]->s;
	const char* dst_port_path = &argv[1]->s;
	
	m_client_hooks->connection(new ConnectionModel(src_port_path, dst_port_path));

	return 0;
}


int
OSCListener::m_disconnection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* src_port_path = &argv[0]->s;
	const char* dst_port_path = &argv[1]->s;

	m_client_hooks->disconnection(src_port_path, dst_port_path);

	return 0;
}


int
OSCListener::m_node_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->node_removal((const char*)&argv[0]->s);
	
	return 0;
}


/** Notification of a new node creation.
 */
int
OSCListener::m_new_node_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* node_path  = &argv[0]->s;
	const int   poly       =  argv[1]->i;
	const char* type       = &argv[2]->s;
	const char* lib_name   = &argv[3]->s;
	const char* plug_label = &argv[4]->s;

	m_receiving_node_model = new NodeModel(node_path);
	m_receiving_node_model->polyphonic((poly == 1));

	PluginModel* pi = new PluginModel();
	pi->set_type(type);
	pi->lib_name(lib_name);
	pi->plug_label(plug_label);
	m_receiving_node_model->plugin_model(pi);
	
	m_receiving_node = true;
	
	return 0;
}


/** Notification of the end of a node transmission (which is a bundle).
 */
int
OSCListener::m_new_node_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->new_node(m_receiving_node_model);
	m_receiving_node = false;
	m_receiving_node_model = NULL;
	
	return 0;
}


/** Notification of a new port creation.
 */
int
OSCListener::m_new_port_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path   = &argv[0]->s;
	const char* type        = &argv[1]->s;
	const char* direction   = &argv[2]->s;
	const char* hint        = &argv[3]->s;
	float       default_val =  argv[4]->f;
	float       min_val     =  argv[5]->f;
	float       max_val     =  argv[6]->f;

	PortType ptype;
	if (!strcmp(type, "AUDIO")) ptype = AUDIO;
	else if (!strcmp(type, "CONTROL")) ptype = CONTROL;
	else cerr << "[OSCListener] WARNING:  Unknown port type received (" << type << ")" << endl;

	PortDirection pdir;
	if (!strcmp(direction, "INPUT")) pdir = INPUT;
	else if (!strcmp(direction, "OUTPUT")) pdir = OUTPUT;
	else cerr << "[OSCListener] WARNING:  Unknown port direction received (" << direction << ")" << endl;

	PortHint phint;
	if (!strcmp(hint, "LOGARITHMIC")) phint = LOGARITHMIC;
	else if (!strcmp(hint, "INTEGER")) phint = INTEGER;
	else if (!strcmp(hint, "TOGGLE")) phint = TOGGLE;
	else phint = NONE;
	
	PortModel* port_model = new PortModel(port_path, ptype, pdir, phint, default_val, min_val, max_val);
	
	if (m_receiving_node) {
		m_receiving_node_model->add_port_model(port_model);
	} else {
		m_client_hooks->new_port(port_model);
	}

	return 0;	
}


/** Notification of a port's removal (destruction).
 */
int
OSCListener::m_port_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path = &argv[0]->s;

	m_client_hooks->port_removal(port_path);

	return 0;
}


/** Notification of a new or updated piece of metadata.
 */
int
OSCListener::m_metadata_update_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* node_path  = &argv[0]->s;
	const char* key        = &argv[1]->s;
	const char* value      = &argv[2]->s;

	m_client_hooks->metadata_update(new MetadataModel(node_path, key, value));

	return 0;	
}


int
OSCListener::m_control_change_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path  = &argv[0]->s;
	float       value      =  argv[1]->f;

	m_client_hooks->control_change(new ControlModel(port_path, value));

	return 0;	
}


/** A plugin info response from the server, in response to a /send_plugins
 */
int
OSCListener::m_plugin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	PluginModel* info = new PluginModel();
	info->lib_name(&argv[0]->s);
	info->plug_label(&argv[1]->s);
	info->name(&argv[2]->s);
	info->set_type(&argv[3]->s);
	
	m_client_hooks->new_plugin(info);

	return 0;	
}

} // namespace LibOmClient
