(* $Id: netstring_str.ml,v 1.2 2000/06/25 21:15:48 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

let lock   = ref (fun () -> ());;
let unlock = ref (fun () -> ());;

let init_mt new_lock new_unlock =
  lock   := new_lock;
  unlock := new_unlock
;;

let protect f =
  !lock();
  try
    let r = f() in
    !unlock();
    r
  with
      x ->
	!unlock();
	raise x
;;

type regexp = Str.regexp;;
type split_result = Str.split_result = Text of string | Delim of string;;

type result =
    { pos : int;
      match_beg : int;
      match_end : int;
      group_beg : int array;
      group_end : int array;
    }
;;

let regexp s =
  protect
    (fun () -> Str.regexp s)
;;

let regexp_case_fold s =
  protect
    (fun () -> Str.regexp_case_fold s)
;;

let quote s =
  protect
    (fun () -> Str.quote s)
;;

let regexp_string s =
  protect
    (fun () -> Str.regexp_string s)
;;

let regexp_string_case_fold s =
  protect
    (fun () -> Str.regexp_string_case_fold s)
;;

let return_result pos n_groups =
  let r =
    { pos = pos;
      match_beg = (try Str.match_beginning() with Not_found -> -1);
      match_end = (try Str.match_end()       with Not_found -> -1);
      group_beg = Array.create n_groups (-1);
      group_end = Array.create n_groups (-1);
    }
  in
  for g = 0 to n_groups - 1 do
    r.group_beg.(g) <- (try Str.group_beginning (g+1) with Not_found -> -1);
    r.group_end.(g) <- (try Str.group_end (g+1)       with Not_found -> -1);
  done;
  r
;;

let string_match ?(groups = 9) pat s pos =
  protect
    (fun () ->
       if Str.string_match pat s pos then
	 Some (return_result pos groups)
       else
	 None
    )
;;

let string_partial_match ?(groups = 9) pat s pos =
  protect
    (fun () ->
       if Str.string_partial_match pat s pos then
	 Some (return_result pos groups)
       else
	 None
    )
;;

let search_forward ?(groups = 9) pat s pos =
  protect
    (fun () ->
       let i = Str.search_forward pat s pos in
       i, return_result pos groups
    )
;;

let search_backward ?(groups = 9) pat s pos =
  protect
    (fun () ->
       let i = Str.search_backward pat s pos in
       i, return_result pos groups
    )
;;

let matched_string result s =
  if result.match_beg < 0 or result.match_end < 0 then raise Not_found;
  String.sub s result.match_beg (result.match_end - result.match_beg)
;;

let match_beginning result =
  if result.match_beg < 0 then raise Not_found;
  result.match_beg
;;

let match_end result =
  if result.match_end < 0 then raise Not_found;
  result.match_end
;;

let matched_group result n s =
  if n < 0 || n >= Array.length result.group_beg then raise Not_found;
  let gbeg = result.group_beg.(n-1) in
  let gend = result.group_end.(n-1) in
  if gbeg < 0 or gend < 0 then raise Not_found;
  String.sub s gbeg (gend - gbeg)
;;

let group_beginning result n =
  if n < 0 || n >= Array.length result.group_beg then raise Not_found;
  let gbeg = result.group_beg.(n-1) in
  if gbeg < 0 then raise Not_found else 
    gbeg
;;

let group_end result n =
  if n < 0 || n >= Array.length result.group_end then raise Not_found;
  let gend = result.group_end.(n-1) in
  if gend < 0 then raise Not_found else 
    gend
;;

let global_replace pat templ s =
  protect
    (fun () ->
       Str.global_replace pat templ s)
;;

let replace_first pat templ s =
  protect
    (fun () ->
       Str.replace_first pat templ s)
;;

let global_substitute ?(groups = 9) pat subst s =
  protect
    (fun () ->
       let xsubst s =
	 let r = return_result 0 groups in
	 subst r s
       in
       Str.global_substitute pat xsubst s)
;;

let substitute_first ?(groups = 9) pat subst s =
  protect
    (fun () ->
       let xsubst s =
	 let r = return_result 0 groups in
	 subst r s
       in
       Str.substitute_first pat xsubst s)
;;

(* replace_matched: n/a *)

let split sep s =
  protect
    (fun () ->
       Str.split sep s)
;;

let bounded_split sep s max =
  protect
    (fun () ->
       Str.bounded_split sep s max)
;;

let split_delim sep s =
  protect
    (fun () ->
       Str.split_delim sep s)
;;

let bounded_split_delim sep s max =
  protect
    (fun () ->
       Str.bounded_split_delim sep s max)
;;

let full_split sep s =
  protect
    (fun () ->
       Str.full_split sep s)
;;

let bounded_full_split sep s max =
  protect
    (fun () ->
       Str.bounded_full_split sep s max)
;;

let string_before = Str.string_before;;
let string_after = Str.string_after;;
let first_chars = Str.first_chars;;
let last_chars = Str.last_chars;;

(* ======================================================================
 * History:
 * 
 * $Log: netstring_str.ml,v $
 * Revision 1.2  2000/06/25 21:15:48  gerd
 * 	Checked thread-safety.
 *
 * Revision 1.1  2000/06/25 20:48:19  gerd
 * 	Initial revision.
 *
 * 
 *)
