(* $Id: netencoding.mli,v 1.5 2001/08/30 19:45:43 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

(**********************************************************************)
(* Several encodings important for the net                            *)
(**********************************************************************)


(**********************************************************************)
(* Base 64 encoding                                                   *)
(**********************************************************************)

(* See RFC 2045 for a description of Base 64 encoding. *)

(* THREAD-SAFETY: 
 * All Base64 functions are reentrant and thus thread-safe.
 *)

module Base64 : sig

  val encode : ?pos:int -> ?len:int -> ?linelength:int -> ?crlf:bool ->
               string -> string
      (* Compute the "base 64" encoding of the given string argument.
       * Note that the result is a string that only contains the characters
       * a-z, A-Z, 0-9, +, /, =, and optionally spaces, CR and LF characters.
       *
       * If pos and/or len are passed, only the substring starting at
       * pos (default: 0) with length len (default: rest of the string)
       * is encoded.
       *
       * The result is divided up into lines not longer than 'linelength' 
       * (without counting the line separator); default: do not divide lines.
       * If 'linelength' is smaller than 4, no line division is performed.
       * If 'linelength' is not divisible by 4, the produced lines are a 
       * bit shorter than 'linelength'.
       *
       * If 'crlf' (default: false) the lines are ended by CRLF; otherwise 
       * they are only ended by LF.
       * (You need the crlf option to produce correct MIME messages.)
       * 
       *)

  val url_encode : ?pos:int -> ?len:int -> ?linelength:int -> ?crlf:bool ->
                   string -> string
      (* Same as 'encode' but use slightly different characters that can be
       * part of URLs without additional encodings.
       * The encoded string consists only of the characters a-z, A-Z, 0-9, 
       * -, /, .
       * 'url_encode' does NOT implement the Base 64 encoding as described
       * in the standard!
       *)

  val encode_substring : string -> pos:int -> len:int -> linelength:int -> 
                         crlf:bool -> string
      (* *** DEPRECATED FUNCTION *** Use 'encode' instead! ***
       *
       * encode_substring s pos len linelen crlf:
       * Encodes the substring at position 'pos' in 's' with length 'len'.
       * The result is divided up into lines not longer than 'linelen' (without
       * counting the line separator).
       * If 'linelen' is smaller than 4, no line division is performed.
       * If 'linelen' is not divisible by 4, the produced lines are a 
       * bit shorter than 'linelen'.
       * If 'crlf' the lines are ended by CRLF; otherwise they are only
       * ended by LF.
       * (You need the crlf option to produce correct MIME messages.)
       *)

  val decode : ?pos:int -> ?len:int -> ?url_variant:bool -> 
               ?accept_spaces:bool -> string -> string
      (* Decodes the given string argument. 
       *
       * If pos and/or len are passed, only the substring starting at
       * pos (default: 0) with length len (default: rest of the string)
       * is decoded.
       * 
       * If url_variant (default: true) is set, the functions also
       * accepts the characters '-' and '.' as produced by 'url_encode'.
       *
       * If accept_spaces (default: false) is set, the function ignores
       * white space contained in the string to decode (otherwise the
       * function fails if it finds white space).
       *)

  val decode_ignore_spaces : string -> string
      (* *** DEPRECATED FUNCTION *** Use 'decode' instead! ***
       *
       * Decodes the string, too, but it is allowed that the string contains
       * whitespace characters.
       * This function is slower than 'decode'.
       *)

  val decode_substring : string -> pos:int -> len:int -> url_variant:bool -> 
                         accept_spaces:bool -> string
      (* *** DEPRECATED FUNCTION *** Use 'decode' instead! ***
       *
       * decode_substring s pos len url spaces:
       * Decodes the substring of 's' beginning at 'pos' with length 'len'.
       * If 'url', strings created by 'url_encode' are accepted, too.
       * If 'spaces', whitespace characters are allowed in the string.
       *)
end

(**********************************************************************)
(* Quoted printable encoding                                          *)
(**********************************************************************)

(* See RFC 2045.
 * This implementation assumes that the encoded string has a text MIME
 * type. Because of this, the characters CR and LF are never protected 
 * by hex tokens; they are copied literally to the output string.
 *)

(* THREAD-SAFETY: 
 * All QuotedPrintable functions are reentrant and thus thread-safe.
 *)

module QuotedPrintable :
  sig
    val encode : ?pos:int -> ?len:int -> string -> string
	(* Encodes the string and returns it.
	 * Note line breaks: 
	 *   No additional soft line breaks are added. The characters CR
	 *   and LF are not represented as =0D resp. =0A. (But other control
	 *   characters ARE encoded.)
	 * Note unsafe characters:
	 *   As recommended by RFC 2045, the characters !\"#$@[]^`{|}~
	 *   are additionally represented as hex tokens.        -- "
	 *
	 * If pos and/or len are passed, only the substring starting at
	 * pos (default: 0) with length len (default: rest of the string)
	 * is encoded.
	 *)

    val encode_substring : string -> pos:int -> len:int -> string
	(* *** DEPRECATED FUNCTION *** Use 'encode' instead! ***
	 * encode_substring s pos len:
	 * Encodes the substring of 's' beginning at 'pos' with length 'len'.
	 *)

    val decode : ?pos:int -> ?len:int -> string -> string
	(* Decodes the string and returns it.
	 * Most format errors cause an Invalid_argument exception.
	 * Note that soft line breaks can be properly decoded although 
	 * 'encode' will never produce them.
	 *
	 * If pos and/or len are passed, only the substring starting at
	 * pos (default: 0) with length len (default: rest of the string)
	 * is decoded.
	 *)

    val decode_substring : string -> pos:int -> len:int -> string
        (* *** DEPRECATED FUNCTION *** Use 'decode' instead! ***
	 * decode_substring s pos len:
	 * Decodes the substring of 's' beginning at 'pos' with length 'len'.
	 *)

  end

(**********************************************************************)
(* Q encoding                                                         *)
(**********************************************************************)

(* See RFC 2047. 
 * The functions behave similar to those of QuotedPrintable. 
 *)

(* THREAD-SAFETY: 
 * All Q functions are reentrant and thus thread-safe.
 *)

module Q :
  sig
    val encode : ?pos:int -> ?len:int -> string -> string
	(* Note:
	 * All characters except alphanumeric characters are protected by
	 * hex tokens.
	 * In particular, spaces are represented as "=20", not as "_".
	 *)

    val decode : ?pos:int -> ?len:int -> string -> string

    val encode_substring : string -> pos:int -> len:int -> string
        (* *** DEPRECATED FUNCTION *** Use 'encode' instead! *** *)

    val decode_substring : string -> pos:int -> len:int -> string
        (* *** DEPRECATED FUNCTION *** Use 'decode' instead! *** *)
  end

(**********************************************************************)
(* B encoding                                                         *)
(**********************************************************************)

(* The B encoding of RFC 2047 is the same as Base64. *)


(**********************************************************************)
(* URL-encoding                                                       *)
(**********************************************************************)

(* Encoding/Decoding within URLs:
 *
 * The following two functions perform the '%'-substitution for
 * characters that may otherwise be interpreted as metacharacters.
 *
 * According to: RFC 1738, RFC 1630
 *
 * Option ~plus: This option has been added because there are some
 * implementations that do not map ' ' to '+', for example Javascript's
 * escape function. The default is "true" because this is the RFC-
 * compliant definition.
 *)

(* THREAD-SAFETY:
 * The Url functions are thread-safe.
 *)

module Url : 
  sig
    val decode : ?plus:bool -> string -> string
	(* Option ~plus: Whether '+' is converted to space. The default
	 * is true. If false, '+' is returned as it is.
	 *)
    val encode : ?plus:bool -> string -> string
	(* Option ~plus: Whether spaces are converted to '+'. The default
	 * is true. If false, spaces are converted to "%20", and
	 * only %xx sequences are produced.
	 *)
  end


(**********************************************************************)
(* HTMLization                                                        *)
(**********************************************************************)

(* Encodes characters that need protection by converting them to
 * entity references. E.g. "<" is converted to "&lt;".
 * As the entities may be named, there is a dependency on the character
 * set. Currently, there are only functions for the Latin 1 alphabet.
 *)

(* THREAD-SAFETY:
 * The Html functions are thread-safe.
 *)

module Html :
  sig
    val encode_from_latin1 : string -> string
	(* Encodes the characters 0-8, 11-12, 14-31, '<', '>', '"', '&',
	 * 127-255. If the characters have a name, a named entity is
	 * preferred over a numeric entity.
	 *)
    val decode_to_latin1   : string -> string
	(* Decodes the string. Unknown named entities are left as they
	 * are (i.e. decode_to_latin1 "&nonsense;" = "&nonsense;").
	 * The same applies to numeric entities greater than 255.
	 *)
  end


(* ======================================================================
 * History:
 * 
 * $Log: netencoding.mli,v $
 * Revision 1.5  2001/08/30 19:45:43  gerd
 * 	Module Url: added the option ~plus
 *
 * Revision 1.4  2000/06/25 22:34:43  gerd
 * 	Added labels to arguments.
 *
 * Revision 1.3  2000/06/25 21:15:48  gerd
 * 	Checked thread-safety.
 *
 * Revision 1.2  2000/03/03 01:08:29  gerd
 * 	Added Netencoding.Html functions.
 *
 * Revision 1.1  2000/03/02 01:14:48  gerd
 * 	Initial revision.
 *
 * 
 *)
