/* nexp.c

   NetworkExpect's main(). main() parses command-line arguments and calls
   Tcl_Main() to initialize the application (via Tcl_AppInit() ) and then
   transfer control to the Tcl interpreter.

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include "tcl_ws-ftypes.h"
#include "nexp_trap.h"

#define PROMPT1 "puts -nonewline \"netexpect> \""
#define PROMPT2 "puts -nonewline \"+> \""

#ifdef TCL_TEST

#include "tclInt.h"

extern int		Procbodytest_Init _ANSI_ARGS_((Tcl_Interp *interp));
extern int		Procbodytest_SafeInit _ANSI_ARGS_((Tcl_Interp *interp));
extern int		TclObjTest_Init _ANSI_ARGS_((Tcl_Interp *interp));
extern int		Tcltest_Init _ANSI_ARGS_((Tcl_Interp *interp));
#ifdef TCL_THREADS
extern int		TclThread_Init _ANSI_ARGS_((Tcl_Interp *interp));
#endif

#endif /* TCL_TEST */

#ifdef TCL_XT_TEST
extern void		XtToolkitInitialize _ANSI_ARGS_((void));
extern int		Tclxttest_Init _ANSI_ARGS_((Tcl_Interp *interp));
#endif

static int rseed;

static const char *cli_commands[256];	/* Array of pointers to Tcl commands
					   specified via -c */
static unsigned curr_command;		/* Index into array of commands */
#define MAX_CLI_COMMANDS (sizeof(cli_commands)/sizeof(cli_commands[0]) - 1)

static int tcl_interactive;

static char *ws_conf_profile; /* libwireshark configuration profile (set
				 with nexp's -C option, just as tshark's
				 -C option) */

char *program_name;

Tcl_Interp *nexp_interp; /* for use by signal handlers who can't figure out
			    the interpreter directly */

static void
version(void)
{
    printf("\
Network Expect Version %s\n\
Copyright (c) 2007, 2008, 2009, 2010, 2011, 2012 Eloy Paris <peloy@netexpect.org>\n\
\n", PACKAGE_VERSION);
}

static void
usage(void)
{
    printf("\
Usage: %s [OPTION...]\n\
  -V                   display version information\n\
  -s                   set a specific random seed\n\
  -c                   execute command before any in the script\n\
  -v                   increase verbosity level\n\
  -t                   change timestamp format\n\
  -C <config profile>  start with specified Wireshark configuration profile\n\
\n", program_name);
}

/*
 *----------------------------------------------------------------------
 *
 * main --
 *
 *	This is the main program for the application.
 *
 * Results:
 *	None: Tcl_Main never returns here, so this procedure never
 *	returns either.
 *
 * Side effects:
 *	Whatever the application does.
 *
 *----------------------------------------------------------------------
 */

int
main(int argc, char **argv)
{
    int c;

    program_name = strrchr(argv[0], '/');
    if (program_name)
	program_name++;
    else
	program_name = argv[0];

    while ( (c = getopt(argc, argv, "hVs:c:vtC:") ) != -1) {
	switch (c) {
	case 'h':
	    usage();
	    return EXIT_SUCCESS;
	case 'v':
	    vflag++;
	    break;
	case 't':
	    tflag--;
	    break;
	case 'V':
	    version();
	    return EXIT_SUCCESS;
	case 's':
	    rseed = atoi(optarg);
	    break;
	case 'c':
	    /*
	     * -c specifies a command to be executed before the script
	     * is run. Several -c options can be specified, and they
	     * are executed in the order in which the appear in the CLI.
	     * Since we don't have the Tcl interpreter yet and we have
	     * not initialized the Network Expect commands yet we just
	     * save the specified command here. We execute the commands
	     * during the initialization of the application (in
	     * Tcl_AppInit() ).
	     */
	    if (curr_command >= MAX_CLI_COMMANDS)
		fprintf(stderr, "Too many -c commands - ignoring\n");
	    cli_commands[curr_command++] = optarg;
	    break;
	case 'C':
	    ws_conf_profile = optarg;
	    break;
	default:
	    usage();
	    exit(EXIT_FAILURE);
	}
    }

    srand(rseed ? rseed : time(NULL) );

    if (optind > 1) {
	/*
	 * There could be leftover arguments. Bypass all the options that
	 * have been processed and let Tcl_Main() deal with it.
	 */
	argc = argc - optind + 1;
	argv[optind - 1] = argv[0];
	argv = &argv[optind - 1];
    }

    /*
     * The following #if block allows you to change how Tcl finds the startup
     * script, prime the library or encoding paths, fiddle with the argv,
     * etc., without needing to rewrite Tcl_Main()
     */

#ifdef TCL_LOCAL_MAIN_HOOK
    extern int TCL_LOCAL_MAIN_HOOK _ANSI_ARGS_((int *argc, char ***argv));
#endif

#ifdef TCL_XT_TEST
    XtToolkitInitialize();
#endif

#ifdef TCL_LOCAL_MAIN_HOOK
    TCL_LOCAL_MAIN_HOOK(&argc, &argv);
#endif

    Tcl_Main(argc, argv, Tcl_AppInit);

    /* Never reached */
    return EXIT_SUCCESS;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_AppInit --
 *
 *	This procedure performs application-specific initialization.
 *	Most applications, especially those that incorporate additional
 *	packages, will have their own version of this procedure.
 *
 * Results:
 *	Returns a standard Tcl completion code, and leaves an error
 *	message in the interp's result if an error occurs.
 *
 * Side effects:
 *	Depends on the startup script.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_AppInit(Tcl_Interp *interp)
{
    unsigned i;
    static int first_time = TRUE;

    if (Tcl_Init(interp) == TCL_ERROR)
	return TCL_ERROR;

#ifdef HAVE_LIBTCLX8_4
    if (Tclx_Init(interp) == TCL_ERROR)
	return TCL_ERROR;
#endif

#ifdef TCL_TEST
#ifdef TCL_XT_TEST
     if (Tclxttest_Init(interp) == TCL_ERROR)
	 return TCL_ERROR;
#endif

    if (Tcltest_Init(interp) == TCL_ERROR)
	return TCL_ERROR;

    Tcl_StaticPackage(interp, "Tcltest", Tcltest_Init,
            (Tcl_PackageInitProc *) NULL);

    if (TclObjTest_Init(interp) == TCL_ERROR)
	return TCL_ERROR;

#ifdef TCL_THREADS
    if (TclThread_Init(interp) == TCL_ERROR)
	return TCL_ERROR;
#endif

    if (Procbodytest_Init(interp) == TCL_ERROR)
	return TCL_ERROR;

    Tcl_StaticPackage(interp, "procbodytest", Procbodytest_Init,
            Procbodytest_SafeInit);
#endif /* TCL_TEST */

    tcl_interactive = !strcmp(Tcl_GetVar(interp, "tcl_interactive", 0), "1");

    /*
     * Call the init procedures for included packages.  Each call should
     * look like this:
     *
     * if (Mod_Init(interp) == TCL_ERROR) {
     *     return TCL_ERROR;
     * }
     *
     * where "Mod" is the name of the module.
     */

    /*
     * Initialize important Tcl variables, their C counterparts, and the
     * variables subsystem before doing anything else. C global variables that
     * can be specified via CLI arguments have their correct values at this
     * point.
     */
    nexp_init_tclvars(interp);

    if (first_time) {
	nexp_init_trap();
	nexpDiagInit();
	nexpLogInit();

	/*
	 * Initialize the libpbuild subsystem.
	 */
	pb_init();

	first_time = FALSE;
    }

    /* save last known interp for emergencies */
    nexp_interp = interp;

    /*
     * Call Tcl_CreateCommand for application-specific commands, if
     * they weren't already created by the init procedures called above.
     */
    nexp_init_cmds(interp);             /* add misc cmds to interpreter */
    nexp_init_expect_cmds(interp);      /* add expect cmds to interpreter */
    nexp_init_send_cmds(interp);        /* add send cmds to interpreter */
    nexp_init_spawn_cmds(interp);
    nexp_init_send_expect_cmd(interp);  /* add send_expect cmd to interpreter */
    nexp_init_send_receive_cmd(interp);
    nexp_init_ghost_cmd(interp);        /* add ghost cmd to interpreter  */
    nexp_init_tgn_cmd(interp);
    nexp_init_iflist_cmd(interp);
    nexp_init_usr_cmds(interp);
    nexp_init_trap_cmds(interp);        /* add trap cmds to interpreter */

    /*
     * Register custom types and their related commands.
     */
    nexp_init_pdu_cmd(interp);
    nexp_init_packet_cmd(interp);
    nexp_init_barray_cmd(interp);
    nexp_init_timeval_cmd(interp);
    nexp_init_numspec_cmds(interp);
    nexp_register_tcl_ws_ftypes();

    /*
     * Specify a user-specific startup file to invoke if the application
     * is run interactively.  Typically the startup file is "~/.apprc"
     * where "app" is the name of the application.  If this line is deleted
     * then no user-specific startup file will be run under any conditions.
     */
    Tcl_SetVar(interp, "tcl_rcFileName", "~/.nexprc", TCL_GLOBAL_ONLY);

    /*
     * Set prompts. See tclsh(1)  for details.
     */
    Tcl_SetVar(interp, "tcl_prompt1", PROMPT1, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tcl_prompt2", PROMPT2, TCL_GLOBAL_ONLY);

    /*
     * Initialize the libpackets module. libwireshark initialization is
     * performed there.
     */
    pkt_init(ws_conf_profile);

    /*
     * Now that everything is initialized we can execute the commands
     * specified via -c switches.
     */
    for (i = 0; cli_commands[i]; i++)
	if (Tcl_Eval(interp, cli_commands[i]) != TCL_OK) {
	    nexpErrorLog( (char *) Tcl_GetVar(interp, "errorInfo",
					      TCL_GLOBAL_ONLY), NULL);
	    nexpErrorLog("\r\n");
	}

    /*
     * This will give us the same behavior that Expect has with the -c
     * switch. The logic is: all the commands specified via -c switches
     * are evaluated before the script (if any) is run. If no script is
     * provided then the commands specified via -c switches are executed
     * and then the execution of the entire program finishes.
     */
    if (cli_commands[0] && tcl_interactive)
	exit(EXIT_SUCCESS);

    return TCL_OK;
}
