!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  kernel.S
!* Purpose: Setup the stack and data areas, and then call the higher
!*          level routines.
!* Entries: Called by loader at offset 0.
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Load assembler macros.
!
#include <macros.inc>
#include <memory.inc>
#include <version.h>


!
!**************************************************************************
!
! Equates for this module:
!
STACKSIZE	equ	4096		! Size of stack
BIOS_SEG	equ	$0040		! BIOS data segment


!
!**************************************************************************
!
! Define the data segment
!
	.data

	extrn	_end_of_data


!
!**************************************************************************
!
! Define the BSS segment
!
	.bss

	extrn	_end_of_bss

	.lcomm	oldints,VECTSIZE	! repository for old interrupt vectors


!
!**************************************************************************
!
! The kernel starts here. At the beginning there is some
! miscellaneous data.
!
	.text

	entry	start

	public	datseg

	extrn	_end_of_text
	extrn	_kernelmain
	extrn	_fatal
	extrn	prnstr
	extrn	term_packet


start:	jmp	krnstart		! the loader jumps to this address

	.align	DATAOFS			! Fill up

! The following value gets filled in by the binary patch program which
! concatenates the kernel with the packet driver programs.

prgadr:	.word	0			! Address of program pointer list

! These values can be used by the binary patch program to strip off the
! BSS segment from the final binary. The BSS segment just contains zeroes
! which dont have to be saved in the tight ROM.

	.word	_end_of_text
	.word	_end_of_data

! Put the version number into the final kernel image. This allows the
! binary patch program to verify that its using the correct kernel
! image.

	.byte	MAJOR_VER		! major version number
	.byte	MINOR_VER		! minor version number

! This variable holds the segment address of the data segment. It has to
! accessible by CS, and therefore has to be within the text segment. Since
! the bootrom code will only run after being loaded into RAM, there is no
! problem writing into this variable.

datseg:	.word	0


!
!**************************************************************************
!
! Now start with the actual kernel code, which sets up the stack, copies
! the data segment into its place, and then calls the higher level code.
!
krnstart:

	cli
	cld

! The boot rom code is completely within the OS loading area. In order
! to avoid overwriting it with the loaded OS we have to copy it into
! a secure memory area right at the end of the physical ram. However,
! dont copy the packet driver which gets loaded into the DOS ram
! area lateron.

	mov	bx,#_end_of_text	! compute total size required for
	add	bx,#_end_of_bss		! kernel in kB
	add	bx,#STACKSIZE + $03FF
	shift	(shr,bx,10)
	add	bx,#(HIGHMEM >> 6)
	int	$12			! determine number of kB in the system
	cmp	ax,bx
	ja	memok			! check if enough memory available
	sti
	mov	bx,#memerr
	call	prnstr
	jmp	near _fatal		! print error message and terminate

memok:	mov	cx,#_end_of_text
	add	cx,#_end_of_data
	mov	bx,cs
	mov	ds,bx			! copy the bootrom code and data into
	mov	bx,#HIGHMEM		! the secure high memory area
	mov	es,bx
	xor	si,si
	xor	di,di
	rep
	movsb

	mov	si,ax			! save AX for later
	mov	cx,#_end_of_bss
	sub	cx,#_end_of_data
	xor	al,al
	rep
	stosb				! clear BSS segment

! Compute new address of data, BSS and stack segment

	mov	dx,#_end_of_text + $000F
	shift	(shr,dx,4)		! adjust to segment boundary
	add	dx,bx
	shift	(shl,si,6)
	sub	si,dx			! set new stack pointer at end of
	shift	(shl,si,4)		! available ram
	sub	si,#2
	and	si,#$FFFE
	mov	ss,dx
	mov	sp,si			! set new stack pointer

! Set new data segment and call new copy of bootrom kernel

	mov	es,dx			! set new data segment
#ifdef IS186
	push	bx
	push	#domain
#else
	mov	ax,#domain
	push	bx
	push	ax
#endif
	mov	ax,cs			! save for later
	retf				! call the new boot rom copy

! The main routine of the kernel gets the farpointer onto the stack,
! which can be found at the beginning of the rom code. But before
! calling the main routine set the reboot flag to indicate warm boot.
! This is necessary so that a Ctrl-Alt-Del keypress reboots the system
! correctly. Also setup the bootrom services interrupt vector.

domain:	xor	bx,bx			! set cleanup interrupt vector
	mov	ds,bx
	xor	si,si
	mov	di,#oldints
	mov	cx,#VECTSIZE / 2	! save old interrupt vectors
	rep
	movsw

	mov	word ptr [SERVICES * 4 + 0],#doservices
	mov	word ptr [SERVICES * 4 + 2],cs
	mov	bx,#BIOS_SEG
	mov	ds,bx
	mov	word ptr [$0072],#$1234	! indicates warm boot

	mov	ds,dx			! set new data segment
	seg	cs
	mov	datseg,dx

	sti				! the packet driver is still in the
	push	ax			! decompressed copy of the bootrom
	seg	cs			! in the OS loading area
	push	prgadr			! push the program pointer list address
	call	near _kernelmain	! now start the game
	add	sp,#4

! If the main program returns restore the interrupt vectors and call the
! bootstrap vector. Also deinitialize any resident program.

	mov	ah,#1
	int	SERVICES		! call the cleanup routine
	int	BOOTSTRAP		! call the bootstrap vector
dolop:	jmp	dolop


! Messages:

memerr:	.asciz	"Base mem err"


!
!**************************************************************************
!
! Interrupt routine to provide different services to the boot image.
! Input:  AH  -  function code
! Output: depends on the function called
! Registers changed: depends on the function called
!
doservices:

	sti
	or	ah,ah
	jnz	dosvc1

! Return installation information

	mov	ax,#$474B		! magic number
	mov	bx,#$0001		! service code
dosvc9:	iret


! Terminate all resident programs and the packet driver. The magic number
! given to the 2Fh interrupt is exactly what the name says: magic ;-))

dosvc1:	dec	ah			! check for function code 1
	jnz	dosvc9

	push	ds
	push	es
	mov	ax,#$CF45
	int	$2F
	seg	cs			! term_packet() needs correct data
	mov	ds,datseg		! segment
	call	term_packet		! terminate packet driver

! Reset the interrupt vector table.

	cli				! make this copying atomic
	cld
	xor	ax,ax
	mov	es,ax
	mov	si,#oldints
	xor	di,di
	mov	cx,#VECTSIZE / 2
	rep
	movsw
	sti

! Return to caller

	pop	es
	pop	ds
	iret

!
!**************************************************************************
!
	end

