#
# This script was written by George A. Theall, <theall@tifaware.com>.
#
# See the Nessus Scripts License for details.
#


desc = "
Synopsis :

The remote web server contains a webmail client. 

Description :

The remote host is running IMP, an open-source PHP-based webmail package
from The Horde Project that provides access to mail accounts via POP3 or
IMAP

See also :

http://www.horde.org/imp/

Risk factor : 

None";


if (description) {
  script_id(12643);
  script_version("$Revision: 1.10 $");
 
  script_name(english:"IMP Detection");
  script_summary(english:"Checks for the presence of IMP");

  script_description(english:desc);
 
  script_category(ACT_GATHER_INFO);
  script_family(english:"CGI abuses");

  script_copyright(english:"This script is Copyright (C) 2004-2007 George A. Theall");

  script_dependencies("horde_detect.nasl");
  script_exclude_keys("Settings/disable_cgi_scanning");
  script_require_ports("Services/www", 80);

  exit(0);
}


include("global_settings.inc");
include("http_func.inc");
include("http_keepalive.inc");


port = get_http_port(default:80);
if (!get_port_state(port)) exit(0);
if (!can_host_php(port:port)) exit(0);


# Horde is a prerequisite.
horde_install = get_kb_item(string("www/", port, "/horde"));
if (isnull(horde_install)) exit(0);
matches = eregmatch(string:horde_install, pattern:"^(.+) under (/.*)$");
if (isnull(matches)) exit(0);
horde_dir = matches[2];


# Loop through directories.
if (thorough_tests) dirs = make_list("/webmail", "/imp", horde_dir+"/imp", "/email", cgi_dirs());
else dirs = make_list(cgi_dirs());

installs = 0;
foreach dir (dirs) {
  # Grab index.php.
  res = http_get_cache(item:string(dir, "/index.php"), port:port);
  if (res == NULL) exit(0);

  # If we see the copyright...
  if ("IMP: Copyright 20" >< res) {
    # Search for version number in a couple of different pages.
    files = make_list(
      "/services/help/?module=imp&show=about",
      "/docs/CHANGES", "/test.php", "/lib/version.phps",
      "/status.php3"
    );

    foreach file (files) {
      # Get the page.
      if ("/services/help" >< file) url = horde_dir + file;
      else url = dir + file;
      req = http_get(item:url, port:port);
      res = http_keepalive_send_recv(port:port, data:req);
      if (res == NULL) exit(0);

      # Specify pattern used to identify version string.
      # - version 4.x
      if (file =~ "^/services/help") {
        pat = ">This is Imp (.+)\.<";
      }
      # - version 3.x
      else if (file == "/docs/CHANGES") {
        pat = "^ *v([0-9]+\..+) *$";
      }
      #   nb: test.php available is itself a vulnerability but sometimes available.
      else if (file == "/test.php") {
        pat = "^ *<li>IMP: +(.+) *</li> *$";
      }
      #   nb: another security risk -- ability to view PHP source.
      else if (file == "/lib/version.phps") {
        pat = "IMP_VERSION', '(.+)'";
      }
      # - version 2.x
      else if (file == "/status.php3") {
        pat = ">IMP, Version (.+)<";
      }
      # - someone updated files but forgot to add a pattern???
      else {
        if (log_verbosity > 1) debug_print("don't know how to handle file '", file, "'!", level:0);
        exit(1);
      }

      # Get the version string.
      matches = egrep(pattern:pat, string:res);
      if (
        matches &&
        (
          # nb: add an extra check in the case of the CHANGES file.
          (file == "/docs/CHANGES" && "IMP " >< res) ||
          1
        )
      ) {
        foreach match (split(matches)) {
          match = chomp(match);
          ver = eregmatch(pattern:pat, string:match);
          if (ver == NULL) break;
          ver = ver[1];

          # Success!
          if (dir == "") dir = "/";
          set_kb_item(
            name:string("www/", port, "/imp"), 
            value:string(ver, " under ", dir)
          );
          installations[dir] = ver;
          ++installs;

          # nb: only worried about the first match.
          break;
        }
      }
      # nb: if we found an installation, stop iterating through files.
      if (installs) break;
    }
  }
  # Scan for multiple installations only if "Thorough Tests" is checked.
  if (installs && !thorough_tests) break;
}

# Report any instances found unless Report verbosity is "Quiet".
if (installs && report_verbosity > 0) {
  if (installs == 1) {
    foreach dir (keys(installations)) {
      # empty - just need to set 'dir'.
    }
    if (ver == "unknown") {
      info = string("An unknown version of IMP was detected on the remote\nhost under the path '", dir, "'.");
    }
    else {
      info = string("IMP ", ver, " was detected on the remote host under the path '", dir, "'.");
    }
  }
  else {
    info = string(
      "Multiple instances of IMP were detected on the remote host :\n",
      "\n"
    );
    foreach dir (keys(installations)) {
      info = info + string("    ", installations[dir], ", installed under '", dir, "'\n");
    }
    info = chomp(info);
  }

  report = string(
    desc,
    "\n\n",
    "Plugin output :\n",
    "\n",
    info
  );
  security_note(port:port, data:report);
}
