// firfilter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "firfilter.h"
#include "delay.h"
#include "valuerequester.h"
#include "request.h"
#include "data.h"

class FIRFilterRequester : public ValueRequester<double> {
    friend class FIRFilter;
protected:
    FIRFilterRequester(FIRFilter *);
    redefined void configureRequest(class Request *);
private:
    FIRFilter* client;
};

FIRFilterRequester::FIRFilterRequester(FIRFilter* c)
    : ValueRequester<double>("Run FIR Filter on Selected Region:",
                    "Gain Factor:",
                    c->gain), client(c) {}
void
FIRFilterRequester::configureRequest(Request* request) {
//     request->appendValue("Fundamental Frequency/Loop Time:",
//                         &client->delayTime,
//                 NonNegativeNumbers);
    ValueRequester<double>::configureRequest(request);
}

//********

double FIRFilter::_savedGain = 1.0;

FIRFilter::FIRFilter(Data* input, Data *output, Data *coefficients)
    : SimpleInputOutputFunction(input, output),
      coeffData(coefficients), _coeffs(nil), _past(nil), gain(_savedGain) {
}

FIRFilter::FIRFilter(Data* input, Data *output)
    : SimpleInputOutputFunction(input, output),
      coeffData(nil), _coeffs(nil), _past(nil), gain(_savedGain) {}

FIRFilter::~FIRFilter() {
    delete [] _past;
    delete [] _coeffs;
    Resource::unref(coeffData);
}

void
FIRFilter::clear() {
    _counter = 0;
    bzero(_past, _length * sizeof(double));
}

Requester *
FIRFilter::createRequester() {
    return new FIRFilterRequester(this);
}

void
FIRFilter::initialize() {
    coeffData->ref();
    _length = coeffData->length();
    _coeffs = new double[_length];
    _past = new double[_length];
    coeffData->getArray(_coeffs, _length, 0);
    Super::initialize();
}

void
FIRFilter::saveConfig() {
    _savedGain = gain;
}

void
FIRFilter::restoreState() {
    Super::restoreState();
    clear();
}

int
FIRFilter::doProcessing() {
    BUG("FIRFilter::doProcessing()");
    int status = true;
    int flushing = needToFlush();
    while((!inQueueEmpty() || flushing) && status == true) {
        status = processInput(flushing);
    }
    moreOutputIf(true);        // let this run until output eof
    return status;
}

int
FIRFilter::doSimpleProcess(double *buffer, int nframes) {
	register double *p = _past;
	register double *cf = _coeffs;
	const int len = _length;
	for (int n=0; n<nframes; ++n) {
		int intap = _counter % len;	// 0 to (len - 1)
    
		p[intap] = buffer[n];			// feed signal into delay
		int outtap = intap - 1;
		double out = 0.0;

		// first loop:  while outtap is >= 0
		int loop1 = outtap + 1;
		int i, c = 0;

		for (i = 0; i < loop1; ++i) {
			out += p[outtap--] * cf[c++];
		}

		/* second loop: while outtap is positive after being wrapped */

		outtap += len;
		for (; i < len; ++i) {
			out += p[outtap--] * cf[c++];
		}
		_counter++;
    	buffer[n] = out * gain;
	}
	return nframes;
}
