// envelopeheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993 - 1999 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       April 21, 1999
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "datafile.h"
#include "envelopeheader.h"
#include "sound.h"
#include "valuesetter.h"

// Envelope Header Methods

EnvelopeHeader::Type EnvelopeHeader::default_HeaderType = EnvelopeHeader::With;
Header::ByteOrder EnvelopeHeader::default_ByteOrder =
    Application::isLittleEndian() ? Header::Little : Header::Big;

EnvelopeHeader::EnvelopeHeader(int npts, int srate)
	: Header(EVP_MAGIC, FloatData, EvpChannels),
	      _samprate(srate), _npoints(npts) {
    if(defaultHeaderType() == None)
		setRaw();
}

EnvelopeHeader::EnvelopeHeader()
	: Header(EVP_MAGIC, FloatData, EvpChannels),
	  _samprate(Sound::defaultSampleRate()), _npoints(EvpMinPoints) {
    if(defaultHeaderType() == None)
		setRaw();
}

boolean
EnvelopeHeader::isMagic() { 
    return magic() == EVP_MAGIC || magic() == EVP_SWAPMAGIC;
}

boolean
EnvelopeHeader::magicIsSwapped() {
    return (magic() == EVP_SWAPMAGIC);
}

boolean
EnvelopeHeader::isLittleEndian() {
    return defaultByteOrder() == Header::Little;
}

const char*
EnvelopeHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid Envelope file magic number (%d != %d)",
		magic(), EVP_MAGIC);
	return msg;
}

int
EnvelopeHeader::readInfo(DataFile *file) {
	EvpStruct head;
	int status = false;
	if (head.read(file)) {
		_samprate = int(head.srate);
		_npoints = head.npoints;
		setDataOffset(head.headersize);
		setDataSize(file->dataSize() - head.headersize);
		nchans = 1;
		data_type = FloatData;
		status = true;
	}
	else
		Application::alert("EnvelopeHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
EnvelopeHeader::checkHeader() {
	int status = false;
	if(data_type != FloatData)
		Application::alert("Invalid envelope data type [must be floating point].");
	else if(nchans != 1)
		Application::alert("Invalid envelope header: nchans must be 1.");
	else status = true;
	return status;
}

int
EnvelopeHeader::writeInfo(DataFile *file) {
	// set offset based on comment length
	int offset = diskHeaderInfoSize()
		+ max(diskHeaderCommentSize(), commentLength());
	setDataOffset(offset);
	EvpStruct head(magic(), dataOffset(), nPoints(), sampleRate());
	return head.write(file);
}

// called by config requester after failed read

void
EnvelopeHeader::reset() {
    data_type = FloatData;
    nchans = 1;
}

// outline methods for nested classes

long
EnvelopeHeader::EvpStruct::readSize() {
    return sizeof(*this) - Super::readSize();
}

ValueSetterBase**
EnvelopeHeader::EvpStruct::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&evpmagic);
    delete setterList[1];
    setterList[1] = newValueSetter(&headersize);
    delete setterList[2];
    setterList[2] = newValueSetter(&npoints);
    delete setterList[3];
    setterList[3] = newValueSetter(&srate);
    return setterList;
}
