// analysisfun.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "analysisfun.h"
#include "data.h"
#include "envelope.h"
#include "request.h"
#include "requester.h"

AnalysisFunction::~AnalysisFunction() {
	Resource::unref(myAnalysis);
}

// None of the analysis functions need to undo

Modifier* AnalysisFunction::createUndo() {
	return nil;
}

void
AnalysisFunction::setAnalysis(Data *analysis) {
	Resource::unref(myAnalysis);
	myAnalysis = analysis;
	if(myAnalysis != nil) {
		myAnalysis->ref();
		// check to be sure analysis data object was fully allocated
		okIf(getAnalysis()->length() != 0);
	}
}

//********

class EnvelopeExtractRequester : public TitledRequester {
	friend class EnvelopeExtracter;
protected:
	EnvelopeExtractRequester(const char* title, EnvelopeExtracter* e)
		: TitledRequester(title), client(e) {}
	redefined void configureRequest(Request *);
private:
	EnvelopeExtracter* client;
};

void
EnvelopeExtractRequester::configureRequest(Request* request) {
	request->appendValue("Envelope Length (min. 8):", &client->envelopeLength,
			    NonNegativeIntegers);
	request->appendChoice("Amplitude Mode:", "|Linear|RMS|Decibel|",
			     &client->envelopeType, true);
}

//********

ChoiceValue EnvelopeExtracter::_savedEnvelopeType = AbsoluteMagnitude;
int EnvelopeExtracter::_savedEnvelopeLength = 1024;

EnvelopeExtracter::EnvelopeExtracter(Data* d)
	: AnalysisFunction(d),
	  envelopeType(_savedEnvelopeType),
	  envelopeLength(_savedEnvelopeLength) {}

void
EnvelopeExtracter::initialize() {
	Envelope* envelope = new Envelope(analysisLength());
	envelope->setChannelName(
		(envelopeType == AbsoluteMagnitude) ? "Abs. Amp."
		: (envelopeType == RMSAmplitude) ? "RMS Amp."
		: "Decibels"
	);
	setAnalysis(envelope);
	Super::initialize();
}

Requester *
EnvelopeExtracter::createRequester() {
	return new EnvelopeExtractRequester("Extract Amplitude Envelope:", this);
}

void
EnvelopeExtracter::saveConfig() {
    _savedEnvelopeType = envelopeType;
    _savedEnvelopeLength = envelopeLength;
}

int
EnvelopeExtracter::doApply(Data *data) {
	data->getEnvelope(getAnalysis(), 0,
	                  EnvelopeType(envelopeType));
	getAnalysis()->Notify();
	return true;
}

//********

class SlopeChangeDetectRequester : public TitledRequester {
	friend class SlopeChangeDetecter;
protected:
	SlopeChangeDetectRequester(const char* title, SlopeChangeDetecter* e)
		: TitledRequester(title), client(e) {}
	redefined void configureRequest(Request *);
private:
	SlopeChangeDetecter* client;
};

void
SlopeChangeDetectRequester::configureRequest(Request* request) {
	request->appendLabel("Search for discontinuity with given slope factor.");
	request->appendValue("Slope change factor:",
			     &client->slopeFactor,
			    Range(1.001, 10000.0),
			    false);
	request->appendValue("Frame size for search:", &client->frameSize,
			     Range(3, client->target()->length()));
#if 0
	request->appendChoice("Search Mode:", "|Positive|Negative|Either|",
			     &client->searchMode, true);
#endif
}

//********

double SlopeChangeDetecter::_savedSlopeFactor = 10.0;
int SlopeChangeDetecter::_savedFrameSize = 0;
ChoiceValue SlopeChangeDetecter::_savedSearchMode = SlopeChangeDetecter::Either;

SlopeChangeDetecter::SlopeChangeDetecter(Data* d)
	: AnalysisFunction(d),
	  slopeFactor(_savedSlopeFactor),
	  frameSize(_savedFrameSize == 0 ? d->length() : _savedFrameSize),
	  searchMode(_savedSearchMode) {}

SlopeChangeDetecter::SlopeChangeDetecter(Data* d,
					 double factor,
					 int framesize,
					 ChoiceValue mode)
		: AnalysisFunction(d), slopeFactor(factor), 
		  frameSize(frameSize), searchMode(mode) {
	initialize();
}

void
SlopeChangeDetecter::initialize() {
	Data* points = new Data(IntData, 1, analysisLength());
	setAnalysis(points);
	Super::initialize();
}

Requester *
SlopeChangeDetecter::createRequester() {
	return new SlopeChangeDetectRequester("Search for Slope Change:", this);
}

void
SlopeChangeDetecter::saveConfig() {
    _savedSlopeFactor = slopeFactor;
    _savedFrameSize = frameSize;
    _savedSearchMode = searchMode;
}

int
SlopeChangeDetecter::analysisLength() { return 256; }

int
SlopeChangeDetecter::doApply(Data *data) {
	int location = 0;
	const int length = data->length();
	Data *points = getAnalysis();
	Data *subsection;
	Range section(0, frameSize - 1);
	while(section.intMax() < length) {
		int offset = 0;
		subsection = data->clone(section);
		subsection->ref();
		if((offset = subsection->findSlopeChange(slopeFactor)) > 0) {
			if(location >= points->length())
				if(!points->changeLength(location * 2))
					break;
			points->set(offset + section.intMin(), location++);
		}
		Resource::unref(subsection);
		section += frameSize / 2;
	}
	return (location > 0);
}
