/*
 *   dspsrv.c -- DSP Manager server routines
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#define  DSPMGR
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
extern char *ini_file;

#include "dspmgr.h"
#include "dspmgri.h"
#include "dspmgrdm.h"
#include "dsppcmem.h"                  // Function prototypes
#include "dspinit.h"                   // Function prototypes
#include "dspdmfnc.h"                  // Function prototypes
#include "dspstruc.h"                  // Function prototypes
#include "dspparse.h"                  // Function prototypes
#include "dspbios.h"                   // Function prototypes
#include "dspabil.h"                   // Function prototypes
#include "dspmlvl.h"                   // DSP version and release level
#include "dspstcmp.h"                  // mwstrcmp function
//#include "mwevents.h"                  // CH05 Low power support
#include "dspsrv.h"
#include "dsperr.h"
#include "port_functions.h"
//char acErrorString[200];

HANDLE                MwDoneEvent = 0;
HANDLE                hMwPipe;
HANDLE                hMwThread;
DWORD                 MwThreadId;
/* @TBD CRITICAL_SECTION      MwTerminateSection; */
BOOL                  MwTerminate = FALSE;
HANDLE                hDrv;
HANDLE                hMwProcess;

BOOL AllocateHeaps( void );
void DestroyHeaps( void );

HANDLE g_hSerializeKernelDriverCalls; 
HANDLE hMwaveInitialized = 0;       

#ifdef DBGLOG
FILE  *MwLog;
#endif

#define TERMINATE_SERVER_THREAD  0xFFFFFFFF
#define TRANSACTION_TIMEOUT      3000

VOID MwError( LPTSTR lpscMsg, DWORD dwError );

DWORD StartMwaveAPIServer( void );
DWORD StopMwaveAPIServer( void );
BOOL  ProccessMwRequest( LPBYTE pReqBuffer, DWORD ReqSize, 
                         LPBYTE pAnsBuffer, PDWORD pAnsSize );

RC ManagerInit(VOID);
RC ConfigureDsp(PRDSP prdsp,USHORT usDD,USHORT usId);
VOID msgprintf( ULONG ulMSG, ... );
RC APIENTRY dspNumToHandle(USHORT dspnumber, PHDSP phdsp);



/**************************************************************************************************/
/*  Static variables (global to this file only)                                                   */
/**************************************************************************************************/
static CRITICAL_SECTION s_csAPIAccess;                  // Serialize access to API processing code
static CRITICAL_SECTION s_csAPIStateVars;               // Serialize access to API state variables
static BOOL s_bAPIEnabled = FALSE;                      // TRUE if API request processing is enabled
static BOOL s_bAPIStarted = FALSE;                      // TRUE if the Mwave subsystem has been started
static unsigned s_uAPIUseCount = 0;                     // Number of calls to Start... without calls to Stop...
//static BOOL s_bServiceProcRegistered = FALSE;           // True if our service procedure has been registered



void SetAPIEnabled( BOOL bEnabled )
{
    EnterCriticalSection( &s_csAPIStateVars );
    s_bAPIEnabled = bEnabled;
    LeaveCriticalSection( &s_csAPIStateVars );
}



BOOL IsAPIEnabled( void )
{
    BOOL bEnabled;

    EnterCriticalSection( &s_csAPIStateVars );
    bEnabled = s_bAPIEnabled;
    LeaveCriticalSection( &s_csAPIStateVars );

    return bEnabled;
}



void SetAPIStarted( BOOL bStarted )
{
    EnterCriticalSection( &s_csAPIStateVars );
    s_bAPIStarted = bStarted;
    LeaveCriticalSection( &s_csAPIStateVars );
}



BOOL IsAPIStarted( void )
{
    BOOL bStarted;

    EnterCriticalSection( &s_csAPIStateVars );
    bStarted = s_bAPIStarted;
    LeaveCriticalSection( &s_csAPIStateVars );

    return bStarted;
}



unsigned GetAPIUseCount( void )
{
    unsigned uUseCount;

    EnterCriticalSection( &s_csAPIStateVars );
    uUseCount = s_uAPIUseCount;
    LeaveCriticalSection( &s_csAPIStateVars );

    return uUseCount;
}



void SetAPIUseCount( unsigned uUseCount )
{
    EnterCriticalSection( &s_csAPIStateVars );
    s_uAPIUseCount = uUseCount;
    LeaveCriticalSection( &s_csAPIStateVars );
}



/**************************************************************************************************/
/*                                                                                                */
/*  StartMwaveSubsystem                                                                           */
/*                                                                                                */
/*  Purpose:    Does full initialization of the Mwave subsystem.                                  */
/*                                                                                                */
/*  Entry:      x( void )                                                                         */
/*                                                                                                */
/*  Exit:       BOOL                                                                              */
/*                                                                                                */
/*  Remarks:    This function does full initialization of the Mwave subsystem.  If the            */
/*              subsystem is already started, this function increases the usage count and         */
/*              returns TRUE.  If the function fails, it returns FALSE.                           */
/*                                                                                                */
/**************************************************************************************************/
BOOL StartMwaveSubsystem( void )
{
    MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveSubsystem entry \n");
    InitializeCriticalSection( &s_csAPIStateVars );
    InitializeCriticalSection( &s_csAPIAccess );
    //
    //  If we're already started, increment use count and return TRUE.
    //
    if (IsAPIStarted()) {
	MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveSubsystem mwave already started\n");
        SetAPIUseCount( GetAPIUseCount() + 1 );
        return TRUE;
    }


    //
    //  Call the main Mwave initialization code.  If successful, enable the API.
    //
    MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveSubsystem calling StartMwaveAPIServer\n");
    if (StartMwaveAPIServer() == OK) {
        SetAPIUseCount( 1 );
        SetAPIStarted( TRUE );
        SetAPIEnabled( TRUE );
	MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveSubsystem Exit return TRUE\n");
        return TRUE;
    } else {
        return FALSE;
    }
}



/**************************************************************************************************/
/*                                                                                                */
/*  StopMwaveSubsystem                                                                            */
/*                                                                                                */
/*  Purpose:    Does full shutdown of the Mwave subsystem.                                        */
/*                                                                                                */
/*  Entry:      x( void )                                                                         */
/*                                                                                                */
/*  Exit:       BOOL                                                                              */
/*                                                                                                */
/*  Remarks:    This function does shuts down the Mwave subsystem and disables the API.  If the   */
/*              use count is greater than 1, this function simply decreases the use count.  If    */
/*              the Mwave subsystem is not started, this function does nothing.  This function    */
/*              returns TRUE if the Mwave subsystem is stopped, else it returns FALSE.            */
/*                                                                                                */
/**************************************************************************************************/
BOOL StopMwaveSubsystem( void )
{
    //
    //  If we're started and the use count is greater than 1, decrement use count and return FALSE.
    //  If we're not started, return TRUE.
    //
    if (IsAPIStarted()) {
        if (GetAPIUseCount() > 1 ) {
            SetAPIUseCount( GetAPIUseCount() - 1 );
            return FALSE;
        }
    } else
        return TRUE;


    //
    //  Shutdown the subsystem:
    //      1. Clear the "API enabled" flag to prevent any more API requests from being accepted.
    //      2. Enter the "API access" critical section to wait for pending API request to be processed.
    //      3. Call StopMwaveAPIServer to do the actual subsystem shutdown.
    //      4. Leave the "API access" critical section.
    //      5. Set the API use count to 0 and set the "API started" flag to FALSE.
    //
    SetAPIEnabled( FALSE );
    EnterCriticalSection( &s_csAPIAccess );
    StopMwaveAPIServer();
    LeaveCriticalSection( &s_csAPIAccess );
    SetAPIUseCount( 0 );
    SetAPIEnabled( FALSE );

    return TRUE;
}



/**************************************************************************************************/
/*                                                                                                */
/*  IssueMwaveAPIRequest                                                                          */
/*                                                                                                */
/*  Purpose:    Processes an Mwave API request.                                                   */
/*                                                                                                */
/*  Entry:      x( pbRequestBuffer, ppbAnswerBuffer )                                             */
/*                  pbRequestBuffer : Pointer to buffer containing Mwave API request.             */
/*                  uRequestSize    : Size of the request.                                        */
/*                  ppbAnswerBuffer : Pointer to a pointer to the output of the API request.      */
/*                  puAnswerSize    : Pointer to a variable to contain the size of the answer.    */
/*                                                                                                */
/*  Exit:       BOOL                                                                              */
/*                                                                                                */
/*  Remarks:    This function processes an Mwave API request that is passed in via a request      */
/*              buffer.  This function allocates an answer buffer to contain the results of the   */
/*              API processing, and it writes the address of the answer buffer to                 */
/*              *ppbAnswerBuffer and the size of the answer to *puAnswerSize.  The caller is      */
/*              responsible for freeing the answer buffer with a call to                          */
/*              FreeMwaveAPIAnswerBuffer.  This function returns TRUE if the request              */
/*              was processed or FALSE if no processing was attempted (as in the case of the API  */
/*              being disabled). If FALSE is returned, the answer buffer and answer size outputs  */
/*              are not valid, and the caller must not use FreeMwaveAPIAnswerBuffer.              */
/*                                                                                                */
/**************************************************************************************************/
BYTE AnswerBuffer[0x10000];
BOOL IssueMwaveAPIRequest( BYTE * pbRequestBuffer, unsigned uRequestSize,
                                  BYTE ** ppbAnswerBuffer, unsigned * puAnswerSize )
{
    BOOL     bReturn;
    BYTE   * pbAnswerBuffer=AnswerBuffer;
    DWORD    uAnswerSize;

    MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest entry\n");
    //
    //  Synchronize access to this code
    //
    if (!IsAPIEnabled()) {
        MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPI request, API not enabled return FALSE\n");
        return FALSE;
    }

    EnterCriticalSection( &s_csAPIAccess );

    if (!IsAPIEnabled()) {
        MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest, API not enabled...leaving\n");
        bReturn = FALSE;
        goto LEAVE;
    }
        
    //
    //  Allocate the answer buffer
    //
    MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest, allocating pbAnswerBuffer size %x\n",ANSWER_BUFFER_SIZE);
    //pbAnswerBuffer = malloc( ANSWER_BUFFER_SIZE );
    if (pbAnswerBuffer == NULL) {
        MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest, allocate of pbAnserBuffer failed\n");
        bReturn  = FALSE;
        goto LEAVE;
    }

    //
    //  Process the request
    //
MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest calling ProcessMwRequest\n");
    if (!ProccessMwRequest( pbRequestBuffer, uRequestSize, pbAnswerBuffer, &uAnswerSize )) {
      MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest return from ProcessMwRequest FAIL\n");
      //free( pbAnswerBuffer );
      bReturn = FALSE;
      goto LEAVE;
    }
MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest, return from ProcessMwrequest OK\n");
    /* @TBD got rid of exception handler */


    *ppbAnswerBuffer = pbAnswerBuffer;
    *puAnswerSize = uAnswerSize;
    bReturn = TRUE;

    //
    //  Release the critical section and return
    //    
  LEAVE:
    LeaveCriticalSection( &s_csAPIAccess );
  MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::IssueMwaveAPIRequest exit bReturn %x\n",bReturn);
    return bReturn;
}



/**************************************************************************************************/
/*                                                                                                */
/*  FreeMwaveAPIAnswerBuffer                                                                      */
/*                                                                                                */
/*  Purpose:    Frees the Mwave API answer buffer returned by IssueMwaveAPIRequest.               */
/*                                                                                                */
/*  Entry:      x( pbAnswerBuffer )                                                               */
/*                  pbAnswerBuffer : Buffer returned by IssueMwaveAPIRequest.                     */
/*                                                                                                */
/*  Exit:       void                                                                              */
/*                                                                                                */
/*  Remarks:    This function frees the Mwave API answer buffer that was allocated and returned   */
/*              by the function IssueMwaveAPIRequest.                                             */
/*                                                                                                */
/**************************************************************************************************/
void FreeMwaveAPIAnswerBuffer( BYTE * pbAnswerBuffer )
{
  // @TBD free( pbAnswerBuffer );
}



/* @TBD  VOID MwError( LPTSTR lpscMsg, DWORD dwError )
{
    HANDLE  hMwEvent;
    CHAR    chMsg[256];
    LPTSTR  lpszStrings[2];

    hMwEvent = RegisterEventSource( NULL, MWAVE_SERVICE_NAME );

    sprintf( chMsg, "%s error: %d", MWAVE_SERVICE_NAME, dwError );
    lpszStrings[0] = chMsg;
    lpszStrings[1] = lpscMsg;

    if (hMwEvent != NULL) {

      ReportEvent( hMwEvent,
                   EVENTLOG_ERROR_TYPE,
                   0,
                   0,
                   NULL,
                   2,
                   0,
                   lpszStrings,
                   NULL );

      (VOID) DeregisterEventSource(hMwEvent);
      }

    //
    //if (dwError != NO_ERROR)
    //  SetEvent(MwDoneEvent);    
    //

    return;
    } */



/* @TBD unsigned MwmgrServiceProc( unsigned uCommand, void * pvArgs )
{
    unsigned uRC;
    HDSP hDSP;
    BOOL bDSPFound;
    MWCL_rCProc_PowerEventParms * pPowerEventArgs;

    bDSPFound = (dspNumToHandle( 0, &hDSP ) == DSP_NOERROR);
    switch (uCommand) {
        case MWCL_CProcCmd_NoOp:
            uRC = MWCL_CProcRes_Success;
            break;

        case MWCL_CProcCmd_PowerEvent:
            pPowerEventArgs = (MWCL_rCProc_PowerEventParms *)pvArgs;
            switch (pPowerEventArgs->uPowerState) {
                case MWCL_PowerState_Suspend:
                    //
                    //  Do suspend preparation
                    if (bDSPFound) {
                        DspBIOS_SaveUartState( hDSP );
                        DspBIOS_Suspend( hDSP );
                    }
                    uRC = MWCL_CProcRes_Success;
                    break;

                case MWCL_PowerState_Resume:
                    //
                    //  Do resume recovery
                    if (bDSPFound) {
                        DspBIOS_Resume( hDSP );
                        dspInit( hDSP, "" );
                        DspBIOS_RestoreUartState( hDSP );
                    }
                    uRC = MWCL_CProcRes_Success;
                    break;

                default:
                    uRC = MWCL_CProcRes_Unsupported;
            }
            break;

        default:
            uRC = MWCL_CProcRes_Unsupported;
    }

    return uRC;
}

*/

DWORD StartMwaveAPIServer( void )
{
    ULONG ulRC;

    //
    // Allocate Manager Heaps
    MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveAPIServer entry\n");
    if (!AllocateHeaps())
        return DSP_ERROR;
		
    // Allocate memory for pg (the shared memory stuctured used in os2)
    if ((pg = (PRG) malloc( sizeof(RG) )) == NULL)
        return DSP_ERROR;

    pg->prmgrTail = NULL;
    pg->prmodCurrentMod = NULL;
    pg->usDaemonFunc = 0xffff;          // illegal function

    // Open Driver
    hDrv=open("/dev/modems/mwave",O_RDWR);
    if (!hDrv) {
      MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveAPIServer, unable to open mwave device driver\n");
      return -1;
    }

    // Initialize Manager.
    MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveAPIServer opened mwave device hDrv %x\n",hDrv);
    ulRC = ManagerInit();
    MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMwaveAPIServer return from ManagerInit ulRC %lx\n",ulRC);

    //
    // Register our Mwave Component Loader service procedure
    /* @TBD if (ulRC == DSP_NOERROR) {
        if (mwclRegisterServiceControlProcXMgr( &MwmgrServiceProc ) == MWCLERROR_NoError)
            s_bServiceProcRegistered = TRUE;
	    } */

    MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::StartMWaveAPIServer exit ulRC %lx\n",ulRC);
    return ulRC;
}



DWORD StopMwaveAPIServer( void )
{
    //
    // Unregister our Mwave Component Loader service procedure
  /* if (s_bServiceProcRegistered) {
        mwclUnregisterServiceControlProcX( &MwmgrServiceProc );
        s_bServiceProcRegistered = FALSE;
	} */
    
    // Stop the DSP
    //DspBIOS_DspOff( hDrv );

    // Close Driver
    /* @TBD if (hDrv)
       CloseHandle( hDrv ); */

    // Deallocate pg
    if (pg)
        free( pg );

    //
    // Deallocate Heaps
    DestroyHeaps();
      
    return OK;
}



BOOL ProccessMwRequest( LPBYTE pReqBuffer, DWORD ReqSize, 
                        LPBYTE pAnsBuffer, PDWORD pAnsSize )
{
    PMWMGR_REQUEST_HEAD pReqHead;
    PMWMGR_ANSWER_HEAD  pAnsHead; 
    LPBYTE              pRequestPart;

    MW_SYSLOG_4(TRACE_MANAGER_SPECIFIC,"dspsrv::ProcessMwRequest entry pAnsBuffer %p pReqBuffer %p ReqSize %lx\n",pAnsBuffer,pReqBuffer,ReqSize);

    if (((DWORD) *pReqBuffer) == TERMINATE_SERVER_THREAD) {
      *pAnsSize = sizeof(DWORD);
      memset(pAnsBuffer,0,*pAnsSize);
      /* @TBD ZeroMemory( (PVOID)pAnsBuffer, *pAnsSize ); */
      return FALSE;
      }

    /* update RG structure */

    pReqHead = (PMWMGR_REQUEST_HEAD) pReqBuffer;
    
    pg->usDaemonFunc = (USHORT) pReqHead->dwRequestCode;

    pRequestPart = pReqBuffer + sizeof(MWMGR_REQUEST_HEAD);

    memcpy( (PVOID) &pg->DAEMONpacket, 
                (PVOID) pRequestPart , 
                pReqHead->dwPacketSize );

    pRequestPart += pReqHead->dwPacketSize;

    if (pReqHead->dwPgBuf1Size > 0) {
      
      /* @TBD CopyMemory( (PVOID) &pg->Buffer1,
                  (PVOID) pRequestPart,
                  pReqHead->dwPgBuf1Size ); */
     memcpy( (PVOID) &pg->Buffer1,
	     (PVOID) pRequestPart,
	     pReqHead->dwPgBuf1Size );

      pRequestPart += pReqHead->dwPgBuf1Size;
      }

    else
      pg->Buffer1[0] = 0;


    if (pReqHead->dwPgBuf2Size > 0) {
      
      memcpy( (PVOID) &pg->Buffer2,
                  (PVOID) pRequestPart,
                  pReqHead->dwPgBuf2Size );

      pRequestPart += pReqHead->dwPgBuf2Size;
      }

    else
      pg->Buffer2[0] = 0;

    if (pReqHead->dwPgBuf3Size > 0)
      
      memcpy( (PVOID) &pg->Buffer3,
                  (PVOID) pRequestPart,
                  pReqHead->dwPgBuf3Size );
    else
      pg->Buffer3[0] = 0;
             
    pg->usBufferSize = (USHORT) pReqHead->dwRetBufSize; 
 
    /* in case of core function failure, set the answer packet to */
    /* be a short DSP_INTERNAL_ERROR answer.                      */

    pAnsHead = (PMWMGR_ANSWER_HEAD) pAnsBuffer;   

    pAnsHead->dwReturnCode = DSP_INTERNAL_ERROR;
    pAnsHead->dwHandle     = 0;
    pAnsHead->dwBufSize    = 0;

    *pAnsSize = sizeof(MWMGR_ANSWER_HEAD);
    
    /* Call the apropiate routine */
MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ProcessMwRequest calling pg->usDaemonFunc %x\n",pg->usDaemonFunc);
    switch (pg->usDaemonFunc) {

       case DSPMDD_DspAllocateSegment :
                 DAEMON_DspAllocateSegment(pAnsHead);
                 break;
       case DSPMDD_DspChangeCPF :
                 DAEMON_DspChangeCPF(pAnsHead);
                 break;
       case DSPMDD_DspChangeDMA :
                 DAEMON_DspChangeDMA(pAnsHead);
                 break;
       case DSPMDD_DspChangeModuleState :
                 DAEMON_DspChangeModuleState(pAnsHead);
                 break;
       case DSPMDD_DspChangeTaskState :
                 DAEMON_DspChangeTaskState(pAnsHead);
                 break;
       case DSPMDD_DspConnectGPC :
                 DAEMON_DspConnectGPC(pAnsHead);
                 break;
       case DSPMDD_DspConnectIPC :
                 DAEMON_DspConnectIPC(pAnsHead);
                 break;
       case DSPMDD_DspConnectITCB :
                 DAEMON_DspConnectITCB(pAnsHead);
                 break;
       case DSPMDD_DspDisconnectIPC :
                 DAEMON_DspDisconnectIPC(pAnsHead);
                 break;
       case DSPMDD_DspDisconnectGPC :
                 DAEMON_DspDisconnectGPC(pAnsHead);
                 break;
       case DSPMDD_DspDisconnectITCB :
                 DAEMON_DspDisconnectITCB(pAnsHead);
                 break;
       case DSPMDD_DspFreeModule :
                 DAEMON_DspFreeModule(pAnsHead);
                 break;
       case DSPMDD_DspFreeSegment :
                 DAEMON_DspFreeSegment(pAnsHead);
                 break;
       case DSPMDD_DspFreeTask :
                 DAEMON_DspFreeTask(pAnsHead);
                 break;
       case DSPMDD_DspInit :
                 DAEMON_DspInit(pAnsHead);
                 break;
       case DSPMDD_DspLabelToAddr :
                 DAEMON_DspLabelToOffset(pAnsHead);
                 break;
       case DSPMDD_DspLoadModule :
                 DAEMON_DspLoadModule(pAnsHead);
                 break;
       case DSPMDD_DspLoadSegment :
                 DAEMON_DspLoadSegment(pAnsHead);
                 break;
       case DSPMDD_DspLoadTask :
                 DAEMON_DspLoadTask(pAnsHead);
                 break;
       case DSPMDD_DspNameToModuleHandle :
                 DAEMON_DspNameToModuleHandle(pAnsHead);
                 break;
       case DSPMDD_DspNameToSegmentHandle :
	   DAEMON_DspNameToSegmentHandle(pAnsHead);
                 break;
       case DSPMDD_DspNameToTaskHandle :
                 DAEMON_DspNameToTaskHandle(pAnsHead);
                 break;
       case DSPMDD_DspQueryDSPInfo :
                 DAEMON_DspQueryDSPInfo(pAnsHead);
                 break;
       case DSPMDD_DspQueryManagerInfo :
                 DAEMON_DspQueryManagerInfo(pAnsHead);
                 break;
       case DSPMDD_DspQueryMiscInfo :
                 DAEMON_DspQueryMiscInfo(pAnsHead);
                 break;
       case DSPMDD_DspQueryModuleInfo :
                 DAEMON_DspQueryModuleInfo(pAnsHead);
                 break;
       case DSPMDD_DspQueryTaskInfo :
                 DAEMON_DspQueryTaskInfo(pAnsHead);
                 break;
       case DSPMDD_DspAbilities :
                 DAEMON_DspAbilities(pAnsHead);
                 break;
       case DSPMDD_DspReset :
                 DAEMON_DspReset(pAnsHead);
                 break;
       case DSPMDD_DspRun :
                 DAEMON_DspRun(pAnsHead);
                 break;

       default:
                 pAnsHead->dwReturnCode = DSP_INTERNAL_ERROR;

       }  /* endswitch */

      if (pAnsHead->dwReturnCode == DSP_NOERROR)
        *pAnsSize = sizeof(MWMGR_ANSWER_HEAD) + pAnsHead->dwBufSize;
      else
        *pAnsSize = sizeof(MWMGR_ANSWER_HEAD);
 
      MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::ProcessMwRequest exit\n");
    return TRUE;
}

RC ManagerInit(VOID)
{
   RC         ulRC = DSP_NOERROR;      // Initialization assumed by code.
   PRDSP      prdspHead;               // ptr to head of DSP list;
   PRDSP      prdsp;                   // ptr to current DSP structure
   PRMGR      prmgr;                   // ptr to current manager structure
   PRMGR      prmgrHead;               // ptr to head of manager list;
   PSZ        pszFileName;             // ptr to name of ISPOS.dsp file
   PSZ        pszBIOSFileName;         // ptr to name of BIOS.dsp file
   USHORT     usNum;                   // Number of DSPs for Current DD
   USHORT     usNum_Total;             // Number of DSPs for all BIOS DD
   USHORT     i;                       // Counter
   USHORT     DD;                      // BIOS DD Number/ID

   CHAR       *ISPOS_name[10] =  {
      "MWAVEOS","MWAVEOS1","MWAVEOS2","MWAVEOS3","MWAVEOS4","MWAVEOS5",
      "MWAVEOS6","MWAVEOS7","MWAVEOS8","MWAVEOS9"
   } ;                                 // Module Instance Names for ISPOS

   usNum = 1;                          // Debug: assume one DSP
   usNum_Total =1;
   DD = 0;
   MW_SYSLOG_4(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit entry usNum %x usNum_Total %x DD %x\n",usNum,usNum_Total,DD);

   // Igor - 08/12/96: I moved eventInit() to here because it was never reached
   #ifdef MWNS_ENABLED
   // dead code for us @MIKE
   MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit entry calling eventInit()\n");
   eventInit();   /* Initialize MWAGENT Callbacks CH02 */
   #endif

         /*******************************************************************/
         /* Add a new manager to the Manager DSP list                       */
         /*******************************************************************/

         if ((ulRC = AddRManagerTail(&pg->prmgrTail, &prmgr)) != DSP_NOERROR)
            return (ulRC);
	 MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit return from AddRManagerTail prmgr=%x\n",(int)prmgr);

         /*******************************************************************/
         /* Create a DSP structure for each DSP and fill in the contents    */
         /*******************************************************************/

         for (i = 0; (i < usNum) && (ulRC == DSP_NOERROR); i++) {

               /*************************************************************/
               /* create a DSP structure                                    */
               /*************************************************************/
	     MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit calling AddRDspTail()\n");
               if ((ulRC = AddRDspTail(&prmgr->MGR_prdspTail, &prdsp)) !=
                  DSP_NOERROR)
                  return (ulRC);
	    MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit AddRDspTail returns prdsp %x\n",(int)prdsp);

	    MW_SYSLOG_4(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit entry calling ConfigureDsp prdsp %x, DD %x, usNum-1 %x\n",
		    (int)prdsp,DD,usNum-1);
               if ((ulRC = ConfigureDsp(prdsp, DD, (USHORT)(usNum-i))) !=
                  DSP_NOERROR)
                  return (ulRC);
	    MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit return from ConfigureDsp()\n");

               prdsp->DSP_prmgr = prmgr; // Point DSP to MGR
            }

   if (usNum_Total == 0) {       // No ISP reported by any of the BIOS DDs
      return (DSP_NO_DSP_FOUND);
   }

   /*************************************************************************/
   /* Initialize all DSPs in the system.                                    */
   /*************************************************************************/
   /* get path\name of DSP operating system from environment                */
   /*************************************************************************/

// if ((pszFileName = getenv("MWOSNAME")) == NULL)
   if ((pszFileName = GetDSPStringAttribute(DSPATTR_OSFILENAME, prdsp)) == NULL)
     return (DSP_FILE_NOT_FOUND);
   MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit pszFileName %s\n",pszFileName);
   prmgrHead = prmgr = pg->prmgrTail->MGR_prmgrNext; // set head ptr
   do {

         i = 0;                        // ISPOS Instance

         prdspHead = prdsp = prmgr->MGR_prdspTail;

         /*******************************************************************/
         /* Traverse the DSP list for this manager                          */
         /*******************************************************************/

         do {
               strcpy(&(prdsp->DSP_ISPOS[0]), ISPOS_name[i]); // Copy Orignal Name
	       MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit prdsp->DSP_ISPOS[0] %s\n",prdsp->DSP_ISPOS);

	       MW_SYSLOG_3(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit calling InitDSP, prdsp %x pszFileName %s, DSP_INIT_ACTIVE\n",
		       (int)prdsp, pszFileName);

               if ((ulRC = InitDSP(prdsp, pszFileName, DSP_INIT_ACTIVE)) !=
                  DSP_NOERROR)
                  return (ulRC);

	       MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit, InitDSP returns ulRC %lx\n",ulRC);

               /*************************************************************/
               /* Record the BIOS available to the Manager                  */
               /*************************************************************/

//             if ((pszBIOSFileName = getenv("MWBIOS")) == NULL)
               if ((pszBIOSFileName = GetDSPStringAttribute(DSPATTR_BIOSFILENAME, prdsp)) == NULL)
                 return (DSP_FILE_NOT_FOUND);

	       MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit, GetDSPStringAttribute DSPATTR_BIOSFILENAME returns %s\n",
		       pszBIOSFileName);

	       MW_SYSLOG_3(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit, InitDSP calling RecordBios pmgr %x pszBIOSFileName %s\n",
		       (int)prmgr,pszBIOSFileName);

               ulRC = RecordBios(prmgr, pszBIOSFileName);

               prdsp = prdsp->DSP_prdspNext; // Move to next DSP
               i = i++;                // Move to next ISPxxx Vitrual Name

            }  while (prdsp != prdspHead);

         prmgr = prmgr->MGR_prmgrNext; // go to next in list
      }  while (prmgr != prmgrHead);   // Back at head yet ?

      MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ManagerInit exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ConfigureDsp                                            */
/*                                                                          */
/* FUNCTION: This routine will get information about a DSP and save that    */
/*           information in the appropriate RDSP structure. This routine    */
/*           will probably call ABIOS to get the info about the DSP         */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP  prdsp - Ptr to the DSP structure to fill in.               */
/*        USHORT usId  - ID number of the DSP. (1-based number)             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ConfigureDsp(PRDSP prdsp,USHORT usDD,USHORT usId)
{
   RC         ulRC;                    // Error return code
   PRMEM      prmem;                   // Ptr to memory structure
   PRHW       prhw;                    // Ptr to HW structure
   PVOID      pBuf = NULL;             // Ptr to Buffer (NULL assumed)
   ULONG      ulHW_Info;              // Count/Name of HW from DSPAbilities
   USHORT     Num_HW;                 // Index into DSPAbilities Structure
   USHORT     Num_ID;                 // Index for Loop adding the same HW Name
   USHORT     usHW_Name;              // Name of HW from DSPABilities
   USHORT     usHW_Count;             // Count for a HW Name from DSPAbilities
   USHORT     usBuf_Size;             // Size of DSPAbilities Query Buffer
   USHORT     len;                    /* CH01 */
   CHAR       szSect[20];             /* CH01 */			  
   CHAR       szOut[20];              /* CH01 */

   MW_SYSLOG_4(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDsp entry prdsp %x usDD %x usId %x\n",(int)prdsp,usDD,usId);
   prdsp->DSP_usID = usId;
   prdsp->DSP_rver.bVersion = 5;
   prdsp->DSP_rver.bRelease = 6;

   /*************************************************************************/
   /* Save Info to associate this with a DSP and BIOS DD                    */
   /*************************************************************************/

   prdsp->DSP_ulBIOSDD_ID = (ULONG)usId; // BIOS DD's Identifier(which ISP)
   prdsp->DSP_DDEntry = pg->BDDInfo[usDD].pfnIDCEntry; // BDD Entry point CH02

   /*************************************************************************/
   /* Query the DSP Abilities                                               */
   /*************************************************************************/


   usBuf_Size = 0;
   if ((ulRC = DspBIOS_Abilities(prdsp, /* Find Size of Buffer required     */
      &usBuf_Size, pBuf)) != DSP_INS_BUFFER)
      return (ulRC);

   /* @TBD if ((pBuf = VirtualAlloc(NULL,     
      (ULONG)usBuf_Size, MEM_COMMIT|MEM_RESERVE,PAGE_READWRITE)) == NULL) */
     if ((pBuf=malloc(usBuf_Size)) == NULL)
      return (DSP_ERROR);

   if ((ulRC = DspBIOS_Abilities(prdsp, /* Get DSP's Abilities Info         */
      &usBuf_Size, pBuf)) != DSP_NOERROR)
      return (ulRC);
   MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDsp return from DspBIOS_Abilities Num_Hw %x\n",
	   ((RDSP_ABILITIES_INFO *)pBuf)->DSPABILITIES_usResv5);

   /*************************************************************************/
   /* Save Information need from the Query Buffer in the DSP Structure      */
   /*************************************************************************/
   /* Loop thru each HW provides in the Abilities Info Query Buffer         */
   /*************************************************************************/

   for (Num_HW = 0; Num_HW < ((RDSP_ABILITIES_INFO *)pBuf)->
      DSPABILITIES_usResv5; Num_HW++) {

         ulHW_Info = ((RDSP_ABILITIES_INFO *)pBuf)->DSPABILITIES_ulResv6
            [Num_HW];
         usHW_Name = (USHORT)(ulHW_Info);/* Lower 16 Bits                   */
         usHW_Count = (USHORT)(ulHW_Info >> 16);/* Upper 16 Bits            */

         /*******************************************************************/
         /* Loop thru Number of each HW ID provided                         */
         /*******************************************************************/

	 MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP usHW_Count %x\n",usHW_Count);
         for (Num_ID = 0; Num_ID < usHW_Count; Num_ID++) {

               /*************************************************************/
               /* Add New element to list of HW this DSP supports           */
               /*************************************************************/

               ulRC = AddRHwTail(&(prdsp->DSP_prhwTail), &prhw);

               /*************************************************************/
               /* Fill in info in the new element as to type of HW          */
               /*************************************************************/

               prhw->HW_prtsk = NULL;
               prhw->HW_ulhwType = (ULONG)usHW_Name;

            }                          /* endfor Num_ID                     */
      }                                /* endfor Num_HW                     */

   /*************************************************************************/
   /* Allocate the CPS.                                                     */
   /*************************************************************************/

   prdsp->DSP_ulCPS = ((RDSP_ABILITIES_INFO *)pBuf)->DSPABILITIES_ulIPS;
   prdsp->DSP_ulCPSFree = ((RDSP_ABILITIES_INFO *)pBuf)->DSPABILITIES_ulIPS;

   prdsp->DSP_ulIMemTotal = (((RDSP_ABILITIES_INFO *)pBuf)->
      DSPABILITIES_ulInstStore);
   MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP, prdsp->DSP_ulIMemTotal %lx\n",prdsp->DSP_ulIMemTotal);

   /*************************************************************************/
   /* convert DataStore to Bytes                                            */
   /*************************************************************************/

   prdsp->DSP_ulDMemTotal = 2*(((RDSP_ABILITIES_INFO *)pBuf)->
      DSPABILITIES_ulDataStore);

   /*************************************************************************/
   /* Pick up Bus Capacity                                                  */
   /*************************************************************************/

   pg->prmgrTail->MGR_ulBusCapacity = pg->prmgrTail->MGR_ulRemBusCapacity =
      (((RDSP_ABILITIES_INFO *)pBuf)->DSPABILITIES_ulSys_Bus_Capacity);

   /*************************************************************************/
   /* CH01 Check if MWAVE.INI file exists and override abilities info       */
   /*************************************************************************/

   sprintf(szSect, "MWAVE,BOARD,slot%d", ((RDSP_ABILITIES_INFO *)pBuf)->
                                             DSPABILITIES_usSlot_Number);
MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP, szSect %s\n",szSect); // @TBD
   len = (USHORT)GetPrivateProfileString(szSect, "ISSize", "",
                                               szOut, sizeof(szOut), ini_file);
MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP, len %x returned\n",len);
 MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigurueDSP, szOut %s\n",szOut);
   if (len!=0) {
       sscanf(szOut, "%ld", &prdsp->DSP_ulIMemTotal);
       MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP prdsp->DSP_ulIMemTotal %lx\n",prdsp->DSP_ulIMemTotal);
   }

   len = (USHORT)GetPrivateProfileString(szSect, "DSSize", "",
                                         szOut, sizeof(szOut), ini_file);
   if (len!=0) {
       if (sscanf(szOut, "%ld", &prdsp->DSP_ulDMemTotal) == 1) {
           prdsp->DSP_ulDMemTotal <<= 1;  /* convert words->bytes */
	   MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP prdsp->DSP_ulDMemTotal %lx\n",prdsp->DSP_ulDMemTotal);
       }


   }

   /* @TBD if (!(ulRC = VirtualFree(pBuf,0,MEM_RELEASE)))
      return (DSP_ERROR); */
   if (pBuf) free(pBuf);

   prdsp->DSP_ulMemXferDMAcb = 0;

   /*************************************************************************/
   /* Initialize DSP data memory                                            */
   /*************************************************************************/

   if ((ulRC = AllocRMemory(&prmem)) == DSP_NOERROR) {

       MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP AllocRMemory returns  data prmem %x\n",(int)prmem);
      /**********************************************************************/
      /* Initialize data memory ADDRESS chain                               */
      /**********************************************************************/

      prdsp->DSP_prmemDTail = prmem;
      prmem->MEM_prmemNext = prmem->MEM_prmemPrev = prmem;

      /**********************************************************************/
      /* Initialize data memory FREE chain                                  */
      /**********************************************************************/

      prdsp->DSP_prmemDFreeTail = prmem;
      prmem->MEM_prmemListNext = prmem->MEM_prmemListPrev = prmem;

      /**********************************************************************/
      /* initialize total free memory size                                  */
      /**********************************************************************/

      prmem->MEM_ulMemSize = prdsp->DSP_ulDMemTotal;
      prmem->MEM_usUseCount = 0;    // AllocRmemory sets it to 1, free MEM
                                        //  must always have a Use Count of 0.

      /**********************************************************************/
      /* Initialize DSP instruction memory                                  */
      /**********************************************************************/

      if ((ulRC = AllocRMemory(&prmem)) == DSP_NOERROR) {

	  MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP AllocRMemory returns instr prmem %x\n",(int)prmem);
         /*******************************************************************/
         /* Initialize instruction memory ADDRESS chain                     */
         /*******************************************************************/

         prdsp->DSP_prmemITail = prmem;
         prmem->MEM_prmemNext = prmem->MEM_prmemPrev = prmem;

         /*******************************************************************/
         /* Initialize instruction memory FREE chain                        */
         /*******************************************************************/

         prdsp->DSP_prmemIFreeTail = prmem;
         prmem->MEM_prmemListNext = prmem->MEM_prmemListPrev = prmem;

         /*******************************************************************/
         /* initialize total free memory size                               */
         /*******************************************************************/

         prmem->MEM_ulMemSize = prdsp->DSP_ulIMemTotal;
         prmem->MEM_usUseCount = 0; // AllocRmemory sets it to 1, free MEM
                                        //  must always have a Use Count of 0.

#if defined(NODSP) || defined(SINGLEPRC)
         /*******************************************************************/
         /* Emulate DSP memory                                              */
         /*******************************************************************/

         if ((ulRC = AllocatePerm(prdsp->DSP_ulIMemTotal *4, (PVOID *)
         &prdsp->DSP_pInstMem)) != DSP_NOERROR)

            return (ulRC);
         if ((ulRC = AllocatePerm(prdsp->DSP_ulDMemTotal, (PVOID *)
            &prdsp->DSP_pDataMem)) != DSP_NOERROR)
            return (ulRC);
	 MW_SYSLOG_3(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP AllocPerm returns prdsp->DSP_pInstMem %x prdsp->DSP_pDataMem %x\n",
		 prdsp->DSP_pInstMem,prdsp->DSP_pDataMem);
#endif
      }
   }
   MW_SYSLOG_2(TRACE_MANAGER_SPECIFIC,"dspsrv::ConfigureDSP Exit ulRC %lx\n",ulRC);

   return (ulRC);
}

/****************************************************************************/
/*                                                                          */
/* Function Name: msgprintf                                                 */
/*                                                                          */
/* FUNCTION: This function accepts a ulong msg identifier and a string and  */
/*           extracts the specified msg resource for use in a printf.       */
/*                                                                          */
/* INPUT:                                                                   */
/*        ULONG  ulMsg - MSG identifier (usually defined in DSPMSG.H)       */
/*                                                                          */
/*                                                                          */
/* OUTPUT: n/a                                                              */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
VOID msgprintf( ULONG ulMSG, ... )
{
	MW_SYSLOG_1(TRACE_MANAGER_SPECIFIC,"A OS2 Message should have been printed.\n");
}

/*------------------------------------------------------------------------*/
/* This section defines the funntions dspNumToHandle and dspInit which    */
/* are not available to the OS/2 daemon.  It also defines the functions   */
/* necessary to serialize the API requests amoung all blocked callers.    */
/*------------------------------------------------------------------------*/

/****************************************************************************/
/*                                                                          */
/* Function Name: dspNumToHandle                                            */
/*                                                                          */
/* FUNCTION: This function converts an integer to an HDSP for internal      */
/*           use of the daemon.                                             */
/*                                                                          */
/* INPUT: USHORT dspnumber                                                  */
/*        PHDSP  address of hdsp                                            */
/*                                                                          */
/* OUTPUT: HDSP                                                             */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY dspNumToHandle(USHORT dspnumber, PHDSP phdsp)
{
   if ((dspnumber > 0 )||(NULL==phdsp)) 
     return(DSP_INV_PARAMETER);

   *phdsp=pg->prmgrTail->MGR_prdspTail;
   return (DSP_NOERROR);
}


