//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: thread.cpp,v 1.3 2002/02/27 15:53:45 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include "thread.h"
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/poll.h>
#include <fcntl.h>

#include "globals.h"
#include "errno.h"

//---------------------------------------------------------
//   Thread
//---------------------------------------------------------

Thread::~Thread()
      {
      }

//---------------------------------------------------------
//   serverloop
//---------------------------------------------------------

static void* loop(void* mops)
      {
      Thread* t = (Thread*) mops;
      t->loop();
      return 0;
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void Thread::start()
      {
      pthread_create(&thread, 0, ::loop, this);
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void Thread::stop()
      {
      pthread_cancel(thread);
      if (pthread_join(thread, 0)) {
            // perror("Failed to join sequencer thread");
            }
      _running = false;
      }

//---------------------------------------------------------
//   Thread
//---------------------------------------------------------

Thread::Thread(bool rt=true, int prio=50, bool ml = true)
      {
      realTimeScheduling = rt;
      lockMemory         = ml;
      realTimePriority   = prio;
      pfd                = 0;
      npfd               = 0;
      maxpfd             = 0;
      _running           = false;
      _pollWait          = -1;

      // create message channels
      int filedes[2];         // 0 - reading   1 - writing
      if (pipe(filedes) == -1) {
            perror("thread:creating pipe4");
            exit(-1);
            }
      toThreadFdr = filedes[0];
      toThreadFdw = filedes[1];

      if (pipe(filedes) == -1) {
            perror("thread: creating pipe5");
            exit(-1);
            }
      fromThreadFdr = filedes[0];
      fromThreadFdw = filedes[1];
      }

//---------------------------------------------------------
//   setRTScheduling
//---------------------------------------------------------

void Thread::setRTScheduling()
      {
      if (realTimeScheduling || lockMemory) {
#ifdef RTCAP
            extern void getCapabilities();
            getCapabilities();
#else
            doSetuid();
#endif
            struct sched_param sp;
		
            printf("set rt %d\n", realTimePriority);
            if (realTimePriority < sched_get_priority_min(SCHED_RR))
                  realTimePriority = sched_get_priority_min(SCHED_RR);
            else if (realTimePriority > sched_get_priority_max(SCHED_RR))
                  realTimePriority = sched_get_priority_max(SCHED_RR);

            sp.sched_priority = realTimePriority;

	      if (sched_setscheduler(0, SCHED_RR, &sp) == 0) {
                  printf("thread: running as realtime process now (priority %d)\n",
                     realTimePriority);
		      }
		else
			perror("WARNING: Can't get realtime priority:");
            if (lockMemory) {
                  if (mlockall(MCL_FUTURE))
                        perror("WARNING: Cannot lock memory:");
                  }
#ifndef RTCAP
            undoSetuid();
#endif
            }
      }

//---------------------------------------------------------
//   addPollFd
//---------------------------------------------------------

void Thread::addPollFd(int fd, int action, void (*handler)(void*,void*), void* p, void* q)
      {
      for (iPoll i = plist.begin(); i != plist.end(); ++i) {
            if ((i->fd == fd) && (i->action == action))
                  return;
            }

      plist.push_back(Poll(fd, action, handler, p, q));

      if (npfd == maxpfd) {
            int n = (maxpfd == 0) ? 4 : maxpfd * 2;
            pfd   = new struct pollfd[n];
            maxpfd = n;
            }
      ++npfd;
      int idx = 0;
      for (iPoll i = plist.begin(); i != plist.end(); ++i, ++idx) {
            pfd[idx].fd     = i->fd;
            pfd[idx].events = i->action;
            }
      }

//---------------------------------------------------------
//   removePollFd
//---------------------------------------------------------

void Thread::removePollFd(int fd, int action)
      {
      for (iPoll i = plist.begin(); i != plist.end(); ++i) {
            if (i->fd == fd && i->action == action) {
                  plist.erase(i);
                  --npfd;
                  break;
                  }
            }
      int idx = 0;
      for (iPoll i = plist.begin(); i != plist.end(); ++i, ++idx) {
            pfd[idx].fd     = i->fd;
            pfd[idx].events = i->action;
            }
      }

//---------------------------------------------------------
//   loop
//---------------------------------------------------------

void Thread::loop()
      {
      pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, 0);
      pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, 0);
      setRTScheduling();

      _running = true;
      for (;;) {
            if (debugMode)          //DEBUG
                  _pollWait = 1;
            int n = poll(pfd, npfd, _pollWait);
            if (n < 0) {
                  fprintf(stderr, "poll failed: %s\n", strerror(errno));
                  if (errno == EINTR)
                        continue;
                  exit(-1);
                  }
            if (n == 0) {       // timeout
                  defaultTick();
                  continue;
                  }
            struct pollfd* p = &pfd[0];
            for (iPoll ip = plist.begin(); ip != plist.end(); ++ip, ++p) {
//                  if (ip != plist.begin())
//                        printf("%d  %d & %d\n", p->fd, ip->action, p->revents);
                  if (ip->action & p->revents)
                        (ip->handler)(ip->param1, ip->param2);
                  }
            }
      }

//---------------------------------------------------------
//   send
//    send request from gui to thread
//    wait until request is processed
//---------------------------------------------------------

bool Thread::sendMsg(const ThreadMsg* m)
      {
      static char c = 0;
      ++c;
      if (_running) {
            int rv = write(toThreadFdw, &m, sizeof(ThreadMsg*));
            if (rv != sizeof(ThreadMsg*)) {
                  perror("Thread::sendMessage(): write pipe failed");
                  return true;
                  }

            ThreadMsg* rm;
            // wait for sequencer to finish operation
//printf("vor read: npfd %d\n", npfd);
            rv = read(fromThreadFdr, &rm, sizeof(ThreadMsg*));
//printf("nach read\n");
            if (rv != sizeof(ThreadMsg*)) {
                  perror("Thread::sendMessage(): read pipe failed");
                  return true;
                  }
            }
      else {
            // if thread is not running (during initialization)
            // process commands directly:
            processMsg(m);
            }
      return false;
      }

//---------------------------------------------------------
//   send
//    send request from gui to thread
//    do __not__ wait until request is processed
//---------------------------------------------------------

bool Thread::sendMsg1(const void* m, int n)
      {
      int rv = write(toThreadFdw, m, n);
      if (rv != n) {
            perror("Thread::sendMessage1(): write pipe failed");
            return true;
            }
      return false;
      }

//---------------------------------------------------------
//   readMsg
//    sequencer reads one gui message
//---------------------------------------------------------

void Thread::readMsg()
      {
      ThreadMsg* p;
      if (read(toThreadFdr, &p, sizeof(p)) != sizeof(p)) {
            perror("Thread::readMessage(): read pipe failed");
            exit(-1);
            }
//printf("vor process\n");
      processMsg(p);
//printf("nach process\n");
      int rv = write(fromThreadFdw, p, sizeof(p));
      if (rv != sizeof(p))
            perror("Thread::readMessage(): write pipe failed");
      }

//---------------------------------------------------------
//   readMsg
//    sequencer reads one gui message
//---------------------------------------------------------

void Thread::readMsg1(int size)
      {
      char buffer[size];
      int n = read(toThreadFdr, buffer, size);
      if (n != size) {
            fprintf(stderr, "Thread::readMessage1(): read pipe failed, get %d, expected %d: %s\n",
               n, size, strerror(errno));
            exit(-1);
            }
      processMsg1(buffer);
      }


