//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: stk.cpp,v 1.2 2002/02/11 10:10:57 muse Exp $
//
//    stk  - stk kit based mono synthesizer
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <sys/time.h>
#include <pthread.h>

#include "mess.h"
#include "ladspa.h"

#include "Object.h"
#include "Clarinet.h"
#include "Flute.h"
#include "Brass.h"
#include "Bowed.h"
#include "Plucked.h"
#include "Mandolin.h"
#include "ModalBar.h"
#include "Rhodey.h"
#include "Wurley.h"
#include "TubeBell.h"
#include "HeavyMtl.h"
#include "PercFlut.h"
#include "BeeThree.h"
#include "Moog1.h"
#include "FMVoices.h"
#include "VoicForm.h"
#include "DrumSynt.h"
#include "Shakers.h"
#include "BowedBar.h"
#include "BlowHole.h"
#include "miditabl.h"

static const int maxPoly = 16;
static const int maxPatches = 20;

static MidiPatch patches[maxPatches] = {
      { 0, 0, 0,  0, "Clarinet" },
      { 0, 0, 0,  1, "Flute"    },
      { 0, 0, 0,  2, "Brass"    },
      { 0, 0, 0,  3, "Bowed"    },
      { 0, 0, 0,  4, "BlowHole" },
      { 0, 0, 0,  5, "Plucked"  },
      { 0, 0, 0,  6, "Mandolin" },
      { 0, 0, 0,  7, "Rhodey"   },
      { 0, 0, 0,  8, "Wurley"   },
      { 0, 0, 0,  9, "TubeBell" },
      { 0, 0, 0, 10, "HeavyMtl" },
      { 0, 0, 0, 11, "PercFlut" },
      { 0, 0, 0, 12, "BeeThree" },
      { 0, 0, 0, 13, "Moog1"    },
      { 0, 0, 0, 14, "FMVoices" },
      { 0, 0, 0, 15, "VoicForm" },
      { 0, 0, 0, 16, "DrumSynt" },
      { 0, 0, 0, 17, "Shakers"  },
      { 0, 0, 0, 18, "BowedBar" },
      { 0, 0, 0, 19, "ModalBar" },
      };

struct Voice {
      Instrmnt* instrument;
      int pitch;
      };

struct Channel {
      Voice voices[maxPoly];
      int program;            // current patch number
      };

//---------------------------------------------------------
//   stk - stk kit based mono synthesizer
//---------------------------------------------------------

class Stk : public Mess {

      Channel channel[16];

      void processEvent(MEvent* ev);
      void write(int, float**, int);
      void noteon(int channel, int pitch, int velo);
      void noteoff(int channel, int pitch);
      void program_change(int channel, int program);
      void pitch_bend(int channel, int data);
      void setCc(int channel, int ctrl, int data);
      void sysex(unsigned char*, int len);

   public:
      Stk(const char* classname);
      ~Stk();
      float* port;          // ladspa output port
      const char* getPatchName(int, int, int, int, MType);
      const MidiPatch* getNextPatch(int ch, const MidiPatch* mp) const;
      };

//---------------------------------------------------------
//   newInstByNum
//---------------------------------------------------------

Instrmnt* newInstByNum(int n)
      {
      fprintf(stderr, "Instrument %d\n", n);
      switch(n) {
            case 0: return new Clarinet(20.0);
            case 1: return new Flute(20.0);
            case 2: return new Brass(20.0);
            case 3: return new Bowed(20.0);
            case 4: return new BlowHole(20.0);
            case 5: return new Plucked(20.0);
            case 6: return new Mandolin(20.0);
            case 7: return new Rhodey;
            case 8: return new Wurley;
            case 9: return new TubeBell;
            case 10: return new HeavyMtl;
            case 11: return new PercFlut;
            case 12: return new BeeThree;
            case 13: return new Moog1;
            case 14: return new FMVoices;
            case 15: return new VoicForm;
            case 16: return new DrumSynt;
            case 17: return new Shakers;
            case 18: return new BowedBar;
            case 19: return new ModalBar;
            default:
                  return 0;
            }
      }

//---------------------------------------------------------
//   Stk
//---------------------------------------------------------

Stk::Stk(const char* cn)
   : Mess(cn, 1)
      {
      for (int c = 0; c < 16; ++c) {
            for (int i = 0; i < maxPoly; ++i) {
                  channel[c].program = 0;
                  channel[c].voices[i].pitch = -1;
                  channel[c].voices[i].instrument = newInstByNum(0);
                  }
            }
      }

//---------------------------------------------------------
//   ~Stk
//---------------------------------------------------------

Stk::~Stk()
      {
      for (int c = 0; c < 16; ++c) {
            for (int i = 0; i < maxPoly; ++i) {
                  delete channel[c].voices[i].instrument;
                  }
            }
      }

//---------------------------------------------------------
//   processEvents
//    process received midi event
//---------------------------------------------------------

void Stk::processEvent(MEvent* ev)
      {
      switch(ev->type()) {
            case SND_SEQ_EVENT_NOTEON:
            case SND_SEQ_EVENT_KEYPRESS:
                  noteon(ev->chan(), ev->dataA(), ev->dataB());
                  break;
            case SND_SEQ_EVENT_NOTEOFF:
                  noteoff(ev->chan(), ev->dataA());
                  break;
            case SND_SEQ_EVENT_PGMCHANGE:
                  program_change(ev->chan(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_PITCHBEND:
                  pitch_bend(ev->chan(), (ev->dataA()<<7) | ev->dataB());
                  break;

            case SND_SEQ_EVENT_CONTROL14:
            case SND_SEQ_EVENT_NONREGPARAM:
            case SND_SEQ_EVENT_REGPARAM:
            case SND_SEQ_EVENT_CONTROLLER:
                  setCc(ev->chan(), ev->dataA(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_SYSEX:
                  sysex(ev->data(), ev->dataLen());
                  break;
            default:
                  break;
            }
      }

//---------------------------------------------------------
//   write
//    synthesize n samples into buffer
//---------------------------------------------------------

void Stk::write(int n, float** ports, int offset)
      {
      float* buffer = ports[0] + offset;
      for (int i = 0; i < n; ++i) {
            for (int c = 0; c < 16; ++c) {
                  for (int k = 0; k < maxPoly; ++k) {
                        if (channel[c].voices[k].pitch != -1)
                              buffer[i] += (double)channel[c].voices[k].instrument->tick();
                        }
                  }
            }
      }

//---------------------------------------------------------
//   noteon
//    midi note on received
//---------------------------------------------------------

void Stk::noteon(int ch, int pitch, int velo)
      {
      if (velo == 0) {
            noteoff(ch, pitch); // treat as note off
            return;
            }
      for (int i = 0; i < maxPoly; ++i) {
            if (-1 == channel[ch].voices[i].pitch) {
                  double lastPitch = __MIDI_To_Pitch[pitch];
                  channel[ch].voices[i].instrument->noteOn(lastPitch, velo*NORM_7);
                  channel[ch].voices[i].pitch   = pitch;
                  break;
                  }
            }
      }

//---------------------------------------------------------
//   noteoff
//    midi note off received
//---------------------------------------------------------

void Stk::noteoff(int ch, int pitch)
      {
      for (int i = 0; i < maxPoly; ++i) {
            if (pitch == channel[ch].voices[i].pitch) {
                  channel[ch].voices[i].instrument->noteOff(64.0);
                  channel[ch].voices[i].pitch = -1;
                  break;
                  }
            }
      }

//---------------------------------------------------------
//   program_change
//---------------------------------------------------------

void Stk::program_change(int ch, int program)
      {
      if (program >= maxPatches)
            return;
      for (int i = 0; i < maxPoly; ++i) {
            channel[ch].voices[i].pitch = -1;
            delete channel[ch].voices[i].instrument;
            channel[ch].voices[i].instrument = newInstByNum(program);
            }
      }

//---------------------------------------------------------
//   pitch_bend
//---------------------------------------------------------

void Stk::pitch_bend(int /*channel*/, int /*data*/)
      {
      }

//---------------------------------------------------------
//   setCc
//---------------------------------------------------------

void Stk::setCc(int /*channel*/, int /*ctrl*/, int /*data*/)
      {
      }

//---------------------------------------------------------
//   sysex
//---------------------------------------------------------

void Stk::sysex(unsigned char*, int /*len*/)
      {
      }

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* Stk::getPatchName(int /*ch*/, int /*hb*/, int /*lb*/, int prog, MType /*type*/)
      {
      for (int i = 0; i < maxPatches-1; ++i) {
            if (patches[i].prog == prog)
                  return patches[i].name;
            }
      return "---";
      }

//---------------------------------------------------------
//   getNextPatch
//---------------------------------------------------------

const MidiPatch* Stk::getNextPatch(int /*ch*/, const MidiPatch* mp) const
      {
      if (mp == 0)
            return &patches[0];
      for (int i = 0; i < maxPatches-1; ++i) {
            if (&patches[i] == mp)
                  return &patches[i+1];
            }
      return 0;
      }

//---------------------------------------------------------
//   inst
//---------------------------------------------------------

static Mess* inst(const char* name)
      {
      return new Stk(name);
      }

//---------------------------------------------------------
//   __init
//---------------------------------------------------------

void __init()
      {
      Mess::initMess(
          1223,
         "Stk",
         "Werner Schweer",
         "stk soft synth",
         "None",
         1,
         inst);
      }

