 /*****************************************************************************
 *                                                                            *
 *  MultiTicker     Author  : Paul Coates, University of Newcastle Upon Tyne  *
 *                  Version : 1.0                                             *
 *                  Date    : 4 Jan 2001                                      *
 *                  Email   : Paul.Coates@ncl.ac.uk                           *
 *                                                                            *
 * Copyright (C) 2001, 2002 Paul Coates
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <applet-widget.h>
#include <ghttp.h>
#include <stdio.h>
#include <pwd.h>

#define MAX_ITEMS 16

class Item
{
	public:
		char title[102];
		char link[502];
		char descr[502];
		int width;
		int sofar;
};

class Image
{
	public:
		char title[102];
		char url[502];
		char link[502];
};

class Site
{
	public:
		char filename[256];
		char url[256];
		char title[102];
		char descr[502];
		char link[502];
		Item items[MAX_ITEMS];
		int num_items;
		Image image;
		int interval;		// time between downloads
		int ticker;			// which ticker is site in
		time_t last;		// last time file was downloaded
		GdkColor col;
};

class Ticker
{
	public:
		int pos;				// current position for scroll
		int total_len;	// total length of all text
};

class MultiTicker
{
	public:
		MultiTicker(int argc, char *argv[]);
		~MultiTicker();

		void draw();
		void alloc_color(GdkColor &col, guint16 r, guint16 g, guint16 b);
		int get_http(Site *s);
		int parse(Site *s);
		int node(Site *s, xmlNodePtr xmlnode);
		int itemnode(Site *s, xmlNodePtr xmlnode);
		int imagenode(Site *s, xmlNodePtr xmlnode);
		void init();
		void update();
		void select(int x, int y);
		void move(int x, int y);
		void show_properties();

		char proxy[502];
		Site *sites;
		int num_sites;
		Ticker *tickers;
		int num_tickers;

		int rate;				// how fast the text scrolls (0 for stop)
		int gap;				// how big the gap is between items

		gint width, height;
		GdkPixmap *canvas;
		GdkCursor *finger;
		GdkCursor *arrow;
		GdkFont *font;
		GdkGC *gc;
		GdkColormap *colormap;
		GdkColor red, green, blue, black;
		GtkWidget *applet;
		GtkWidget *draw_area;
};

int main(int argc, char *argv[])
{
	new MultiTicker(argc, argv);
	applet_widget_gtk_main();
	return 0;
}

static void about_window(AppletWidget *applet, gpointer data)
{
	const gchar *authors[] = { "Paul Coates <Paul.Coates@ncl.ac.uk>", NULL };

	GtkWidget *about;
	about = gnome_about_new("MultiTicker", "1.0", "Copyright 2001",
		authors, "A scrolling text box used to report messages, alerts from monitoring software or news items from your favourite web site. Messages should be written as RDF files.", NULL);
	gtk_widget_show(about);
}

static gint expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	MultiTicker *m = (MultiTicker*)data;

	if (m->canvas!=NULL)
	{
		gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			m->canvas, event->area.x, event->area.y, event->area.x, event->area.y,
			event->area.width, event->area.height);
	}
	return FALSE;
}

static gint configure_event(GtkWidget *w, GdkEventConfigure *e, gpointer d)
{ ((MultiTicker*)d)->draw(); return TRUE; }
static gint button_press_event(GtkWidget *w, GdkEventButton *e, gpointer d)
{ ((MultiTicker*)d)->select((int) e->x, (int) e->y); return TRUE; }
static gint button_release_event(GtkWidget *w, GdkEventButton *e, gpointer d)
{ return TRUE; }
static gint enter_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{ ((MultiTicker*)data)->rate = 0; return TRUE; }
static gint leave_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{ ((MultiTicker*)data)->rate = 1; return TRUE; }
static gint move_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{ ((MultiTicker*)data)->move((int) event->x, (int) event->y); return TRUE; }
gint repaint(gpointer data)
{ ((MultiTicker*)data)->draw(); return TRUE; }
gint reload(gpointer data)
{ ((MultiTicker*)data)->update(); return TRUE; }

MultiTicker::MultiTicker(int argc, char *argv[])
{
	canvas = NULL;
	width = 320;
	height = 24;
	rate = 1;
	gap = 16;

	applet_widget_init("multiticker", NULL, argc, argv, NULL, 0, NULL);

	applet = applet_widget_new("multiticker_applet");
	draw_area = gtk_drawing_area_new();
	gtk_widget_set_events(draw_area, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_POINTER_MOTION_MASK);
	gtk_drawing_area_size(GTK_DRAWING_AREA(draw_area), width, height);
	applet_widget_add(APPLET_WIDGET(applet), draw_area);

	gtk_widget_show_all(applet);

	gtk_signal_connect (GTK_OBJECT (draw_area), "expose_event",
		(GtkSignalFunc) expose_event, this);
	gtk_signal_connect (GTK_OBJECT(draw_area),"configure_event",
		(GtkSignalFunc) configure_event, this);
	gtk_signal_connect (GTK_OBJECT (draw_area), "button_press_event",
		(GtkSignalFunc) button_press_event, this);
	gtk_signal_connect (GTK_OBJECT (draw_area), "button_release_event",
		(GtkSignalFunc) button_release_event, this);
	gtk_signal_connect (GTK_OBJECT (draw_area), "enter_notify_event",
		(GtkSignalFunc) enter_cb, this);
	gtk_signal_connect (GTK_OBJECT (draw_area), "leave_notify_event",
		(GtkSignalFunc) leave_cb, this);
	gtk_signal_connect (GTK_OBJECT (draw_area), "motion_notify_event",
		(GtkSignalFunc) move_cb, this);

	applet_widget_register_stock_callback(APPLET_WIDGET(applet), "about",
		GNOME_STOCK_MENU_ABOUT, "About ...", about_window, this);

////////////////////////////////// resources

	font     = gdk_font_load("-*-helvetica-medium-r-*-*-10-*-*-*-*-*-*-*");
	gc       = gdk_gc_new(applet->window);
	colormap = gdk_colormap_get_system();
	alloc_color(red, 65535, 0, 0);
	alloc_color(green, 0, 65535, 0);
	alloc_color(blue, 0, 0, 65535);
	alloc_color(black, 0, 0, 0);
	arrow = gdk_cursor_new(GDK_ARROW);
	finger = gdk_cursor_new(GDK_HAND2);

	init();

	gtk_timeout_add(10, repaint, this);
	gtk_timeout_add(2000, reload, this);
}

MultiTicker::~MultiTicker()
{
	if (gc) gdk_gc_unref(gc);
}

void MultiTicker::alloc_color(GdkColor &col, guint16 r, guint16 g, guint16 b)
{
	col.red   = r;
	col.green = g;
	col.blue  = b;
	gdk_colormap_alloc_color(colormap, &col, FALSE, TRUE);
}

void MultiTicker::draw()
{
	int i, j;

	if (draw_area->window == NULL) return;
	width = draw_area->allocation.width;
	height = draw_area->allocation.height;
	if (canvas == NULL)
	{
		if (canvas) gdk_pixmap_unref(canvas);
		canvas = gdk_pixmap_new(draw_area->window, width, height, -1);
	}

	gdk_draw_rectangle(canvas,
		draw_area->style->bg_gc[GTK_WIDGET_STATE (draw_area)],
		TRUE, 0, 0, width, height);

	gdk_gc_set_foreground(gc, &black);
	gdk_draw_rectangle(canvas, gc, FALSE, 0, 0, width-1, height-1);
	for (i=0; i<num_tickers; i++)
	{
		tickers[i].pos+=rate;	
		if (tickers[i].pos>tickers[i].total_len)
			tickers[i].pos-=tickers[i].total_len;
	}
	for (i=0; i<num_sites; i++)
	{
		for (j=0; j<sites[i].num_items; j++)
		{
			gdk_gc_set_foreground(gc, &sites[i].col);
			gdk_draw_string(canvas, font, gc, sites[i].items[j].sofar-tickers[sites[i].ticker-1].pos, 10*sites[i].ticker, sites[i].items[j].title);
			gdk_gc_set_foreground(gc, &black);
			gdk_draw_arc(canvas, gc, TRUE, sites[i].items[j].sofar-tickers[sites[i].ticker-1].pos-(gap/2)-2, (10*sites[i].ticker)-5, 4, 4, 0, 64 * 360);
		}
		for (j=0; j<sites[i].num_items; j++)
		{
			gdk_gc_set_foreground(gc, &sites[i].col);
			gdk_draw_string(canvas, font, gc, tickers[sites[i].ticker-1].total_len+sites[i].items[j].sofar-tickers[sites[i].ticker-1].pos, 10*sites[i].ticker, sites[i].items[j].title);
			gdk_gc_set_foreground(gc, &black);
			gdk_draw_arc(canvas, gc, TRUE, tickers[sites[i].ticker-1].total_len+sites[i].items[j].sofar-tickers[sites[i].ticker-1].pos-(gap/2)-2, (10*sites[i].ticker)-5, 4, 4, 0, 64 * 360);
		}
	}
	gdk_draw_pixmap(draw_area->window, gc, canvas, 0, 0, 0, 0, width, height);
}

int MultiTicker::get_http(Site *s)
{
	struct passwd *p;
	char fname[256];
	FILE *f;

	p = getpwuid(getuid());
	sprintf(fname, "%s/.multiticker/%s", p->pw_dir, s->filename);
	f = fopen(fname, "w");
	if (f)
	{
		ghttp_request *request = NULL;
		request = ghttp_request_new();
		if (strcmp(proxy, "")!=0)
			ghttp_set_proxy(request, proxy);
		ghttp_set_uri(request, s->url);
		ghttp_set_header(request, http_hdr_Connection, "close");
		ghttp_prepare(request);
		ghttp_process(request);
		fwrite(ghttp_get_body(request), ghttp_get_body_len(request), 1, f);
		ghttp_request_destroy(request);
		fclose(f);
	}
	return TRUE;
}

int MultiTicker::parse(Site *s)
{
	struct passwd *p;
	char fname[256];
	xmlDocPtr doc;

	p = getpwuid(getuid());
	sprintf(fname, "%s/.multiticker/%s", p->pw_dir, s->filename);
	doc = xmlParseFile(fname);

	if (!doc || !doc->root)
	{
		xmlFreeDoc(doc);
		return FALSE;
	}
	s->num_items = 0;
	node(s, doc->root);
	xmlFreeDoc(doc);
	return TRUE;
}

int MultiTicker::node(Site *s, xmlNodePtr xmlnode)
{
	xmlNodePtr n;

	n = xmlnode->childs;
	while (n != NULL)
	{
		if (strcasecmp((char*)n->name, "item")==0)
		{
			if (s->num_items<MAX_ITEMS)
				itemnode(s, n);
		}else
		if (strcasecmp((char*)n->name, "image")==0)
		{
			imagenode(s, n);
		}else
		if ((strcasecmp((char*)n->name, "title")==0) && (n->childs!=NULL))
		{
			strcpy(s->title, (char*)n->childs->content);
		}else
		if ((strcasecmp((char*)n->name, "link")==0) && (n->childs!=NULL))
		{
			strcpy(s->link, (char*)n->childs->content);
		}else
		if ((strcasecmp((char*)n->name, "description")==0) && (n->childs!=NULL))
		{
			strcpy(s->descr, (char*)n->childs->content);
		}else
			node(s, n);
		n = n->next;
	}
	return TRUE;
}

int MultiTicker::itemnode(Site *s, xmlNodePtr xmlnode)
{
	xmlNodePtr n;

	n = xmlnode->childs;
	while (n != NULL)
	{
		if ((strcasecmp((char*)n->name, "title")==0) && (n->childs!=NULL))
			strcpy(s->items[s->num_items].title, (char*)n->childs->content);
		if ((strcasecmp((char*)n->name, "link")==0) && (n->childs!=NULL))
			strcpy(s->items[s->num_items].link, (char*)n->childs->content);
		if ((strcasecmp((char*)n->name, "description")==0) && (n->childs!=NULL))
			strcpy(s->items[s->num_items].descr, (char*)n->childs->content);
		s->items[s->num_items].width = gdk_string_width(font, s->items[s->num_items].title);
		n = n->next;
	}
	s->num_items++;
	return TRUE;
}

int MultiTicker::imagenode(Site *s, xmlNodePtr xmlnode)
{
	xmlNodePtr n;

	n = xmlnode->childs;
	while (n != NULL)
	{
		if ((strcasecmp((char*)n->name, "title")==0) && (n->childs!=NULL))
			strcpy(s->image.title, (char*)n->childs->content);
		if ((strcasecmp((char*)n->name, "url")==0) && (n->childs!=NULL))
			strcpy(s->image.url, (char*)n->childs->content);
		if ((strcasecmp((char*)n->name, "link")==0) && (n->childs!=NULL))
			strcpy(s->image.link, (char*)n->childs->content);
		n = n->next;
	}
	return TRUE;
}

void MultiTicker::init()
{
	struct passwd *p;
	char fname[256];
	char line[1024];
	char buf1[1024];
	char buf2[1024];
	char buf3[1024];
	FILE *f;
	int i, val, val2;

	p = getpwuid(getuid());
	sprintf(fname, "%s/.multiticker/index", p->pw_dir);

	strcpy(proxy, "");
	num_sites = 0;
	num_tickers = 1;
	f = fopen(fname, "r");
	if (f)
	{
		while(fscanf(f, "%[^\n]\n", line)==1)
		{
			if (sscanf(line, "SOURCE %[^\n]", buf1)==1) num_sites++;
		}
		sites = new Site[num_sites];
		fseek(f, 0L, SEEK_SET);
		num_sites = 0;
		while(fscanf(f, "%[^\n]\n", line)==1)
		{
			if (sscanf(line, "PROXY %[^\n]", buf1)==1)
			{
				strcpy(proxy, buf1);
			}else
			if (sscanf(line, "TICKERS %d", &val)==1)
			{
				num_tickers = val;
			}else
			if (sscanf(line, "SOURCE %[^ ] %[^ ] %d %d %[^\n]",
						buf1, buf2, &val, &val2, buf3)==5)
			{
				strcpy(sites[num_sites].filename, buf1);
				strcpy(sites[num_sites].url, buf2);
				sites[num_sites].interval = val;
				sites[num_sites].ticker = val2;
				gdk_color_parse(buf3, &sites[num_sites].col);
				gdk_colormap_alloc_color(colormap, &sites[num_sites].col, FALSE, TRUE);
				sites[num_sites].last = 0;
				num_sites++;
			}
		}
		fclose(f);
	}

	tickers = new Ticker[num_tickers];
	for (i=0; i<num_tickers; i++)
	{
		tickers[i].pos = 0;
	}
}

void MultiTicker::update()
{
	time_t now = time(NULL);
	int i, j, count;

	for (i=0; i<num_sites; i++)
	{
		if ((now > sites[i].last+sites[i].interval) && (get_http(&sites[i])))
		{ parse(&sites[i]); sites[i].last = now; }
	}

	for (i=0; i<num_tickers; i++)
		tickers[i].total_len=0;

	for (i=0; i<num_sites; i++)
	{
		count = tickers[sites[i].ticker-1].total_len;
		for (j=0; j<sites[i].num_items; j++)
		{
			sites[i].items[j].sofar = count;
			count += sites[i].items[j].width+gap;
		}
		tickers[sites[i].ticker-1].total_len = count;
	}
}

void MultiTicker::select(int x, int y)
{
	int i, j, xx, yy;

	for (i=0; i<num_sites; i++)
	{
		xx = tickers[sites[i].ticker-1].pos + x;
		if (xx>tickers[sites[i].ticker-1].total_len)
			xx-=tickers[sites[i].ticker-1].total_len;
		yy = (sites[i].ticker-1)*10;
		for (j=0; j<sites[i].num_items; j++)
		{
			if ((y>=yy) && (y<=yy+10) && (xx >= sites[i].items[j].sofar) &&
					(xx <= sites[i].items[j].sofar+sites[i].items[j].width) &&
					(strcmp(sites[i].items[j].link, "")!=0))
			{
//				printf("%s\n", sites[i].items[j].title);
				gnome_url_show(sites[i].items[j].link);
			}
		}
	}
}

void MultiTicker::move(int x, int y)
{
	int i, j, xx, yy, hit=0;

	for (i=0; i<num_sites; i++)
	{
		xx = tickers[sites[i].ticker-1].pos + x;
		if (xx>tickers[sites[i].ticker-1].total_len)
			xx-=tickers[sites[i].ticker-1].total_len;
		yy = (sites[i].ticker-1)*10;
		for (j=0; j<sites[i].num_items; j++)
		{
			if ((y>=yy) && (y<=yy+10) && (xx >= sites[i].items[j].sofar) &&
					(xx <= sites[i].items[j].sofar+sites[i].items[j].width))
				hit = 1;
		}
	}
	if (hit==0)
		gdk_window_set_cursor(draw_area->window, arrow);
	else
		gdk_window_set_cursor(draw_area->window, finger);
}

