
/*  @(#)main.c 1.3 01/01/25
 *
 *  Takes a mail file, a news article or an ordinary file
 *  and pretty prints it on a Postscript printer.
 *
 *  Copyright (c) Steve Holden and Rich Burridge.
 *                All rights reserved.
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this software; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "option.h"
#include "header.h"
#include "input.h"
#include "print.h"

int main                       (int, char **);

static void printfile          (void);
static void process_postscript (void);


int
main(int argc, char **argv)
{
    FILE *fp;
    int n = 0;                   /* Index into filename array. */
    int total;

    Header();
    Input();
    Option();
    Print();

    get_appname(argc, argv);     /* Get application name for resources. */
    read_resources();            /* Read resources from merged database. */
    get_options(argc, argv);     /* Read and process command line options. */
    adjust_paper_size();         /* Adjust paper page and line lengths. */

    if (proname[0] == '\0') {
        SPRINTF(proname, "%s/%s", prologue, ptype);
    }

    if (!int_opt_val(O_NUMNAMES)) {
        if (int_opt_val(O_TOPRINT)) {
            set_output(makecmd("stdin"));
        } else {
            set_output(stdout);        /* Send output to standard output. */
        }
        show_prologue(proname);        /* Send prologue file to output. */
        set_input(stdin);              /* Get input from standard input. */
        set_current_filename("stdin");
        printfile();                   /* Pretty print *just* standard input. */
    } else {
        while (n < int_opt_val(O_NUMNAMES)) {
            for (total = 0; total < int_opt_val(O_COPIES); total++) {
                set_current_filename(get_filename(n));
                if (int_opt_val(O_TOPRINT)) {
                    set_output(makecmd(current_filename()));
                } else {
                    set_output(stdout);   /* Send output to standard output. */
                }

                if (n == 0 || int_opt_val(O_TOPRINT)) {
                    show_prologue(proname);     /* Output prologue file. */
                }
                if ((fp = fopen(current_filename(), "r")) == NULL) {
                    FPRINTF(stderr, "%s: cannot open %s\n",
                                    str_opt_val(O_APPNAME), current_filename());
                    continue;
                }
                set_input(fp);
                set_column_count(0);
                set_page_number(1);  /* Initialise current page number. */
                set_eof(0);          /* Reset in case more files to print. */
                reset_headers();
                printfile();         /* Pretty print current file. */
            }
            n++;
        }
    }
    show_trailer();                  /* Send trailer file to output. */
    return(0);
/*NOTREACHED*/
}


static void
printfile(void)    /* Create PostScript to pretty print the current file. */
{
    char *line;
    int blankslate;    /* Nothing set up for printing. */
    int eop;           /* Set if ^L (form-feed) found. */

    if (int_opt_val(O_NUMBER)) {
        set_line_number(0);        /* Reset line counter. */
    }
    readline();
    if (end_of_file()) {
        FPRINTF(stderr, "mp: empty input file, nothing printed\n");
        exit(1);
    }
 
    if (!int_opt_val(O_TEXTDOC)) {
        parse_headers(0);          /* Parse headers of mail or news article. */
    }
    init_setup();                  /* Set values for remaining globals. */

    Start(IS_FILE);
    Start(IS_PAGE);                /* Output initial definitions. */
    blankslate = 0;
    eop        = 0;

/* Print the document. */

    if (doc_type() != DO_TEXT) {
        show_headers(0);
    }

    while (!end_of_file()) {
        line = current_line();
        if (blankslate) {
            Start(IS_FILE);
            Start(IS_PAGE);    /* Output initial definitions. */
            blankslate = 0;
        }

        if (int_opt_val(O_CONTENT) && int_opt_val(O_FOLDER) &&
            message_len() <= 0) {

/*  If the count has gone negative, then the Content-Length is wrong, so go
 *  back to looking for "\nFrom".
 */
            if (message_len() < 0) {
                set_int_opt_val(O_CONTENT, 0);
            } else if ((hdr_equal(FROM_HDR) || hdr_equal(FROMHDR)) &&
                    isupper((int) line[0])) {
                eop = 0;
                set_line_count(int_opt_val(O_PLEN));
                reset_headers();
                parse_headers(0);
                show_headers(0);
            } else {
                set_int_opt_val(O_CONTENT, 0);
            }
        }

        if (!int_opt_val(O_CONTENT) && int_opt_val(O_FOLDER) &&
            (!(int_opt_val(O_ELM) && hdr_equal(FROM_HDR)) ||
              (int_opt_val(O_ELM) && hdr_equal(FROMHDR))) &&
              isupper((int) line[0])) {
            eop = 0;
            set_line_count(int_opt_val(O_PLEN));
            reset_headers();
            parse_headers(0);
            show_headers(0);
        }
        if (int_opt_val(O_DIGEST) &&
            (hdr_equal(FROMHDR) || hdr_equal(DATEHDR) ||
             hdr_equal(SUBJECTHDR)) && isupper((int) line[0])) {
            set_line_count(int_opt_val(O_PLEN));
            parse_headers(1);
            show_headers(1);
        }

        if (int_opt_val(O_PRINT_PS) && hdr_equal(POSTSCRIPT_MAGIC)) {
            if (int_opt_val(O_NUMCOLS)) {
                End(IS_COL);
            }
            End(IS_PAGE);
            End(IS_FILE);
            process_postscript();
            blankslate = 1;
        } else if (int_opt_val(O_FOLDER) && end_of_page()) {
            eop = 1;
        } else {
            if (eop == 1) {
                set_eop(1);
            }
            show_text(T_PLAIN, NULL, line);
            eop = 0;
        }

        if (int_opt_val(O_CONTENT)) {
            set_message_len(message_len() - line_len());
        }

        readline();
    }    

    if (!blankslate) {
        if (int_opt_val(O_NUMCOLS)) {
            End(IS_COL);
        }
        End(IS_PAGE);
        End(IS_FILE);
    }
    close_input();
}


static void
process_postscript(void)
{
    char *line;
    int firstline = 1;    /* To allow a newline after the first line. */

    Start(IS_PAGE);
    while (!hdr_equal(FROMHDR)    && !hdr_equal(DATEHDR) &&
           !hdr_equal(SUBJECTHDR) && !end_of_file()) {
        line = current_line();
        write_output(line);
        if (firstline) {
            write_output("\n");
        }
        firstline = 0;
        if ((line = get_line()) == NULL) {
            set_eof(1);
        }
    }
    End(IS_PAGE);
}
