/* A simple synchronous XML-RPC client written in C. */

#include "motion.h"

#include <xmlrpc.h>
#include <xmlrpc_client.h>

#define NAME "XML-RPC-C Motion control"

#define MOTION_URL "http://localhost:8080"

void die_if_fault_occurred (xmlrpc_env *env)
{
	if (env->fault_occurred) {
		fprintf(stderr, "XML-RPC Fault: %s (%d)\n",
			env->fault_string, env->fault_code);
		exit(1);
	}
}

void info (char *url) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int i;
	char *version;

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.info", "()");
	die_if_fault_occurred(&env);
	xmlrpc_parse_value(&env, result, "(si)", &version, &i);
	die_if_fault_occurred(&env);
	printf("Motion version string: %s\n", version);
	printf("Number of threads: %d\n", i);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void conf_list (char *url) {
	xmlrpc_env env;
	xmlrpc_value *result;
	xmlrpc_value *struct_conf;
	char *option, *type, *help;
	int i, size;

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.conf.list", "()");
	die_if_fault_occurred(&env);
	size = xmlrpc_array_size(&env, result);
	die_if_fault_occurred(&env);
//	xmlrpc_parse_value(&env, result, "i", &i);
	printf("number of config options: %d\n", size);
	for (i=0; i<size; i++) {
		struct_conf=xmlrpc_array_get_item(&env, result, i);
		die_if_fault_occurred(&env);
		xmlrpc_parse_value(&env, struct_conf, "{s:s,s:s,s:s,*}",
		    "option", &option,
		    "type", &type,
		    "help", &help);
		printf("%s\n%s %s\n\n", help, option, type);
	}
	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void conf_write (char *url) {
	xmlrpc_env env;
	xmlrpc_value *result;

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.conf.write", "()");
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void conf_get (char *url, char *thread, char *param) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;
	int ret;
	char *str;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.conf.get", "(is)", threadnr, param);
	die_if_fault_occurred(&env);
	xmlrpc_parse_value(&env, result, "(is)", &ret, &str);
	if (!env.fault_occurred && ret==0) {
		printf("%s\n", str);
	} else {
		printf("Failed!\n");
		exit(1);
	}
	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void conf_set (char *url, char *thread, char *param, char *arg) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.conf.set", "(iss)", threadnr, param, arg);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void detection_pause (char *url, char *thread) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.detection.pause", "(i)", threadnr);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void detection_resume (char *url, char *thread) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.detection.resume", "(i)", threadnr);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void track_auto(char *url, char *thread, char *val) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr, value;

	sscanf(thread, "%d", &threadnr);
	sscanf(val, "%d", &value);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.track.auto", "(ii)", threadnr, value);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void track_pan(char *url, char *thread, char *val) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr, value;

	sscanf(thread, "%d", &threadnr);
	sscanf(val, "%d", &value);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.track.pan", "(ii)", threadnr, value);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void track_tilt(char *url, char *thread, char *val) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr, value;

	sscanf(thread, "%d", &threadnr);
	sscanf(val, "%d", &value);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.track.tilt", "(ii)", threadnr, value);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void track_set(char *url, char *thread, char *val, char *val2) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr, pan, tilt;

	sscanf(thread, "%d", &threadnr);
	sscanf(val, "%d", &pan);
	sscanf(val2, "%d", &tilt);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.track.set", "(iii)", threadnr, pan, tilt);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}


void action_makemovie (char *url, char *thread) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.action.makemovie", "(i)", threadnr);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void action_snapshot (char *url, char *thread) {
	xmlrpc_env env;
	xmlrpc_value *result;
	int threadnr;

	sscanf(thread, "%d", &threadnr);

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.action.snapshot", "(i)", threadnr);
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

void action_quit (char *url) {
	xmlrpc_env env;
	xmlrpc_value *result;

	xmlrpc_env_init(&env);
	result = xmlrpc_client_call(&env, url, "motion.action.quit", "()");
	die_if_fault_occurred(&env);

	/* Dispose of our result value. */
	xmlrpc_DECREF(result);

	/* Clean up our error-handling environment. */
	xmlrpc_env_clean(&env);
}

int main (int argc, char** argv)
{
	xmlrpc_env env;

	/* Start up our XML-RPC client library. */
	xmlrpc_client_init(XMLRPC_CLIENT_NO_FLAGS, NAME, VERSION);

	/* Initialize our error-handling environment. */
	xmlrpc_env_init(&env);

	if (argc==1) {
		printf("Usage\n");
		printf("motion-control info\n");
		printf("motion-control conf list\n");
		printf("motion-control conf get [threadnr] [parameter]\n");
		printf("motion-control conf set [threadnr] [parameter] [value]\n");
		printf("motion-control conf write\n");
		printf("motion-control action makemovie [threadnr]\n");
		printf("motion-control action snapshot [threadnr]\n");
		printf("motion-control action quit\n");
		printf("motion-control detection pause [threadnr]\n");
		printf("motion-control detection resume [threadnr]\n");
		printf("motion-control track auto [threadnr] [value]\n");
		printf("motion-control track set [threadnr] [pan] [tilt]\n");
		printf("motion-control track pan [threadnr] [value]\n");
		printf("motion-control track tilt [threadnr] [value]\n");
	}
	else if (argc==2 && !strcmp(argv[1], "info")) {
		info(MOTION_URL);
	}
	else if (argc==3 && !strcmp(argv[1], "conf") && !strcmp(argv[2], "list")) {
		conf_list(MOTION_URL);
	}
	else if (argc==3 && !strcmp(argv[1], "conf") && !strcmp(argv[2], "write")) {
		conf_write(MOTION_URL);
	}
	else if (argc==5 && !strcmp(argv[1], "conf") && !strcmp(argv[2], "get")) {
		conf_get(MOTION_URL, argv[3], argv[4]);
	}
	else if (argc==6 && !strcmp(argv[1], "conf") && !strcmp(argv[2], "set")) {
		conf_set(MOTION_URL, argv[3], argv[4], argv[5]);
	}
	else if (argc==4 && !strcmp(argv[1], "action") && !strcmp(argv[2], "makemovie")) {
		action_makemovie(MOTION_URL, argv[3]);
	}
	else if (argc==4 && !strcmp(argv[1], "action") && !strcmp(argv[2], "snapshot")) {
		action_snapshot(MOTION_URL, argv[3]);
	}
	else if (argc==3 && !strcmp(argv[1], "action") && !strcmp(argv[2], "quit")) {
		action_quit(MOTION_URL);
	}
	else if (argc==4 && !strcmp(argv[1], "detection") && !strcmp(argv[2], "pause")) {
		detection_pause(MOTION_URL, argv[3]);
	}
	else if (argc==4 && !strcmp(argv[1], "detection") && !strcmp(argv[2], "resume")) {
		detection_resume(MOTION_URL, argv[3]);
	}
	else if (argc==5 && !strcmp(argv[1], "track") && !strcmp(argv[2], "auto")) {
		track_auto(MOTION_URL, argv[3], argv[4]);
	}
	else if (argc==5 && !strcmp(argv[1], "track") && !strcmp(argv[2], "pan")) {
		track_pan(MOTION_URL, argv[3], argv[4]);
	}
	else if (argc==5 && !strcmp(argv[1], "track") && !strcmp(argv[2], "tilt")) {
		track_tilt(MOTION_URL, argv[3], argv[4]);
	}
	else if (argc==6 && !strcmp(argv[1], "track") && !strcmp(argv[2], "set")) {
		track_set(MOTION_URL, argv[3], argv[4], argv[5]);
	}
	else {
		printf("Invalid command.\n");
	}

	xmlrpc_client_cleanup();

	return 0;
}
