/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.27 2001/12/30 22:54:25 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "datatypes/match/datatype.h"
#include "datatypes/sublist/datatype.h"
#include "datatypes/split/datatype.h"
#include "misc.h"

#include "plugin_config.h"

int mplugins_processor_web_dlinit(mconfig *ext_conf) {
	config_processor *conf = NULL;
	
	conf = malloc(sizeof(config_processor));
	memset(conf, 0, sizeof(config_processor));
	
	conf->page_type		= mlist_init();
	
	conf->ignore_host	= mlist_init();
	conf->ignore_url	= mlist_init();
	conf->ignore_ua		= mlist_init();
	conf->ignore_hostmask	= mlist_init();
	
	conf->hide_url		= mlist_init();
	conf->hide_referrer	= mlist_init();
	conf->hide_host		= mlist_init();
	conf->hide_brokenlinks	= mlist_init();
	conf->hide_robots	= mlist_init();
	conf->hide_extension	= mlist_init();
	
	conf->group_os		= mlist_init();
	conf->group_ua		= mlist_init();
	conf->group_hosts	= mlist_init();
	conf->group_brokenlinks	= mlist_init();
	conf->group_referrer	= mlist_init();
	conf->group_url		= mlist_init();
	conf->group_searchstrings = mlist_init();
	conf->group_robots	= mlist_init();
	conf->group_extension	= mlist_init();
	
	conf->searchengines	= mlist_init();
	
	conf->splitby		= mlist_init();
	conf->split_def		= mlist_init();
	
	conf->search_file	= NULL;
	
	conf->host_cache = malloc(sizeof(m_cache));
	conf->host_cache->entry = NULL;
	conf->host_cache->last_tstmp = 0;
	
	ext_conf->plugin_conf = conf;
	
	return 0;
}

int mplugins_processor_web_dlclose(mconfig *ext_conf) {
	config_processor *conf = ext_conf->plugin_conf;
	int fd;
	char *fn;
	char buf[256];
	
	if (conf->host_cache_max > 0) {
		int i;
		for (i = 0; i < conf->host_cache_max; i++) {
			if (conf->host_cache->entry[i]) {
				if (conf->host_cache->entry[i]->key) free(conf->host_cache->entry[i]->key);
				if (conf->host_cache->entry[i]->value) free(conf->host_cache->entry[i]->value);
				free(conf->host_cache->entry[i]);
			}
		}
		free(conf->host_cache->entry);
	}
	free(conf->host_cache);
	
	fn = malloc(strlen(ext_conf->outputdir) + strlen("/searchstrings") + 1);
	sprintf(fn, "%s/searchstrings", ext_conf->outputdir);
	if ((fd = open(fn, O_RDONLY))) {
		int i, n, lines = 0;
		while ((n = read(fd, buf, sizeof(buf))) > 0) {
			for (i = 0; i < n; i++) {
				if (buf[i] == '\n') {
					lines++;
				}
			}
		}
		
		if (ext_conf->debug_level > 0) {
			printf("Searchengines: %s/%s contains %d unprocessed searchstrings\n", 
			       ext_conf->outputdir, 
			       "searchstrings", 
			       lines);
			if (lines > 100) {
				printf("               It's a good time to send them to jan@kneschke.de \n");
				printf("               for inclusion into the official distribution.\n");
			}
		}
		
		close(fd);
	}
	free(fn);
	
	mlist_free(conf->page_type);
	
	mlist_free(conf->ignore_host);
	mlist_free(conf->ignore_url);
	mlist_free(conf->ignore_ua);
	mlist_free(conf->ignore_hostmask);
	
	mlist_free(conf->hide_url);
	mlist_free(conf->hide_referrer);
	mlist_free(conf->hide_host);
	mlist_free(conf->hide_brokenlinks);
	mlist_free(conf->hide_extension);

	mlist_free(conf->group_os);
	mlist_free(conf->group_ua);
	mlist_free(conf->group_hosts);
	mlist_free(conf->group_referrer);
	mlist_free(conf->group_brokenlinks);
	mlist_free(conf->group_url);
	mlist_free(conf->group_searchstrings);
	mlist_free(conf->group_extension);
	
	mlist_free(conf->searchengines);
	mlist_free(conf->splitby);
	mlist_free(conf->split_def);
	
	if (conf->search_file) free(conf->search_file);
	
	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;
	
	return 0;
}

int mplugins_processor_web_parse_config(mconfig *ext_conf, const char *cf, const char *section) {
	config_processor *conf = ext_conf->plugin_conf;
	
	const mconfig_values config_values[] = {
	/* rewritings */
		{"groupreferrer", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_referrer)},
		{"groupos",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_os)},
		{"groupua",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_ua)},
		{"grouphosts",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_hosts)},
		{"groupbrokenlinks", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_brokenlinks)},
		{"groupurl",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_url)},
		{"groupsearchstrings",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_searchstrings)},
		{"grouprobots",	M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_robots)},
		{"groupextension", M_CONFIG_TYPE_SUBSTITUTE,	M_CONFIG_VALUE_APPEND, &(conf->group_extension)},
	/* simple matches */
		{"hideurl",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_url)},
		{"hidereferrer", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_referrer)},
		{"hidehost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_host)},
		{"hidebrokenlinks", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_brokenlinks)},
		{"hiderobots",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_robots)},
		{"hideextension", M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->hide_extension)},
		{"ignorehost",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_host)},
		{"ignoreurl",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_url)},
		{"ignoreua",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->ignore_ua)},
		{"ignorehostmask",	M_CONFIG_TYPE_STRING_LIST,	M_CONFIG_VALUE_APPEND, &(conf->ignore_hostmask)},
		{"pagetype",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->page_type)},
		{"splitby",	M_CONFIG_TYPE_STRING_LIST,	M_CONFIG_VALUE_APPEND, &(conf->splitby)},
	/* integers */
		{"debug_searchengines", M_CONFIG_TYPE_INT, M_CONFIG_VALUE_OVERWRITE, &(conf->debug_searchengines)},
		{"debug_visits", M_CONFIG_TYPE_INT,     M_CONFIG_VALUE_OVERWRITE, &(conf->debug_visits)},
		{"debug_resolver", M_CONFIG_TYPE_INT,     M_CONFIG_VALUE_OVERWRITE, &(conf->debug_resolver)},
		{"visit_timeout", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->visit_timeout)},
		{"debug_timing", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->debug_timing)},
		{"maxhostcacheentries", M_CONFIG_TYPE_INT,    M_CONFIG_VALUE_OVERWRITE, &(conf->host_cache_max)},
	/* strings */
		{"searchengines", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->search_file)},
		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};
	
	if (conf == NULL) return -1;
	
	return mconfig_parse_section(ext_conf, cf, section, config_values);
}

struct field_value {
	char	*string;
	int	type;
};

int mplugins_processor_web_set_defaults(mconfig *ext_conf) {
	config_processor *conf = ext_conf->plugin_conf;
	
	if (conf->visit_timeout <= 0) {
		conf->visit_timeout = 1800;
	}
	
	if (conf->host_cache_max < 0) conf->host_cache_max = 0;
	
	if (conf->host_cache_max > 0) {
		int i;
		conf->host_cache->entry = malloc(sizeof(m_cache_entry *) * conf->host_cache_max);
		for (i = 0; i < conf->host_cache_max; i++) {
			conf->host_cache->entry[i] = malloc(sizeof(m_cache_entry));
			conf->host_cache->entry[i]->key = NULL;
			conf->host_cache->entry[i]->value = NULL;
			conf->host_cache->entry[i]->tstmp = -1;
		}
	}
	
	if (conf->search_file) {
		char *group = NULL;
/* reading the searchengines */
		int linenumber = 0;
		FILE *f;
		char buf[255];
		
		
		int valid_match = 0;
		int invalid_match = -1;
		
		if (!(f = mfopen (ext_conf, conf->search_file, "r"))) {
			fprintf(stderr, "can't open %s: %s\n", conf->search_file, strerror(errno));
			return -1;
		}
		
		while (fgets(buf, sizeof(buf)-1, f)) {
			char *c1, *c2;
			
			if (buf[strlen(buf)-1] != '\n') {
				fprintf(stderr, "%s: %s\n", _("No newline at end of line or line longer then 255 charaters"), buf);
				return -1;
			}
			buf[strlen(buf)-1] = '\0';
			
			linenumber++;
			
			if (!strlen(buf) || *buf == '#') continue;
			
			c1 = buf;
			
			if (*buf == '[' && buf[strlen(buf)-1] == ']') {
				char *key = buf + 1;
				char *c1 = key;
			
				while (*c1 && *c1 != ']') c1++;
				*c1 = '\0';
				
				if (group) free(group);
				group = malloc (strlen(key)+2);
				strcpy(group, key);
				
				/* add the '=' for the searchstring match */
				group[strlen(key)] = '=';
				group[strlen(key)+1] = '\0';
			} else if ((c2 = strchr(buf, ','))) {
				char *c3;
				/* SubList */
				mdata *data;
				/* Match */
				mdata *s_eng;
				mlist *p = conf->searchengines;
					
				*c2 = '\0';
				c3 = ++c2;
				
				while (p) {
					data = p->data;
					
					if (!data || !strcmp(data->key, group)) break;
					
					p = p->next;
				}
				
				/* we haven't found a searchparam group */
				if (!p) {
					data = mdata_SubList_create(group, NULL);
					
					p = conf->searchengines;
					
					/* walk to the end of the list */
					while (p->next) p = p->next;
					
					/* a new list item */
					p->next = mlist_init();
					/* set the link */
					p->next->prev = p;
					
					/* step forward */
					p = p->next;
					
					/* insert the group key */
					p->data = data;
				} else if (!p->data) {
					data = mdata_SubList_create(group, NULL);
					
					p->data = data;
				} else {
					data = p->data;
				}
				
			/* do some checks on the second string */
				if (strlen(c3) > 2 && 
					c3[0] == '"' && 
					c3[strlen(c3)-1] == '"') {
			/* ok, a group information */
			
					s_eng = mdata_Match_create(buf,c3);
				} else {
				/* otherwise it must be a number */
					char *e;
					int key = strtol(c3,&e,10);
					char conf_buf[64];
					
					/* FIXME: check if 'e' is a number */
					
					sprintf(conf_buf, "%d", (key >= 0) ? valid_match++ : invalid_match--);
					if (ext_conf->debug_level > 2 && key < 0) {
						fprintf(stderr, "%s.%d: ignoring searchengine on request: %s\n", __FILE__, __LINE__, buf);
					}
					
					s_eng = mdata_Match_create(buf,conf_buf);
				}
				if (s_eng == NULL) {
					fprintf(stderr, "%s %i: %s\n", _("Corrupt searchengine definition in line"), linenumber, buf);
				} else {
					mlist_append(data->data.sublist.sublist, s_eng);
				}
				
				
			} else {
				fprintf(stderr, "%s %i: %s\n", _("Corrupt searchengine definition in line"), linenumber, buf);
			}
		}
		fclose (f);
		
		if (group) free(group);
		if (ext_conf->debug_level > 0)
			fprintf(stderr, "%s\n", _("Reading searchengines - finished"));
	}
	
	if (conf->splitby) {
		mlist *l;
		pcre *match;
		const char *errptr;
		int erroffset = 0;
		
		l = conf->splitby;
		
		if ((match = pcre_compile("^([a-z]+),\"(.*)\",(.+)$",
			0, &errptr, &erroffset, NULL)) == NULL) {
			fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
			return -1;
		}
		
		/* doing some tests with the string */
		while (l) {
			mdata *data = l->data;
#define N 20 + 1
			int ovector[3 * N], n, i;
			
			if (!data) break;
			
			/* the dawn string */
			if ((n = pcre_exec(match, NULL, data->key, strlen(data->key), 0, 0, ovector, 3 * N)) < 0) {
				if (n == PCRE_ERROR_NOMATCH) {
					fprintf(stderr, "%s.%d: (splitby) string doesn't match: %s\n", __FILE__, __LINE__, data->key);
				} else {
					fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
				}
				return -1;
			}
			
#undef N
			if (n >= 3) {
				const char **list;
				/* Split */
				mdata *ins_data = NULL;
				
				const struct field_value field_values[] = {
				/* local server */
					{ "srvhost",	M_SPLIT_FIELD_SRVHOST },
					{ "srvport",	M_SPLIT_FIELD_SRVPORT },
				/* remote client */
					{ "requser",	M_SPLIT_FIELD_REQUSER },
					{ "requrl",	M_SPLIT_FIELD_REQURL },
					{ "reqhost",	M_SPLIT_FIELD_REQHOST },
				/* client's referrer */
					{ "refurl",	M_SPLIT_FIELD_REFURL },
				
				/* default */
					{ "default",	M_SPLIT_FIELD_DEFAULT },
					{ NULL,		M_SPLIT_FIELD_UNSET }
				};
				
				pcre_get_substring_list(data->key, ovector, n, &list);
				
				i = 0;
				while (field_values[i].string) {
					if (!strcmp(field_values[i].string, list[1]))
						break;
					i++;
				}
				
				if (field_values[i].string) {
					ins_data = mdata_Split_create(list[3], field_values[i].type, list[2]);
					
					if (ext_conf->debug_level > 2)
						fprintf(stderr, "%s.%d: using splitter for \"%s\" type %d\n", __FILE__, __LINE__, list[2], field_values[i].type);
					
					if (ins_data) {
						mlist_append(conf->split_def, ins_data);
					} else {
						fprintf(stderr, "%s.%d: the definition for the splitter couldn't be created\n", __FILE__, __LINE__);
					}
				} else {
					fprintf(stderr, "%s.%d: the requested key isn't supported: %s\n", __FILE__, __LINE__, list[1]);
				}
				
				free(list);
			} else {
				fprintf(stderr, "%s.%d: too few fields matched: %d\n", __FILE__, __LINE__, n);
			}
			
			l = l->next;
		}
	}
	
	return 0;
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_processor_web_dlinit;
	func->dlclose = mplugins_processor_web_dlclose;
	func->parse_config = mplugins_processor_web_parse_config;
	func->set_defaults = mplugins_processor_web_set_defaults;
	func->get_next_record = NULL;
	func->insert_record = mplugins_processor_web_insert_record;
	func->gen_report = NULL;
        func->gen_history = NULL;
	
	return 0;
}
