#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "mlocale.h"
#include "mhash.h"

//#define MTIMER_ENABLED	
#ifdef MTIMER_ENABLED	
#include "misc.h"
#endif

#include "datatypes/count/datatype.h"

#include "web.h"
#include "plugin_config.h"
#include "pictures.h"
#include "template.h"
#include "generate.h"

#define HIGHLIGHT	1
#define GROUPING	2
#define VISITS		4
#define INDEX		8
#define BROKEN_LINK	16
#define PERCENT		32
#define RESOLVE_TLD     64
#define VISITS_TRAFFIC  128
#define SORT_BY_KEY     256
#define VIEW_DURATION   512
#define TIME            1024
#define SORT_BY_VCOUNT	2048

#define BOTTOM_THRESHOLD	16

char *get_duration_string(time_t t) {
	static char str[255];
	time_t days    = (t/60/60/24);
	time_t hours   = (t/60/60)%24;
	time_t mins    = (t/60)%60;
	time_t secs    = (t)%60;

	if (days) {
		sprintf(str, "%ld %s %.2ld:%.2ld:%.2ld", days, days==1?_("day"):_("days"), hours, mins, secs);
	} else {
		sprintf(str, "%.2ld:%.2ld:%.2ld", hours, mins, secs);
	}

	return str;
}

char *htmlencode(const char *s) {
	char *p;
	char *q = NULL;
	int q_len = 0;
	if (!s) return NULL;
	
	q_len = strlen(s) * 2 + 1;
	q = malloc(q_len);

	p = q;
	*p = '\0';
	
	while (*s) {
		switch(*s) {
			case '<':
				*p = '&';
				*(++p) = 'l';
				*(++p) = 't';
				*(++p) = ';';
				break;
			case '>':
				*p = '&';
				*(++p) = 'g';
				*(++p) = 't';
				*(++p) = ';';
				break;
			default:
				*p = *s;
				break;
		}
		*(++p) = '\0';
		s++;
		
		if (strlen(q) > (q_len - 4)) {
			q_len += 128;
			q = realloc(q, q_len);
			
			p = q + strlen(q);
		}
	}
	
	return q;
}

int show_mhash_web (mconfig *ext_conf, tmpl_main *tmpl, mhash *h, int count, int opt) {
	int i;
	config_output *conf = ext_conf->plugin_conf;
	long sum = 0;
	double sum_vcount = 0;
	mdata **md;
#ifdef MTIMER_ENABLED	
	static mtimer timer;
	
	MTIMER_START(timer);
#endif
	if (!h) {
		fprintf(stderr, "%s.%d: no hash for me\n", __FILE__, __LINE__);
		return 0;
	}
	
	sum = mhash_sumup(h);
	if (opt & VISITS && opt & PERCENT)
		sum_vcount = mhash_sumup_vcount(h);
	
	if (opt & SORT_BY_KEY) {
		md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);
	} else if (opt & SORT_BY_VCOUNT) {
		md = mhash_sorted_to_marray(h, M_SORTBY_VCOUNT, M_SORTDIR_DESC);
	} else {
		md = mhash_sorted_to_marray(h, M_SORTBY_COUNT, M_SORTDIR_DESC);
	}
	
	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			char *enc_url = NULL;
			int cut_url = 0;
			unsigned int c = 0;
			char buf[255];
			
			enc_url = htmlencode(data->key);

			cut_url = (strlen(enc_url) > 40) /*&& !conf->dont_cut_urls*/;
			
			if (cut_url) {
				enc_url[40] = '\0';
			}
			
			if (opt & INDEX) {
				sprintf(buf,"%d", i+1);
				
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
			}

			c = mdata_get_count(data);

			tmpl_set_current_block(tmpl, "report_cell");
			tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
			if (opt & TIME) {
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", get_duration_string(c));
			} else {
				sprintf(buf,"%d", c);
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
			}
			tmpl_parse_current_block(tmpl);
			
			if (opt & PERCENT && sum) {
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
				sprintf(buf,"%.2f", c * 100.0 / sum);
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
			}
			
			if (opt & VISITS && data->type == M_DATA_TYPE_VISITED ) {
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
				
				/* if the vcount is used as 'traffic' */
				if (opt & VISITS_TRAFFIC) {
					char c = ' ';
					double tr = data->data.visited.vcount;
					if (tr > 1024) {tr /= 1024; c = 'k';}
					if (tr > 1024) {tr /= 1024; c = 'M';}
					if (tr > 1024) {tr /= 1024; c = 'G';}
					sprintf(buf,"%.2f&nbsp;%cB", tr, c);
				} else {
					sprintf(buf,"%d", data->data.visited.vcount);
				}
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
				/* percentage display for second numeric column */
				if (opt & PERCENT && sum_vcount) {
					tmpl_set_current_block(tmpl, "report_cell");
					tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
					sprintf(buf,"%.2f", data->data.visited.vcount * 100.0 / sum_vcount);
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
					tmpl_parse_current_block(tmpl);
				}
			}

			if (opt & VIEW_DURATION && data->type == M_DATA_TYPE_VISITED ) {
				/* use count as duration and vcount as hits per page view */
				
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
				sprintf(buf, "%d", data->data.visited.vcount);
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
				
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", data->data.visited.vcount ?
					     get_duration_string(data->data.visited.count / data->data.visited.vcount) :
					     "--");
				tmpl_parse_current_block(tmpl);
			}

			if ((opt & GROUPING) && mdata_is_grouped(data)) {
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
				tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "grouping");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", enc_url);
				if (cut_url) tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "...");
				tmpl_parse_current_block(tmpl);
				
				tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
			} else {
				if (opt & HIGHLIGHT) {
					char *s;
					tmpl_set_current_block(tmpl, "report_cell");
					tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
					if (conf->assumedprotocol == NULL || strstr(data->key, "://")) {
						s = malloc(strlen(data->key) + 
							   strlen(enc_url) + 
							   (cut_url ? 3 : 0) + 
							   strlen("<a href=\"\"></a>") +
							   1);
						sprintf(s,"<a href=\"%s\">%s</a>%s", 
							data->key, 
							enc_url, cut_url ? "..." : "");
					} else {
						s = malloc(strlen(conf->assumedprotocol) +
							   strlen(conf->hostname) +
							   (*data->key == '/' ? 0 : 1) + 
							   strlen(data->key) + 
							   strlen(enc_url) + 
							   (cut_url ? 3 : 0) + 
							   strlen("<a href=\"://\"></a>") +
							   1);
						sprintf(s,"<a href=\"%s://%s%s%s\">%s</a>%s", 
							conf->assumedprotocol,
							conf->hostname,
							*data->key == '/' ? "" : "/", 
							data->key, 
							enc_url, 
							cut_url ? "..." : "");
					}
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", s);
					tmpl_parse_current_block(tmpl);
					free(s);
				} else {
					/* added option to resolve tlds on the fly */
					if (opt & RESOLVE_TLD) {
						char *c = htmlencode(misoname(data->key));
						tmpl_set_current_block(tmpl, "report_cell");
						tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
						tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", c);
						tmpl_parse_current_block(tmpl);
						free(c);
					} else {
						tmpl_set_current_block(tmpl, "report_cell");
						tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
						tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", enc_url);
						if (cut_url) tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "...");
						tmpl_parse_current_block(tmpl);
					}
				}
			}

			if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
				struct tm *_tm;
				char timebuf[32] = "";
				
				if (data->data.brokenlink.referrer && 
				    (0 != strcmp(data->data.brokenlink.referrer, "-"))) {
					
					free(enc_url);
					enc_url = htmlencode(data->data.brokenlink.referrer);
					cut_url = strlen(enc_url) > 40;
					
					if (cut_url) {
						enc_url[40] = '\0';
					}
						
					tmpl_set_current_block(tmpl, "report_cell");
					tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "<a href=\"");
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", data->data.brokenlink.referrer);
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "\">");
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", enc_url);
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "</a>");
					if (cut_url) tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "...");
					tmpl_parse_current_block(tmpl);
				} else {
					tmpl_set_current_block(tmpl, "report_cell");
					tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "-");
					tmpl_parse_current_block(tmpl);
				}

				_tm = localtime(&(data->data.brokenlink.timestamp));
				if (strftime(timebuf, sizeof(timebuf)-1, "%x", _tm) == 0) {
					fprintf(stderr, "output::modlogan.show_mhash: strftime failed\n");
				}
				tmpl_set_current_block(tmpl, "report_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", timebuf);
				tmpl_parse_current_block(tmpl);
			}

			free(enc_url);
			
			tmpl_set_current_block(tmpl, "report_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "report_cell");
		}
	}
	
	free(md);
#ifdef MTIMER_ENABLED	
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);
	
	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return 0;
}

mlist * get_next_element(mhash *h) {
	mlist *ret = NULL;
	int max = 0, i;
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				if ( mdata_get_count(data) > max) {
					max = mdata_get_count(data);
					ret = l;
				}
			}
			l = l->next;
		}
	}
	/* invert the value */
	if (ret) {
		mdata_set_count(ret->data, -mdata_get_count(ret->data));
	}
	
	return ret;
}

int cleanup_elements(mhash *h) {
	int i;
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				if (mdata_get_count(data) > 0) {
/*					fprintf(stderr, "%s.%d: count (%d) > 0 in cleanup\n", 
						__FILE__, __LINE__,
						mdata_get_count(data));*/
				} else {
					mdata_set_count(data, -mdata_get_count(data));
				}
				
			}
			l = l->next;
		}
	}
	
	return 0;
}

int show_visit_path (mconfig *ext_conf, tmpl_main *tmpl,  mhash *h, int count, int opt) {
	int i = 0;
	mlist *l;
	long sum;
	char buf[255];

	if (!h) return 0;
	
	sum = mhash_sumup(h);
	
	while ((l = get_next_element(h)) && i < count) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			int c;
			
			c = -data->data.sublist.count;
			
			i++;
			
			tmpl_set_current_block(tmpl, "report_cell");
			tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
			sprintf(buf, "%d", i);
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_cell");
			sprintf(buf, "%d", c);
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_cell");
			sprintf(buf, "%.2f", c * 100.0 / sum);
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_cell");
			tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
			tmpl_clear_var(tmpl, "TABLE_ROW_CONTENT");

			while (sl) {
				if (sl->data) {
					mdata *sld = sl->data;
					
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", sld->key);
					tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", "<br />");
				}
				
				sl = sl->next;
			}
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "report_cell");
		}
	}
	
	cleanup_elements(h);
	
	return 0;
}

int show_status_mhash (mconfig *ext_conf, tmpl_main *tmpl, mhash *h, int count) {
	int i;
	mdata **md;
	char buf[255];
	
	if (!h) return 0;
	
	md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];
	
		if (data) {
			tmpl_set_current_block(tmpl, "report_cell");
			tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
			sprintf(buf, "%d", data->data.count.count);
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_cell");
			tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", data->key);
			tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", " - ");
			tmpl_append_var(tmpl, "TABLE_ROW_CONTENT", mhttpcodes(strtol(data->key, NULL, 10)));
			
			tmpl_parse_current_block(tmpl);
			
			tmpl_set_current_block(tmpl, "report_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "report_cell");
		}
	}
	
	free(md);

	return 0;
}

mhash *get_entry_pages(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (data->type != M_DATA_TYPE_SUBLIST) {
					M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
						 "datatype not a sublist: %d - %s\n", data->type, data->key);
					return NULL;
				}
			
				if (sl && sl->data) {
					mdata *sld = sl->data;
					mdata *insd;
					
					insd = mdata_Count_create(sld->key, 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(ret, insd);
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_exit_pages(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (sl) {
					/* go to the last element */
					while (sl->next) sl = sl->next;
					
					if (sl->data) {
						mdata *sld = sl->data;
						mdata *insd;
						
						insd = mdata_Count_create(sld->key, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(ret, insd);
					}
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_visit_path_length(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (sl) {
					/* go to the last element */
					long c = 0;
					char str[255];
					mdata *insd;
					while (sl) {
						c++;
						sl = sl->next;
					}
					
					sprintf(str, "%5ld", c);
					
					insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(ret, insd);
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_visit_duration(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
			
				if (sl) {
				/* go to the last element */
					char str[255];
					mdata *insd;
					mdata *sld = sl->data;
					
					if (sld) {
						time_t t1 = sld->data.brokenlink.timestamp, t2;
						
						while (sl->next) 
							sl = sl->next;
						t2 = sl->data->data.brokenlink.timestamp;
						
						if (t2 - t1) { 
							sprintf(str, "%5ld %s", (t2 - t1) / 60, _("min"));
						} else {
							sprintf(str, " < 1 %s", _("min"));
						}
				
						insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(ret, insd);
					}
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

double get_visit_full_path_length(mhash *h) {
	double d = 0;
	int i;

	if (!h) return 0;
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				d += mlist_count(data->data.sublist.sublist);
			}
			l = l->next;
		}
	}
	
	return d;
}

double get_visit_full_duration(mhash *h) {
	double d = 0;
	int i;
	if (!h) return 0;
	
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
			
				if (sl) {
					/* go to the last element */
					mdata *sld = sl->data;
					
					if (sld) {
						time_t t1 = sld->data.brokenlink.timestamp, t2;
						
						while (sl->next) 
							sl = sl->next;
						t2 = sl->data->data.brokenlink.timestamp;
				
						d += (t2 - t1);
					}
				}
			}
			l = l->next;
		}
	}
	
	return d;
}

mhash *get_path_length(mhash *h) {
	mhash *ret;
	mlist *l;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	while ((l = get_next_element(h))) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			
			if (sl) {
				/* go to the last element */
				long c = 0;
				char str[255];
				mdata *insd;
				while (sl) {
					c++;
					sl = sl->next;
				}
				
				sprintf(str, "%5ld", c);
				
				insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}
	
	cleanup_elements(h);
	
	return ret;
}



#define M_REPORT_REQ_URL      "web_request_url"
#define M_REPORT_REQ_URL_TRAFFIC      "web_request_url_traffic"
#define M_REPORT_REQ_PROT     "web_request_protocol"
#define M_REPORT_REQ_METH     "web_request_method"
#define M_REPORT_STATUS_CODES "web_request_status"
#define M_REPORT_EXTENSION    "web_request_extensions"
#define M_REPORT_EXTENSION_TRAFFIC "web_request_extensions_traffic"

#define M_REPORT_REF_URL      "web_user_referer"
#define M_REPORT_OS           "web_user_os"
#define M_REPORT_OS_VISITS    "web_user_os_visits"
#define M_REPORT_HOSTS        "web_user_host"
#define M_REPORT_HOSTS_VISITS "web_user_host_visits"
#define M_REPORT_USERAGENT    "web_user_useragent"
#define M_REPORT_USERAGENT_VISITS    "web_user_useragent_visits"
#define M_REPORT_BOOKMARKS    "web_user_bookmarks"
#define M_REPORT_COUNTRIES    "web_user_countries"
#define M_REPORT_VIEW_DURATION "web_user_view_duration"

#define M_REPORT_ENTRY_PAGES  "web_visit_entry_pages"
#define M_REPORT_EXIT_PAGES   "web_visit_exit_pages"
#define M_REPORT_VISIT_PATH   "web_visit_path"
#define M_REPORT_VISIT_PATH_LENGTH "web_visit_path_length"
#define M_REPORT_VISIT_DURATION "web_visit_duration"

#define M_REPORT_INDEXED      "web_robot_indexed_pages"
#define M_REPORT_ROBOTS       "web_robot_names"
#define M_REPORT_SEARCH_ENGINE "web_robot_searchengine"
#define M_REPORT_SEARCH_STRINGS "web_robot_seachstrings"

#define M_REPORT_BROKEN_LINKS "web_server_broken_links"
#define M_REPORT_INTERNAL_ERROR "web_server_internal_errors"
#define M_REPORT_VHOSTS       "web_server_vhosts"

#define M_REPORT_SUMMARY      "web_generic_summary"
#define M_REPORT_HOURLY       "web_generic_hourly"
#define M_REPORT_DAILY        "web_generic_daily"

/**
 * data container for the coloums of the tables
 * 
 */

typedef struct {
	const char *name;  /*< visible name of the coloumn */
	const char *class; /*< CSS class, may be NULL */
} fields_def;

/**
 * data container for generic reports 
 * 
 */

typedef struct {
	char *key; /*< type of report */
	const char *title; /*< visible title of the table */
	int options;       /*< options for show_mhash */
	int show_graph;    /*< for the graph ? (conf->show...) */
	char * (*draw_graph)(mconfig * ext_conf, mstate * state, const char *subpath);
	                   /*< graph function (may be NULL) */
	fields_def fields[5];
	                   /*< coloumns */
} reports_def;

/**
 * 
 * @return copy of the report-def to generate_web
 */

const reports_def *get_reports_web () {
	static reports_def * r_static = NULL;
	const reports_def reports[] = {
		/* 0 */
		  /* id             title */
		{ M_REPORT_REQ_URL, _("Requested URL's"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 1 */
		{ M_REPORT_REF_URL, _("Referring URL's"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Referrer"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 2 */
		{ M_REPORT_VIEW_DURATION, _("View Durations"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION | TIME,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 3 */
		{ M_REPORT_OS, _("Used Operating Systems"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL }
				
			}
		},
		/* 4 */
		{ M_REPORT_HOSTS, _("Hosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL }
			}
		},
		/* 5 */
		{ M_REPORT_ENTRY_PAGES, _("Entry Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Entry Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 6 */
		{ M_REPORT_EXIT_PAGES, _("Exit Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Exit Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 7 */
		{ M_REPORT_INDEXED, _("Indexed Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Indexed Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 8 */
		{ M_REPORT_USERAGENT, _("Used Browsers"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL }
			}
		},
		/* 9 */
		{ M_REPORT_REQ_PROT, _("Used Request Protocol"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Protocol"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 10 */
		{ M_REPORT_REQ_METH, _("Used Request Method"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Method"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 11 */
		{ M_REPORT_ROBOTS, _("Robots"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Robot"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 12 */
		{ M_REPORT_BOOKMARKS, _("Bookmarked Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Bookmarked Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 13 */
		{ M_REPORT_BROKEN_LINKS, _("Missing File / Broken Link"),
			GROUPING | INDEX | BROKEN_LINK | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 14 */
		{ M_REPORT_INTERNAL_ERROR, _("Internal Errors"),
			INDEX | BROKEN_LINK,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 15 */
		{ M_REPORT_SEARCH_STRINGS, _("SearchStrings"),
			INDEX | PERCENT | GROUPING,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Search String"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 16 */
		{ M_REPORT_SEARCH_ENGINE, _("SearchEngines"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
				
			{
				{ _("Hits"), "hits" },
				{ _("Search Engine"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 17 */
		{ M_REPORT_EXTENSION, _("Extensions"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				1,
				create_pic_ext,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Extenstions"), NULL },
				{ NULL, NULL }
			}
		},
		/* 18 */
		{ M_REPORT_VISIT_DURATION, _("Time Per Visit"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vd,
			{
				{ _("Visits"), "visits" },
				{ _("Time per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 19 */
		{ M_REPORT_VISIT_PATH_LENGTH, _("Visit Path Length"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vpl,
			
			{
				{ _("Visits"), "visits" },
				{ _("Pages per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 20 */
		{ M_REPORT_COUNTRIES, _("Countries"),
			VISITS | INDEX | PERCENT | RESOLVE_TLD,
				1,
				create_pic_countries,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Country"), NULL },
				{ NULL, NULL }
			}
		},
		/* 21 */
		{ M_REPORT_VHOSTS, _("Vhosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				1,
				create_pic_vhost,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Vhost"), NULL },
				{ NULL, NULL }
			}
		},
		/* 22 */
		  /* id             title */
		{ M_REPORT_REQ_URL_TRAFFIC, _("Requested URL's by Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 23 */
		{ M_REPORT_EXTENSION_TRAFFIC, _("Extensions by Traffic"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				1,
				create_pic_ext_traffic,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Extenstions"), NULL },
				{ NULL, NULL }
			}
		},
		/* 24 */
		{ M_REPORT_OS_VISITS, _("Used Operating Systems by Visits"),
			GROUPING | VISITS | INDEX | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL }
				
			}
		},
		/* 25 */
		{ M_REPORT_HOSTS_VISITS, _("Hosts by Visits"),
			GROUPING | INDEX | VISITS | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL }
			}
		},
		/* 26 */
		{ M_REPORT_USERAGENT_VISITS, _("Used Browsers by Visits"),
			GROUPING | VISITS | INDEX | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL }
			}
		},

		{0, NULL, 0, 0, NULL,
			{
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		}
	};
	
	if (r_static == NULL) {
		r_static = malloc(sizeof(reports));
		memcpy(r_static, reports, sizeof(reports));
	}
	
	return r_static;
}

char * generate_web_visit_path(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_web *staweb = NULL;
	
	if (state == NULL)
		return NULL;
	
	if (state->ext == NULL)
		return NULL;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;
	
	staweb = state->ext;
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "#");
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Visits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "visits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "%");
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Visit Path"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
		
	tmpl_clear_block(tmpl, "header_cell");

	show_visit_path(ext_conf, tmpl, staweb->visits,
			max,
			HIGHLIGHT | GROUPING | INDEX | PERCENT);

	sprintf(buf, "%d", 4);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Visit Path"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - the template */
	tmpl_free(tmpl);
	
	return s;
}

char * generate_web_status_codes(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	tmpl_main *tmpl;
	char *s, *ref, buf[255], *fn;
	mstate_web *staweb = NULL;
	
	if (state == NULL)
		return NULL;
	
	if (state->ext == NULL)
		return NULL;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;
	
	staweb = state->ext;
	
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	ref = create_pic_status(ext_conf, state, subpath);
	
	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Status Code"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
		
	tmpl_clear_block(tmpl, "header_cell");

	show_status_mhash(ext_conf, tmpl, staweb->status_hash, max);

	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Status Code"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - the template */
	tmpl_free(tmpl);
	
	return s;
	
}

char * generate_web_summary(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	int i, last_day = 1;
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_web *staweb = NULL;
	data_WebHistory sumdat, maxdat;
	unsigned int s_200, s_304;
	unsigned int min, sec;
	double d = 0;
	
	char c;
	double tr;
	
	if (state == NULL)
		return NULL;
	
	if (state->ext == NULL)
		return NULL;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;
	
	staweb = state->ext;
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	sumdat.files	= maxdat.files		= 0;
	sumdat.xfersize	= maxdat.xfersize	= 0;
	sumdat.hits	= maxdat.hits		= 0;
	sumdat.hosts	= maxdat.hosts		= 0;
	sumdat.pages	= maxdat.pages		= 0;
	sumdat.visits	= maxdat.visits		= 0;
	
	/* count the values */
	for ( i = 0; i < 31; i++) {
		if (staweb->days[i].hits) last_day = i+1;
		sumdat.files	+= staweb->days[i].files;
		sumdat.xfersize	+= staweb->days[i].xfersize;
		sumdat.hits	+= staweb->days[i].hits;
		sumdat.hosts	+= staweb->days[i].hosts;
		sumdat.pages	+= staweb->days[i].pages;
		sumdat.visits	+= staweb->days[i].visits;
		
		if (maxdat.files < staweb->days[i].files)
			maxdat.files	= staweb->days[i].files;
		if (maxdat.hits < staweb->days[i].hits) 
			maxdat.hits	= staweb->days[i].hits;
		if (maxdat.hosts < staweb->days[i].hosts) 
			maxdat.hosts	= staweb->days[i].hosts;
		if (maxdat.pages < staweb->days[i].pages) 
			maxdat.pages	= staweb->days[i].pages;
		if (maxdat.visits < staweb->days[i].visits) 
			maxdat.visits	= staweb->days[i].visits;
		if (maxdat.xfersize < staweb->days[i].xfersize) 
			maxdat.xfersize	= staweb->days[i].xfersize;
	}
	
	maxdat.hosts = sumdat.hosts = mhash_count(staweb->host_hash);
	

	/* Totals */
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Total Hits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hits");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.hits);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Total Files"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "files");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.files);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Total Pages"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "pages");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.pages);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Total Hosts"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hosts");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.hosts);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Total Visits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "visits");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.visits);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Traffic"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "traffic");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	c = ' ';
	tr = sumdat.xfersize;
	if (tr > 1024) {tr /= 1024; c = 'k';}
	if (tr > 1024) {tr /= 1024; c = 'M';}
	if (tr > 1024) {tr /= 1024; c = 'G';}
	sprintf(buf,"%.2f&nbsp;%cB", tr, c);
	
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	sprintf(buf, "%d", 2);
	tmpl_set_var(tmpl, "TABLE_ROW_COLSPAN", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	/* averages and maximums */
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_clear_var(tmpl, "TABLE_ROW_COLSPAN");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "&nbsp;");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("avg"));
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("max"));
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "header_cell");
	
	/*  */
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hits per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hits");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.hits / last_day);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", maxdat.hits);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Files per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "files");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.files / last_day);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", maxdat.files);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Pages per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "pages");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.pages / last_day);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", maxdat.pages);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hosts per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hosts");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);
	
	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.hosts / last_day);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "---");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Visits per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "visits");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", sumdat.visits / last_day);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%ld", maxdat.visits);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Traffic per Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "traffic");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	c = ' ';
	tr = sumdat.xfersize / last_day;
	if (tr > 1024) {tr /= 1024; c = 'k';}
	if (tr > 1024) {tr /= 1024; c = 'M';}
	if (tr > 1024) {tr /= 1024; c = 'G';}
	sprintf(buf,"%.2f&nbsp;%cB", tr, c);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	c = ' ';
	tr = maxdat.xfersize;
	if (tr > 1024) {tr /= 1024; c = 'k';}
	if (tr > 1024) {tr /= 1024; c = 'M';}
	if (tr > 1024) {tr /= 1024; c = 'G';}
	sprintf(buf,"%.2f&nbsp;%cB", tr, c);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	
	if (sumdat.visits) {
		double allvisitduration = get_visit_full_duration(staweb->visits);
		
		d = (allvisitduration / sumdat.visits);
		min = d / 60;
		sec = (int)floor(d) % 60;
	} else {
		min = 0;
		sec = 0;
	}
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Time per visit"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%d:%02d %s", min, sec, _("min"));
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "---");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	if (sumdat.visits) {
		double allvisitlength = get_visit_full_path_length(staweb->visits);
		d = (double)allvisitlength / sumdat.visits;
	} else {
		d = 0;
	}
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Pages per visit"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "pages");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%.2f", d );
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "---");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	if (sumdat.pages && staweb->views) {
		double allviewduration = mhash_sumup(staweb->views);
		d = (double)allviewduration / sumdat.pages;
	} else {
		d = 0;
	}
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Duration per page"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", get_duration_string(d));
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "---");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	s_200 = mhash_get_value(staweb->status_hash, "200");
	s_304 = mhash_get_value(staweb->status_hash, "304");
	
	d = ((double)s_304/(s_200+s_304)) * 100;
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Cache Hit ratio"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_var(tmpl, "TABLE_ROW_CLASS");
	
	tmpl_set_current_block(tmpl, "report_cell");
	sprintf(buf, "%.2f%%", d);
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
	tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "---");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "report_cell");
	tmpl_clear_block(tmpl, "header_cell");
	
	sprintf(buf, "%d", 3);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Summary"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - the template */
	tmpl_free(tmpl);
	
	return s;

}

char * generate_web_daily(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	int i, last_day = 1;
	tmpl_main *tmpl;
	char *s, *ref, buf[255], *fn;
	mstate_web *staweb = NULL;
	
	if (state == NULL)
		return NULL;
	
	if (state->ext == NULL)
		return NULL;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;
	
	staweb = state->ext;
	
	for ( i = 0; i < 31; i++) 
		if (staweb->days[i].hits) last_day = i+1;
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	ref = create_pic_31_day(ext_conf, state, subpath);
		
	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}
	
	/* header */
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Day"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Files"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "files");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Pages"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "pages");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Visits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "visits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("KBytes"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
		
	tmpl_clear_block(tmpl, "header_cell");
	
	for ( i = 0; i < last_day; i++) {
		char c;
		double tr;
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%d", i+1);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "left");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->days[i].hits);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->days[i].files);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->days[i].pages);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->days[i].visits);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		c = ' ';
		tr = staweb->days[i].xfersize;
		if (tr > 1024) {tr /= 1024; c = 'k';}
		if (tr > 1024) {tr /= 1024; c = 'M';}
		if (tr > 1024) {tr /= 1024; c = 'G';}
		sprintf(buf,"%.2f&nbsp;%cB", tr, c);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_row");
		tmpl_parse_current_block(tmpl);
		
		tmpl_clear_block(tmpl, "report_cell");
	}
	
	sprintf(buf, "%d", 6);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Daily Statistics"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - the template */
	tmpl_free(tmpl);
	
	return s;
}

char * generate_web_hourly(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	int i;
	tmpl_main *tmpl;
	char *s, *ref, buf[255], *fn;
	mstate_web *staweb = NULL;
	
	if (state == NULL)
		return NULL;
	
	if (state->ext == NULL)
		return NULL;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;
	
	staweb = state->ext;
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	ref = create_pic_24_hour(ext_conf, state, subpath);
			
	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}

	/* header */
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hour"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Hits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "hits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Files"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "files");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Pages"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "pages");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Visits"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "visits");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "header_cell");
	tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("KBytes"));
	tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);
	
	tmpl_set_current_block(tmpl, "report_row");
	tmpl_parse_current_block(tmpl);
		
	tmpl_clear_block(tmpl, "header_cell");
	
	for ( i = 0; i < 24; i++) {
		char c;
		double tr;
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%d", i);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->hours[i].hits);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->hours[i].files);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->hours[i].pages);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		sprintf(buf, "%ld", staweb->hours[i].visits);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_cell");
		c = ' ';
		tr = staweb->hours[i].xfersize;
		if (tr > 1024) {tr /= 1024; c = 'k';}
		if (tr > 1024) {tr /= 1024; c = 'M';}
		if (tr > 1024) {tr /= 1024; c = 'G';}
		sprintf(buf,"%.2f&nbsp;%cB", tr, c);
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", buf);
		tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
		tmpl_set_var(tmpl, "TABLE_ROW_ALIGN", "right");
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_row");
		tmpl_parse_current_block(tmpl);
		
		tmpl_clear_block(tmpl, "report_cell");
	}
	
	
	sprintf(buf, "%d", 6);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Hourly Statistics"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - the template */
	tmpl_free(tmpl);
	
	return s;
}


char * generate_web(mconfig * ext_conf, mstate * state, const char *current, int max, const char *subpath) {
	int i;
	mstate_web *staweb = NULL;
	int w = 0, j;
	char *s = NULL;
	char buf[255];
	tmpl_main *tmpl;
	const reports_def *reports;
	char *fn;
	mhash *data;
#ifdef MTIMER_ENABLED		
	static mtimer timer;
	
	MTIMER_START(timer);
#endif	
	if (state == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state = NULL\n");
		return NULL;
	}
	
	if (state->ext == NULL) {
		M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state->ext = NULL, (%d, %d, %d)\n", 
			 state->year,
			 state->month,
			 state->ext_type
			 );
		return NULL;
	}
	
	if (state->ext_type != M_STATE_TYPE_WEB) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state extension != web\n");
		return NULL;
	}
	
	staweb = state->ext;
	
	reports = get_reports_web();
	
	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, current)) 
			break;
	}
	
	/* unknown report */
	if (reports[i].key == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here\n", current);
		return NULL;
	}
	
	switch(i) {
	case 0: data = staweb->req_url_hash; break;
	case 1: data = staweb->ref_url_hash; break;
	case 2: data = staweb->views; break;
	case 3: data = staweb->os_hash; break;
	case 4: data = staweb->host_hash; break;
	case 5: data = get_entry_pages(staweb->visits); break; /* free me */
	case 6: data = get_exit_pages(staweb->visits); break; /* free me */
	case 7: data = staweb->indexed_pages; break;
	case 8: data = staweb->ua_hash; break;
	case 9: data = staweb->req_prot_hash; break;
	case 10: data = staweb->req_meth_hash; break;
	case 11: data = staweb->robots; break;
	case 12: data = staweb->bookmarks; break;
	case 13: data = staweb->status_missing_file; break;
	case 14: data = staweb->status_internal_error; break;
	case 15: data = staweb->searchstring; break;
	case 16: data = staweb->searchsite; break;
	case 17: data = staweb->extension; break; 
	case 18: data = get_visit_duration(staweb->visits); break; /* free me */
	case 19: data = get_visit_path_length(staweb->visits); break; /* free me */
	case 20: data = staweb->country_hash; break;
	case 21: data = staweb->vhost_hash; break;
	case 22: data = staweb->req_url_hash; break;
	case 23: data = staweb->extension; break; 
	case 24: data = staweb->os_hash; break;
	case 25: data = staweb->host_hash; break;
	case 26: data = staweb->ua_hash; break;
	default:
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here - what's up ??\n", current);
		return NULL;
	}
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	/* only create the report if we have data */
	if (mhash_count(data)) {
		if (reports[i].show_graph && reports[i].draw_graph) {
			char *ref = reports[i].draw_graph(ext_conf, state, subpath);
			
			if (ref && strlen(ref)) {
				tmpl_set_var(tmpl, "IMAGE", ref);
			}
		}
		
		/* calculate the number coloumns */
		for (w = 0; reports[i].fields[w].name != NULL; w++);
		if (reports[i].options & INDEX) w++;
		if (reports[i].options & PERCENT) w++;
		
		if (reports[i].options & INDEX) {
			tmpl_set_current_block(tmpl, "header_cell");
			tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "#");
			tmpl_parse_current_block(tmpl);
		}
		
		for (j = 0; reports[i].fields[j].name != NULL; j++) {
			tmpl_set_current_block(tmpl, "header_cell");
			
			if (reports[i].fields[j].class) {
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", reports[i].fields[j].name);
				tmpl_set_var(tmpl, "TABLE_ROW_CLASS", reports[i].fields[j].class);
			} else {
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", reports[i].fields[j].name);
				tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
			}
			tmpl_parse_current_block(tmpl);
			
			if (j == 0 && reports[i].options & PERCENT) {
				tmpl_set_current_block(tmpl, "header_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "%");
				tmpl_parse_current_block(tmpl);
			}
		}
		
		tmpl_set_current_block(tmpl, "report_row");
		tmpl_parse_current_block(tmpl);
		
		tmpl_clear_block(tmpl, "header_cell");
	
#if 0
		fprintf(stderr, "%s.%d: %d (%s - %d)\n", __FILE__, __LINE__, i, reports[i].title, max);
#endif
		if (show_mhash_web(ext_conf, tmpl, data, max, reports[i].options)) {
			fprintf(stderr, "show mhash web failed for '%s'\n", current);
		}
		
		tmpl_clear_var(tmpl, "TABLE_ROW_ALIGN");
		if (max > BOTTOM_THRESHOLD) {
			if (reports[i].options & INDEX) {
				tmpl_set_current_block(tmpl, "header_cell");
				tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "#");
				tmpl_parse_current_block(tmpl);
			}
			
			for (j = 0; reports[i].fields[j].name != NULL; j++) {
				tmpl_set_current_block(tmpl, "header_cell");
				
				if (reports[i].fields[j].class) {
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", reports[i].fields[j].name);
					tmpl_set_var(tmpl, "TABLE_ROW_CLASS", reports[i].fields[j].class);
				} else {
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", reports[i].fields[j].name);
					tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none");
				}
				tmpl_parse_current_block(tmpl);
				
				if (j == 0 && reports[i].options & PERCENT) {
					tmpl_set_current_block(tmpl, "header_cell");
					tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", "%");
					tmpl_parse_current_block(tmpl);
				}
			}
			
			tmpl_set_current_block(tmpl, "report_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "header_cell");
		}
	} else {
		/* generate a dummy page if we have no data to process */
		w = 1;
		
		tmpl_set_current_block(tmpl, "header_cell");
		tmpl_set_var(tmpl, "TABLE_ROW_CONTENT", _("Sorry, no data to display"));
		tmpl_parse_current_block(tmpl);
		
		tmpl_set_current_block(tmpl, "report_row");
		tmpl_parse_current_block(tmpl);
		tmpl_clear_block(tmpl, "header_cell");
	}
	
	
	sprintf(buf, "%d", w);
	tmpl_set_var(tmpl, "TABLE_TITLE", reports[i].title);
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	s = tmpl_replace(tmpl);
	
	/* cleanup */
	
	/* - data */
	switch(i) {
	case 5: 
	case 6: 
	case 18: 
	case 19: mhash_free(data);
	}
	
	/* - the template */
	tmpl_free(tmpl);

#ifdef MTIMER_ENABLED	
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);
	
	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return s;
}

int register_reports_web (tmpl_reports *r) {
	int i, j;
	
	const reports_def *reports = get_reports_web();
	const char *bla[] = { 
		M_REPORT_DAILY, 
			M_REPORT_HOURLY,
			M_REPORT_STATUS_CODES,
			M_REPORT_VISIT_PATH,
			M_REPORT_SUMMARY,
			NULL };
	
	/* set the 'registered-reports-pointer' to the first unused element */
	for (i = 0; i < M_TMPL_MAX_REPORTS && r[i].key != NULL; i++);
	
	/* copy the reports */
	for (j = 0; reports[j].key && i < M_TMPL_MAX_REPORTS; j++, i++) {
		r[i].key = reports[j].key;
		r[i].func = generate_web;
		r[i].title = reports[j].title;
	}
	
	if (i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[0];
		r[i].func = generate_web_daily;
		r[i].title = _("Daily Statistics");
	}
	
	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[1];
		r[i].func = generate_web_hourly;
		r[i].title = _("Hourly Statistics");
	}
	
	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[2];
		r[i].func = generate_web_status_codes;
		r[i].title = _("Status Codes");
	}
	
	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[3];
		r[i].func = generate_web_visit_path;
		r[i].title = _("Visit Path");
	}
	
	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[4];
		r[i].func = generate_web_summary;
		r[i].title = _("Summary");
	}
	
	return 0;
}

void generate_web_cleanup() {
	free((void *)get_reports_web());
}
