/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: misc.c,v 1.16 2001/10/22 21:08:10 ostborn Exp $
*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "misc.h"

/**
 * convert a hex value to a integer
 * 
 * 'F' -> 15
 * 
 * @param c hex caracter
 * @return the corresponding integer
 */

int hex2int(unsigned char c) {
	c = tolower(c);
	
	c = (c >= '0' && c <= '9') ? c-'0' : c-'a'+10;
	
	return (c);
}

/**
 * convert URL-encoded string into a plain c-string
 * 
 * takes care of non printable characters and replaces them by '_'.
 * The orignal string is desctroyed !!
 * 
 * @param url URL-encoded string
 * @return the decoded string
 */

char *urlescape(char *url) {
	unsigned char *l = url;
	unsigned char *o = url;
	
	while (*l) {
		if ( *l == '+') {
			*o = ' ';
		} else if (*l == '%') {
			if (isxdigit(l[1]) && isxdigit(l[2])) {
				unsigned char c;
				c = (hex2int(l[1]) << 4) + hex2int(l[2]);
				
				if (c < 32 || c == 127 ) c = '_';
				
				*o = c;
				
				l += 2;
			} else if (l[1] == '%'){
				*o = *l;
				l++;
			} else {
				*o = *l;
			}
		} else {
			*o = *l;
		}
		o++;
		l++;
	}
	
	*o = '\0';
	
	return url;
}

/**
 * Checks if the specified string is a HTML color tripple
 * 
 * @param col a HTML-color tripple (e.g.: "#FFEE76")
 * @return 0 for failure, 1 for success
 */

int is_htmltripple(const char *col) {
	if (!col) return 0;
	if (strlen(col) != 7) return 0;
	
	if (*col != '#') return 0;
	
	col++;
	
	while(*col) {
		if (!isxdigit(*col)) return 0;
		col++;
	}
	
	return 1;
}


/**
 * Converts a HTML-tripple to a internal RGB-struct
 * 
 * @param html3 a HTML-color tripple (e.g.: "#FFEE76")
 * @param rgb pointer to a internal rgb structure
 * @return 0 for failure, 1 for success
 */
int html3torgb3(const char *html3, rgb_tripple * rgb) {
	if (!is_htmltripple(html3)) return 0;
	
	rgb->r = (hex2int(html3[1]) << 4) + hex2int(html3[2]);
	rgb->g = (hex2int(html3[3]) << 4) + hex2int(html3[4]);
	rgb->b = (hex2int(html3[5]) << 4) + hex2int(html3[6]);
	
	return 1;
}

/**
 * the m// operator from perl
 * 
 * @param pattern the compiled pcre pattern
 * @param str the string to match
 * @return 0 for failure, 1 for success
 */

int strmatch(pcre *pattern, pcre_extra *study, const char *str) {
#define N 20 + 1
	int ovector[3 * N], n;
	
	if ((n = pcre_exec(pattern, study, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n != PCRE_ERROR_NOMATCH) {	
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		} 
		return 0;
	}
	
#undef N	
	return 1;
}

char *substitute(mconfig *ext_conf, pcre *match, pcre_extra *study, char *subst, const char *str) {
	char *name = NULL;	/* name if the state -> directory-name */
	
#define N 20
	int ovector[3 * N], n;

/* the dawn string */
	if ((n = pcre_exec(match, study, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n != PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
			return NULL;
		}
	}

/* the string has matched somehow */
	if (n >= 0) {
		char *s;	/* */
		int sl = 0;	/* calculated string length */
		int _err = 0;	/* error flag */
		const char **list;
		
		pcre_get_substring_list(str, ovector, n, &list);
		
		s = subst;
		
/* check if we can safely create the state-name */
		while (*s && _err == 0) {
			if (*s == '$') {
				if (*(s+1) == '_') s++;
				if (*(s+1) == '^') s++;
				if (isdigit(*(s+1))) {
					int d = *(s+1) - '0';
					
					if (d >= n || d < 0) {
						_err = 1;
						fprintf(stderr, "%s.%d: match-index out of range: %s - %d\n", __FILE__, __LINE__, subst, d);
					} else {
						sl += strlen(list[d]);
						s += 2;
					}
				} else {
					fprintf(stderr, "%s.%d: $ isn't followed by a digit: %s\n", __FILE__, __LINE__, subst);
					_err = 1;
				}
			} else {
				sl++;
				s++;
			}
		}
		
		if (_err == 0) {
			char *dst;
		/* the string is ok. copy the string accordingly to the def */
			s = subst;
			
			name = malloc(sl+1);
			*name = '\0';
						
			dst = name;
			
			while (*s) {
				if (*s == '$') {
					int doupper = 0;
					int dolower = 0;
					int d;
					if (*(s+1) == '_') { dolower = 1; s++; }
					if (*(s+1) == '^') { doupper = 1; s++; }
					d = *(s+1) - '0';
					
					M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_PROCESSING, M_DEBUG_LEVEL_VERBOSE,
						 "catting [%s] (%d): %s\n", subst, d, list[d]);
						
					if ( d <= n ) {
						if (!dolower && !doupper)
							strcpy(dst, list[d]);
						else {
							int strcasecpy=0;
							for(strcasecpy = 0;;strcasecpy++) {
								if (dolower)
									*(dst+strcasecpy) = tolower(*(list[d]+strcasecpy));
								else
									*(dst+strcasecpy) = toupper(*(list[d]+strcasecpy));
								if (*(list[d]+strcasecpy) == '\0') break;
							}
						}
					
						dst += strlen(list[d]);
						s += 2;
					} else {
				/* just a paranoid check */
						fprintf(stderr, "%s.%d: ALARM !!! REPORT ME: (should have already been already detected)\n", __FILE__, __LINE__);
						fprintf(stderr, "%s.%d: requested index '$%d' is put of range (0-%d). check your splitby definition", __FILE__, __LINE__, d, n);
						fprintf(stderr, "%s.%d: using $%d as of the directoryname instead.", __FILE__, __LINE__, d);
						*dst++ = *s++;
						*dst = '\0';
					}
				} else {
					*dst++ = *s++;
					*dst = '\0';
				}
			}
			
		/* just a paranoid check */
			if (strlen(name) > sl)
				fprintf(stderr, "%s.%d: ALARM !!! REPORT ME: strlen(name) > sl [ %d > %d ]. possible SEGV\n", __FILE__, __LINE__, strlen(name), sl);
			
		}
		pcre_free(list);
	}
	
	return name;
}

static unsigned char hexchars[] = "0123456789ABCDEF";

/*** URL-Encode a string - allocates memory for new string, don't forget to free() ! ***/
char *url_encode(const char *s) {
	/* x and y denote the position in source respective destination string */
	register int x, y;
	unsigned char *str;
	
	if (!s) return NULL;

	/* allocate memory for encoded string */
	str = (unsigned char *) malloc(3 * strlen(s) + 1);
	/* loop until end of string */
	for (x = 0, y = 0; s[x] != '\0'; x++, y++) {
		str[y] = (unsigned char) s[x];
		/* spaces are replaced with + */
		if (str[y] == ' ') {
			str[y] = '+';
		/* anything except regular chars is hex-encoded */
		} else if ((str[y] < '-') ||
			   (str[y] < 'A' && str[y] > '9') ||
			   (str[y] > 'Z' && str[y] < 'a' && str[y] != '_') ||
			   (str[y] > 'z')) {
			str[y++] = '%';
			str[y++] = hexchars[(unsigned char) s[x] >> 4];
			str[y] = hexchars[(unsigned char) s[x] & 15];
		}
	}
	str[y] = '\0';
	return ((char *) str);
}

/*** URL-DEcode a string - allocates memory for new string, don't forget to free() ! ***/
char* url_decode(const char *s) {
	unsigned char *str;
	/* work pointers */
	const unsigned char *data;
	unsigned char *dest;
	
	if (!s) return NULL;
	
	/* allocate memory for decoded string */
	str = (unsigned char *) malloc(strlen(s) + 1);

	data = s;
	dest = str;

	/* until the end of the string */
	while (*data != '\0') {
		if (*data == '+')
			*dest = ' ';
		else if (*data == '%' && isxdigit((int) *(data + 1)) && isxdigit((int) *(data + 2))) {
			*dest = (hex2int(data[1]) << 4) + hex2int(data[2]);
			data += 2;
		} else
			*dest = *data;
		data++;
		dest++;
	}
	/* add a string terminator */
	*dest = '\0';
	return str;
}

